package ins.api;

import ins.inr.*;
import ins.namespace.*;

import java.net.*;
import java.io.*;
import java.util.*;
import java.awt.*;
					
/**
 * Is a generic remote application that can access an INR.  Custom
 * applications should extend this class.  
 *
 * Generally, an application may want to:
 * <ul>
 *  <li> extend Application, and perhaps calling its constructor 
 *    with super() to set the dsr if it is different than dsr.dnsdomain
 *  <li> announce its name to the world -- startAnnouncer() can start
 *    a periodic announcer, while other methods can add to/
 *    remove from/suspend/restart this announcer. Early binding
 *    attributes can also be announced this way.
 *  <li> send packets with late-binding -- use the various sendMessage methods
 *  <li> receive late-binding packets -- override receivePacket
 *  <li> connect to an early-binding host -- use getHostByiName
 *  <li> become an early-binding host -- addTCPEarlyBinding or startAnnouncer
 *  <li> discover other services based on a filter -- use discoverNames
 *  <li> find location - use getLocation
 *  <li> get incremental service updates as they happen -- needs implementation
 * </ul>
 *
 * In general, the Application will contact the DSR and cache INR information
 * for each vspace it needs to use. It is possible to have it peer to a 
 * single, specific INR by using the appropriate constructor or calling
 * the proper methods, however.
 *
 * @author Anit Chakraborty, Jeremy Lilley */

public class Application
{	
    public int port; //the port that we are actually listening on
    protected DatagramSocket socket; // socket to access the UDP port
    public InetAddress localhost;
    public INRManager inrmanager;
    public NameSpecifier hostNS;
    protected NameAnnouncer name_announcer = null;
    protected Object sendLock = new Object();
    protected ReceiveManager rm = null;
    protected DatagramPacket dpack;
    protected String DSRSearchString=null;
    
    InetAddressPort dsrAddress = null;

    String defaultVspace = "default";

    static Attribute vspace_attr = new Attribute("vspace");
    static Attribute control_msg = new Attribute("control-msg");
    static Value discoverVal = new Value("discovery");
    static Value earlybindVal = new Value("early-binding");

    static AVelement control_discover = 
	new AVelement(control_msg, discoverVal);
    static AVelement control_earlybind = 
	new AVelement(control_msg, earlybindVal);
    
    DatagramSocket nonINSsocket; // socket for sending/receiving non-INS
                                 // packets, e.g. DSR, location
    
    String location=null;  // last location heard

    InetAddressPort singlePeeringAddress;
    boolean singlePeering;
	
    protected int time_out = 2500;

    // CONSTRUCTORS
    // there are 3 constructors:
    //   - one that uses dns to find DSR = dsr.domainname
    //   - one that takes a string DSR name
    //   - one that bypasses the DSR and peers to a specific INR

    // it would be trivial to add constructors that changed the 
    // port number that it's set up on from a random to a pre-specified
    // one. however, there are no applications which would benefit from
    // that currently and it just adds complexity.
    // if there is a compelling reason to add it, that could very well
    // be easily implemented 
    // (just replace the -1 in the constructor's call to initApplication)


    /** Creates a new Application The application listens on a random
     *  port and uses standard DNS
     *  bootstrapping techniques to locate the DSR.
     * @exception Exception Something went wrong.  */
    public Application() throws Exception 
    {
	initApplication(-1, null, null);
    }

    /**
     * Creates a new Application with the DSR pre-specified.
     * @param dsr_name The name of the machine that runs the DSR 
     *      we will be contacting.
     * @exception Exception Something went wrong.
     */
    public Application (String dsr_name) throws Exception 
    {
	initApplication(-1, dsr_name, null);
    }

    /**
     * Creates a new Application that peers to a single, specific INR
     * and does no DSR communication.
     * It's possible also to enable single peering by calling
     * enableSinglePeering() later.
     * @param inrname name of INR
     * @param inrport port of INR
     * @exception Exception Something went wrong.
     */    
    public Application (String inrname, int inrport)
	throws Exception
    {
	InetAddress ia = InetAddress.getByName(inrname);
	initApplication (-1, null, new InetAddressPort(ia, inrport));
    }

    ///////////////////////////////
    
    /**
     * Initializes the application. 
     * Sets up initial IP Address information and network connections.  
     * @param window If this is true, then the library will display a 
     *   window which shows all messages printed by printStatus.
     * @param p The UDP port the node will listen on, or -1 if any port.
     * @param dsr_name The name of the machine that runs the DSR we will 
     *    be contacting, or null to use DNS lookup to find the DSR.
     * @exception Exception Something when wrong.
     */
    public void initApplication (int p, String dsr_name,
				 InetAddressPort singlePeeringAddress) 
	throws Exception
    {
	inrmanager = new INRManager();

	// Get our IP address	
	try {
	    localhost = InetAddress.getLocalHost();
	} catch (UnknownHostException e) {
	    printStatus("Couldn't get local host name");
	    throw new Exception("Couldn't get local host name " +
				e.getMessage());
	}
	
	
	// Open a socket on the port p, or any port if p < 0
	try {
	    if (p < 0) {
		socket = new DatagramSocket();
		p = socket.getLocalPort();
	    } else {
		socket = new DatagramSocket(p);
	    }
	} catch (SocketException se) {
	    throw new Exception("Couldn't open socket.");
	}
	
	printStatus("Listening on port " + p + ".");
	this.port = p;
	nonINSsocket = new DatagramSocket();

	// now that we have the socket ready to send and receive on, 
	// let's set our source that we send to dsr's
	setHostNS();
	
	// transfer single peering info
	this.singlePeeringAddress = singlePeeringAddress;
	this.singlePeering = (singlePeeringAddress != null);

	DSRSearchString=dsr_name;

	// if we are not single peering, ping the DSR
	if (!singlePeering) {
	    findDSR(dsr_name);  // go find the dsr and add it to the inrmanager
	}
	
	// we can start the ReceiveManager
	//rm = new ReceiveManager(this);	
	startReceiveManager();
    }

    /**
     * This method is added in order for different applications
     * to be able to use different ReceiveManagers
     */
    public void startReceiveManager() throws Exception  {
	rm = new ReceiveManager(this);	
    }
    
    /**
     * Enables single-peering, e.g. the client peers to one INR only.
     * In this case, the next INR that would normally be contacted
     * becomes the peer
     */
    public void enableSinglePeering()
    { 
	singlePeering = true; 
    }

    /**
     * Enables single-peering to a specific INR. Returns whether
     * that INR could set (but does not guarantee successful 
     * communication with the INR).
     */
    public boolean enableSinglePeering(String inr_address, int port)
    { 
	try {
	    InetAddress ia = InetAddress.getByName(inr_address);
	    singlePeeringAddress = new InetAddressPort(ia, port);  
	    singlePeering = true; 
	    return true;
	} catch (Exception e) { singlePeering = false; return false; }
    }

    /**
     * Disables single peering, and uses/caches information on the
     * INRs from the DSRs
     */
    public void disableSinglePeering()
    { 
	singlePeering = false; 
	singlePeeringAddress = null; 
    }

    /**
     * Sends a packet to the vspace specified in dNS
     * If the vspace in dNS is dsr, then it will ignore sNS and send 
     * the standard sNS that is supposed to be sent to dsrs according to spec.
     * @param sNS NameSpecifier This is the source NameSpecifier to be sent
     * @param dNS NameSpecifier This is the destination name specifier
     * @param data byte[] This is the data in the payload of the
     *   packet to be sent. If data is null, sendMessage will send a 0
     *   length payload
     * @return boolean Returns true upon successful completion, 
     *   and false on failure.  */
    public boolean sendMessage (NameSpecifier sNS, NameSpecifier dNS,
				byte[] data) 
    {
	return sendMessage(new Packet(sNS, dNS, data));
    }


    /**
     * Sends a packet to the vspace specified in dNS
     * If the vspace in dNS is dsr, then it will ignore sNS and send 
     * the standard sNS that is supposed to be sent to dsrs according to spec.
     * @param sNS NameSpecifier This is the source NameSpecifier to be sent
     * @param dNS NameSpecifier This is the destination name specifier
     * @param multicast Send to all?
     * @param data byte[] This is the data in the payload of the packet 
     *   to be sent.If data is null, sendMessage will send a 0 length payload
     * @return boolean Returns true upon successful completion,      *   and false on failure.  */
    public boolean sendMessage (NameSpecifier sNS, NameSpecifier dNS, 
				boolean multicast,
				byte[] data) 
    {
	return sendMessage(new Packet(sNS, dNS, multicast, data));
    }

    /**
     * Sends a packet to the vspace specified in dNS
     * If the vspace in dNS is dsr, then it will ignore sNS and send 
     * the standard sNS that is supposed to be sent to dsrs according to spec.
     * @param sNS NameSpecifier This is the source NameSpecifier to be sent
     * @param dNS NameSpecifier This is the destination name specifier
     * @param data byte[] This is the data in the payload of the
     *   packet to be sent. If data is null, sendMessage will send a 0
     *   length payload
     * @return boolean Returns true upon successful completion, 
     *   and false on failure.  */
    public boolean sendMessage (NameSpecifier sNS, NameSpecifier dNS,
				byte[] data, boolean announceSourceNS) 
    {
	if (announceSourceNS)
	    return sendMessage
		(new Packet(sNS, dNS, name_announcer.announcer, data));
	else
	    return sendMessage(new Packet(sNS, dNS, data));
    }


    // use the one below for route inference

    /**
     * Sends a packet to the vspace specified in dNS
     * If the vspace in dNS is dsr, then it will ignore sNS and send 
     * the standard sNS that is supposed to be sent to dsrs according to spec.
     * @param sNS NameSpecifier This is the source NameSpecifier to be sent
     * @param dNS NameSpecifier This is the destination name specifier
     * @param multicast Send to all?
     * @param data byte[] This is the data in the payload of the packet 
     *   to be sent.If data is null, sendMessage will send a 0 length payload
     * @param announceSourceNS boolean This indicates whether INR should
     *   announce the existence of sNS to the network
     * @return boolean Returns true upon successful completion, 
     *   and false on failure.  */
    public boolean sendMessage (NameSpecifier sNS, NameSpecifier dNS, 
				boolean multicast,
				byte[] data, boolean announceSourceNS) 
    {
	if (announceSourceNS)
	    return sendMessage(new Packet(sNS, dNS, name_announcer.announcer, 
		multicast, data));
	else
	    return sendMessage(new Packet(sNS, dNS, multicast, data));
    }

    
    /**
     * This method allows the application to directly send a packet 
     * that has already been formed and addressed.
     * @param p The packet to be sent.
     * @return boolean Returns true upon successful completion, and
     *   false on failure.  
     */
    
    public boolean sendMessage(Packet p) {
	NameSpecifier dNS = p.dNS;
	NameSpecifier sNS = p.sNS;
	byte[] data  = p.data;

	//System.out.println("sending: "+p.toString());
	
	String vspace = null;
	// First thing we need to do is find the vspace that 
	// we are to sent this packet to
	vspace = dNS.getVspace();
	if (vspace == null)
	    vspace=defaultVspace;
       
	
	InetAddressPort addr;

	if (singlePeering) {
	    if (singlePeeringAddress == null) {
		singlePeeringAddress = 
		    this.inrmanager.getResolverForVspace(vspace, this);
		if (singlePeeringAddress == null) return false;
	    }
	    addr = singlePeeringAddress;
	} else { 
	    addr = this.inrmanager.getResolverForVspace(vspace, this);
	    if (addr==null) return false;
	}

	// create the bytes to send
	byte[] bytestosend = p.toBytes();
	
	//now create the datagram packet to send
	DatagramPacket dpacket = 
	    new DatagramPacket(bytestosend, bytestosend.length, 
			       addr.addr, addr.port);
	
	//now send the packet
	try {
	    sendDatagramPacket(dpacket);
	    // printStatus 
	    //   ("Sent Packet to "+addr.addr+":"+addr.port);
	} catch (IOException e) {
	    if (e.getMessage().equals("Connection refused")) {
		printStatus("Connection refused from " + addr.addr+
			    " on port "+addr.port);
	    } 
	    else {
		printStatus("Error sending message: " + e.getMessage());
	    }
	    return false;
	}
	return true;
    }
    

    /**
     * sends a packet to the dsr at dsrAddress and waits until 
     * timeout for a response. 
     * @param inPacket packet to send DSR, addressed to the DSR
     * @param outPacket packet to receive DSR's message in
     * @param timeout Maximum time to wait in ms
     * @return boolean Whether a packet could be sent and a response received
     */
    protected boolean contactDSR(DatagramPacket inPacket, 
			 DatagramPacket outPacket, int timeout) 
    {
	try {
	    synchronized (nonINSsocket) {
		nonINSsocket.setSoTimeout(timeout);
		
		nonINSsocket.send(inPacket);
		nonINSsocket.receive(outPacket);
	    }
	} catch (Exception e) { 
	    printStatus("contacting DSR: "+e.toString());return false; 
	}

	return true;
    }


    /**
     * This method will ask the DSR for the address of a particular vspace,
     * and it is only called by the INRManager.
     * @param vspace The vspace that we are trying to lookup
     * @return InetAddressPort If successful, will return the address
     *     info, else will return null.  */
    protected InetAddressPort requestVspace(String vspace) 
	throws Exception 
    {
	if (singlePeering) 
	    if (singlePeeringAddress!=null) return singlePeeringAddress;

	try { 
	    if (this.dsrAddress==null)
		findDSR(DSRSearchString); 
	}
	catch (Exception e) { printStatus("couldn't find dsr"); return null; }
	
	int sequenceNo = getNextSequenceNo();
	byte[] bytes = ins.dsr.DSRRequest.
	    makeGetVspaceResolvers(vspace, sequenceNo).toBytes();

	DatagramPacket sPacket = new DatagramPacket (bytes, bytes.length,
						    dsrAddress.addr, 
						    dsrAddress.port);

	DatagramPacket rPacket = new DatagramPacket (new byte[10000], 10000);


	if (!contactDSR(sPacket, rPacket, 1000)) return null;

	ins.dsr.DSRResponse response = ins.dsr.DSRResponse.readWireform
	    (rPacket.getData(), 0, rPacket.getLength());

	if (response==null) {
	    printStatus("Malformed DSR Response");
	    return null;
	}
	    
	if (response.getSequenceNo()!=sequenceNo) {
	    printStatus("Got DSR Response with bad sequence number");
	    return null;
	}
	
	Vector v = response.getVector();
	ins.dsr.Node n = null;
	InetAddressPort vspaceaddr = null;
	boolean pingSuccessful = false;


	while (!pingSuccessful && v.size()>0)
	{
	    int elem = (int)(Math.random()*v.size()); // choose random INR
	    n = (ins.dsr.Node)v.elementAt(elem);
	    vspaceaddr = new InetAddressPort(n.getInetAddress(), 
					     n.getUDPport(), n.getTTL());
	    pingSuccessful = pingINR(vspaceaddr, vspace, 1000);

	    if (!pingSuccessful) 
		v.removeElementAt(elem);
	    
	}

	return vspaceaddr;
    }
    
    /**
     * The first thing that the application needs to do is find the 
     * InetAddress for the DSR. this function does a look up on the dsr. 
     * Note: Currently we contact the DSR at
     * port 5678 as a default!
     * @param dsr_name The name of the host that runs the dsr.
     * @exception UnknownHostException This is thrown when the 
     *      string passed to us is not a valid hostname.
     */
    protected void findDSR(String dsr_name) throws Exception
    {
	if (dsr_name == null) {
	    String hostName = localhost.getHostName(); // throws exception
	    int dotIndex = hostName.indexOf('.');
	    dsr_name = "dsr"+hostName.substring(dotIndex);
	}

	if (dsr_name.equals("localhost")){
	    dsr_name = localhost.getHostName();	
	}

	//this is where we hard code the address
        this.dsrAddress = new InetAddressPort 
	    (InetAddress.getByName(dsr_name), 5678);

	
	int sequenceNo = getNextSequenceNo();

	byte [] sendB = ins.dsr.DSRRequest.makePing(sequenceNo).toBytes();
	DatagramPacket iP =  new DatagramPacket
	    (sendB, sendB.length, dsrAddress.addr, dsrAddress.port);
	DatagramPacket oP = new DatagramPacket(new byte[512], 512);

	if (!contactDSR(iP, oP, 1000)) 
	    throw new Exception
		("Could not contact DSR at "+dsrAddress.toString());

	if (ins.dsr.DSRResponse.readWireform
	    (oP.getData(), 0, oP.getLength()).
	    getSequenceNo() != sequenceNo) 
	    throw new Exception("Got back wrong sequence number");

    }

    

    
    /**
     * This provides an abstraction to the socket so that only one
     * thread may send at a time. (Threadsafes the socket)
     */
    protected void sendDatagramPacket(DatagramPacket packet) 
	throws IOException 
    {
	synchronized(this.sendLock) {
	    socket.send(packet) ;	
	}
    }
    
    
    /**
     * This will set the name specifier that we use to identify 
     * our host/port.
     */
    protected void setHostNS() {
	try {
	    this.hostNS = 
		new NameSpecifier ("[host="+
				   InetAddress.getLocalHost().getHostAddress()+
				   "][UDPport="+this.port+"]");
	} catch (Exception e) { ; }
    }
    

    /**
     * Creates a Thread object that can announce to the dsr its route.
     * This must be called before sending out the first packet!  If an
     * announcer already exists, it will add the names to the existing
     * announcer.
     * @param source The name of the source or application that the 
     * application would like to advertise to the world.
     */
    public void startAnnouncer(NameSpecifier source[]) {

	if (name_announcer == null) {
	    HostRecord hostRecord = new HostRecord();
	    hostRecord.UDPport = (short)port;

	    int [] metrics = new int[source.length];
	    for (int i=0; i<metrics.length; i++)
		metrics[i] = NameAnnouncer.DEFAULT_METRIC;

	    name_announcer = 
		new NameAnnouncer(this, source, metrics, 
				  NameAnnouncer.DEFAULT_PERIOD, 
				  hostRecord);
	} else {
	    for (int i=0; i<source.length; i++) 
		name_announcer.addAnnouncement
		    (source[i], NameAnnouncer.DEFAULT_METRIC);
	}
    }


    /**
     * EARLY-BINDING VERSION of startAnnouncer.
     * Creates a Thread object that can announce to the dsr its route.
     * This must be called before sending out the first packet!  If an
     * announcer already exists, it will add the names to the existing
     * announcer.
     * @param source The name of the source or application that the 
     * @param transporttype Early binding transport type
     * @param ebport  Early binding port number
     * application would like to advertise to the world.
     */
    public void startAnnouncer(NameSpecifier source[], 
			       String transporttype, short ebport) {

	if (name_announcer == null) {
	    HostRecord hostRecord = new HostRecord();
	    hostRecord.UDPport = (short)port;
	    hostRecord.addTransport("TCP",ebport);

	    int [] metrics = new int[source.length];
	    for (int i=0; i<metrics.length; i++)
		metrics[i] = NameAnnouncer.DEFAULT_METRIC;

	    name_announcer = 
		new NameAnnouncer(this, source, metrics, 
				  NameAnnouncer.DEFAULT_PERIOD, 
				  hostRecord);
	} else {
	    for (int i=0; i<source.length; i++) 
		name_announcer.addAnnouncement
		    (source[i], NameAnnouncer.DEFAULT_METRIC);
	}
    }


    /**
     * Creates a Thread object that can announce to the dsr its route.
     * This must be called before sending out the first packet!  If an
     * announcer already exists, it will add the names to the existing
     * announcer.
     * @param source The name of the source or application that the 
     * application would like to advertise to the world.
     * @param metric The application controlled metric.
     * @param period Length of time (in ms) between announcements to the INR.  
     */
    public void startAnnouncer(NameSpecifier source[], int metric[], int period) {

	if (name_announcer == null) {
	    HostRecord hostRecord = new HostRecord();
	    hostRecord.UDPport = (short)port;

	    name_announcer = 
		new NameAnnouncer(this, source, metric, period, hostRecord);
	} else {
	    for (int i=0; i<source.length; i++) 
		name_announcer.addAnnouncement(source[i], metric[i]);
	}
    }


    /**
     * Creates a Thread object that can announce to the dsr its route.
     * This version of the method allows a HostRecord besides 
     *     the default to be used, such as for early binding
     * @param source The name of the source or application that the
     *   application would like to advertise to the world.
     * @param metric The application controlled metric
     * @param period Length of time (in ms) between announcements to the INR.  
     * @param hostRecord Alternate hostRecord, as useful for doing
     *          early-binding.
     */
    public void startAnnouncer
	(NameSpecifier source[], int metric[], int period,
	 HostRecord hostRecord) 
    {

	if (name_announcer == null) {

	    name_announcer = 
		new NameAnnouncer(this, source, metric, period, hostRecord);
	} else {
	    for (int i=0; i<source.length; i++) 
		name_announcer.addAnnouncement(source[i], metric[i]);
	}
    }

    /**
     * Changes the HostRecord being announced to include early binding
     * on the given TCP port. The name announcer needs to be started,
     * and the changes are sent immediately.
     * @param tcpport Port on which the early binding service resides
     */
    public void addTCPEarlyBinding(short tcpport) 
    {
	if (name_announcer == null) return;

	this.name_announcer.addTCPEarlyBinding(tcpport);
	this.name_announcer.cycleThread();
	
    }
    

    /**
     * This merely suspends the announcer.
     */    
    public void suspendAnnouncer() {
		name_announcer.halt();	
    }
    

    /**
     * This starts the announcer again.
     */
    public void resumeAnnouncer() {
	name_announcer.restart();
    }


    public void sendExpiringAnnouncement(NameSpecifier ns, int ttl)
    {
	if (name_announcer != null) 
	    name_announcer.sendExpiringAnnouncement(ns, ttl);
	else 
	    printStatus("Implementation ERROR: sendExpiringAnnouncement "+
			"requires startAnnouncer to be called");
    }

    /**
     * If the application would like to advertise a new announcement, this will
     * add the source and metric in the advertisment.
     * @param ns The NameSpecifier we would like to advertise.
     * @metric metric The application-controlled metric
     */
    public void addAnnouncement(NameSpecifier ns, int metric)
    {
	if (name_announcer!=null)
	    name_announcer.addAnnouncement(ns, metric);
    }	
    
    /**
     * If the application would like to remove a specific advertisement, 
     *  this will remove the namespecifier from the advertisement list.
     * @param ns The NameSpecifier to remove from the advertisement list.
     */
    public void removeAnnouncement(NameSpecifier ns) {
	if (ns != null && name_announcer!=null) {
	    name_announcer.removeAnnoucement(ns);
	}	
    }

    /**
     * Changes the advertised application metric on a single exact
     *  namespecifier in the announcement list.
     * @param ns The NameSpecifier to modify an entry for
     * @param metric The application metric to add
     */
    public void changeAppMetric(NameSpecifier ns, int metric) {
	if (ns != null && name_announcer!=null) {
	    name_announcer.changeAppMetric(ns, metric);
	}	
    }

    /**
     * Changes the advertised application metric on all the 
     *  namespecifiers in the announcement list.
     * @param metric The application metric to add
     */
    public void changeAppMetric(int metric) {
	if (name_announcer!=null) {
	    name_announcer.changeAppMetric(metric);
	}	
    }

    /**
     * Changes the advertised application metrics, "efficient version."
     * The array must be the exact length of the number of namespecifiers
     * being announced, and the metrics will be associated with namespecifiers
     * in the order they were added.
     */
    public void changeAppMetrics(int[] metrics)
    {
	name_announcer.changeAppMetrics(metrics);
    }

    /**
     * Adds a method that is to be called every time before the
     * names are announced.
     * Particularly useful for changing application metrics before
     * every announcement.
     * Calling with null disables this, which is the default.
     */
    public void setPreNameAnnouncer(PreNameAnnouncer preNameAnnouncer)
    {
	if (name_announcer == null) {
	    System.out.println("APP BUG: setPreNameAnnouncer needs to be called _after_ startAnnouncer");
	} else {
	    name_announcer.setPreNameAnnouncer(preNameAnnouncer);
	}
    }


    
    /**
     * Changes the time between annoucements.
     * @param period The time between announcements.
     */
    public void setAnnouncementPeriod(int period) {
	name_announcer.setAnnouncementPeriod(period);
    }
    

    protected void printStatus(String s)
    {
	System.out.println(s);
    }

    
    /**
     * Application should override this function to receive a Packet.
     * The Receiving thread will call this function in the application
     * when it receives a packet that is for the application.
     * Keep code in this function relatively short, since no further
     *  packets can be received by this application until this terminates.
     * @param p The packet that was received by the ReceiveManager on behalf
     * of the Application.
     */
    public void receivePacket(Packet p) {
	System.out.println("received packet");
    }


    // use this one by default:

    /**
     * Discover names in the system.
     * This function looks up all the NameSpecifiers known that match ns (ns
     * is used as a filter). This function is intended to return immediatly.
     * @param ns       Filter to match records by
     * @return Set of NameSpecifiers, may be empty, but not null.
     */
    public NameSpecifier[] discoverNames(NameSpecifier ns) 
    {
	return discoverNames(ns, true);
    }


    // differences here are more applicable to aggregate vspace-level
    // browsers:

    /**
     * Discover names in the system.
     * This function looks up all the NameSpecifiers known that match ns (ns
     * is used as a filter). This function is intended to return immediatly.
     * @param ns       Filter to match records by
     * @param discoverFull If hierarchical vspaces are being used, should
     *                     we see everything underneath, or just child folders?
     *           (true returns everything underneath that matches [default],
     *            false returns names of child vspaces)
     * @return Set of NameSpecifiers, may be empty, but not null.
     */
    public NameSpecifier[] 
	discoverNames(NameSpecifier ns, boolean discoverFull) 
    {
	System.out.println("*** discoverNames on "+ns);
	
	Vector v=new Vector();
	
	// Modification allowing subclasses not to use vspaces
	ns = addVSpace(ns);
	//String vspace=ns.getVspace();
	//if (vspace==null) {
	//  vspace=defaultVspace; 
	//  ns = new NameSpecifier(ns);
	//  ns.addAVelement(new AVelement(vspace_attr, new Value(vspace)));
	//}
	
	byte [] nsBytes = ns.toString().getBytes();

	byte [] data  = new byte[11+nsBytes.length];
	int sequenceNo = getNextSequenceNo();
	ins.inr.Conversion.insertInt(data, 0, sequenceNo);

	System.arraycopy(this.localhost.getAddress(), 0, data, 4, 4);
	ins.inr.Conversion.insertIntLow16(data, 8, this.port);
	data[10] = (byte)(discoverFull? 1: 0);
	System.arraycopy(nsBytes, 0, data, 11, nsBytes.length);

	NameSpecifier dNS =new NameSpecifier();
	// Modification allowing subclasses not to use vspaces
	//dNS.addAVelement(ns.getAVelement(vspace_attr));
	AVelement avelement = ns.getAVelement(vspace_attr);
	if ( avelement != null)
	    dNS.addAVelement(ns.getAVelement(vspace_attr));

	dNS.addAVelement(control_discover);
	
	Packet p = 
	    //rm.waitForReply(new Packet(hostNS, dNS, data), sequenceNo, 2500);
	    rm.waitForReply(new Packet(hostNS, dNS, data), sequenceNo, time_out);
	if (p==null) {
	    printStatus("Couldn't send request or got null return packet");
	    return new NameSpecifier[0];
	}

	// the stuff below reads the return packets and constructs the 
	//   proper results - it's basically a series of
	data = p.data;
	int ct = 4;

	while (ct < data.length) {
	    int len = ins.inr.Conversion.extract16toIntLow16(data, ct);

	    if (ct + 2 + len > data.length) break; // don't overflow the array
	    if (len == 0) break;

	    String str = new String(data, ct+2, len);
	    ct += (2+len);
	    
	    NameSpecifier n;
	    try { n = new NameSpecifier(str); v.addElement(n); }
	    catch (CantParseString e) { }
	}

	NameSpecifier[] res = new NameSpecifier[v.size()];
	v.copyInto(res);
	return res;
    }


    /**
     * Subclasses should override this method to simply return ns,
     * if vspaces shouldn't be automatically added to discovery
     * requests
     */
    protected NameSpecifier addVSpace(NameSpecifier ns) {

	String vspace=ns.getVspace();
	if (vspace==null) {
	    vspace=defaultVspace; 
	    ns = new NameSpecifier(ns);
	    ns.addAVelement(new AVelement(vspace_attr, new Value(vspace)));
	}
	return ns;

    }


    /** 
     * Performs early-binding. That is, it will return a set of
     * HostRecords matching ns, possibly being empty. 
     * @param ns Filter to match records by
     * @param all All matching records, or just the best-matching
     *            (if all is false, at most one record will be returned)
     * @return Set of hostrecords, possibly empty, but not null.
     */
    public EarlyBindRecord[] getHostByiName(NameSpecifier ns, boolean all) {

	Vector v=new Vector();
	
	// Modification allowing subclasses not to use vspaces
	ns = addVSpace(ns);
	//String vspace=ns.getVspace();
	//if (vspace==null) {
	//  vspace=defaultVspace; 
	//  ns = new NameSpecifier(ns);
	//  ns.addAVelement(new AVelement(vspace_attr, new Value(vspace)));
	//}

	byte [] nsBytes = ns.toString().getBytes();

	byte [] data  = new byte[11+nsBytes.length];

	int sequenceNo = getNextSequenceNo();
	ins.inr.Conversion.insertInt(data, 0, sequenceNo);
	System.arraycopy(this.localhost.getAddress(), 0, data, 4, 4);
	ins.inr.Conversion.insertIntLow16(data, 8, this.port);
	data[10] = (byte)(all?1:0);
	System.arraycopy(nsBytes, 0, data, 11, nsBytes.length);

	NameSpecifier dNS =new NameSpecifier();
	// Modification allowing subclasses not to use vspaces
	//dNS.addAVelement(ns.getAVelement(vspace_attr));
	AVelement avelement = ns.getAVelement(vspace_attr);
	if ( avelement != null)
	    dNS.addAVelement(ns.getAVelement(vspace_attr));

	dNS.addAVelement(control_earlybind);

	Packet p = 
	    //rm.waitForReply(new Packet(hostNS, dNS, data), sequenceNo, 2500);
	    rm.waitForReply(new Packet(hostNS, dNS, data), sequenceNo, time_out);
	if (p==null) {
	    printStatus("Couldn't send request or got null return packet");
	    return new EarlyBindRecord[0];
	}

	System.out.println("response is "+p.data.length+" bytes");
	data = p.data;
	int ct = 4;

	int bestMetric = Integer.MIN_VALUE;
	EarlyBindRecord bestRec = null;
	
	while (ct < data.length) {
	    int metric = ins.inr.Conversion.extract32toInt(data, ct);
	    int len = ins.inr.Conversion.extract16toIntLow16(data, ct+4);

	    if (ct + 6 + len > data.length) break; // don't overflow the array
	    if (len == 0) break;
	    
	    try { 
		EarlyBindRecord ebr=
		    new EarlyBindRecord
		    (new HostRecord(data, ct+6, len), metric);
		if (all) 
		    { v.addElement(ebr); }
		else { 
		    if (metric>bestMetric) 
			{ bestMetric=metric; bestRec=ebr; }
		}
	    } catch (Exception e) { ; }

	    ct += (6+len);	    
	}

	if (all) {
	    EarlyBindRecord[] res = new EarlyBindRecord[v.size()];
	    v.copyInto(res);
	    return res;
	} else {
	    if (bestRec == null) { return new EarlyBindRecord[0]; }
	    else { return new EarlyBindRecord[]{bestRec}; }
	}
    }


    public String getDefaultVspace()
    {
	return defaultVspace;
    }

    

    /**
     * If the Application wants to find its location, then it should call
     * this function once.  The ClientLibrary will try to go and find
     * the location if a locationmanager is available (running on this machine)
     * If the location manager isn't running on this machine, then we will 
     * return a null after a  timeout.  
     * @return NameSpecifier This string contains the location, null if we 
     *       never receive anything.
     *  This looks like [location=floor5[room=504]][vspace=wind]
     */
    public NameSpecifier getLocationNameSpecifier() 
    {
	// packet for receiving data
	DatagramPacket dpacket=new DatagramPacket(new byte[512],512);

	try {
	    synchronized (nonINSsocket) {
		nonINSsocket.setSoTimeout(1000);
		
		nonINSsocket.send(new DatagramPacket(new byte[2], 2, 
						     this.localhost, 5644));
		nonINSsocket.receive(dpacket);
	    }
	} catch (Exception e) { 
	    location = null;
	    return null; 
	}

	location = new String(dpacket.getData(), 0, dpacket.getLength());

	try {
	    NameSpecifier locationNS = new NameSpecifier(location);
	    return locationNS;
	}
	catch (Exception e)
	{
	    return null;
	}
    }


    /**
     * ---- use getLocationNameSpecifier ordinarily now...
     * If the Application wants to find its location, then it should call
     * this function once.  The INRClientLibrary will try to go and find
     * the location if a locationmanager is available (running on this machine)
     * If the location manager isn't running on this machine, then we will 
     * return a null after a  timeout.  
     * @param timeout The time the application is willing to wait for 
     *       the location (in milliseconds)
     * @return String This string contains the location, null if we 
     *       never receive anything.
     */
    public String getLocation(int timeout) 
    {
	// packet for receiving data
	DatagramPacket dpacket=new DatagramPacket(new byte[512],512);

	try {
	    synchronized (nonINSsocket) {
		nonINSsocket.setSoTimeout(timeout);
		
		nonINSsocket.send(new DatagramPacket(new byte[0], 0, 
						     this.localhost, 5644));
		nonINSsocket.receive(dpacket);
	    }
	} catch (Exception e) { 
	    location = null;
	    return null; 
	}

	location = new String(dpacket.getData(), 0, dpacket.getLength());

	return location;
    }


    
    public boolean pingINR(InetAddressPort inr, String vspace, int timeout) 
    {
	System.out.println("pinging INR "+inr);

	// packet for receiving data
	DatagramPacket dpacket=new DatagramPacket(new byte[512],512);

	NameSpecifier destNS = new NameSpecifier("[control-msg=ping][vspace="
						 +vspace+"]");
	Packet p = new Packet(hostNS, destNS, false, new byte[0]);
	byte [] data = p.toBytes();

	try {
	    synchronized (nonINSsocket) {
		nonINSsocket.setSoTimeout(timeout);		
		nonINSsocket.send(new DatagramPacket(data, data.length, 
						     inr.addr, inr.port));
		nonINSsocket.receive(dpacket);
	    }
	} catch (Exception e) { 
	    return false;
	}

	return true;
    }

    /**
     * Stops the running threads if any, releases the sockets, etc.
     * Meant if you want to kill activity by this application.
     * (don't explicitly link this to finalize unless you are sure
     *  this is appropriate for the given application)
     */
    public void cleanUp()
    {
	// first, kill receiveManager thread if applicable
	if (rm.rmRunning)
	{
	    rm.rmRunning = false;
	    try {
		nonINSsocket.send(new DatagramPacket
				  (new byte[0], 0, localhost, port));
	    } catch (Exception e) { ; }
	}

	if (name_announcer.nmRunning)
	{
	    name_announcer.nmRunning = false;
	    name_announcer.cycleThread();
	}

	if (socket != null) 
	{
	    socket.close();
	    socket = null;
	}
	if (nonINSsocket != null)
	{
	    nonINSsocket.close();
	    nonINSsocket = null;
	}
    }


    
    /* routine for getting unique sequential sequenceNumbers */
    protected static int getNextSequenceNo() 
    { 
	int result;
	synchronized (sequenceNoLock) {
	    sequenceNo++; 
	    result = sequenceNo;
	}
	return result;
    }

    protected static int sequenceNo=(int)System.currentTimeMillis();

    protected static Object sequenceNoLock=new Object();

	
}
