package ins.api;

import java.net.*;
import java.util.*;

/**
 * This class manages all of the ip address for the various vspaces an 
 * application may want to listen to.  
 * We can add, remove, and change ip addresses for any vspace.
 * 
 * @author Anit Chakraborty
 */
public class INRManager
{
	
    private Hashtable inrs; //this will hash vspaces versus ip address
    
    // CONSTRUCTORS	
    
    /**
     * Initializes a new INRManager
     */
    protected INRManager() 
    {
	inrs = new Hashtable();
    }
    
    //METHODS
    
    /**
     * Adds a new vspace to the INRmanager
     * @param vspace The vspace to add.
     * @param addr The InetAddress the vspace maps to
     * @return boolean If the vspace was already in the Manager 
     *   returns true, else returns false.
     */
    
    protected boolean addVspace (String vspace, InetAddressPort addr) 
    {
	Object temp = null;
	
	temp = inrs.put(vspace, addr);
	if (temp == null)
	    return false;
	else
	    return true;			
    }
    
    /**
     * Removes a vspace from the INRManager
     * @param vspace The vspace to remove.
     * @return InetAddress If the vspace was already in the Manager 
     *     returns the InetAddress, else returns null.
     */
    protected InetAddressPort removeVspace (String vspace) 
    {
	InetAddress addr;
	
	return (InetAddressPort) inrs.remove(vspace);
    }
    
    /**
     * Looks up a vspace from the INRManager
     * @param vspace The vspace to lookup.
     * @return InetAddress If the vspace was already in the Manager 
     * returns the InetAddress, else returns null.
     * If the entry is expired, it returns null.
     */
    private InetAddressPort lookupVspace (String vspace) 
    {
	InetAddressPort i = (InetAddressPort) inrs.get(vspace);
	if (i==null) return null;

	if (i.isExpired()) return null;
	else return i;
    }
    
    /** Looks up the InetAddressPort for a Resolver for a given vspace
     * In general, this information is cached with a time to live
     * in the application's inrmanager, but it may need to be 
     * explicitly requested (and may very well still return null).
     * @param vspace Vspace to look up
     * @return InetAddressPort reference to the INR to send to
     */
    InetAddressPort getResolverForVspace(String vspace, Application app) 
    {
	InetAddressPort addrp = lookupVspace(vspace);
	if (addrp == null){
	    //before we return false, let's ask the DSR for the vspace..
	    try{
		addrp = app.requestVspace(vspace);
	    } catch (Exception e) {
		// we had a socket exception, probably couldn't set the timeout
		// so we threw an exception instead rather than block forever
		// waiting for the dsr to return to us
		return null;
	    }
	    if (addrp == null) {
		//we still dont have a vspace, return false
		return null;
	    }
	    else {
		//we received a vspace, we can add it to the inrmanager
		addVspace(vspace, addrp);
	    }
	}
	return addrp;
    }
	
}
