package ins.api;

import java.util.*;
import java.io.*;

/**
 * This class implements an object which keeps track of a configuation.
 *
 * Some applications currently require some manual configuration --
 *  this automates the process of reading a configuration file with
 *  a given base name through an appropriate search path and reading
 *  a set of attributes and values. 
 *
 * A INSConfig file looks like:
 *     attribute1 value1
 *     attribute2 value2
 *     building   ne43
 *     floor      5
 *       ...
 * 
 * Since it just represents manual configuration for an application,
 * there are no rules on what the data means, except as the application
 * defines and interprets it.
 */

public class INSConfig
{

    // VARIABLES
    
    Vector pairs;  // Vector of type Pair
    
    // CONSTRUCTORS

    /**
     * Creates a new INSConfig object based on its name.
     * @param name The name of the configuration file.
     * @exception Exception Something went wrong.
     */
    public INSConfig(String name)
	throws Exception
    {
	File file = findFile(name);
	initFromFile(file);
    }

    /**
     * Creates a new INSConfig object from a File.
     * @param file The File to create it from.
     * @exception Exception Something went wrong.
     */
    public INSConfig(File file)
	throws Exception
    {
	initFromFile(file);
    }

    // METHODS

    /**
     * Finds a configuration File based on its name.
     * First looks in the current directory, then in the user's home directory,
     * and then in /etc/ (or equivalent). 
     * Adds ".conf" to the name.
     * @param name The name.
     * @return The File.
     * @exception Exception Something went wrong.
     */
    private File findFile(String name)
	throws Exception
    {
	// This defines the order of things to be checked.
	String[] paths = new String[3];
	paths[0] = System.getProperty("user.dir");  // current directory
	paths[1] = System.getProperty("user.home"); // home directory
	paths[2] = File.separator + "etc";          // /etc or equivalent

	for (int i = 0; i < paths.length; i++) {
	    String path = paths[i];

	    if (path != null) {
		File file;
				
		String hname = path + File.separator +
		    name + ".conf";
		
		file = new File(hname);
		
		if (tryFile(file)) {
		    return(file);
		}
	
	    }
	}
	File file = new File(name + ".conf");
	
	if (tryFile(file)) {
	    return(file);
	}
	
	throw new Exception("Couldn't find a configuration file for " + name +
			    " anywhere.");
    }
    
    /**
     * Tests to see if the file exists.
     * @param file The file to be tested.
     * @return true if it is, false if it doesn't.
     * @exception Exception It exists, but can't be read or isn't normal.
     */
    private boolean tryFile(File file)
	throws Exception
    {
	if (file.exists()) {
	    if (! file.isFile()) {
		throw new Exception("Error while reading " + 
				    file.getCanonicalPath() + ":\n" +
				    "File is not normal " +
				    "(may be directory).");
	    }
	    
	    if (! file.canRead()) {
		throw new Exception("Error while reading " +
				    file.getCanonicalPath() + ":\n" +
				    "Can't read the file.");
	    }
	    
	    return true;
	    
	} else {
	    
	    return false;
	}
	
    }
    
    /**
     * Initializes a new INSConfig object from a File.
     * @param file The File to create it from.
     * @exception Exception Something went wrong.
     */
    private void initFromFile(File file)
	throws Exception
    {
	pairs = new Vector();
	
	BufferedReader br = new BufferedReader(new FileReader(file));
	
	while (true) {
	    
	    // Get the next line.
	    String line = br.readLine();
	    
	    // Check to see if we're done.
	    if (line == null)
		break;

	    if (line.length() == 0) 
		continue;
	    
	    // Check if the line is a comment.
	    if (line.charAt(0) == '#')
		continue;
	    
	    // Get the index of the first space.
	    int space = line.indexOf(' ');
	    
	    Pair p = new Pair();
	    if (space == -1) {
				// No space, just a keyword.
		p.keyword = line;
		p.value = new String();
	    } else {
		p.keyword = line.substring(0, space);
		p.value = line.substring(space + 1);
	    }
	    
	    // Add it to the Vector of Pairs
	    pairs.addElement(p);
	}
    }
    
    /**
     * Gets the first value associated with a particular keyword.
     * @param keyword The keyword to get the value for.
     * @returns The value, or null if not found.
     */
    public String getValue(String keyword)
    {
	for (Enumeration e = pairs.elements(); e.hasMoreElements(); ) {
	    
	    Pair p = (Pair)e.nextElement();
	    
	    if (keyword.equalsIgnoreCase(p.keyword)) {
		return(p.value);
	    }
	}
	
	return(null);
    }

    /**
     * Gets the values associated with a particular keyword.
     * @param keyword The keyword to get the values for.
     * @returns An array of values.
     */
    public String[] getValues(String keyword)
    {
	Vector valvec = new Vector(); // Vector of String

	for (Enumeration e = pairs.elements(); e.hasMoreElements(); ) {
	    
	    Pair p = (Pair)e.nextElement();
	    
	    if (keyword.equalsIgnoreCase(p.keyword)) {
		valvec.addElement(p.value);
	    }
	}
	
	String[] values = new String[valvec.size()];
	valvec.copyInto(values);

	return(values);
    }
    

    /**
     * Gets an Enumeration of the values associated with a particular keyword.
     * @param keyword The keyword to get the values for.
     * @returns An Enumeration of values.
     */
    public Enumeration values(String keyword)
    {
	Vector valvec = new Vector(); // Vector of String

	for (Enumeration e = pairs.elements(); e.hasMoreElements(); ) {
	    
	    Pair p = (Pair)e.nextElement();
	    
	    if (keyword.equalsIgnoreCase(p.keyword)) {
		valvec.addElement(p.value);
	    }
	}
	
	return(valvec.elements());
    }

    // INNER CLASSES
    private class Pair {
	public String keyword;
	public String value;
    }
    
}
