package ins.api;

import ins.inr.*;
import ins.namespace.*;
import java.util.*;
import java.io.IOException;
import java.net.*;

/**
 * This simply runs a thread that will announce to the INR 
 * 
 * @author Anit Chakraborty, Jeremy Lilley
 */

public class NameAnnouncer extends Thread
{
    protected boolean run = false;
    protected boolean running = false;
    protected Object lock = new Object();
    Application app;
    long announcer;
    HostRecord hr;
    int sleep_time;
    NameAnnouncement nannouncements[];
    NameSpecifier ns[];
    int arrlength;
    PreNameAnnouncer preNameAnnouncer=null;

    boolean nmRunning = true;

    public static final int DEFAULT_METRIC = 1;
    public static final int DEFAULT_PERIOD = 25000;
    
    static Attribute vspaceAttr = new Attribute("vspace");
    
    /**
     * Added a constructor that initializes everything
     * except the announcements themselves
     * It leaves the name announcer in the stoped state
     * @author: Magdalena Balazinska
     */ 
    public NameAnnouncer(Application app, int announcement_period) {

	NameSpecifier routingNS, sourceNS;
	byte[] name_update_bytes;
	this.app = app;
	this.sleep_time = announcement_period;
	if (this.sleep_time < 10000)
	    this.sleep_time = 15000;
	
	// create a unique id
	byte[] addr = app.localhost.getAddress();
	Date now = new Date();
	announcer = addr[0];
	announcer |= (addr[1] <<  8) & 0xFF00l;
	announcer |= (addr[1] << 16) & 0xFF0000l;
	announcer |= (addr[1] << 24) & 0xFF000000l;
	announcer |= (now.getTime() << 32) & 0x0000FFFF00000000l;
	announcer |= (app.port << 56) & 0xFFFF000000000000l;
	running = false;
    }
    
    public NameAnnouncer (Application app, NameSpecifier ns[],
			  int metric[], int announcement_period,
			  HostRecord hr) {
	
	String vspace;
	NameSpecifier routingNS, sourceNS;
	byte[] name_update_bytes;
	
	this.app = app;
	this.sleep_time = announcement_period;
	if (this.sleep_time < 10000)
	    this.sleep_time = 15000;

	this.hr = hr;
	
	// create a unique id
	byte[] addr = app.localhost.getAddress();
	Date now = new Date();
	announcer = addr[0];
	announcer |= (addr[1] <<  8) & 0xFF00l;
	announcer |= (addr[1] << 16) & 0xFF0000l;
	announcer |= (addr[1] << 24) & 0xFF000000l;
	announcer |= (now.getTime() << 32) & 0x0000FFFF00000000l;
	announcer |= (app.port << 56) & 0xFFFF000000000000l;
	
	// create the host record
	// now this hostrecord is universal for all of our routing updates.
	
	// make the nannouncements array
	nannouncements = new NameAnnouncement[ns.length+5];
	// keep track of which ns refers to which nannouncements
	this.ns = new NameSpecifier[ns.length + 5];
	
	this.arrlength = ns.length;
	
	for (int i = 0; i < ns.length; i ++) {
	    this.ns[i] = ns[i];
	    nannouncements[i] = createNameAnnouncement(ns[i], metric[i],
						       NameUpdate.ADD_NAME,
						       sleep_time*2);
	}

	running = true;

	start();
    }

    /**
     * Sets a routine to be called before name announcements get
     * sent out. Null cancels this.
     */
    void setPreNameAnnouncer(PreNameAnnouncer preNameAnnouncer)
    {
	this.preNameAnnouncer = preNameAnnouncer;
    }


    void addTCPEarlyBinding(short tcpport) 
    {
	this.hr.addTransport("TCP", tcpport);
	changeAnnouncedHostRecord(this.hr);
    }


    void changeAnnouncedHostRecord(HostRecord hr) 
    {
	this.hr = hr;

	synchronized(lock) {
	    for (int i = 0; i < ns.length; i ++) {
		if (ns[i] != null) 
		    nannouncements[i] = 
			createNameAnnouncement
			(ns[i], nannouncements[i].getAppMetric(),
			 NameUpdate.ADD_NAME, sleep_time*2);
	    }
	}
    }
    
    
    /**
     * Another internal helper function, when given a ns and metric
     * it will construct a packet that we can send out as a routing
     * update
     * @param ns The NameSpecifier we want to advertise in this routing update.
     * @param metric Metric for this new packet
     * @return NameAnnouncement A pre-wireframe version of this advertisement
     *      that can transformed into a real DatagramPacket at transmission 
     *      time (so that we can bind to vspace name than resolver address)
     */
    protected NameAnnouncement 
	createNameAnnouncement(NameSpecifier ns, int metric, byte updatetype,
			       int ttl) 
    {
	NameUpdate name_update;
	byte[] name_update_bytes;
	NameSpecifier sourceNS, routingNS;
	
	//find the vspace that we are advertising for
	String vspace = ns.getVspace();
	if (vspace == null)
	    vspace = this.app.getDefaultVspace();
	
	// Create the name specifier indicating routing update
	routingNS = new NameSpecifier(new String("[control-msg=announcement"));
	routingNS.setVspace(vspace);
	
	// Create the source ns for sending routing-update packet.
	// In source ns we need to put our host and port for receiving purposes
	// (Buffer doesn't have info from which neighbor the packet comes from
	// so RoutingProtocol can deduce from which neighbor the packet from
	// by the host and port info in this ns)
	StringBuffer str = new StringBuffer();
	str.append("[host = ");
	str.append(app.localhost.getHostAddress());
	// Note: getHostName() will use ip addr if DNS name doesn't exist
	// but better use getHostAddress() since Java for Windows doesn't
	// correctly get the name if the ip suddenly changes
	// (it may caches the name somewhere)
	str.append("][UDPport = ");
	str.append(app.port);
	str.append("]");
	
	sourceNS = new NameSpecifier(str.toString());
	sourceNS.setVspace(vspace);
	
	name_update = new NameUpdate(ns, metric, 0, 0, hr, announcer,
				     updatetype, ttl);	
	name_update_bytes = name_update.toBytes();

	Packet packet = new Packet(sourceNS, routingNS, 
				   Packet.toAll,
				   name_update_bytes);
	
	// lookup the vspace that we need to send this to.
	byte[] bytestosend = packet.toBytes();

	return new NameAnnouncement(bytestosend, vspace);
    }

    
    InetAddressPort getResolverForVspace(String vspace) {
	return this.app.inrmanager.getResolverForVspace(vspace, this.app);
    }

    /**
     * This will take a new namespecifier and metric and add it to our
     * advertisement store.  After creating the packet to send, we will 
     */
    protected void addAnnouncement(NameSpecifier ns, int metric) {	
	NameAnnouncement na;
	
	na = createNameAnnouncement(ns, metric, NameUpdate.ADD_NAME, 
				    sleep_time*2);
	
	synchronized(lock){
	    addAnnouncement (ns, na, metric);
	}
    }
    
    /**
     * This is a internal function that will  modify our
     * advertisements store.  It will do this and expand our
     * store if we need to and copy all the data into it again. 
     */
    
    protected void addAnnouncement(NameSpecifier ns, NameAnnouncement na,
				 int metric) 
    {
	
	if (this.arrlength < this.ns.length) {
	    this.ns[this.arrlength] = ns;
	    this.nannouncements[this.arrlength] = na;
	    this.arrlength ++;
	    
	    return;
	}
	
	//if we reached here, this means that we didnt' find an empty spot
	//create a new array.
	NameSpecifier newns[] = new NameSpecifier[this.ns.length + 5];
	NameAnnouncement newnas[] = new NameAnnouncement[this.ns.length+ 5];
	
	for (int i = 0; i < newns.length; i++) {
	    if (i < this.ns.length){
		newns[i] = this.ns[i];
		newnas[i] = this.nannouncements[i];
	    }
	    else {
		newns[i] = null;
		newnas[i] = null;
	    }
	    
	}
	this.ns = newns;
	this.nannouncements = newnas;
	
	this.ns[this.arrlength] = ns;
	this.nannouncements[this.arrlength] = na;
	this.arrlength ++;
    }
    
    protected boolean changeAppMetric(NameSpecifier ns, int metric) 
    {
	synchronized (lock) {
	    for (int i = 0 ; i < this.arrlength; i++) {
		if (this.ns[i].equals(ns)){
		    this.nannouncements[i].setAppMetric(metric);
		    return true;
		}
	    }
	}
	return false;
    }

    protected boolean changeAppMetrics(int [] metrics) 
    {
	synchronized (lock) {
	    for (int i =0; i<metrics.length; i++)
		this.nannouncements[i].setAppMetric(metrics[i]);
	}
	return true;
    }

    protected boolean changeAppMetric(int nsnumber, int metric) 
    {
	synchronized (lock) {
	    this.nannouncements[nsnumber].setAppMetric(metric);
	    return true;
	}
    }


    protected void changeAppMetric(int metric) 
    {
	synchronized (lock) {
	    for (int i = 0 ; i < this.arrlength; i++) {
		this.nannouncements[i].setAppMetric(metric);
	    }
	}
    }


    protected void removeAnnoucement(NameSpecifier ns) {
	int i;
	synchronized (lock){
	    for (i = 0 ; i < this.arrlength; i++) {
		if (this.ns[i].equals(ns)){
		    break;
		}
	    }
	    
	    //now copy everything back by one.
	    this.arrlength --;
	    for (;i < this.arrlength; i++) {
		this.ns[i] = this.ns[i+1];
		this.nannouncements[i] = this.nannouncements[i+1];
	    }
	}
	sendNegativeAnnouncement(ns);
    }

    public void sendExpiringAnnouncement(NameSpecifier ns, int ttl)
    {
	NameAnnouncement na = 
	    createNameAnnouncement(ns, 5, NameUpdate.ADD_NAME, ttl);

	if (na == null) return;

	DatagramPacket dp = na.produceDatagramPacket(this);
	if (dp == null) return;

	try {
	    app.sendDatagramPacket(dp);
	} catch (IOException e) { 
	    app.printStatus("...ah! could not make it!"); 
	} 
    }

    /** 
     * Sends a REMOVE_NAME announcement for a namespecifier
     */
    public void sendNegativeAnnouncement(NameSpecifier ns) {

	NameAnnouncement na = 
	    createNameAnnouncement(ns, 5, NameUpdate.REMOVE_NAME, 
				   sleep_time*2);
	if (na == null) return;

	DatagramPacket dp = na.produceDatagramPacket(this);
	if (dp == null) return;

	app.printStatus("Sending negative announcement to inr");
	try {
	    app.sendDatagramPacket(dp);
	} catch (IOException e) { 
	    app.printStatus("...ah! could not make it!"); 
	} 
    }
    
    boolean insleep=false;

    public void run () 
    {
	while (nmRunning) 
	{
	    int subtime = 0;

	    if (running) 
	    {
		long time = System.currentTimeMillis();

		if (this.preNameAnnouncer != null) {
		    this.preNameAnnouncer.preNameAnnounce();
		}

		// if we are running send the route update..
		// else wait indefintiely		
		synchronized (lock) {
		    
		    for (int i=0 ; i < this.arrlength; i++) {
			try {	
			    if (this.ns[i] != null) {
				DatagramPacket dp = nannouncements[i].
				    produceDatagramPacket(this);
				if (dp!=null)
				    app.sendDatagramPacket(dp);
				// app.printStatus
				//     ("Sent routing update to inr");
			    }
			} catch (Exception e) {
			    app.printStatus
				("Error sending routing update to inr: "+
				 e.getMessage());
			}		    
		    }
		}

		subtime = (int)(System.currentTimeMillis()-time);
	    }

	    try {
		insleep=true;
		sleep(this.sleep_time-subtime);
		insleep=false;
	    } catch (InterruptedException e) { 
		insleep=false;
	    }
	}
	System.out.println("exiting nm thread");
    }
    
    public void setAnnouncementPeriod (int period) {
	this.sleep_time = period;	
    }
	
    
    public void halt() {
	running = false;
    }
    
    public void restart() {
	running = true;
    }

    void cycleThread() {
	if (insleep) this.interrupt();
    }
    
}



/** Encapsulates a vspace name and the byte[] representation 
 * of a Packet to advertise a namespecifier to that vspace.
 * We choose this rather than a DatagramPacket, because the 
 * resolver node to advertise to may very well change. */
class NameAnnouncement { 

    byte [] data;
    String vspace;
    
    NameAnnouncement(byte[] data, String vspace) 
    {
	this.data= data; 
	this.vspace = vspace; 
    }

    void setAppMetric(int metric)
    {
	int ofs = Packet.getDataOffset(data);
	NameUpdate.mutateMetricInWireform(data, ofs, metric);
    }

    int getAppMetric()
    {
	int ofs = Packet.getDataOffset(data);
	return NameUpdate.getMetricInWireform(data, ofs);
    }

    
    /** May return null if the resolver to send to can't be found */
    DatagramPacket produceDatagramPacket(NameAnnouncer ra) 
    {
	InetAddressPort iap = ra.getResolverForVspace(vspace);

	if (iap == null) return null;
	
	return new DatagramPacket(data, data.length,
				  iap.addr, iap.port);
    }

}
