package camera;

import java.util.*;
import java.io.*;
import java.net.*;

import ins.api.*;
import ins.inr.Packet;
import ins.namespace.*;

/** 
 * This class implements the camera Receiver application.
 */
public class Receiver extends Application
{
    // VARIABLES
    final static int DEBUG = 0;     // 0 = no debug msg

    NameSpecifier sourceNS;         // template to create source NS's from
    NameSpecifier mSourceNS;
    NameSpecifier destNS;	    // template to create dest NS's from
    ReceiverFrame receiverFrame;    // the GUI
    AVelement location;             // The location we're getting pictures for

    String name;

    static Attribute serviceAttr = new Attribute("service");
    static Value cameraVal = new Value("camera");
    static Attribute locationAttr = new Attribute("location");

    boolean subscribed;
    boolean requested;

    // CONSTRUCTORS
    
    /**
     * args[0] is expected to hold the NameSpecifier to receive
     */

    public Receiver(String [] args) throws Exception
    {
	super();
	init(args);
    }

    public Receiver(String dsrname, String [] args)
	throws Exception
    {
	super(dsrname);
	init(args);
    }

    public Receiver(String inrname, int inrport, String [] args)
	throws Exception
    {
	super(inrname, inrport);
	init(args);
    }



    /**
     * Initialization function called by the constructor.
     */	
    private void init(String [] args) throws Exception
    {
	printStatus("Initializing");

	NameSpecifier cameraNS = new NameSpecifier(args[0]);

	// Records the name specifier we've been spawned in response to
	location = cameraNS.getAVelement(locationAttr);
	printStatus("location is " + location);
	name = "camera.Receiver: " + location.toString();

	String vspace = cameraNS.getVspace();

	// Construct a source name specifier with the
	// service, entity, and node (if possible) attributes
	// location will get put on at request time.
		
	sourceNS = new NameSpecifier("[service=camera]"+
				     "[entity=receiver]");
	mSourceNS = new NameSpecifier("[service=camera]"+
				      "[entity=subscriber]");
		
	
	AVelement ave = new AVelement(new Attribute("node"),
				      new Value(localhost.getHostAddress()));
	ave.addAVelement(new AVelement(new Attribute("port"),
				       new Value(Integer.toString(port))));

	sourceNS.addAVelement(ave);
	mSourceNS.addAVelement(ave);


	sourceNS.setVspace(vspace);
	mSourceNS.setVspace(vspace);
	mSourceNS.addAVelement(location);
		
	// Construct a destination name specifier with the
	// service and entity attributes.
	// location will get put on at request time.
		
	destNS = new NameSpecifier("[service=camera]"+
				   "[entity=transmitter]");
	destNS.setVspace(vspace);
	destNS.addAVelement(location);
		
	// Create new ReceiverFrame for displaying image
	receiverFrame = new ReceiverFrame(this,
					  getBottomValue(location).toString());

	this.subscribed = true;
	this.requested = false;

	startAnnouncer(new NameSpecifier[]{sourceNS, mSourceNS});

	// turn on the subscriber at startup
	receiverFrame.doSubscribe();
    }

    private Value getBottomValue(AVelement a)
    {
	while (! a.isLeaf()) {
	    Enumeration e = a.getAVelements();
	    a = (AVelement)e.nextElement();
	}

	return(a.getValue());
    }


    /** 
     * Process a received packet.
     */
    public void receivePacket(Packet packet) {

	// Check to make sure this is actually a packet 
	// we want to receive.
	
	// Make sure [service=camera]
	AVelement sAVE = packet.dNS.getAVelement(serviceAttr);
	if (! sAVE.getValue().equals(cameraVal))
	    return;
	
	// Ok, it's probably for us. Update.

	if (!this.subscribed) {
	    if (!this.requested) {
		printStatus("Dropping unsubscribed, unrequested packet");
		return;
	    }
	    this.requested = false;
	}
	    
	receiverFrame.updateImage(packet.data);
    }
	
    /** 
     * Send a request packet for a picture at location.
     */
    public void sendRequest() 
    {	
	printStatus("sending request to "+destNS);

	this.requested = true;

	sendMessage(sourceNS, destNS, new byte[0]);
    }

    public synchronized void subscribe() 
    {
	if (!this.subscribed)
	    addAnnouncement(mSourceNS, 0);

	this.subscribed = true;
    }

    public synchronized void unsubscribe() 
    {
	if (this.subscribed)
	    removeAnnouncement(mSourceNS);

	this.subscribed =false;
    }

	
    public static void printHelpMessage() 
    {
	System.out.println("Syntax:");
	System.out.println("\tjava camera.Receiver [receive what camera] "+
			   "[-d DSR name] [-p specific-INR-to-peer-with "+
			   "its-portnum]");
	System.out.println("\nExamples:\n\tjava camera.Receiver "+
			   "[service=camera][entity=transmitter]"+
			   "[location=floor5[room=504]][vspace=wind]"+
			   " -d localhost");
	System.out.println("\tjava camera.Receiver [service=camera]"+
			   "[entity=transmitter][location=floor5[room=504]]"+
			   "[vspace=wind] -p fenway.lcs.mit.edu 1234");
	System.exit(-1);
    }
    

    public static void main(String args[])
    {
	String receiverString = "[service=camera][entity=transmitter]"+
	                 "[location=floor5[room=504]][vspace=wind]";

	String dsrname = null;
	String inrname = null;
	int inrport = -1;
	
	// process command line

	for (int argnum = 0; argnum < args.length; argnum++)
	{
	    String arg = args[argnum];
	    if (arg == null) continue;

	    if (argnum == 0 && !arg.startsWith("-")) {
		receiverString = arg;
	    } 

	    else if (arg.equals("-d"))
	    {
		argnum++;
		if (argnum>=args.length) 
		    printHelpMessage();
	     
		dsrname = args[argnum];
	    }

	    else if (arg.equals("-p"))
	    {
		argnum+=2;
		if (argnum>=args.length)
		    printHelpMessage();

		inrname = args[argnum-1];
		try { inrport = Integer.parseInt(args[argnum]); }
		catch (NumberFormatException e) { inrport = -1; }		
	    } else 
		printHelpMessage();
	}

	try { 
	    String [] appargs = new String[]{ receiverString };

	    if (inrport>-1)
		new Receiver(inrname, inrport, appargs);
	    else if (dsrname != null)
		new Receiver(dsrname, appargs);
	    else
		new Receiver(appargs);

	}
	catch (Exception e) { e.printStackTrace(); }
    }
}


