package camera;

import ins.api.*;
import ins.namespace.*;
import ins.inr.Packet;
import java.util.*;
import java.net.*;
import java.io.*;

/** 
 * This class implements the Camera Transmitter application.
 */
public class Transmitter extends Application
    implements Runnable
{
    // VARIABLES
    
    INSConfig conf;     // stores the Transmitter configuration
    
    AVelement location;     // location this camera is at
    String imagefile;       // file to read to get a picture
    String command;         // command to run to get a picture
    String periodiccommand; // command to run to get a picture initially
                                // (or every few hundred times, to adjust
                                //  the lighting level, which saves the camera
                                //  from doing that every time)
    int timesSampled=0;     //  how many times pictures have been taken
                                // used to run initcommand every few times
                                //   to readjust lighting level
    int periodicity=32;     // how often the lighting level is readjusted
    int multicastperiod=1000; // how long between multicasts
    
    NameSpecifier [] sourceNSes;  // source NS to use on outgoing packets
    NameSpecifier [] mDestNSes;   // multicast dest NS
    
    static Attribute serviceAttr = new Attribute("service");
    static Attribute entityAttr = new Attribute("entity");
    static Value cameraVal = new Value("camera");
    static Value transmitterVal = new Value("transmitter");


    // CONSTRUCTORS
    public Transmitter() throws Exception
    {
	super();
	init();
    }

    public Transmitter(String dsrname) throws Exception
    {
	super(dsrname);
	init();
    }

    public Transmitter(String inrname, int inrport) throws Exception
    {
	super(inrname, inrport);
	init();
    }

	
    /** 
     * Initialization function called by the constructors.
     *  - initialize name specifiers
     *  - create in, out buffers for communication with WIND Manager
     *  - connect to the WIND Manager
     *  - initialize client list
     *  - create camera frame
     *  - evoke both main thread and listener thread
     */	
    public void init() 
    {
	String line;
	BufferedReader br;
	
	printStatus("Initializing.");

	// Read the configuration file.
	try {
	    conf = new INSConfig("Transmitter");
	} catch (Exception e) {
	    printStatus("Problem with Transmitter "+
			"configuration:\n" +
			e.getMessage());
	    e.printStackTrace();
	    return;
	}
	
	// Retrieve parameters from configuration file
	String strloc;
	strloc = conf.getValue("location");
	imagefile = conf.getValue("imagefile");
	command = conf.getValue("command");
	periodiccommand = conf.getValue("periodiccommand");
	String vspacesstr = conf.getValue("vspaces");

	// extract vspaces to a list
	Vector vspaces = new Vector(2);	
	if (vspacesstr != null) {
	    StringTokenizer st = new StringTokenizer(vspacesstr, " ,\t");
	    while (st.hasMoreTokens())
		vspaces.addElement(st.nextToken());
	}

	String tmp = conf.getValue("periodicity");
	if (tmp != null) {
	    try {
		periodicity = Integer.parseInt(tmp);
	    } catch (Exception e) { periodicity = 32; }
	} else { periodicity = 32; }


	tmp = conf.getValue("multicastperiod");
	if (tmp != null) {
	    try {
		multicastperiod = Integer.parseInt(tmp);
	    } catch (Exception e) { multicastperiod = 1000; }
	} else { multicastperiod = 1000; }


	if (strloc == null) {
	    printStatus("Couldn't read location");
	    return;
	}
	
	location = new AVelement(strloc);
	printStatus("location is " + location);
	
	if (imagefile == null) {
	    imagefile = (File.separator + "tmp" + 
			 File.separator + "camera.gif");
	}
	
	printStatus("imagefile is " + imagefile);
	
	if (command != null) {
	    printStatus("command is " + command);
	}
	
	if (periodiccommand == null)
	    periodiccommand = command;
	
	// Construct the name specifiers.
	
	// Construct sourceNS.
	
	// Start with the service and entity attributes.
	NameSpecifier sourceNS = new NameSpecifier("[service=camera]"+
						   "[entity=transmitter]"+
						   "[floorplanshow=true]");
	
	sourceNS.addAVelement(location); // Add in the location attribute.

	AVelement ave = new AVelement(new Attribute("node"),
				      new Value(localhost.getHostAddress()));
	ave.addAVelement(new AVelement(new Attribute("port"),
				       new Value(Integer.toString(port))));


	sourceNS.addAVelement(ave); 

	// Construct mDestNS.
	
	// Start with the service and entity attributes.
	NameSpecifier mDestNS = new NameSpecifier("[service=camera]"+
						  "[entity=subscriber]");
	// Add in the location attribute.
	mDestNS.addAVelement(location);
	

	// construct versions of the namespecifiers all the different vspaces

	// no explicit vspace
	if (vspaces.size() == 0)
	{
	    sourceNSes = new NameSpecifier[]{sourceNS};
	    mDestNSes = new NameSpecifier[]{mDestNS};
	} else {
	    int size = vspaces.size();

	    sourceNSes = new NameSpecifier[size];
	    mDestNSes = new NameSpecifier[size];

	    for (int i = 0; i<size; i++)
	    {
		String vspace = (String)vspaces.elementAt(i); 

		NameSpecifier src = new NameSpecifier(sourceNS);
		src.setVspace(vspace);
		sourceNSes[i] = src;

		NameSpecifier dst = new NameSpecifier(mDestNS);
		dst.setVspace(vspace);
		mDestNSes[i] = dst;
	    }
	}

	startAnnouncer(sourceNSes);
		
	// Start the multicast transmitting thread
	new Thread(this).start();
    }	


    /** 
     * Process a request packet.
     */
    public void receivePacket(Packet packet) {

	// Check to make sure this is actually a packet 
	// we want to receive.
	
	// Make sure [service=camera]
	AVelement sAVE = packet.dNS.getAVelement(serviceAttr);
	if (! sAVE.getValue().equals(cameraVal))
	    return;
	    
	// Make sure [entity=transmitter]
	AVelement eAVE = packet.dNS.getAVelement(entityAttr);
	if (! eAVE.getValue().equals(transmitterVal))
	    return;
			
	// Make sure [location=(ourlocation)]
	AVelement lAVE;
	lAVE = packet.dNS.getAVelement(location.getAttribute());
	if (! lAVE.equals(location))
	    return;
		       
	// Ok, it's probably for us. Send out a response.
	printStatus("sending response");
	sendResponse(packet.sNS);
    }

    /** 
     * Returns the picture to be sent
     */
    private byte[] getPicture()
    {
	byte[] data = null; // To store data for the packet (the image).
	
	// Run the command, if we have one.
	if (command != null) {
	    String cmd = ((timesSampled % periodicity)==0)? 
		periodiccommand : command;
	    
	    printStatus("Retrieving image via " + cmd);

	    Runtime rt = Runtime.getRuntime();
	    
	    Process proc = null;
	    try {
		proc = rt.exec(cmd);
		data = getProcessOutput(proc, 64000);
	    } catch (IOException e) {
		printStatus("Couldn't exec " + command);
	    }
	    timesSampled++;

	} else {
	    printStatus("Retriving image from " + imagefile);
	    
	    // Find the file and lastModified time.
	    File imageFile = new File(imagefile);
	    
	    data = new byte[(int)imageFile.length()]; 
	    try {
				// Read the file.
		FileInputStream in
		    = new FileInputStream(imageFile);
		in.read(data);
		in.close();
	    } catch (Exception e) {
		printStatus("Couldn't read imagefile.");
		return(null);
	    }
	}
	
	return(data);
    }


    /** 
     * Send out a response packet.
     */
    private void sendResponse(NameSpecifier to)
    {
	byte[] data = getPicture();
	
	printStatus("sending anycast to "+to.toString());
	
	sendMessage(sourceNSes[0], to, (data==null? new byte[0]: data));
    }
    
    
    /** 
     * Send out a multicast packet.
     */
    synchronized void sendMulticast()
    {
	byte[] data = getPicture();
	data = (data==null? new byte[0]: data);
	
	printStatus("sending Multicast");
	
	for (int i = 0; i<sourceNSes.length; i++)
	    sendMessage(sourceNSes[i], mDestNSes[i], true, data);
    }
    
    /**
     * Loops forever to multicast the packets
     */
    public void run() 
    {
	while (true) {
	    try {	    
		sendMulticast();
		Thread.currentThread().sleep(multicastperiod);
	    } catch (Exception e) {
		e.printStackTrace();
	    }
	}
    }

    
    private byte[] getProcessOutput(Process proc, int maxbuf) 
	throws IOException
    {
	InputStream is = proc.getInputStream();
	
	byte[] buf = new byte[maxbuf];
	
	int len, offset = 0;
	while (true) {
	    len = is.read(buf, offset, 1024);

	    if (len == -1) {
		break;
	    } else {
		offset += len;
	    }
	}
	
	is.close();

	byte[] data = new byte[offset];
	System.arraycopy(buf,0,data,0,offset);

	return(data);
    }


    public static void printHelpMessage() 
    {
	System.out.println("Syntax:");
	System.out.println("\tjava camera.Transmitter "+
			   "[-d DSR name] [-p specific-INR-to-peer-with "+
			   "its-portnum]");
	System.out.println("\nExamples:\n\tjava camera.Transmitter "+
			   "-d localhost");
	System.out.println("\tjava camera.Transmitter -p "+
			   "fenway.lcs.mit.edu 1234");
	System.exit(-1);
    }
    

    public static void main(String args[])
    {

	String dsrname = null;
	String inrname = null;
	int inrport = -1;
	
	// process command line

	for (int argnum = 0; argnum < args.length; argnum++)
	{
	    String arg = args[argnum];
	    if (arg == null) continue;

	    if (arg.equals("-d"))
	    {
		argnum++;
		if (argnum>=args.length) 
		    printHelpMessage();
	     
		dsrname = args[argnum];
	    }

	    else if (arg.equals("-p"))
	    {
		argnum+=2;
		if (argnum>=args.length)
		    printHelpMessage();

		inrname = args[argnum-1];
		try { inrport = Integer.parseInt(args[argnum]); }
		catch (NumberFormatException e) { inrport = -1; }		
	    } else 
		printHelpMessage();
	}

	try { 
	    if (inrport>-1)
		new Transmitter(inrname, inrport);
	    else if (dsrname != null)
		new Transmitter(dsrname);
	    else
		new Transmitter();
	}
	catch (Exception e) { e.printStackTrace(); }
    }


}

