package floorplan;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.border.*;

import java.awt.*;
import java.awt.event.*;
import java.util.Enumeration;
import java.util.Vector;

import ins.api.*;
import ins.namespace.*;
import ins.inr.*;

public class Display extends JFrame
{
    private Floorplan floorPlan;
    private MapPanel mapPane;
    private Vector icons;
    public Vector iconNames;

    static Attribute serviceAttr = new Attribute("service");
    static Attribute locationAttr = new Attribute("location");

    /**
     * Displays a floorplan map, where context is the description, like:
     *   [location=floor5[room=504]][vspace=wind]
     */
    public Display(Floorplan fp, String context)
    {
	super("Floorplan: " + context);

	floorPlan = fp;
	icons = new Vector();
	iconNames = new Vector();

	mapPane = new MapPanel(fp.mapContents);
	getContentPane().add(mapPane, BorderLayout.CENTER);
	fp.mapContents = null;

        addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {
		floorPlan.exit();	
		//System.exit(0);
            }
        });

        //Finish setting up the frame, and show it.
        pack();
        setVisible(true);
    }


    /**
     * Adds a new service to the display
     * @param strIcon Popup description of the service
     * @param xpos    Position on the map
     * @param ypos    Position on the map
     * @param handler What application gets called (e.g. camera.Receiver)
     * @param ns      The namespecifier
     */
    public void addService(String strIcon, int xpos, int ypos,
	String handler, String[] args, NameSpecifier ns)
    {
	java.net.URL url = this.getClass().getResource(strIcon);
	if (url==null) {
	    floorPlan.printStatus("Icon for "+strIcon+" not found");
	    return;
	}

	ImageIcon icon = new ImageIcon(url);
	JButton b = new JButton(icon);
	b.setBorder(null);

	b.addActionListener(new ButtonAction(floorPlan, handler, args));

	// Just have the service and botton location part as the ToolTipText
	AVelement serviceAVE = ns.getAVelement(serviceAttr);
	if (serviceAVE == null) return;
	Value serviceValue = serviceAVE.getValue();

	AVelement locationAVE = ns.getAVelement(locationAttr);
	if (locationAVE == null) return;
	AVelement bottomAVE = getBottomAVelement(locationAVE);
	String tttext = serviceValue.toString() + " ";

	for (Enumeration e = serviceAVE.getAVelements();
	     e.hasMoreElements(); ) 
	{
	    AVelement nextAVE = (AVelement)e.nextElement();
	    tttext = tttext + nextAVE.toString();
	}

	tttext = tttext + " @ " + 
	    bottomAVE.getAttribute() + "=" +
	    bottomAVE.getValue();
	b.setToolTipText(tttext);

	mapPane.add(b,0);
	b.setBounds(xpos, ypos, icon.getIconWidth(), icon.getIconHeight());

	b.setVisible(true);	
	icons.addElement(b);
	iconNames.addElement(ns);
    }	


    public void addZoom(String strIcon, int xpos, int ypos,
	String handler, String [] args, NameSpecifier ns)
    {
	ImageIcon icon = new ImageIcon(this.getClass().getResource(strIcon));
	JButton b = new JButton(icon);
	b.setBorder(null);

	b.addActionListener(new ButtonAction(floorPlan, handler, args));

	// Just have the service and botton location part as the ToolTipText
	NameSpecifier ttns = new NameSpecifier(args[0]); //DON'T USE just NS
	AVelement locationAVE = ttns.getAVelement(locationAttr);
	if (locationAVE == null) return;
	AVelement bottomAVE = getBottomAVelement(locationAVE);

	String tttext = "zoom in @ " + 
	    bottomAVE.getAttribute() + "=" +
	    bottomAVE.getValue();
	b.setToolTipText(tttext);

	mapPane.add(b,0);
	b.setBounds(xpos,ypos,icon.getIconWidth(),icon.getIconHeight());
	//System.out.println("Floorplan: NEW Zoom Added !!");
	b.setVisible(true);

	icons.addElement(b);
	iconNames.addElement(ns);
    }	


    public void checkExpiration(Vector newNames, Vector iconCoords)
    {
	for (int i=0; i<iconNames.size(); i++)
	{
	    NameSpecifier ns = (NameSpecifier)iconNames.elementAt(i);
	    if (newNames.indexOf(ns) >= 0)  continue;

	    // Otherwise, remove icon
	    System.out.println(" --> Floorplan removing expire icon!");
	    JButton b = (JButton)icons.elementAt(i);
	    b.setVisible(false);
	    mapPane.remove(b);
	    icons.removeElementAt(i);
	    iconNames.removeElementAt(i);
	    iconCoords.removeElementAt(i);

	    // don't increment i in the next iteration because we
	    // just remove element at i, everything after i will shift down
	    i--;
	}
    }   


    private AVelement getBottomAVelement(AVelement a)
    {
	Enumeration e = a.getAVelements();

	while(e.hasMoreElements()) {
	    a = (AVelement)e.nextElement();
	    e = a.getAVelements();
	}

	return(a);
    }

}


class MapPanel extends JPanel
{
    MapPanel(byte [] mapContents)
    {
	super();
	ImageIcon floorImg = new ImageIcon(mapContents);
	JLabel map = new JLabel(floorImg);

	setLayout(null);
	add(map, -1);
	map.setBounds(10,10,floorImg.getIconWidth(), floorImg.getIconHeight());

	setPreferredSize(new Dimension(floorImg.getIconWidth()+20,
				       floorImg.getIconHeight()+20));
	setMinimumSize(new Dimension(floorImg.getIconWidth()+20, 
				     floorImg.getIconHeight()+20));
    }

}

class ButtonAction implements ActionListener
{
	String handler;
	String [] args;
	Floorplan fp;

	ButtonAction(Floorplan f, String h, String [] args)
	{
		fp = f;
		handler = h;
		this.args = args;
	}

	public void actionPerformed(ActionEvent e) {
		fp.runHandler(handler, args);
	}
}	
