package floorplan;

import ins.api.*;
import ins.inr.*;
import ins.namespace.*;

import java.net.*;
import java.util.*;
import java.io.*;
import javax.swing.UIManager;
import java.lang.reflect.*;

public class Floorplan extends Application
    implements Runnable
{
    // VARIABLES
    final static int DEBUG = 0;		// 0=no debug msg

    Display display;		// GUI floorplan display
    Thread mainThread;		// the main thread
    FloorplanConf conf;		// floorplan configuration 

    NameSpecifier locationSourceNS;

    String context;		// location context

    // for communication with LocationServer
    HostRecord locServer;	// for early binding to LocationServer
    boolean ebAvail;        	// is early-binding (eb) available

    AVelement locationAVE;

    // location stuff
    boolean doinglocation = true;  // checking for location?
    NameSpecifier location=null;          // current location
    NameSpecifier lastLocation=null;
    boolean foundBefore =false;

    String name;
    String vspace;

    static Attribute serviceAttr = new Attribute("service");
    static Attribute dataAttr = new Attribute("data");
    static Attribute locationAttr = new Attribute("location");
    static Value mobileHostVal = new Value("mobile-host");
    static Value coordsVal = new Value("coords");

    // applications can advertise whether they want to be displayed in 
    // the floorplan or not... this is probably a clumsy attribute name
    // It gets around the problem of receivers being recognized as "services"
    // (this crept up in the new system because I implemented the other
    //  apps as having announcers rather than using route interference -jjl)
    static Attribute floorplanshowAttr = new Attribute("floorplanshow");


    String dsrname=null;       // to be stored to help invoke other apps
    String inrname=null;
    int inrport=-1;

    byte [] mapContents=null;

    boolean fpRunning = true;



    // CONSTRUCTOR
    public Floorplan(String [] args) throws Exception
    {
	super();
	init(args);
    }

    public Floorplan(String dsrname, String [] args) throws Exception
    {
	super(dsrname);
	this.dsrname = dsrname;
	init(args);
    }

    public Floorplan(String inrname, int inrport, String [] args) 
	throws Exception
    {
	super(inrname, inrport);
	this.inrname = inrname;
	this.inrport = inrport;
	init(args);
    }


    //////////////////////

    public void init(String [] args) 
    {
	if (args.length >= 1)
	   context = args[0];
	else
	   context = "[location=floor5][vspace=wind]";

	System.out.println("**************************\ninput context: "+
			   context);

	NameSpecifier contextNS = new NameSpecifier(context);
	vspace = contextNS.getVspace();

	// Modified by Magda to support lack of vspace
	if (vspace==null)
	    vspace = "wind";
	// END modif by Magda

	AVelement lave = contextNS.getAVelement(locationAttr);
	contextNS = new NameSpecifier();
	contextNS.addAVelement(lave);
	contextNS.setVspace(vspace);
	context = contextNS.toString();

	name = "Floorplan:" + context;

	locationSourceNS = 
	    new NameSpecifier("[service=location]"+
			      "[entity=client][node="+
			      localhost.getHostAddress()+
			      "[port="+port+"]]");
	if (vspace != null)
	    locationSourceNS.setVspace(vspace);
	
	startAnnouncer(new NameSpecifier [] { locationSourceNS });


	printStatus("Reading Floorplan.conf file ...");
	// Read configuration file for floorplan
	try{
	    conf = new FloorplanConf("Floorplan");
	} catch (Exception e) {
	    printStatus(e.getMessage());
	    System.exit(-1);
	}
	
	// Check java version
	String vers = System.getProperty("java.version");
	if (vers.compareTo("1.1.6") < 0) {
	    printStatus("WARNING: Floorplan must be run with a " +
			       "1.1.6 or higher version VM!!!");
	}
	
	// Force Sysmon to come up in the Java Platform L&F
	try {
	    UIManager.setLookAndFeel
		(UIManager.getCrossPlatformLookAndFeelClassName());
	
	    // if want to force to use Windows L&F, comment out above and 
	    // uncomment below:
	    // UIManager.setLookAndFeel("com.sun.java.swing.plaf.windows.WindowsLookAndFeel");
	    // If want the Java L&F instead, comment out one line below
	
	    // If want the System L&F instead, comment out the above line and
	    // uncomment the following:
	    // UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
	} catch (Exception exc) {
	    printStatus("Error loading L&F: " + exc);
	}
	
	mainThread = new Thread(this);
	mainThread.start();
    }



    public void run()
    {
	int WAITTIME = 3000;
	NameSpecifier allNS =
	    new NameSpecifier("[service=*][vspace="+vspace+"]");
	Vector savePrev = new Vector();
	Vector serviceKnown = new Vector();
	Vector iconCoords = new Vector();

	// Block until location server is discovered
	printStatus("Waiting to discover location server ...");	
	
	
	// Get early-binding information of the LocationServer
	ebAvail = false;
	EarlyBindRecord[] ebrset = 
	    getHostByiName(new NameSpecifier
			   ("[service=location][vspace="+vspace+"]"), false);

	printStatus("Got the early binding record ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"+
	ebrset.length);


	for (int i=0; i<ebrset.length; i++)
	{
	//////?????????????????????????????????

		printStatus("Examining the early binding record");
	

	///???????????????????????????????????????
	    HostRecord hr = ebrset[i].getHostRecord();
		 printStatus(hr.toString());
	    if (hr.getPortForTransport("TCP") > -1) {
		locServer = hr;
		ebAvail = true;
printStatus("early binding available to map server");
		break;
	    }
	}
   
	// Get the map from location server
	// requestMap("[location=mit[building=ne43[floor=5]]]");
	while (!requestMap(context, vspace)) {
	    printStatus("requesting map for \""+context+"\", "+vspace);
	    printStatus("Error: cannot get the map... retrying!");
	    try {
		Thread.sleep(5000); // sleep for 5 seconds
	    } catch (Exception e) {}
	}

	// Display the floor plan
	printStatus("Setting up floorplan display...");	
	display = new Display(this, context);

	while (fpRunning) {  
	    try {
		savePrev = serviceKnown;
		NameSpecifier [] nses = discoverNames(allNS, true);

		// put in vector (at least the ones that are supported by
		//                the floorplan => have the floorplanshowAttr)
		serviceKnown = new Vector(nses.length);
		synchronized (serviceKnown) {
		    for (int i=0; i<nses.length; i++)
			if (nses[i].getAVelement(floorplanshowAttr) != null)
			    serviceKnown.addElement(nses[i]);
		}
		
		System.out.println("serviceKnown = "+serviceKnown.toString());
		
		// location checks
		if (doinglocation) 
		    WAITTIME = doLocation(serviceKnown);
		
		// Check new discovery
		checkDiscovery(savePrev, serviceKnown, iconCoords);
		
		// Check icon expirations
		display.checkExpiration(serviceKnown, iconCoords);
	    } catch (Exception e) {
		printStatus("error: "+e.toString());
		e.printStackTrace();
	    }

	    try { Thread.sleep(WAITTIME);}
	    catch (InterruptedException e) {
		printStatus
		    ("Exception: thread can't sleep in the run() method");
	    }
	}
    }


    /**
     * Check the location of this current device
     */
    private int doLocation(Vector serviceKnown)
    {
	int WAITTIME=3000;

	location = getLocationNameSpecifier();
	System.out.println("location = "+location);
	
	if (!foundBefore) {
	    if (location!=null) {
		XYPos xypos = 
		    requestCoord(location);
		if (((xypos.x>=0) && (xypos.y>=0))) 
		    foundBefore=true;
	    }
	}
		
	if (foundBefore) {
	    if (location!=null) {
		if (location.equals(lastLocation)) {
		    NameSpecifier mobileHost = new NameSpecifier(location);
		    mobileHost.addAVelement
			(new AVelement(serviceAttr, mobileHostVal));	    
		    serviceKnown.addElement(mobileHost);
		} else { WAITTIME = 5; }			
	    } 
	}
	lastLocation = location;

	return WAITTIME;
    }


    /** 
     * Displays all the new icons 
     *  -- savePrev consists of all the old/previous NS's
     *  -- serviceKnown is all the new/current NS's
     */
    private void checkDiscovery(Vector savePrev, Vector serviceKnown,
				Vector iconCoords)
    {
	NameSpecifier ns;
	AVelement ave;
	String location, service;
	XYPos xypos;
	String[] args;
	
	for (int i=0; i<serviceKnown.size(); i++) {
	    ns = (NameSpecifier)serviceKnown.elementAt(i);
	    
	    // if already displayed before, skip it
	    if (savePrev.indexOf(ns) > -1) 
		continue;
	    	    	    
	    // Extract information from the source name specifier,
	    // and tell the GUI about the new service.	    
	    ave = ns.getAVelement(serviceAttr);
	    
	    // if no service attr. skip
	    if (ave == null) 
		continue;
	    
	    service = ave.getValue().toString();
	    
	    ave = ns.getAVelement(locationAttr);
	    
	    // if no location attr. skip
	    if (ave == null) 
		continue;

	    // copy to a new one just for manipulation next
	    ave = new AVelement(ave);

	    NameSpecifier reqNS = new NameSpecifier();
	    reqNS.addAVelement(ave);
	    reqNS.setVspace(vspace);
	    	    
	    switch (draw(ave,new AVelement(context))) {
	    case 0:             // NOTHING
		break;
	    case 1:		// SERVICE		
		// Add service icon
		location = ave.getValue().toString();
		
		String[] svcRecord = conf.getServiceRecord(service);
		if (svcRecord == null) continue;
				
		xypos = requestCoord(reqNS);

		// if unknown location
		if ((xypos.x<0) || (xypos.y<0)) continue;
		
		// if (xpos, ypos) already has an icon
		//    use some offset for y coordinate
		while (iconCoords.indexOf(xypos) >= 0)
		    xypos.y = xypos.y + 32;
		
		iconCoords.addElement(xypos);
		
		printStatus("***New Service discovered***");
		printStatus(" diplayed at (x,y) = (" + xypos.x + "," +
			    xypos.y + ")");
				
		display.addService(svcRecord[0], xypos.x, xypos.y, 
				   svcRecord[1], 
				   new String[] {ns.toString()}, ns);
		
		break;
		
	    case 2:                   // ZOOM DOT
		// zoom
		location = ave.getValue().toString();
				
		xypos = requestCoord(reqNS);
		
		// if unknown location
		if ((xypos.x<0) || (xypos.y<0)) continue;
		
		// if (xpos, ypos) already has an icon
		//    don't add duplicate icons
		if (iconCoords.indexOf(xypos) < 0) {
		    iconCoords.addElement(xypos);

		    NameSpecifier tmpns = new NameSpecifier();
		    tmpns.addAVelement(ave);

		    // Modified by Magda to support advertisements
		    // without vspaces
		    // tmpns.setVspace(ns.getVspace());
		    String virtual_space=ns.getVspace();
		    if (vspace==null)
			virtual_space = vspace;
		    tmpns.setVspace(virtual_space);
		    // END modif by Magda

		    display.addZoom("ins-dot.gif", xypos.x, xypos.y, 
				    "floorplan.Floorplan", 
				    new String[] { tmpns.toString() }, ns);
		}
	    }
	}
    }
    


    /** 
     * Requests Coordinates from the location server
     *  @param ns NameSpecifier containing location and vspace
     */
    private XYPos requestCoord(NameSpecifier ns)
    {
	NameSpecifier coordNS = 
	    new NameSpecifier("[service=location]" +
			      "[entity=server]");
	
	if (ns==null) 
	    return new XYPos(-1,-1);
  
	// Modified by Magda to support advertisements
	// without vspaces
	// coordNS.setVspace(ns.getVspace());
	String virtual_space=ns.getVspace();
	if (virtual_space==null)
	    virtual_space = vspace;
	coordNS.setVspace(virtual_space);
	// END modif by Magda

	NameSpecifier requestNS = new NameSpecifier(ns);
	requestNS.addAVelement(new AVelement(dataAttr, coordsVal));

	String requestStr = requestNS.toString();
       
	// we tried to do some caching here, but it didn't work right

	printStatus("Sending location request: " + coordNS.toString());	
	Packet p = contactServer(locationSourceNS, coordNS, 
				 (requestStr.getBytes()), 1500, 2);
	if (p==null)
	    return new XYPos(-1,-1);

	String strXY = new String(p.data);

	printStatus("response is (X Y) = " + strXY);	

	StringTokenizer st = new StringTokenizer(strXY);
	XYPos xypos;
	if (st.countTokens() ==2 ) {
	    String strX = (String)st.nextElement();
	    String strY = (String)st.nextElement();
	    int xpos = Integer.parseInt(strX);
	    int ypos = Integer.parseInt(strY);
	    xypos = new XYPos(xpos, ypos);

	} else {
	    xypos = new XYPos(-1,-1);
	}

	return xypos;
    }


    /**
     * Requests a map for a given location,
     *   calling the early-binding version (requestMapEB)
     * if applicable.
     */
    private boolean requestMap(String location, String vspace)
    {
	if (ebAvail)
	    return requestMapEB(location, vspace);

	NameSpecifier mapNS = 
	    new NameSpecifier("[service=location]" +
			      "[entity=server]");
	mapNS.setVspace(vspace);

	NameSpecifier requestNS = new NameSpecifier("[data=map]");
	requestNS.addAVelement(new AVelement(location));

	printStatus("Sending request for map: " + requestNS.toString());

	//works up to this !!!!!
	printStatus(locationSourceNS.toString());

	Packet p = contactServer(locationSourceNS, mapNS, 
				 (requestNS.toString().getBytes()), 
				 1500, 2);
	if (p==null) {
	 	printStatus("got null packet");
	    return false;
	}
	mapContents = p.data;

	return true;
    }


    /**
     * Does an early-binding request for a map to the locationserver
     */
    private boolean requestMapEB(String location, String vspace)
    {
	NameSpecifier mapNS = 
	    new NameSpecifier("[service=location]" +
			      "[entity=server][vspace="+vspace+"]");
	mapNS.addAVelement(new AVelement(location));

	mapContents = null;

	printStatus("Establishing TCP connection to LocationServer...");
	Socket socket = null;
        PrintWriter out = null;
        InputStream in = null;

	short port = locServer.getPortForTransport("TCP");
		printStatus("Connecting to port"+port);
        try {
	    StringBuffer strbuf = new StringBuffer();
	    strbuf.append((locServer.address[0]) & 0xFF);strbuf.append(".");
	    strbuf.append((locServer.address[1]) & 0xFF);strbuf.append(".");
	    strbuf.append(locServer.address[2] & 0xFF);strbuf.append(".");
	    strbuf.append(locServer.address[3] & 0xFF);
	
	    // Establish TCP connection to LocationServer
		 	printStatus("IP address of server : "+strbuf.toString());
			printStatus("Port number" + port);
            socket = new Socket(strbuf.toString(), port);



            out = new PrintWriter(socket.getOutputStream(), true);
            in = socket.getInputStream();

	    // Send a request for image
	    printStatus("Sending request for map: " + mapNS.toString());
	    out.println(mapNS.toString());

	    // Receive the image 	
	    printStatus("...waiting for map picture ...");
	    byte [] buffer = new byte[4];

	  


	    int numread = in.read(buffer);
	    if (numread != 4)
		 return false;
	   
 	   //Check this out ????




	    int len = ( (int)(buffer[0] & 0xFF) | (int)((buffer[1]&0xFF)<<8) |
			(int)((buffer[2] & 0xFF) <<16) | (int)((buffer[3] & 0xFF)<<24) );


	printStatus("Length of data packet : "+len);

	    if (len>1048576) // greater than a meg ==> too big
		return false;
	    
	    buffer = new byte[len];

		 numread=0;
		 do{
	    	numread += in.read(buffer, numread ,(len-numread));
			printStatus("Read of data packet : "+numread);
		 }while(numread < len);

	    mapContents = buffer;

	    out.close();
	    in.close();
	    socket.close();
        } catch (UnknownHostException e) {
            printStatus("Don't know about Location Server's address");

            return false;
        } catch (IOException e) {
            System.err.println
		("Couldn't get I/O for the connection to LocationServer");
	    return false;
        } catch (Exception e) {
	    printStatus("Couldn't wrap map picture into file.");
	    return false;
	}

	return true;
    }


    /**
     * Make requests to the LocationServer go through a single
     * serialized point... this is the mechanism for sending 
     * a request and getting back a response
     * (at least for coordinates, not for early-binding graphics)
     */
    Object processLock = new Object();
    Packet receivedPacket = null;
    
    public void receivePacket(Packet packet)
    {
	 printStatus("Received packet");


	synchronized (processLock) {
	    receivedPacket = packet;
	    processLock.notifyAll();
	}
    }

    /**
     * Contacts a server with a data packet and waits for the first
     * received response. Uses the processLock to single thread the whole
     * process (i.e. as long as all communication goes through this, there
     * should be not conflicts)
     */
    public Packet contactServer(NameSpecifier sourceNS, NameSpecifier destNS,
				byte [] data, int wait, int maxtries)
    {
	Packet packet = null;
	int tries = 0;

	do {
	    tries++;
	   synchronized(processLock) {
		boolean keepGoing = true;
		
		receivedPacket = null;
		
		if (!sendMessage(sourceNS, destNS, data)) {
		    printStatus("could not send request to gateway");
		    keepGoing = false;
		} 
		
		try {
		    if (keepGoing) 
			 {
			 processLock.wait(); //??????????
			 printStatus("Waiting on th eprocess lock");
			 }
		} catch (Exception e) {
		    keepGoing = false;
		}
		
		if (keepGoing){
	 	packet = receivedPacket;

		}
		
		receivedPacket = null;
	   }
	} while (packet==null && tries<maxtries);
	
	return packet;
    }


    /**
     *  draws and returns whether just a dot (2) or the full icon (1)
     */
    public static int draw(AVelement service, AVelement context)
    {
	while (! context.isLeaf()) {
	    Enumeration e = context.getAVelements();
	    AVelement newContext = (AVelement)e.nextElement();

	    AVelement newService = 
		service.getAVelement(newContext.getAttribute());
	    if (newService == null) {
		return 0;
	    }

	    if (! newContext.getValue().equals(newService.getValue())) {
		return 0;
	    }

	    context = newContext;
	    service = newService;
	}

	Enumeration e = service.getAVelements();
	if (!e.hasMoreElements()) return 0;
	AVelement lastService = (AVelement)e.nextElement();

	if (lastService.isLeaf()) {
	    return 1;    // if it's a leaf, draw the actual icon  
	} else {
	    // make it a leaf
	    lastService.removeAllAVelements();
	    return 2;    // if it's not, just draw the dot
	}
    }


    /**
     * Invokes a helper application
     */
    public void runHandler(String handler, String [] args)
    {
	System.out.println("***********************************");
	System.out.println("******* HANDLER *****************");
	System.out.println("******* "+handler+"; "+args[0]);

	if (!handler.equals("none"))
	{
	    try {
		// Find the class
		Class t = Class.forName(handler);
			    
		// Find the constructors
		Constructor[] cs = t.getConstructors();
		
		Class stringarrayClass = (new String[0]).getClass();
		Class stringClass = (new String()).getClass();
		Class intClass = Integer.TYPE;
		
		// find correct constructor, matching the way we 
		// were called
		for (int i=0; i< cs.length; i++)
		{		
		    Constructor c = cs[i];
		    Class [] params = c.getParameterTypes();

		    if (inrname !=null)
		    {
			if (params.length != 3)
			    continue;
			
			if ( (!stringClass.equals(params[0])) ||
			     (!intClass.equals(params[1])) ||
			     (!stringarrayClass.equals(params[2])) )
			    continue;
			
			c.newInstance(new Object[] {inrname, 
						    new Integer(inrport), 
						    args});
		    }
		    if (dsrname != null)
		    {
			if (params.length != 2)
			    continue;
			
			if ( (!stringClass.equals(params[0])) ||	
			     (!stringarrayClass.equals(params[1])) )
			    continue;
			
			c.newInstance(new Object[] {dsrname, args});
		    } else {
			if (params.length != 1)
			    continue;
			
			if (!stringarrayClass.equals(params[0]))
			    continue;
			
			c.newInstance(new Object[] {args});
		    }
		    break;
		}   
	    } catch (Exception e) {
		printStatus("Could not instantiate "+handler+": "+
			    e.toString());
		e.printStackTrace();
	    }
	} else {
	    // if none, we just ignore the mouse click
	    System.out.println("uh, ignoring...");
	}
    }

    void exit()
    {
	fpRunning = false;
	display.setVisible(false);
	display.setEnabled(false);
	display.dispose();
	display = null;
	
	cleanUp();
    }

    public void printStatus(String s)
    {
	super.printStatus(s);
    }


    public static void printHelpMessage() {
	System.out.println("Syntax:");
	System.out.println("\tjava floorplan.Floorplan [starting location] "+
			   "[-d DSR name] [-p specific-INR-to-peer-with "+
			   "its-portnum]");
	System.out.println("\nExamples:\n\tjava floorplan.Floorplan "+
			   "[location=mit][vspace=wind] -d 127.0.0.1");
	System.out.println("\tjava floorplan.Floorplan [location=floor5]"+
			   "[vspace=wind] -p fenway.lcs.mit.edu 1234");
	System.exit(-1);
    }
    

    public static void main(String [] args)
    {
	
	String startingLocation = "[location=mit][vspace=wind]";
	String dsrname = null;
	String inrname = null;
	int inrport = -1;
	
	// process command line

	for (int argnum = 0; argnum < args.length; argnum++)
	{
	    String arg = args[argnum];
	    if (arg == null) continue;

	    if (argnum == 0 && !arg.startsWith("-")) {
		startingLocation = arg;
	    } 

	    else if (arg.equals("-d"))
	    {
		argnum++;
		if (argnum>=args.length) 
		    printHelpMessage();
	     
		dsrname = args[argnum];
	    }

	    else if (arg.equals("-p"))
	    {
		argnum+=2;
		if (argnum>=args.length)
		    printHelpMessage();

		inrname = args[argnum-1];
		try { inrport = Integer.parseInt(args[argnum]); }
		catch (NumberFormatException e) { inrport = -1; }		
	    } else 
		printHelpMessage();
	}

	try { 
	    String [] appargs = new String[]{ startingLocation };

	    if (inrport>-1)
		new Floorplan(inrname, inrport, appargs);
	    else if (dsrname != null)
		new Floorplan(dsrname, appargs);
	    else
		new Floorplan(appargs);
	}
	catch (Exception e) { e.printStackTrace(); }
    }
}



// helper classes 


class XYPos
{
    public int x;
    public int y;

    public XYPos(int x, int y)
    {
	this.x = x;
	this.y = y;
    }

    public boolean equals(Object obj)
    {
	XYPos xypos = (XYPos)obj;
	if ((xypos.x == x) && (xypos.y == y))
		return true;
	else
		return false;
    }
}


