package floorplan;

import java.util.Vector;
import java.util.Enumeration;
import java.util.StringTokenizer;
import java.io.File;
import java.io.BufferedReader;
import java.io.FileReader;

/**
 * This class implements an object which keeps track of a configuration.
 */
public class FloorplanConf
{

    // VARIABLES
    
    Vector records;  // Vector of type Record
    
    // CONSTRUCTORS

    /**
     * Creates a new Configuration object based on its name.
     * @param name The name of the configuration file.
     * @exception Exception Something went wrong.
     */
    public FloorplanConf(String name)
	throws Exception
    {
	File file = findFile(name);
	initFromFile(file);
    }

    /**
     * Creates a new Configuration object from a File.
     * @param file The File to create it from.
     * @exception Exception Something went wrong.
     */
    public FloorplanConf(File file)
	throws Exception
    {
	initFromFile(file);
    }

    // METHODS

    /**
     * Finds a configuration File based on its name.
     * First looks in the current directory, then in the user's home directory,
     * and then in /etc/ (or equivalent). Tries in "inat" before ".inat".
     * Adds ".conf" to the name.
     * @param name The name.
     * @return The File.
     * @exception Exception Something went wrong.
     */
    private File findFile(String name)
	throws Exception
    {
	// This defines the order of things to be checked.
	String[] paths = new String[3];
	paths[0] = System.getProperty("user.dir");  // current directory
	paths[1] = System.getProperty("user.home"); // home directory
	paths[2] = File.separator + "etc";          // /etc or equivalent

	for (int i = 0; i < paths.length; i++) {
	    String path = paths[i];

	    if (path != null) {
		File file;
		
		String vname = path + File.separator +
		    "inat" + File.separator +
		    name + ".conf";
		
		file = new File(vname);
		
		if (tryFile(file)) {
		    return(file);
		}
		
		String hname = path + File.separator +
		    ".inat" + File.separator +
		    name + ".conf";
		
		file = new File(hname);
		
		if (tryFile(file)) {
		    return(file);
		}
	
	    }
	}

	File file = new File(name + ".conf");
	
	if (tryFile(file)) {
	    return(file);
	}

	throw new Exception("Couldn't find a configuration file for " + name +
			    " anywhere.");
    }
    
    /**
     * Tests to see if the file exists.
     * @param file The file to be tested.
     * @return true if it is, false if it doesn't.
     * @exception Exception It exists, but can't be read or isn't normal.
     */
    private boolean tryFile(File file)
	throws Exception
    {
	if (file.exists()) {
	    if (! file.isFile()) {
		throw new Exception("Error while reading " + 
				    file.getCanonicalPath() + ":\n" +
				    "File is not normal " +
				    "(may be directory).");
	    }
	    
	    if (! file.canRead()) {
		throw new Exception("Error while reading " +
				    file.getCanonicalPath() + ":\n" +
				    "Can't read the file.");
	    }
	    
	    return true;
	    
	} else {
	    
	    return false;
	}
	
    }
    
    /**
     * Initializes a new Configuration object from a File.
     * @param file The File to create it from.
     * @exception Exception Something went wrong.
     */
    private void initFromFile(File file)
	throws Exception
    {
	records = new Vector();
	BufferedReader br = new BufferedReader(new FileReader(file));
	Enumeration enum;
        StringTokenizer st;
	
	while (true) {
	    
	    // Get the next line.
	    String line = br.readLine();
	    
	    // Check to see if we're done.
	    if (line == null)
		break;
	    
	    // Check if the line is a comment.
	    if (line.charAt(0) == '#')
		continue;

	    // Parse line	    
	    st = new StringTokenizer(line);
	    if (st.countTokens() <= 0) continue;

	    Record r = new Record();
	    r.service = (String)st.nextElement();
	    System.out.print(r.service + "\t");

	    if (st.hasMoreElements())
		r.icon = (String)st.nextElement();
	    else
		r.icon = "ins-default.gif";
	    System.out.print(r.icon + "\t");

	    if (st.hasMoreElements())
		r.handler = (String)st.nextElement();
	    else
		r.handler = "floorplan.DefHandler";
	    System.out.println(r.handler);

/*	    // Get the index of the first space.
	    int space = line.indexOf(' ');
	    
	    Pair p = new Pair();
	    if (space == -1) {
				// No space, just a keyword.
		p.keyword = line;
		p.value = new String();
	    } else {
		p.keyword = line.substring(0, space);
		p.value = line.substring(space + 1);
	    }
*/	    
	    // Add it to the Vector of Pairs

	    records.addElement(r);
	}
    }
    
    /**
     * Gets the first value associated with a particular keyword.
     * @param keyword The keyword to get the value for.
     * @returns The value, or null if not found.
     */
    public String[] getServiceRecord(String service)
    {
	for (Enumeration e = records.elements(); e.hasMoreElements(); ) {
	    
	    Record r = (Record)e.nextElement();
	    
	    if (service.equalsIgnoreCase(r.service)) {
		String[] result = new String[2];
		result[0] = r.icon;
		result[1] = r.handler;
		return(result);
	    }
	}
	
	return(null);
    }
   
    // INNER CLASSES
    private class Record {
	public String service;
	public String icon;
	public String handler;
    }
    
}
