package locationserver;

import java.util.*;
import java.io.*;
import java.net.*;
import ins.api.*;
import ins.inr.*;
import ins.namespace.*;

/** 
 * This class implements the LocationServer application
 *
 * Currently handles two types of incoming requests:
 *   dNS: [service=location] [entity=server][vspace=myvspace]
 *         
 *  Packet contents = "[data=map][location=mit
 *                     [building=ne43[floor=5[room=504]]]]"
 * To which it replies with a gif map of the floor
 *
 *   dNS: [service=location] [entity=server][vspace=myvspace]
 *         
 *   Packet contents = "[data=coords][location=mit
 *                      [building=ne43[floor=5[room=504]]]]"
 * To which it replies with the X Y coordinates in the form "X Y"
 *
 * The system currently assumes that the system serves all the
 * locations in its virtual spaces, as specified by LocationServer.conf
 */
public class LocationServer extends Application
{
    // VARIABLES
    final static int DEBUG = 0;		// 0 = no debug msg
					// higher # = more msgs
    String name;
    INSConfig conf;     // stores the Transmitter configuration
    
    NameSpecifier sourceNS; // source NS to use on sending packets
    
    String locationDir=null;  // base place for location searches
    String imageName=null;    // expected name for the image in a directory
    String configName=null;   // expected name for the configuration file
                              // containing the coordinates in the parent
    ServerSocket ebSocket;    // early-binding socket;

    static Attribute serviceAttr = new Attribute("service");
    static Attribute entityAttr = new Attribute("entity");
    static Attribute locationAttr = new Attribute("location");
    static Attribute dataAttr = new Attribute("data");

    static Value locationVal = new Value("location");
    static Value serverVal = new Value("server");
    static Value mapVal = new Value("map");
    static Value coordsVal = new Value("coords");
    
    // CONSTRUCTORS
    
    /**
     * Creates a new LocationServer
     */
    public LocationServer() throws Exception
    {
	super();
	init();
    }

    public LocationServer(String dsrname) throws Exception
    {
	super(dsrname);
	init();
    }

    public LocationServer(String inrname, int inrport) throws Exception
    {
	super(inrname, inrport);
	init();
    }


    
    /** 
     * Initialization function called by the constructors.
     *  - initialize name specifiers
     *  - create in, out buffers for communication with WIND Manager
     *  - connect to the WIND Manager
     *  - evoke listener thread
     */	
    public void init() 
    {  
	name ="LocationServer";
	printStatus("Initializing.");

	// Construct sourceNS.
	
	// Start with the service and entity attributes.
	sourceNS = new NameSpecifier("[service=location]"+
				     "[entity=server]");
	
	AVelement ave = new AVelement(new Attribute("node"),
				      new Value(localhost.getHostAddress()));
	ave.addAVelement(new AVelement(new Attribute("port"),
				       new Value(Integer.toString(port))));
	sourceNS.addAVelement(ave);

	NameSpecifier [] nses = readINSConfig(sourceNS);      	
	sourceNS = nses[0]; // yes, there is a guaranteed nses[0]

	printStatus("Established location server as: "+sourceNS.toString());

	// Create early-binding information

	HostRecord hr = new HostRecord();
	hr.UDPport = (short)port;
	try {
	    ebSocket = new ServerSocket(0);
	    hr.addTransport("TCP", (short)ebSocket.getLocalPort());
		 printStatus("********************** craeted EB socket" + (short)ebSocket.getLocalPort()  );
	} catch (IOException e) {;}
	
	//startAnnouncer(nses);  

	// Evoke the TCP server thread
	new Thread(new TCPServer(ebSocket, this)).start();


	startAnnouncer(nses, "TCP",(short)ebSocket.getLocalPort() );
		 printStatus("********************** announcing EB socket" + (short)ebSocket.getLocalPort()  );
    }
 

  private NameSpecifier [] readINSConfig(NameSpecifier sourceNS)  
    {	
	NameSpecifier [] result;
	Vector resultv=new Vector(2);

	// Read the configuration file.
	try {
	    conf = new INSConfig("LocationServer");
	} catch (Exception e) {
	    printStatus("Problem with LocationServer "+
			"configuration:\n" +
			e.getMessage());
	}
	
	// Retrieve parameters from configuration file, if they exist
	if (conf != null) {
	    locationDir = conf.getValue("locationDir");	   
	    
	    imageName = conf.getValue("imageName");
	    configName = conf.getValue("configName");

	    String vspaces = conf.getValue("vspaces");
	    		
	    StringTokenizer st = new StringTokenizer(vspaces, ", ");
		
	    if (!st.hasMoreTokens()) {
		NameSpecifier ns = new NameSpecifier(sourceNS);
		resultv.addElement(ns);
	    } else {
		    do {
			String vspace = st.nextToken();
			NameSpecifier ns = new NameSpecifier(sourceNS);
			ns.setVspace(vspace);
			resultv.addElement(ns);
		    } while (st.hasMoreTokens());
		}
	    
	    result = new NameSpecifier[resultv.size()];
	    resultv.copyInto(result);
	} else {
	    result = new NameSpecifier[]{sourceNS};
	}

	if (locationDir == null) {
	    locationDir = System.getProperty("user.dir") +
		File.separator + "locdir";
	}
	if (imageName == null) imageName = "image.gif";		
	if (configName == null) configName = "locations.conf";
	
	return result;
    }


    
    /** 
     * Process a request packet.
     */
    public void receivePacket(Packet packet) {
	// Check to make sure this is actually a packet 
	// we want to receive.
	
	AVelement ave, ave_service;
	
	// Make sure [service=location]
	ave_service = packet.dNS.getAVelement(serviceAttr);
	if (! ave_service.getValue().equals(locationVal))
	    return;
    
	// Make sure [entity=server]
	ave = packet.dNS.getAVelement(entityAttr);
	if (! ave.getValue().equals(serverVal))
	    return;
		
	NameSpecifier ns = new NameSpecifier(new String(packet.data));
	printStatus("+++++++++++++++++++++++++++++++++++++++++++++++++++");
	printStatus(ns.toString());

	AVelement dataAVE = ns.getAVelement(dataAttr);
	
        // map
	if (mapVal.equals(dataAVE.getValue())) {
	    String pathName = lookupPath(ns);
	    
	    if (pathName!=null) 
		sendImage(pathName, packet.sNS);
	    else
		printStatus("Picture not found");
	}
	
        // coordinates
        else if (coordsVal.equals(dataAVE.getValue())) {
	    String pathName = lookupPath(ns);
	    
	    if (pathName!=null) 			
		sendCoordinates(parentPath(pathName), 
				lastPathValue(pathName),
				packet.sNS);
	    else
		printStatus("Coordinates not found");
	    
	} 

	else { printStatus("Bad Location Request"); }
    }
    
    
    /**
     * Sends the imageName image from the specified place within
     * the directory hierarchy to the correct destination.
     */
    private void sendImage(String directory, NameSpecifier clientNS)
    {
	byte[] data; // To store data for the packet (the image).
	
	// Find the file and lastModified time.
	File imageFile = new File(directory + 
				  File.separator + 
				  imageName);
	
	data = new byte[(int) imageFile.length()];


	try {  
	    // Read the file.
	    FileInputStream in = new FileInputStream(imageFile);
	    in.read(data);
	    in.close();
	} catch (Exception e) {
	    printStatus("Couldn't read imagefile.");
	    return;
	}  
	
 printStatus("Retreived the image, starting to send it");
	sendMessage(sourceNS, clientNS, data);
    }
    

    /**
     * Sends the imageName image from the specified place within
     * the directory hierarchy to the correct destination.
     */
    private void sendCoordinates(String directory, String lastValue,
				 NameSpecifier clientNS)
    {		
	// Get the full configuration file name for the coordinates
	File confFile = new File(directory + 
				 File.separator + 
				 configName);
	
	// Read the configuration file.
	try {
	    conf = new INSConfig(confFile);
	} catch (Exception e) {
	    printStatus("Could not retrieve the coordinates from" +
			confFile + ": " +
			e.getMessage());
	}		
	
	// Retrieve coordinates from the file
	String posS = null;
	if (conf != null) {
	    posS = conf.getValue(lastValue);
	    if (posS==null && lastValue.startsWith("0"))
		posS= conf.getValue(lastValue.substring(1));
	}

	String xPos = "-1";
	String yPos = "-1";

	if (posS !=null) {
	    StringTokenizer st = new StringTokenizer(posS, "\t ,");
	    
	    if (st.hasMoreTokens()) 
		xPos = st.nextToken();

	    boolean tokensDone = false;
	    while ((st.hasMoreTokens()) && (!tokensDone)) {
		yPos = st.nextToken();
		tokensDone = (!yPos.equals(""));
	    }
	}
	
	// if (DEBUG >=2) 
	    printStatus("Found at (" + xPos + " " + yPos + ") ");
	
	// Create a name specifier containing the x,y position
	NameSpecifier realDestNS = new NameSpecifier(clientNS);
		
	// Send the packet
	// note: for now packet MUST be toAll since
	//   there are all different Floorplans (mit, mit/ne43, mit/ne43/5)
	//   if use toAny, map will go to any of them
	// Specifically, if packet is destined to mit/ne43/5,
	// the way name-lookup works will match to all three above
	// since more specific name matches all less-specific names in the 
	// routeTree and accumulate all entries while going down to the
	// leaf nodes... [see alg. in paper -> returns UNION with root]

	sendMessage(sourceNS, realDestNS, true, 
		    (xPos+" "+yPos).getBytes());
	
    }


    public void printStatus(String string) {
	super.printStatus(string);
    }
    

    /**
     * Creates a path name to find the location specified in the 
     * name specifier. "Flattens out" the hierarchy, assuming that
     * there is only a single location specified.
     * Assumes the top level attribute for the location is "organization"
     * Given: [location=floor[room=504]]
     * It will produce: 
     *     locationDir + /location/floor5/room/504
     */
    String lookupPath(NameSpecifier ns) 
    {
	// assumes the top-level attribute is "location"
	AVelement ave = ns.getAVelement(locationAttr);
	String path = locationDir;
	
	// goes down the tree from domain, assuming one child per level
	while (ave!= null)  {
	    path += File.separator + ave.getAttribute() +
		File.separator + ave.getValue();
	    
	    Enumeration e = ave.getAVelements();
	    if (e.hasMoreElements()) 
		ave = (AVelement) e.nextElement();
	    else 
		ave = null;
	}
	
	return path;
    }

    /**
     * Given a path, this will prune off the last attribute/value pair,
     * essentially by erasing after the last two slashes in the path.
     */
    private String parentPath(String path) 
    {
	int index;
	if (path == null) return null;
	
	if ((index = path.lastIndexOf(File.separator)) == -1)
	    return null;

	if ((index = path.lastIndexOf(File.separator,index-1)) == -1)
	    return null;
	
	return path.substring(0,index);
    }

    private String lastPathValue(String path) 
    {
	int index;

	if (path == null) return null;
	
	if ((index = path.lastIndexOf(File.separator)) == -1)
	    return null;
	
	if (index+1 < path.length())
	    return path.substring(index+1);
	else
	    return null;
    }

    public static void printHelpMessage()
    {
	System.out.println("Syntax:");
	System.out.println("\tjava locationserver.LocationServer "+
			   "[-d DSR name] [-p specific-INR-to-peer-with "+
			   "its-portnum]");
	System.out.println("\nExamples:\n\tjava locationserver.LocationServer"+
			   " -d 127.0.0.1");
	System.out.println("\tjava locationserver.LocationServer "+
			   "-p fenway.lcs.mit.edu 1234");
	System.exit(-1);

    }


    public static void main(String [] args)
    {
	String dsrname = null;
	String inrname = null;
	int inrport = -1;
	
	// process command line

	for (int argnum = 0; argnum < args.length; argnum++)
	{
	    String arg = args[argnum];
	    if (arg == null) continue;

	    if (arg.equals("-d"))
	    {
		argnum++;
		if (argnum>=args.length) 
		    printHelpMessage();
	     
		dsrname = args[argnum];
	    }

	    else if (arg.equals("-p"))
	    {
		argnum+=2;
		if (argnum>=args.length)
		    printHelpMessage();

		inrname = args[argnum-1];
		try { inrport = Integer.parseInt(args[argnum]); }
		catch (NumberFormatException e) { inrport = -1; }		
	    } else 
		printHelpMessage();
	}

	try { 
	    if (inrport>-1)
		new LocationServer(inrname, inrport);
	    else if (dsrname != null)
		new LocationServer(dsrname);
	    else
		new LocationServer();
	}
	catch (Exception e) { e.printStackTrace(); }
    }

}



class TCPServer implements Runnable
{
    // VARIABLES
    ServerSocket socket = null;
    boolean listening = true;
    LocationServer locServer;

    TCPServer(ServerSocket s, LocationServer ls)
    {
	socket = s;
	locServer = ls;
    }

    public void run()
    {
	System.out.print("LocationServer: TCP binding listens to port # ");
	
	System.out.println(socket.getLocalPort());
	
	while (listening)
	    {
		try {
		    new TCPServerThread(socket.accept(), locServer).start();
		} catch (IOException e) {;}
	    }
	try { socket.close(); }
	catch (IOException e) {;}
    }
}


class TCPServerThread extends Thread {
    private Socket socket = null;
    private LocationServer locServer;
    private BufferedOutputStream out;
    private BufferedReader in;
    
    public TCPServerThread(Socket s, LocationServer ls) 
    {
        super("TCPServerThread");
        socket = s;
	locServer = ls;
    }

    public void run() 
    {
	if (LocationServer.DEBUG >=2) 
	    locServer.printStatus("Serving a client ...");
        try {
            out = new BufferedOutputStream(socket.getOutputStream());
            in = new BufferedReader
		(new InputStreamReader(socket.getInputStream()));
	    
            String inputLine;
            if ((inputLine = in.readLine()) != null) {
		if (LocationServer.DEBUG >=2) 
		    locServer.printStatus("... client requests: " + inputLine);
		processRequest(inputLine);
            }
	    
	    if (LocationServer.DEBUG >=2) 
		locServer.printStatus("Closing connection.");
	    out.close();
            in.close();
            socket.close();
	    
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /** 
     * Process a request
     */
    private void processRequest(String request) {
	// Check to make sure this is actually a packet 
	// we want to receive.
	System.out.println("Processing the request");


	AVelement ave, ave_service;
	NameSpecifier ns = new NameSpecifier(request);
	
	// Make sure [service=location]
	ave_service = ns.getAVelement(LocationServer.serviceAttr);
	 System.out.println(ave_service.getValue());
	 System.out.println(LocationServer.locationAttr);


	 ////???????
	 
 	// Make sure [service=location]
	/*ave_service = packet.dNS.getAVelement(serviceAttr);
	if (! ave_service.getValue().equals(locationVal))
	    return;
    
	// Make sure [entity=server]
	ave = packet.dNS.getAVelement(entityAttr);
	if (! ave.getValue().equals(serverVal))
	    return;
		
		 */


if (!ave_service.getValue().toString().equals(LocationServer.locationAttr.toString())){
		 System.out.println("Request test1 faild");


	    return;

	} 

	 System.out.println("Request test1 passed");

	// Make sure [entity=server]
	ave = ns.getAVelement(LocationServer.entityAttr);
	 System.out.println(ave.toString());
	 System.out.println(LocationServer.serverVal.toString());

	if (! ave.getValue().equals(LocationServer.serverVal)) {
	 System.out.println("Request test2 fail");

	    return;

	}
	
	// asume they want the map (late-binding is fine for coordinates)
	String pathName = locServer.lookupPath(ns);
	
	if (pathName!=null)  {
	System.out.println("Starting to send the file");
	    sendImage(pathName);
	}
	else
	    locServer.printStatus("Picture not found");

	// don't bother supporting coordinates with early binding
    }

   /**
     * Sends the imageName image from the specified place within
     * the directory hierarchy to the correct destination.
     */
    private void sendImage(String directory)
    {
	byte[] data; // To store data for the packet (the image).
	
	// Find the file and lastModified time.
	File imageFile = new File(directory + 
				  File.separator + 
				  locServer.imageName);


	int len = (int) imageFile.length();
	data = new byte[len+4];
	data[0] = (byte)(len & 0xff);       data[1] = (byte)((len>>8)  & 0xff);
	data[2] = (byte)((len>>16) & 0xff); data[3] = (byte)((len>>24) & 0xff);
	locServer.printStatus(" Length of the image file  " +  len);
	try {
	    // Read the file.
	    FileInputStream in = new FileInputStream(imageFile);
	    in.read(data,4,data.length-4);
	    in.close();
	} catch (Exception e) {
	    locServer.printStatus("Couldn't read imagefile.");
	    return;
	}
	  	locServer.printStatus("Read the image trying to send it");
	// send the image to client
	try {
	    if (LocationServer.DEBUG >=1) 
	        locServer.printStatus("Sending image of " + directory);

		locServer.printStatus("-------------------   "+ data.length);
	    out.write(data, 0, data.length);
	    out.flush();
	    if (LocationServer.DEBUG >=1) 
		locServer.printStatus("... completed!");
	} catch (IOException e) {
	    locServer.printStatus(e.getMessage());
	}

    }
}





