package secureprinter;

import java.io.*;
import java.net.*;
import java.util.*;

import ins.api.*;
import ins.namespace.*;
import ins.inr.Packet;
import ins.inr.HostRecord;

/** 
 * This class implements the LPR Client application, which 
 * is the graphical viewer in the floorplan application that
 * interacts with the LPRGateway.
 */
public class LPRClient extends SecureApp
{

    static String keyprefix = "clikeyfile";

    // VARIABLES

    String printer;             // the printer we're a client for
    String location;            // the location we're a client for
    NameSpecifier sourceNS;     // source NS for sent packets
    NameSpecifier sourceLocNS;  // source NS for submit-location sent packets
    NameSpecifier checkSourceNS;  // dest NS for all check packets
    NameSpecifier submitSourceNS; // dest NS for submit-name packets
    NameSpecifier submitLocNS;  // dest NS for submit-location packets
    NameSpecifier removeSourceNS; // dest NS for all remove packets
    NameSpecifier destNS, destLocNS;
    LPRGUI gui;                 // the graphical user interface

    // So we don't have to always remake these
    static Attribute serviceAttr = new Attribute("service");
    static Attribute entityAttr = new Attribute("entity");
    static Attribute actionAttr = new Attribute("action");
    static Attribute locationAttr = new Attribute("location");
    static Attribute nameAttr = new Attribute("name");
    static Attribute jobAttr = new Attribute("job");
    static Value checkValue = new Value("check");
    static Value submitValue = new Value("submit");
    static Value removeValue = new Value("remove");
    static Value clientValue = new Value("client");
    static Value printerValue = new Value("printer");

    static int jobnum=1;        // job number to avoid resubmitting the 
                                // same job inside the gateway
                                // Static because unique over hostname

    // CONSTRUCTORS
    
    public LPRClient(String [] args) throws Exception
    {
	super(null, false, keyprefix);
	init(args);
    }

    public LPRClient(String dsrname, String [] args)
	throws Exception
    {
	super(null, false, keyprefix, dsrname);
	init(args);
    }

    public LPRClient(String inrname, int inrport, String [] args)
	throws Exception
    {
	super(null, false, keyprefix, inrname, inrport);
	init(args);
    }


    /////////////////////


    /**
     * Initialization function called by the constructor.
     * @param args the arguments
     */	
    public void init(String [] args) throws Exception
    {
	printStatus("Initializing");
	
	NameSpecifier printerNS = new NameSpecifier(args[0]);

	AVelement locationAVE;
	String vspace=printerNS.getVspace();

	// Get the printer name
	AVelement iAVE1 = printerNS.getAVelement(entityAttr);
	AVelement iAVE2 = iAVE1.getAVelement(nameAttr);
	printer = iAVE2.getValue().toString();

	// Get the printer location
	locationAVE = printerNS.getAVelement(locationAttr);
	AVelement bottomAVE = getBottomAVelement(locationAVE);
	location = bottomAVE.getAttribute() + " " + bottomAVE.getValue();

	printStatus("printer is " + printer);

	// Setup the source NS for all packets
	sourceNS = new NameSpecifier("[service=printer]"+
				     "[entity=client" +
				     "[name=" + printer + "]]");
	sourceNS.setVspace(vspace);

	sourceLocNS = new NameSpecifier("[service=printer]"+
					"[entity=client]");
	sourceLocNS.addAVelement(locationAVE);
	sourceLocNS.setVspace(vspace);

	
	AVelement nAVE = new AVelement(new Attribute("node"),
				      new Value(localhost.getHostAddress()));
	nAVE.addAVelement(new AVelement(new Attribute("port"),
				       new Value(Integer.toString(port))));

	sourceNS.addAVelement(nAVE);
	sourceLocNS.addAVelement(nAVE);


	// Prototype destination NS
	destNS = new NameSpecifier("[service=printer][group=61337749927541422376238073610434650711341502519629251355895508262086379307371:72234469145070839385182367648059200400426877218784883639080522901986499135319]"+
				   "[entity=spooler" +
				   "[name=" + printer + "]]");
	destNS.setVspace(vspace);

	destLocNS = new NameSpecifier("[service=printer]"+
				   "[entity=spooler]");
	destLocNS.addAVelement(locationAVE);
	destLocNS.setVspace(vspace);

	// Destination for queue checks
	checkSourceNS = new NameSpecifier(sourceNS);
	AVelement entityAVE1 = checkSourceNS.getAVelement(entityAttr);
	entityAVE1.addAVelement(new AVelement(actionAttr, checkValue));
	checkSourceNS.addAVelement(new AVelement(opAttr, checkValue));
	
	// Destination for name job submissions
	submitSourceNS = new NameSpecifier(sourceNS);
	AVelement entityAVE2 = submitSourceNS.getAVelement(entityAttr);
	entityAVE2.addAVelement(new AVelement(actionAttr, submitValue));
	submitSourceNS.addAVelement(new AVelement(opAttr, submitValue));

	// Destination for location job submissions
	submitLocNS = new NameSpecifier("[service=printer]"+
					"[entity=client]");
	AVelement entityAVE3 = submitLocNS.getAVelement(entityAttr);
	entityAVE3.addAVelement(new AVelement(actionAttr, submitValue));
	submitLocNS.addAVelement(locationAVE);
	submitLocNS.setVspace(vspace);
	submitLocNS.addAVelement(new AVelement(opAttr, submitValue));

	// Destination for job removals
	removeSourceNS = new NameSpecifier(sourceNS);
	AVelement entityAVE4 = removeSourceNS.getAVelement(entityAttr);
	entityAVE4.addAVelement(new AVelement(actionAttr, removeValue));
	removeSourceNS.addAVelement(new AVelement(opAttr, removeValue));


	startAnnouncer(new NameSpecifier[]{sourceNS, sourceLocNS});

	// GUI
	gui = new LPRGUI(this, printer, location);
	
	gui.doUpdate();
    }


    /*
     * Requests are single threaded/blocking in the LPR Client,
     * which is "good enough," since it's all tied to a GUI
     * that is typically limited by the user
     */

    Object processLock = new Object();
    Packet receivedPacket = null;

    // this has been changed so that receivePacket performs handshaking     
    // public void receivePacket(Packet packet)
    public void respondToPacket(Packet packet)
    {
	synchronized (processLock) {
	    receivedPacket = packet;
	    processLock.notifyAll();
	}
    }

    /**
     * Contacts the LPRGateway with a data packet and waits for the first
     * received response. Uses the processLock to single thread the whol
     * process (i.e. as long as all communication goes through this, there
     * should be not conflicts)
     */
    public Packet contactGateway(NameSpecifier sourceNS, NameSpecifier destNS,
				 byte [] data, int wait, int maxtries)
    {
	Packet packet = null;
	int tries = 0;

	do {
	    tries++;
	    synchronized(processLock) {
		boolean keepGoing = true;
		
		receivedPacket = null;
		
		if (!sendMessage(sourceNS, destNS, data)) {
		    printStatus("could not send request to gateway");
		    keepGoing = false;
		} 
		
		try {
		    if (keepGoing) processLock.wait(wait);
		} catch (Exception e) {
		    keepGoing = false;
		}
		
		if (keepGoing) packet = receivedPacket;
		
		receivedPacket = null;
	    }
	} while (packet==null && tries<maxtries);
	
	return packet;
    }


    /**
     * Checks the queue of our printer.
     * @return an array of strings contain the print job status
     */
    public String[] checkQueue()
    {
	Packet p = contactGateway(checkSourceNS, destNS, new byte[0], 3000, 3);
	
	if (p == null) {
	    String [] errorMsg = new String[1];
	    errorMsg[0] = "Fail checking printer's queue!";
	    return (errorMsg);
	}

	// ADDED for GUI --NGF
	if (Arrays.equals(p.data, unAuthStr.getBytes())) {
	    String[] retval = new String[1];
	    retval[0] = unAuthStr;
	    return (retval);
	}
	// END ADDED CODE
	
	if (isToUs(p) && isCheckResponse(p)) {
	    String response = new String(p.data);
	    
	    StringTokenizer st = new StringTokenizer(response, "\n");
	    String[] retval = new String[st.countTokens()];
	    
	    for (int j = 0; j < retval.length; j++) {
		retval[j] = st.nextToken();
	    }
	    
	    return(retval);
	}
	else {
	    String [] errorMsg = new String[1];
	    errorMsg[0] = "Received response invalid";
	    return (errorMsg);
	}
    }

    /**
     * Checks if the packet is a response to a checkQueue.
     * @return true if it is, false otherwise
     */
    private boolean isCheckResponse(Packet packet)
    {
	AVelement serviceAVE;
	serviceAVE = packet.sNS.getAVelement(serviceAttr);
	if (! serviceAVE.getValue().equals(printerValue)) {
	    return false;
	}

	AVelement entityAVE = packet.sNS.getAVelement(entityAttr);	

	// Make sure it's to [action=check]
	AVelement actionAVE = entityAVE.getAVelement(actionAttr);
	if (actionAVE == null) return false;
	if (! actionAVE.getValue().equals(checkValue)) {
	    return false;
	}

	// It's ok, I guess!
	return true;
    }

    /**
     * Submits a job to our printer.
     * @param file the File to submit
     * @return the status of the submission
     */
    public String submitJob(File file, boolean toLocation)
    {
        HostRecord lprGateway;	// early-binding information
	Packet o;
	NameSpecifier jobSourceNS, submitNS;

	if (toLocation) {
	    jobSourceNS = submitLocNS;
	    submitNS = destLocNS;
	} else {
	    jobSourceNS = submitSourceNS;
	    submitNS = destNS;
	}

	// Get early-binding information of the printer
	EarlyBindRecord[] ebrset = getHostByiName(submitNS, false);
	for (int i=0; i<ebrset.length; i++) 
	{
	    lprGateway = ebrset[i].getHostRecord();
	    
	    // If early binding is avail., use it
	    if (lprGateway.getPortForTransport("TCP") > -1)
		return submitJobEB(file, lprGateway, submitNS.toString());
	}

	// Otherwise, use late binding
	FileInputStream f;
	try {
	    f = new FileInputStream(file);
	} catch (FileNotFoundException e) {
	    return("File not found.");
	}

	byte[] buf = new byte[32678];

	int len;
	try {
	    len = f.read(buf);
	} catch (IOException e) {
	    return("Problem reading from file: " + e.getMessage());
	}

	byte[] data = new byte[len+4];
	System.arraycopy(buf, 0, data, 4, len);
	data[0] = (byte)(jobnum & 0xff);
	data[1] = (byte)((jobnum>>8) & 0xff);
	data[2] = (byte)((jobnum>>16) & 0xff);
	data[3] = (byte)((jobnum>>24) & 0xff);
	jobnum++;

	// Submitting job
	Packet p = contactGateway(jobSourceNS, submitNS, data, 3000, 3);

	if (p!= null)
	    if ((isToUs(p) && isJobResponse(p)))
		return (new String(p.data));

	return("Error submitting job");
    }


    /**
     * Submits a job to our printer using early binding.
     * @param file the File to submit
     * @param hostRecord the early binding information
     * @return the status of the submission
     */
    public String submitJobEB(File file, HostRecord lprGateway, 
			      String ns)
    {
	String responseString = "";

	printStatus("Establishing TCP connection to LPRGateway...");
	
	Socket socket = null;
        DataOutputStream out = null;
        BufferedReader in = null;

	int port = lprGateway.getPortForTransport("TCP");
        try {
	    StringBuffer strbuf = new StringBuffer();
	    strbuf.append(lprGateway.address[0]);strbuf.append(".");
	    strbuf.append(lprGateway.address[1]);strbuf.append(".");
	    strbuf.append(lprGateway.address[2]);strbuf.append(".");
	    strbuf.append(lprGateway.address[3]);

	    // Establish TCP connection to LocationServer
            socket = new Socket(strbuf.toString(), port);
            out = new DataOutputStream(socket.getOutputStream());
            in = new BufferedReader(new InputStreamReader(
                                    socket.getInputStream()));

	    // Send a print request 
	    printStatus("Sending print request: " + ns.toString());
	    out.writeInt(ns.length());	
	    out.write(ns.getBytes());

	    // Send the bytes to be printed	
	    FileInputStream f;
	    f = new FileInputStream(file);

	    printStatus("Sending the print job...");

	    out.writeInt((int)file.length());

	    byte[] buf = new byte[4096];
	    int numRead = f.read(buf);
	    while (numRead != -1)
	    {
		out.write(buf, 0, numRead);
		numRead = f.read(buf);
	    }

	    printStatus("... completed!");

	    responseString = in.readLine();

	    // Close the connection and IO
	    f.close();
	    out.close();
	    in.close();
	    socket.close();

        } catch (UnknownHostException e) {
            printStatus("Don't know about LPRGateway's address");
            System.exit(1);
        } catch (IOException e) {
            System.err.println("I/O Error: " + e.getMessage());
            System.exit(1);
	} 

	System.out.println("returning response: "+responseString);
	return(responseString);
    }


    /**
     * Checks if the packet is a response to a submitJob.
     * @return true if it is, false otherwise
     */
    private boolean isJobResponse(Packet packet)
    {
	AVelement serviceAVE;
	serviceAVE = packet.sNS.getAVelement(serviceAttr);
	if (! serviceAVE.getValue().equals(printerValue)) {
	    return false;
	}

	AVelement entityAVE = packet.sNS.getAVelement(entityAttr);

	// Make sure it's to [action=submit]
	AVelement actionAVE;
	actionAVE = entityAVE.getAVelement(actionAttr);
	if (actionAVE == null) return false;
	if (! actionAVE.getValue().equals(submitValue)) {
	    return false;
	}

	// It's ok, I guess!
	return true;
    }

   
    
    /**
     * Submits a job to our printer.
     * @param job the job to remove
     * @return the status of the submission
     */
    public String removeJob(int job)
    {
	String jobString = Integer.toString(job);
	byte[] data = jobString.getBytes();

	Packet p = contactGateway(removeSourceNS, destNS, data, 3000, 3);

	if (isToUs(p) && isRemoveResponse(p)) {
	    String response = new String(p.data);

	    return(response);
	} else {
	    return(new String());
	}
    }

    /**
     * Checks if the packet is a response to a removeJob.
     * @return true if it is, false otherwise
     */
    private boolean isRemoveResponse(Packet packet)
    {
	AVelement serviceAVE;
	serviceAVE = packet.sNS.getAVelement(serviceAttr);
	if (! serviceAVE.getValue().equals(printerValue)) {
	    return false;
	}

	AVelement entityAVE = packet.sNS.getAVelement(entityAttr);

	// Make sure it's to [action=remove]
	AVelement actionAVE;
	actionAVE = entityAVE.getAVelement(actionAttr);
	if (actionAVE == null) return false;
	if (! actionAVE.getValue().equals(removeValue)) {
	    return false;
	}

	// It's ok, I guess!
	return true;
    }


    /**
     * Checks if the packet is to us.
     * @return true if it is, false otherwise
     */
    private boolean isToUs(Packet packet)
    {
	// Make sure it's to [service=printer]
	AVelement serviceAVE;
	serviceAVE = packet.dNS.getAVelement(serviceAttr);
	if (! serviceAVE.getValue().equals(printerValue)) {
	    return false;
	}

	// Make sure it's to [entity=client]
	AVelement entityAVE;
	entityAVE = packet.dNS.getAVelement(entityAttr);
	if (! entityAVE.getValue().equals(clientValue)) {
	    return false;
	}

	// Make sure it's to [name=<ourprinter>]
	// AVelement nameAVE;
	// nameAVE = entityAVE.getAVelement(nameAttr);
	// if (! nameAVE.getValue().equals(new Value(printer))) {
	//	    return false;
	// }

	// Guess it's to us
	return true;
    }

    private AVelement getBottomAVelement(AVelement a)
    {
	Enumeration e = a.getAVelements();

	while(e.hasMoreElements()) {
	    a = (AVelement)e.nextElement();
	    e = a.getAVelements();
	}

	return(a);
    }

    public static void printHelpMessage() 
    {
	System.out.println("Syntax:");
	System.out.println("\tjava printer.LPRClient [talk to what printer] "+
			   "[-d DSR name] [-p specific-INR-to-peer-with "+
			   "its-portnum][-k keyfile-prefix]");
	System.out.println("\nExamples:\n\tjava printer.LPRClient "+
			   "[service=printer][entity=spooler[name=turkey]]"+
			   "[location=floor5[room=516]][vspace=wind]"+
			   " -d localhost");
	System.out.println("\nExamples:\n\tjava printer.LPRClient "+
			   "[service=printer][entity=spooler[name=turkey]]"+
			   "[location=floor5[room=516]][vspace=wind] "+
			   "-p fenway.lcs.mit.edu 1234");
	System.exit(-1);
    }


    public static void main(String args[])
    {

	String printerString=("[service=printer][entity=spooler[name=turkey]]"+
			      "[location=floor5[room=517]][vspace=wind]");
	String dsrname = null;
	String inrname = null;
	int inrport = -1;
	
	// process command line

	for (int argnum = 0; argnum < args.length; argnum++)
	{
	    String arg = args[argnum];
	    if (arg == null) continue;

	    if (argnum == 0 && !arg.startsWith("-")) {
		printerString = arg;
	    } 

	    else if (arg.equals("-d"))
	    {
		argnum++;
		if (argnum>=args.length) 
		    printHelpMessage();
	     
		dsrname = args[argnum];
	    }

	    else if (arg.equals("-k"))
	    {
		argnum++;
		if (argnum>=args.length) 
		    printHelpMessage();
	     
		keyprefix = args[argnum];
	    }

	    else if (arg.equals("-p"))
	    {
		argnum+=2;
		if (argnum>=args.length)
		    printHelpMessage();

		inrname = args[argnum-1];
		try { inrport = Integer.parseInt(args[argnum]); }
		catch (NumberFormatException e) { inrport = -1; }		
	    } else 
		printHelpMessage();
	}

	try { 
	    String [] appargs = new String[]{ printerString };

	    if (inrport>-1)
		new LPRClient(inrname, inrport, appargs);
	    else if (dsrname != null)
		new LPRClient(dsrname, appargs);
	    else
		new LPRClient(appargs);
	}
	catch (Exception e) { e.printStackTrace();}
    }


}
