package secureprinter;

import ins.api.*;
import ins.inr.*;
import ins.namespace.*;

import java.io.*;
import java.util.*;
import java.net.*;

/** 
 * This class implements the LPR Gateway application, which is the
 * INS proxy to the printers.
 * 
 * It periodically advertises the available printers and their loads
 * to the system and interprets all the submitting/checking/removing.
 *
 * This also uses the TCPServer class for its early binding.
 */

public final class LPRGateway extends SecureApp
    implements PreNameAnnouncer
{

    public static final String keyprefix = "servkeyfile";
    public static final String aclfile = "aclfile";

    // VARIABLES
    final static int DEBUG = 123;		// 0 = no debug msg
    				                // higher # = more msgs

    static Attribute serviceAttr = new Attribute("service");
    static Attribute entityAttr = new Attribute("entity");
    static Attribute nameAttr = new Attribute("name");
    static Attribute locationAttr = new Attribute("location");
    static Attribute nodeAttr = new Attribute("node");
    static Attribute actionAttr = new Attribute("action");
    static Value printerVal = new Value("printer");
    static Value spoolerVal = new Value("spooler");
    static Value submitVal = new Value("submit");
    static Value checkVal = new Value("check");
    static Value removeVal = new Value("remove");

    // for determining printer metrics
    StringIntList printerMetricBoundaries;
    int [] printerMetrics;

    INSConfig conf;     // stores the configuration
    Vector printers;        // Vector of inner-class Printer

    Hashtable submittedjobs; // remembers which jobs have already been 
    // submitted to avoid resending on loss of an acknowledgement

    

    ServerSocket ebSocket;  // early-binding socket;

    // CONSTRUCTORS
	
    public LPRGateway(String aclfile, String keyprefix) throws Exception
    {
	super(aclfile, true, keyprefix);
	initgateway();
    }

    public LPRGateway(String dsrname, String aclfile, String keyprefix)
	throws Exception
    {
	super(aclfile, true, keyprefix, dsrname);
	initgateway();
    }

    public LPRGateway(String inrname, int inrport,
		      String aclfile, String keyprefix) throws Exception
    {
	super(aclfile, true, keyprefix, inrname, inrport);
	initgateway();
    }



    ///////////

    void initgateway() 
    {
	printStatus("Initializing.");

	// Read the configuration file.
	try {
	    conf = new INSConfig("LPRGateway");
	} catch (Exception e) {
	    printStatus("Problem with configuration:\n" +
			e.getMessage() +
			".  Exiting.");
	    return;
	}

	// Get the root location that's appended to everything in this file
	String root = conf.getValue("root");

	// If there's no root supplied, append nothing.
	if (root == null) {
	    root = new String();
	}

	// Creating early binding information
	HostRecord hr = new HostRecord();
	hr.UDPport = (short)port;
	try {
	    ebSocket = new ServerSocket(0);
	    hr.addTransport("TCP", (short)ebSocket.getLocalPort());
	} catch (IOException e) {;}
	
	
	submittedjobs = new Hashtable();
	printers = new Vector();

	Vector printerNSes = new Vector();

	printerMetricBoundaries = new StringIntList();

	// Get the printers
	for (Enumeration e = conf.values("printer"); e.hasMoreElements(); ) {
	    String s = (String)e.nextElement();

	    // The first word is the printer name, the second is the location
	    int space1 = s.indexOf(' ');
	    int space2 = s.indexOf(' ',space1+1);

	    if (space1 == -1 || space2 == -1) {
		continue;
	    }
	    
	    String name = s.substring(0, space1);
	    String loc = s.substring(space1 + 1, space2);
	    String vspaces = s.substring(space2+1);

	    StringTokenizer st = new StringTokenizer(vspaces, ", \n\t\r\f");

	    Printer p = new Printer(name, (root==null? loc: root + "/" + loc));

	    printers.addElement(p);

	    // Announce the printer
	    
	    NameSpecifier printerNS = 
		new NameSpecifier("[service=printer][entity=spooler" +
				  "[name=" +p.getName() + "]]"+
				  "[floorplanshow=true]");
	    printerNS.addAVelement(p.getLocation());
	    

	    if (!st.hasMoreTokens()) {
		printerNSes.addElement(printerNS);
		p.setNS(printerNS);
	    } else {
		String vspace = st.nextToken();
		printerNS.setVspace(vspace);
		printerNSes.addElement(printerNS);
		
		p.setNS(printerNS);
		printStatus("Serving printer " + name + " at " + loc +
			    "in vspace "+vspace);
	    }
	    while (st.hasMoreTokens())
	    {
		String vspace = st.nextToken();
		printStatus("\t\t vspace "+vspace);
		printerNS = new NameSpecifier(printerNS);
		printerNS.setVspace(vspace);
		printerNSes.addElement(printerNS);
	    }
	    
	    // put boundary on list
	    printerMetricBoundaries.appendElement
		(new StringInt(name,printerNSes.size()));
	}
	
	NameSpecifier [] nsArray = new NameSpecifier[printerNSes.size()];
	printerNSes.copyInto(nsArray);
	printerMetrics = new int[printerNSes.size()];
	determinePrinterMetrics();

	startAnnouncer(nsArray, printerMetrics, 40000, hr);
	setPreNameAnnouncer(this); // yes this should be after the prev. line
	
	// Evoke the TCP server thread
	new Thread(new TCPServer(ebSocket, this)).start();
    }	
        

    /**
     * Called whenever a packet is received
     *
     * Expected:
     *   Source NameSpecifier: [service=printer][entity=client[action=x]]...
     *   Destination NameSpecifier: [service=printer] [entity=spooler
     *                       then [name=printername]] under entity
     *                          or [location=location] at base level
     *
     * The [action=x] ideally should either be in the destination or not
     * in the name-specifier at all. But, it is not in the destination,
     * since we are not advertising [service=printer[action=submit]...]
     *        [action=check] and [action=remove] versions of our name.
     * (And since there is no provision for advertising [action=*],
     *  particularly with Andrew Lau's hacked nametrees).
     * This probably should go in the packet's data, but oh well.
     */
    public void respondToPacket(Packet packet)
//      public void receivePacket(Packet packet)
    {
	printStatus("Received: " + packet);
	
	NameSpecifier dNS = packet.dNS;
	
	// Make sure it's to [service=printer]
	AVelement serviceAVE = dNS.getAVelement(serviceAttr);
	if (!serviceAVE.getValue().equals(printerVal))
	    return;
	
	// Make sure it's to [entity=spooler]
	AVelement entityAVE = dNS.getAVelement(entityAttr);
	if (! entityAVE.getValue().equals(spoolerVal))
	    return;
	
	// Get the printer name
	AVelement nameAVE = entityAVE.getAVelement(nameAttr);
	String name;
	if (nameAVE != null) {
	    name = nameAVE.getValue().toString();
	} else {
	    // find it by location
	    AVelement locationAVE = 
		dNS.getAVelement(locationAttr);
	    
	    name = getBestPrinter(locationAVE);
	}
		
	// See what the action is. 
	// Valid actions are:
	//   submit = submit a job to the queue
	//   check  = check the status of the queue
	//   remove = remove a job from the queue
	AVelement EentityAVE = packet.sNS.getAVelement(entityAttr);
	if (EentityAVE == null)
	    return;

	AVelement actionAVE = EentityAVE.getAVelement(actionAttr);

	/////////// <-- this is all too subtle here, we're picking off
	//  the [action=x] from the name so that we can send to the return
	// address
	EentityAVE.removeAVelement(actionAVE);
	Value action = actionAVE.getValue();
	
	if (action.equals(submitVal)) {
	    
	    int jobnum=0;

	    if (packet.data.length>=4)
		jobnum = ((int)packet.data[0] | 
			  ((int)packet.data[1]<<8) | 
			  ((int)packet.data[2]<<16) |
			  ((int)packet.data[3]<<24));
			    
	    if (DEBUG >= 1) 
		printStatus("Submitting print job " + jobnum + 
			" to " + name + ".");

	    submit(name, packet.data, packet.sNS, 
		   (jobnum==0? null: new Integer(jobnum)));

	} else	if (action.equals(checkVal)) {
	    if (DEBUG >= 2) 
		printStatus("Checking print queue of " + name + ".");
	    check(name, packet.sNS);

	} else	if (action.equals(removeVal)) {
	    if (DEBUG >= 1) 
		printStatus("Removing job from print queue of " + name + ".");
	    remove(name, packet.data, packet.sNS);

	} else {
	    if (DEBUG >= 1) 
		printStatus("Unknown action " + action);
	}
    }

    

    private void submit(String name, byte[] job, NameSpecifier fromNS, 
			Object jobnum)
    {
	byte[] response = null;

	// For a submission, we want to reply directly to the node that
	// submitted it.

	// For now we'll just send back the printer name, but may want to
	// send the location as well.

	// Has the job already been submitted? (and we just lost the ACK?)
	// submittedjobs is a Hashtable of the job numbers and expiration
	//   times, to check if in fact the job has been submitted.
	Object timeData = null;
	Date now = new Date();

	if (jobnum != null) {
	    timeData = submittedjobs.get(jobnum);
	    
	    // if there is an entry, check if it has expired
	    if ( (timeData != null) && (now.after((Date)timeData)) ) {
		submittedjobs.remove(jobnum); // remove it if it has
		timeData = null;
	    }
	}

	// so now, (timeData == null) if there was no entry (never submitted)
	//       or if the entry expired (submitted too long ago/wrap-around)

	if (timeData == null) {
	    if (jobnum != null) 
		submittedjobs.put(jobnum, new Date(now.getTime()+30000));
	    
	    Runtime rt = Runtime.getRuntime();
	    
	    Process proc = null;
	    try {
		proc = rt.exec("lpr -P" + name);
		
		OutputStream os = proc.getOutputStream();
		
		os.write(job);
		
		os.close();
	    } catch (IOException e) {
		printStatus("Couldn't exec lpr");
		response = (new String("Couldn't exec lpr")).getBytes();
	    }
	    
	    if (proc != null) {
		try {
		    response = getProcessOutput(proc, 64000);
		} catch (IOException e) {
		    String responseString = "Problem reading from lpr to " + name;
		    response = responseString.getBytes();
		}
	    }
	    
	    if (response.length == 0) {
		String responseString = "Job submitted to " + name + "."; 
		response = responseString.getBytes();
	    }

	} else {
	    String responseString = "Job " + jobnum.toString() + 
		" already submitted to " + name + ".";
	    response = responseString.getBytes();
	}
	
	NameSpecifier ourNS = new NameSpecifier("[service=printer" +	
						"[entity=spooler"+
						"[action=submit]"+
						"[name=" + name + "]]");

	sendMessage(ourNS, fromNS, response);
    }


    private void check(String name, NameSpecifier fromNS)
    {
	printStatus("----RUNNING CHECK("+name+", "+fromNS.toString()+");");

	Runtime rt = Runtime.getRuntime();

	Process proc;
	try {
	    proc = rt.exec("lpq -P" + name);
	} catch (IOException e) {
	    printStatus("Couldn't exec lpq");
	    System.out.println("mal...");
	    return;
	}

	byte[] data;
	try {
	    data = getProcessOutput(proc, 64000);
	} catch (IOException e) {
	    data = (new String("Problem reading from lpq")).getBytes();
	}

	System.out.println(data);

	NameSpecifier ourNS = new NameSpecifier("[service=printer]" +
						"[entity=spooler"+
						"[name=" + name + "]" +
						"[action=check]]");

	sendMessage(ourNS, fromNS, data);
    }


    private void remove(String name, byte[] jobByte, NameSpecifier fromNS)
    {
	byte[] response = null;

	// For a removal, we want to reply directly to the node that
	// submitted it.

	// For now we'll just send back the printer name, but may want to
	// send the location as well.

	String jobString = new String(jobByte);
	// String -> int -> String conversion is for security (ha!)
	int job = Integer.parseInt(jobString);

	Runtime rt = Runtime.getRuntime();

	Process proc = null;
	try {
	    proc = rt.exec("lprm -P" + name + " " + job);
	} catch (IOException e) {
	    printStatus("Couldn't exec lprm");
	    response = (new String("Couldn't exec lprm")).getBytes();
	}

	if (proc != null) {
	    try {
		response = getProcessOutput(proc, 64000);
	    } catch (IOException e) {
		response = (new String("Problem reading from lpr")).getBytes();
	    }
	}

	NameSpecifier ourNS = new NameSpecifier("[service=printer]" +
						"[entity=spooler"+
						"[name=" + name + "]" +
						"[action=remove]]");

	sendMessage(ourNS, fromNS, response);
    }


    public void printStatus(String str)
    {
	super.printStatus(str);
    }


    static byte[] getProcessOutput(Process proc, int maxbuf) 
	throws IOException
    {
	InputStream is = proc.getInputStream();
	
	byte[] buf = new byte[maxbuf];
	
	int len, offset = 0;
	while (true) {
	    len = is.read(buf, offset, 1024);

	    if (len == -1) {
		break;
	    } else {
		offset += len;
	    }
	}
	
	is.close();

	byte[] data = new byte[offset];
	System.arraycopy(buf,0,data,0,offset);

	return(data);
    }
    

    public static void printHelpMessage() 
    {
	System.out.println("Syntax:");
	System.out.println("\tjava printer.LPRGateway "+
			   "[-d DSR name] [-p specific-INR-to-peer-with "+
			   "its-portnum]");
	System.out.println("\nExamples:\n\tjava printer.LPRGateway "+
			   "-d localhost");
	System.out.println("\tjava printer.LPRGateway -p "+
			   "fenway.lcs.mit.edu 1234");
	System.exit(-1);
    }


    public static void main(String args[])
    {
	String dsrname = null;
	String inrname = null;
	int inrport = -1;
	
	// process command line

	for (int argnum = 0; argnum < args.length; argnum++)
	{
	    String arg = args[argnum];
	    if (arg == null) continue;

	    if (arg.equals("-d"))
	    {
		argnum++;
		if (argnum>=args.length) 
		    printHelpMessage();
	     
		dsrname = args[argnum];
	    }

	    else if (arg.equals("-p"))
	    {
		argnum+=2;
		if (argnum>=args.length)
		    printHelpMessage();

		inrname = args[argnum-1];
		try { inrport = Integer.parseInt(args[argnum]); }
		catch (NumberFormatException e) { inrport = -1; }		
	    } else 
		printHelpMessage();
	}

	try { 
	    if (inrport>-1)
		new LPRGateway(inrname, inrport, aclfile, keyprefix);
	    else if (dsrname != null)
		new LPRGateway(dsrname, aclfile, keyprefix);
	    else
		new LPRGateway(aclfile, keyprefix);
	}
	catch (Exception e) { e.printStackTrace();}
    }


    /** updates the printerMetrics list appropriately */
    void determinePrinterMetrics()
    {
	StringInt si = printerMetricBoundaries.nextElement();
	if (si == null) return;

	for (int lastnum = 0; lastnum < printerMetrics.length; )
	{
	    int metric = getPrinterMetric(si.s);
	    System.out.println("metric for "+si.s+" is "+metric);

	    for (int i = lastnum; i < si.i; i++)
		printerMetrics[i] = metric;

	    lastnum = si.i;
	    si = si.nextElement();		
	    if (si==null) break;
	}
    }


    static int getPrinterMetric(String name)
    {
	Runtime rt = Runtime.getRuntime();
	if (DEBUG >=2)
	   System.out.println("PRINTER Check: "+name);
	

	Process proc;

	try {
	    proc = rt.exec("lpq -P" + name);
	} catch (IOException e) {
	    System.out.println("Couldn't exec lpq");
	    return Integer.MAX_VALUE; 
	}

	byte[] data;
	try {
	    data = getProcessOutput(proc, 64000);
	} catch (IOException e) {
	    data = (new String("Problem reading from lpq")).getBytes();
	}

	ByteArrayInputStream bais = new ByteArrayInputStream(data);
	InputStreamReader ir = new InputStreamReader(bais);
	BufferedReader br = new BufferedReader(ir);
	
	String firstLine;
	try {
	    firstLine = br.readLine();
	} catch (Exception e) {
	    System.out.println("Error reading first line");
	    return(Integer.MAX_VALUE);
	}

	int metric;
	if (firstLine == null) { // no message in the printer status
	    metric = Integer.MAX_VALUE;
	} else if (firstLine.equals("no entries")) {
	    metric = 0;
	} else if (firstLine.startsWith("Printer Error")) {
	    metric = 10000;
	} else if (firstLine.startsWith(name + "@")) {
	    // This is a reflected queue, for example see lpq -Pset

	    int leadingSpace = firstLine.indexOf(' ');
	    int trailingSpace = firstLine.indexOf(' ', leadingSpace + 1);

	    String numJobsString = firstLine.substring(leadingSpace + 1,
						       trailingSpace).trim();

	    int numJobs = 0;
	    try {
		numJobs = Integer.parseInt(numJobsString);
	    } catch (NumberFormatException e) { return Integer.MAX_VALUE; }

	    metric = numJobs * 10;
	} else {
	    // This is a normal queue, for example see lpq -Pgi

	    // Seek to the queue list, which is immediately preceeded
	    // by a line which starts with "Rank  Owner "...

	    while (true) {
		String line;
		try {
		    line = br.readLine();
		} catch (IOException e) {
		    System.out.println("Error seeking to queue list");
		    return Integer.MAX_VALUE;
		}

		if (line == null) break;;
		if (line.indexOf("Rank")>-1) {
		    break;
		}
	    }

	    // Count the number of entries in the queue after this.

	    int numJobs = 0;
	    while (true) {
		String line;
		try {
		    line = br.readLine();
		} catch (IOException e) {
		    System.out.println("Error seeking to queue list");
		    return Integer.MAX_VALUE;
		}
		
		if (line == null || line.equals("")) {
		    break;
		}

		numJobs++;
	    }

	    metric = numJobs * 10;
	}

	return(metric);
    }
    
    Vector getPrinters(AVelement l)
    {
	Vector ret = new Vector(); // Vector of String

	for (Enumeration e = printers.elements(); e.hasMoreElements(); ) {
	    Printer cur = (Printer)e.nextElement();

	    if (cur.getLocation().equals(l)) {
		ret.addElement(cur);
	    }
	}

	return(ret);
    }

    String getBestPrinter(AVelement l)
    {
	Vector v = getPrinters(l);

	if (v.size() == 0) return null;
	if (v.size() == 1) return ((Printer)v.elementAt(0)).getName();

	Printer best=null;
	int bestMetric = Integer.MAX_VALUE;

	for (Enumeration e = v.elements(); 
	     e.hasMoreElements(); )
	{
	    Printer p = (Printer)e.nextElement();   
	    int metric = p.getAppMetric(this);
	    
	    if (metric <= bestMetric) {
		best = p;
		bestMetric = metric;
	    }
	}
		
	return(best.getName());
    }


    public void preNameAnnounce()
    {
	System.out.println("modifying metrics");
	determinePrinterMetrics();
	changeAppMetrics(printerMetrics);
    }

    /******************************************/
    /* inner classes ... */

    final class StringIntList
    {
	StringInt si;

	boolean hasMoreElements()
	{
	    return (si != null);
	}

	StringInt nextElement()
	{
	    return si;
	}

	void appendElement(StringInt si)
	{
	    if (this.si==null) {
		this.si = si;
		return;
	    }

	    StringInt lastparent=this.si;
	    while (lastparent.next != null)
		lastparent = lastparent.next;
	
	    lastparent.next = si;
	}
    }

    class StringInt
    {
	String s;
	int i;

	StringInt next;

	boolean hasMoreElements()
	{
	    return (next != null);
	}

	StringInt nextElement()
	{
	    return next;
	}

	StringInt(String s, int i) 
	{
	    this.s = s;
	    this.i = i;
	    this.next = null;
	}

	StringInt(String s, int i, StringInt next)
	{
	    this.s = s;
	    this.i = i;
	    this.next = next;
	}
    }



    /** 
     * Printer class to contain a printer 
     */

    final class Printer
    {
	private String name;
	private AVelement location;
	private NameSpecifier ns;

	public Printer(String n, String l)
	{
	    name = n;

	    StringTokenizer st = new StringTokenizer(l, "/");
	    // Should do more error handling on st.nextToken() calls

	    // Get the first AVelement, since it's the root.
	    String ra = st.nextToken();
	    String rv = st.nextToken();
	    location = new AVelement(new Attribute(ra), new Value(rv));

	    // Get the rest of the AVelements
	    AVelement parent = location;
	    while (st.hasMoreTokens()) {
		String a = st.nextToken();
		String v = st.nextToken();
		AVelement ave = new AVelement(new Attribute(a), new Value(v));
		parent.addAVelement(ave);
		parent = ave;
	    }
	}

	public void setNS(NameSpecifier ns)
	{
	    this.ns = ns;
	}

	public int getAppMetric(Application app)
	{
	    if (ns == null)
		return Integer.MAX_VALUE;

	    EarlyBindRecord [] ebr = app.getHostByiName(ns, false);

	    if (ebr.length == 0)
		return Integer.MAX_VALUE;

	    return ebr[0].getMetric();
	}

	public String getName()
	{
	    return(name);
	}

	public AVelement getLocation()
	{
	    return(location);
	}
	
    }
}
