package secureprinter;

import ins.namespace.*;
import java.net.*;
import java.io.*;
import java.util.*;

/**
 * This is the TCP server for LPRGateway, and its early-binding requests.
 */

class TCPServer implements Runnable
{
    // VARIABLES
    ServerSocket socket = null;
    boolean listening = true;
    LPRGateway lprGateway;

    TCPServer(ServerSocket s, LPRGateway l)
    {
	socket = s;
	lprGateway = l;
    }

    public void run()
    {
	lprGateway.printStatus("TCP binding listens to port # ");
	lprGateway.printStatus(Integer.toString(socket.getLocalPort()));

	while (listening)
	{
	    try {
		new TCPServerThread(socket.accept(), lprGateway).start();
	    } catch (IOException e) {;}
	}
	try { socket.close(); }
	catch (IOException e) {;}
    }
}


class TCPServerThread extends Thread {
    private Socket socket = null;
    private LPRGateway lprGateway;
    private PrintWriter out;
    private DataInputStream in;

    public TCPServerThread(Socket s, LPRGateway l) {
        super("TCPServerThread");
        socket = s;
	lprGateway = l;
    }

    public void run() {
	if (LPRGateway.DEBUG >= 2) 
	   lprGateway.printStatus("Serving a client ...");
        try {
            out = new PrintWriter(socket.getOutputStream(), true);
            in = new DataInputStream(socket.getInputStream());

            String inputLine;
	    int nsLength = in.readInt();
	    byte[] nsBytes = new byte[nsLength];

	    in.readFully(nsBytes);
	    inputLine = new String(nsBytes);
	    if (LPRGateway.DEBUG >= 2) 
		lprGateway.printStatus("... client submits: " + inputLine);
	    processRequest(inputLine);

	    if (LPRGateway.DEBUG >= 2) 
		lprGateway.printStatus("Closing connection.");
	    out.close();
            in.close();
            socket.close();

        } catch (IOException e) {
	    lprGateway.printStatus(e.getMessage());
            e.printStackTrace();
        }
    }

    /** 
     * Process a request
     */
    private void processRequest(String request) {

	NameSpecifier ns = new NameSpecifier(request);

	// Make sure it's to [service=printer]
	AVelement serviceAVE = ns.getAVelement(LPRGateway.serviceAttr);
	if (! serviceAVE.getValue().equals(LPRGateway.printerVal))
	    return;

	// Make sure it's to [entity=spooler]
	AVelement entityAVE = ns.getAVelement(LPRGateway.entityAttr);
	if (! entityAVE.getValue().equals(LPRGateway.spoolerVal))
	    return;
	
	// Get the printer name
	AVelement nameAVE = entityAVE.getAVelement(LPRGateway.nameAttr);
	String name;
	if (nameAVE != null) {
	    name = nameAVE.getValue().toString();
	} else {
	    // find it by location
	    AVelement locationAVE;
	    locationAVE = ns.getAVelement(LPRGateway.locationAttr);

	    name = lprGateway.getBestPrinter(locationAVE);
	}

	// See what the action is. 
	// Valid actions are:
	//   submit = submit a job to the queue
	//   check  = check the status of the queue (NO TCP)
	//   remove = remove a job from the queue (NO TCP)

	//AVelement actionAVE = serviceAVE.getAVelement(LPRGateway.actionAttr);
	//	Value action = actionAVE.getValue();
	//if (action.equals(LPRGateway.submitVal)) {   

	    if (LPRGateway.DEBUG >= 1) 
		lprGateway.printStatus("Submitting print job to " + name + 
				       "...");
	    submit(name);
	
    }


    private void submit(String name)
    {
	byte[] response = null;
	String responseString = "";

	Runtime rt = Runtime.getRuntime();

	// Get bytes to be printer from client and submit the bytes
	// to the spooler	   
	Process proc = null;
	try {
	    proc = rt.exec("lpr -P" + name);
	    OutputStream os = proc.getOutputStream();

	    byte[] buffer = new byte[4096];

	    int total = in.readInt();
	    int numRead;
	    int current = 0;
	    while (current < total)
	    {
		numRead = in.read(buffer);
		os.write(buffer, 0, numRead);
		current += numRead;

		//lprGateway.printStatus("reading from client and sending to printer ...");
		//lprGateway.printStatus(Integer.toString(numRead));
	    }

	    os.close();
	    if (LPRGateway.DEBUG >= 1) 
		lprGateway.printStatus(new String("... " + total + " bytes submitted!"));

	} catch (IOException e) {
	    lprGateway.printStatus("Couldn't exec lpr");
	    lprGateway.printStatus(e.getMessage());
	    responseString = new String("Couldn't exec lpr");
	}

	// Get the response msg from spooler	    
	if (proc != null) {
	    try {
		response = lprGateway.getProcessOutput(proc, 64000);
		responseString = new String(response);
	    } catch (IOException e) {
		responseString = "Problem reading from lpr to " + name;
	    }
	}
	    
	if (response.length == 0) {
	    responseString = "Job submitted to " + name + "."; 
	}

	// Send response to client
	out.println(responseString);
	if (LPRGateway.DEBUG >= 2) 
	    lprGateway.printStatus("response is: " + responseString);

    }
}
