package tests;

import ins.api.*;
import ins.namespace.*;
import ins.inr.*;
import java.io.*;
import java.util.*;
import metadatagenerator.*;
import cryptix.util.core.Hex;
import cryptix.provider.md.*;

/** 
 * This application is meant to allows searching 
 * in a distributed file-system such as CFS.
 * <p> 
 * It allows to index some mp3 files and to query
 * for mp3 files.
 * <p>
 * When files are inserted, they create unique pairs of 
 * {name-specifier,host-record} with each host-record holding
 * the key necessary to extract a file. There is one limitation.
 * The uniqueness of a resource/document is determined by the 
 * {name-specifier,announcing-app} pair. Hence if one instance of 
 * SearchCFS advertises many documents under the SAME name-specifier,
 * they will be considered as the same document by the Twine network
 * @author Magdalena Balazinska
 */

public class SearchCFS extends TwineMultiApp {

    // Frequency must be greater then 10s otherwise
    // The name-announcer will set it to 15 seconds
    public static int DEFAULT_FREQUENCY = 4200000;
    public int frequency = DEFAULT_FREQUENCY;
    protected MpegMetaDataExtractor mpegExtractor = new MpegMetaDataExtractor();
    protected Vector announcers = new Vector();
    protected int bytesForSize = 2;

    //-----------------------------------------------
    /**  
     * Constructors
     */
     public SearchCFS(String dsrName) throws Exception {
	 super(dsrName);
	 time_out = 15000; // Long time_out for debugging only
	 this.enableSinglePeering();
     }

     public SearchCFS(String resolverName, int port) throws Exception {
	 super(resolverName,port);
	 time_out = 15000; // Long time_out for debugging only
	 this.enableSinglePeering();
     }


    public void receivePacket(Packet p) {
	System.out.println("/ Received Packet:\n"+p.toString());
    }

    /**
     * I didn't change the implementation of the NameAnnouncer, so
     * once it is started the frequence will not change
     */
    public void setFrequency(int freq) {
	frequency = freq;
	if (name_announcer != null)
	    setAnnouncementPeriod(freq);
    }

    //-----------------------------------------------
    /**  
     * Some util functions
     */
    public HostRecord makeOneHostRecord(byte[] data) {

	HostRecord hr = new HostRecord();
	hr.UDPport = (short)port;
	hr.addData(data);
	return hr;

    }

    public NameSpecifier createMetaData (String fileName) {
	return mpegExtractor.extractDescrFromFile(fileName);
    }

    /**
     * This tokenizer considers white spaces part of tokens
     * allowing file names to contain spaces.
     */
    public StreamTokenizer prepareSimpleTokenizer(FileReader fr) {

	StreamTokenizer tokenizer = new StreamTokenizer(fr); 
	tokenizer.resetSyntax();
	tokenizer.wordChars(0,Character.MAX_VALUE);
	tokenizer.whitespaceChars('\r','\r');
	tokenizer.whitespaceChars('\n','\n');
	tokenizer.eolIsSignificant(false); // It's just a separator
	return tokenizer;

    }

    /**
     */
    public void startAdvertising(Vector nsv, Vector hrv) {

	assert nsv.size() == hrv.size();
	NameSpecifier[] nsa = new NameSpecifier[nsv.size()];
	HostRecord[] hra = new HostRecord[hrv.size()];
	int[] metric = new int[nsv.size()];

        for ( int i = 0; i < nsv.size(); i++) {
		nsa[i] = (NameSpecifier)nsv.elementAt(i);
		hra[i] = (HostRecord)hrv.elementAt(i);
		metric[i] = 2;
	}

	startAnnouncer(nsa,metric,hra,frequency);

    }

    //-----------------------------------------------
    /**  
     * Indexing files identified with individual keys
     * The input to this function is the name of a file 
     * which contains a list of {fileName, key} pairs.
     * Currently only mp3 files are supported.
     */
    public void insertFiles (String fileList) {

	Vector nameSpecsV = new Vector();
	Vector hrsV = new Vector();

	try {
	    FileReader fr = new FileReader(fileList);
	    StreamTokenizer tokenizer = prepareSimpleTokenizer(fr);

	    while ( tokenizer.nextToken() != StreamTokenizer.TT_EOF ) {

		// Extracting the name of the file to index
		String fileName = tokenizer.sval;
		fileName = fileList.substring(0,fileList.lastIndexOf('/')+1) + fileName;
		tokenizer.nextToken();

		// The application data will contain the length of the 
		// key and the key itself
		byte[] key = tokenizer.sval.getBytes();
		byte[] data = new byte[key.length+bytesForSize];
		int index = 0;
		Conversion.insertIntLow16(data,index,key.length);
		index +=bytesForSize;
		System.arraycopy(key,0,data,index,key.length);

		// Preparing info for the indexed file
		HostRecord hr = makeOneHostRecord(data);
		NameSpecifier ns = createMetaData(fileName);
		if ( (ns != null) && (hr != null)) {
		    nameSpecsV.addElement(ns);
		    hrsV.addElement(hr);
		}
	    }
	}  catch ( FileNotFoundException e ) {
	    System.err.println("Index file " + fileList + " not found");
	    return;
	} catch ( IOException e ) {
	    System.err.println("IOException reading file " + fileList);
	    return;
	}
	startAdvertising(nameSpecsV,hrsV);

    } 



    //-----------------------------------------------
    /**  
     * Index a set of files specified by a directory and one 
     * global key (the key of the root node of the directory).
     * The application data associated with each file
     * will be the key for the directory and a relative path to 
     * the file
     */
    public void insertDir (String keyFile, String dir) {

	byte[] key = readKeyFromFile(keyFile);
	if (key == null)
	    return;

	Vector nameSpecsV = new Vector();
        Vector hrsV = new Vector();
	
	try {
	    File file = new File(dir);
	    processOneDir(file,dir,key,nameSpecsV,hrsV);
	} catch (Exception e) {
	    System.out.println("\nError examining directory.");
	    return;
	}

	startAdvertising(nameSpecsV,hrsV);

    }

    public void processOneDir(File file, String topLevelDir, byte[] topLevelKey, 
			      Vector nameSpecsV, Vector hrsV) throws Exception {

	FileFilter ff = null; // No file filtering
	File[] list = file.listFiles(ff); 
	for (int i = 0; i < list.length; i++) {
	    File f = list[i];
	    if ( f.isDirectory())
		processOneDir(f,topLevelDir,topLevelKey,nameSpecsV,hrsV);
	    else
		processOneFile(f,topLevelDir,topLevelKey,nameSpecsV,hrsV);
	}
    } 

    public void  processOneFile(File file, String topLevelDir, byte[] topLevelKey,
				Vector nameSpecsV, Vector hrsV) throws Exception {

	String fullFileName = file.toString();

	String fileName = fullFileName.substring(topLevelDir.length(),fullFileName.length());
	System.out.println("\nInserting file: " + fileName);

	byte[] fileKey = new byte[bytesForSize+topLevelKey.length+fileName.length()];
		
	// Specifying key size
	int index = 0;
	Conversion.insertIntLow16(fileKey,index,topLevelKey.length);
	index +=bytesForSize;

	// Inserting directory key
	System.arraycopy(topLevelKey,0,fileKey,index,topLevelKey.length);
	index += topLevelKey.length;

	// Inserting path to file
	System.arraycopy(fileName.getBytes(),0,fileKey,index,fileName.length());

	// Inserting into the system
	HostRecord hr = makeOneHostRecord(fileKey);
	NameSpecifier ns = createMetaData(fullFileName);
	if ( (ns != null) && ( hr != null) ) {
	    nameSpecsV.addElement(ns);
	    hrsV.addElement(hr);
	}
    }

    public byte[] readKeyFromFile(String keyFile) {

	// For all files in list
	byte[] key;
	int MAX_SIZE = 200;
	try {
	    FileReader fr = new FileReader(keyFile);
	    StreamTokenizer tokenizer = prepareSimpleTokenizer(fr);
	    tokenizer.nextToken();
	    key = tokenizer.sval.getBytes();
	} catch ( FileNotFoundException e ) {
	    System.err.println("Index file " + keyFile + " not found");
	    return null;
	} catch ( IOException e ) {
	    System.err.println("IOException reading file " + keyFile);
	    return null;
	}
	return key;
    }


    //-----------------------------------------------
    /**  
     * Inserts name-specs associated with some key
     * This function is used for experiments only. All name-specifiers
     * are inserted under the same key.
     * @param keyString The key identifying these name-specifiers
     * @param file file containing a list of name-specifiers
     */
    public void insertNameSpecs (String keyString, String file) {

	byte[] key = cryptix.util.core.Hex.fromString(keyString);
        Vector nameSpecsV = new Vector();
        Vector hrsV = new Vector();

	try {
	    FileReader fr = new FileReader(file);
	    StreamTokenizer tokenizer = prepareSimpleTokenizer(fr);

	    while ( tokenizer.nextToken() != StreamTokenizer.TT_EOF ) {

		String nameSpecString = tokenizer.sval;
		byte[] data = new byte[key.length+bytesForSize];
		int index = 0;
		Conversion.insertIntLow16(data,index,key.length);
		index +=bytesForSize;
		System.arraycopy(key,0,data,index,key.length);

		HostRecord hr = makeOneHostRecord(data);
		NameSpecifier ns = new NameSpecifier(nameSpecString);
		if ( (ns != null) && ( hr != null) ) {
		    nameSpecsV.addElement(ns);
		    hrsV.addElement(hr);
		}
	    }
	}  catch ( FileNotFoundException e ) {
	    System.err.println("Index file " + file + " not found");
	    return;
	} catch ( IOException e ) {
	    System.err.println("IOException reading file " + file);
	    return;
	}
	startAdvertising(nameSpecsV,hrsV);

    } 


    //-----------------------------------------------
    /**  
     * Early bind queries
     */
    public void find (StringTokenizer st, boolean toall) { 

	NameSpecifier n1 =new NameSpecifier(st.nextToken());
	System.out.println("/ searching all: "+n1.toString());
	EarlyBindRecord [] eb = getHostByiName(n1,toall);
	System.out.println(" results length: "+eb.length);
	for (int i=0; i<eb.length; i++) {
	    System.out.println(" "+i+") "+eb[i].toString());
	    HostRecord hr = eb[i].getHostRecord();
	    byte[] data = hr.getData();
	    if ( data != null)
		printData(data);
	}
    }

    
    //-----------------------------------------------
    /**  
     * Displays results of early-bind queries
     */
    public void printData(byte[] data) {

	int index = 0;
	int keyLength = Conversion.extract16toIntLow16(data,index);
	byte[] key = new byte[keyLength];
	index += bytesForSize;
	System.arraycopy(data,index,key,0,keyLength);
	index += keyLength;
	byte[] p = new byte[data.length-index];
	System.arraycopy(data,index,p,0,p.length);
	String path = new String(p);
	System.out.print("With key: " + new String(key));
	if ( path != null )
	    System.out.println(path +  "\n");
	else 
	    System.out.println("\n");
    }

    //-----------------------------------------------
    /**  
     * Queries for complete name-specifiers
     */
    public void findInfo (StringTokenizer st) {

	NameSpecifier n1 =new NameSpecifier(st.nextToken());
	boolean discoverFull = (st.hasMoreElements());
	System.out.println("/ discoverNames: "+n1.toString());
	NameSpecifier []nses=discoverNames(n1,discoverFull);
	System.out.println(" results length: "+nses.length);
	for (int i=0; i<nses.length; i++) {
	    System.out.println(" "+i+") "+nses[i].toString());
	}
    } 

    
    //-----------------------------------------------
    /**  
     */
    public static void usage() {

	System.out.println("Usage:");
	System.out.println("java tests.TwineTester [-d dsrName] [-n INR:port] ");
	System.out.println("-d : allows to specify a non-local dsr");
	System.out.println("-n : allows to specify a TwineResolver directly");
	System.exit(0);

    }

    //-----------------------------------------------
    /**  
     */
     public static SearchCFS createFromOptions(String args[]) {

	NameSpecifier ns = null;
	String dsr = "localhost";
	String resolver = null;
	int port = -1;
	byte[] data = null;

	try {
	    for (int i = 0; i < args.length; i++) {
		String arg = args[i];
		String arg2 = null;
		if ( arg.equals("-d")) {
		    dsr = args[i+1];
		} else if ( arg.equals("-n")) {
		    arg2 = args[i+1];
		    int index = arg2.indexOf(':');
		    resolver = arg2.substring(0,index);
		    port = Integer.parseInt(arg2.substring(index+1,arg2.length()));
		    System.out.println("\n Connecting to: " + resolver 
				       + " on port " +port);
		} 
	    }
	} catch (Exception e) {
	    System.out.println(e.toString());
	    usage();
	}
	    
	// Create test application and start announcing name
	SearchCFS searchCFS = null;
	try { 
	    if ( (resolver != null) && (port != -1))
		searchCFS = new SearchCFS(resolver,port);
	    else 
		searchCFS = new SearchCFS(dsr);
	} catch (Exception e) { 
	    System.out.println("Bad: "+e.toString());
	    System.exit(0);
	}
	return searchCFS;	
    }



    //-----------------------------------------------
    /**  
     * 
     */
    public static void main(String args[]) {
	
	SearchCFS searchCFS = createFromOptions(args);
	
	BufferedReader r = new BufferedReader
	    (new InputStreamReader(System.in));
	String response=null;
	
	// If a file was specified with -f then advertise all
	// name-specifiers in this file.
	for (int i = 0; i < args.length; i++) {
	    if ( args[i].equals("-f") ) {
		i++;
		System.out.println("\nInserting file with name-specifiers" + args[i]);
		searchCFS.insertNameSpecs("12345",args[i]);
		try {
		    // Wait for 45 seconds for data to get inserted
		    Thread.sleep(45000);
		} catch (Exception e) {		    
		}
		System.exit(0);
	    }
	}

	while (true) {
	
	    try {
		response = r.readLine();

		StringTokenizer st = new StringTokenizer(response);
		
		if (st.hasMoreTokens()) {
		    String t = st.nextToken();
		    
		    if (t.equals("setFrequency")) {
			searchCFS.setFrequency(Integer.parseInt(st.nextToken()));
			System.out.println("/ Update frequency set to: " + searchCFS.frequency);
		    }
		    else if (t.equals("insertFiles")) {
			String indexFile = st.nextToken();
			searchCFS.insertFiles(indexFile);
			System.out.println("/ Added file info to index ");
		    
		    } else  if (t.equals("insertDir")) {
			String keyFile = st.nextToken();
			String dir = st.nextToken();
			searchCFS.insertDir(keyFile,dir);
			System.out.println("/ Added file info to index ");
		    
		    }  else  if (t.equals("insertNameSpecs")) {
			String key =  st.nextToken();
			String file = st.nextToken();
			searchCFS.insertNameSpecs(key,file);
			System.out.println("/ Index updated");
		    
		    } else if (t.equals("findAll")) {
			boolean toall = true;
			searchCFS.find(st,toall);
		    } else if (t.equals("findAny")) {
			boolean toall = false;
			searchCFS.find(st,toall);			
		    } else if (t.equals("findInfo")) {
			searchCFS.findInfo(st);			
		    } else if ( t.equals("quit") || t.equals("q") ) {
			System.exit(0);
		    }
		}
	    } catch (Exception e) { 
		System.out.println("error: "+e.toString()); 
		e.printStackTrace();
	    }
	}
	
    }


}
