package tests;

import ins.api.*;
import ins.namespace.*;
import ins.inr.*;
import java.io.*;
import java.util.*;

/** 
 * This file is a test application for Twine resolvers.
 *
 * The file is strongly inspired from ins.api.AppTests.java
 * and is also meant to be hacked as needed for testing purposes
 * <p>
 * Current features:
 * <ul>
 * <li>addAnnouncement
 * <li>removeAnnouncement
 * <li>addTCPEarlyBinding
 * <li>suspendAnnouncer
 * <li>resumeAnnouncer
 * <li>getLocation
 * <li>cleanUp
 * <li>discoverNames
 * <li>getHostByiName
 * <li>sendMessage
 * <li>sm
 * <li>smm
 * <li>smn
 * </ul>
 * All that needs to be set is the name of the INR and the 
 * advertised name as command line parameters. e.g.<br>
 *      java tests.TwineTester -n INR_addr:INR_port '[service=hi]'
 * 
 * @author Magdalena Balazinska
 * @author Kalpak Kothari
 * @version 1.0
 */

public class TwineTester extends TwineApp {

    static NameSpecifier ns = null;
    static int curnum = 0;
    static TestTask testTask;
    long myAppuid = -1;
    static boolean INTERACTIVE = true;
    static boolean AUTOSEND = false;
    protected static boolean DISPLAY_OUT = true; // print packets to screen
    protected TwineLogger log;
    static int BURSTRATE = 1;	// 99% normal rate, 1% burst rate

    //-----------------------------------------------
    /**  
     * Find resolver using a DSR
     */
     public TwineTester(String dsrName) throws Exception {
	 super(dsrName);
	 time_out = 3500; // Long time_out for debugging only
	 this.enableSinglePeering();

	 byte[] addr = localhost.getAddress();
	 myAppuid = (long)(Conversion.extract32toInt(addr, 0));
	 myAppuid = (myAppuid << 16) | ((long)(super.port) & 0xFFFFl);

	 log = new TwineLogger(""+myAppuid,"App-"+myAppuid+".log");

	 if(AUTOSEND) {
	     testTask = new TestTask(this);
	     testTask.start();
	 }
     }

    //-----------------------------------------------
    /**  
     * Connect directly to a resolver
     */
     public TwineTester(String resolverName, int port) throws Exception {
	 super(resolverName,port);
	 time_out = 3500; // Long time_out for debugging only
	 this.enableSinglePeering();

	 byte[] addr = localhost.getAddress();
	 myAppuid = (long)(Conversion.extract32toInt(addr, 0));
	 myAppuid = (myAppuid << 16) | ((long)(super.port) & 0xFFFFl);

	 log = new TwineLogger(""+myAppuid,"App-"+myAppuid+".log");

	 if(AUTOSEND) {
	     testTask = new TestTask(this);
	     testTask.start();
	 }
     }


    //-----------------------------------------------
    /**  
     * Display/log received packet
     * @param p received packet
     */
     public void receivePacket(Packet p) {
	long time = System.currentTimeMillis();
	if(DISPLAY_OUT)
	    System.out.println("/ Received Packet:\n"+p.toString());
	String data = new String(p.data);
	if(DISPLAY_OUT)
	    System.out.println("Data:\n"+data+"\n");
	log.printStatus("RCV: " + (p.all?"all ":"any ") + data + " " + time, 
			TwineLogger.IMPORTANT_MSG);
     }


    //-----------------------------------------------
    /**  
     * Start periodic name announcement
     */
    public void startAnnouncer(NameSpecifier source[], int metric[], int period, byte[] data) {
		
	HostRecord hr = new HostRecord();
	hr.UDPport = (short)port;
	hr.addData(data);
	startAnnouncer(source,metric,period,hr);

    }

    //-----------------------------------------------
    /**  
     * 
     */
    public static void usage() {

	System.out.println("Usage:");
	System.out.println("java tests.TwineTester [-d dsrName] [-n INR:port] [-k data] nameSpecifier ");
	System.out.println("-d : allows to specify a non-local dsr");
	System.out.println("-n : allows to specify a TwineResolver directly");
	System.out.println("-k : allows to specify application level data for this resource");
	System.out.println("-i : non-interactive mode");
	System.out.println("--enable-send : auto-send test packets");
	System.out.println("--disable-out: disable packet output to screen");
	System.out.println("--burst-rate rate: set burst rate for sending packets (default: "+BURSTRATE+")"); 
	System.out.println("nameSpecifier : the intentional name of this resource");
	System.exit(0);

    }

    //-----------------------------------------------
    /**  
     * 
     */
     public static TwineTester createFromOptions(String args[]) {



	String dsr = "localhost";
	String resolver = null;
	int port = -1;
	byte[] data = null;

	try {
	    for (int i = 0; i < args.length; i++) {
		String arg = args[i];
		String arg2 = null;
		if ( arg.equals("-d")) {
		    dsr = args[i+1];
		} else if ( arg.equals("-n")) {
		    arg2 = args[i+1];
		    int index = arg2.indexOf(':');
		    resolver = arg2.substring(0,index);
		    port = Integer.parseInt(arg2.substring(index+1,arg2.length()));
		    System.out.println("\n Connecting to: " + resolver + " on port " +port);
		} else if ( arg.startsWith("[")) {
		    ns = new NameSpecifier(arg);
		} else if (arg.equals("-k")) {		    
		    data = args[i+1].getBytes();
		}
		else if (arg.equals("--disable-out")) {
		    DISPLAY_OUT = false;
		}
		else if (arg.equals("-i")) {
		    INTERACTIVE = false;
		}
		else if (arg.equals("--enable-send")) {
		    AUTOSEND = true;
		}
		else if (arg.equals("--burst-rate")) {
		    arg2 = args[i+1];
		    BURSTRATE = Integer.parseInt(arg2);
		}
	    }
	} catch (Exception e) {
	    System.out.println(e.toString());
	    usage();
	}
	    
	// If no NameSpecifier created, print help and quit
	if ( ns == null)
	    usage();

	// Create test application and start announcing name
	TwineTester t1 = null;
	try { 
	    if ( (resolver != null) && (port != -1))
		t1= new TwineTester(resolver,port);
	    else 
		t1= new TwineTester(dsr);

	    if ( data != null )
		t1.startAnnouncer(new NameSpecifier[]{ns},new int[]{2}, 10000,data);
	    else
		t1.startAnnouncer(new NameSpecifier[]{ns},new int[]{2}, 10000);

	    System.out.println("announcing "+ns.toString());
	} 
	catch (Exception e) { 
	    System.out.println("Bad: "+e.toString());
	    System.exit(0);
	}

	return t1;	
    }



    //-----------------------------------------------
    /**  
     * 
     */
    public static void main(String args[]) {
	
	TwineTester t1 = createFromOptions(args);

	while(true) {

	    if(INTERACTIVE) {
	
		BufferedReader r = new BufferedReader
		    (new InputStreamReader(System.in));
		String response=null;

		while (true) {
		    try {
			response = r.readLine();

			StringTokenizer st = new StringTokenizer(response);
		
			if (st.hasMoreTokens()) {
			    String t = st.nextToken();
		    
			    if (t.equals("addAnnouncement")) {
				NameSpecifier n1 =new NameSpecifier(st.nextToken());
				int ttl = Integer.parseInt(st.nextToken());
				t1.addAnnouncement(n1,ttl);
				System.out.println("/ addAnnouncement: "+
						   n1.toString()+" "+
						   Integer.toString(ttl));
			    } else if (t.equals("removeAnnouncement")) {
				NameSpecifier n1 =new NameSpecifier(st.nextToken());
				t1.removeAnnouncement(n1);
				System.out.println("/ removeAnnouncement: "+
						   n1.toString());
			    } else if (t.equals("addTCPEarlyBinding")) {

				int port = Integer.parseInt(st.nextToken());
				t1.addTCPEarlyBinding((short)port);
				System.out.println("/ addTCPEarlyBinding: "+
						   Integer.toString(port));
			    } else if (t.equals("suspendAnnouncer")) {
				t1.suspendAnnouncer();
				System.out.println("/ suspendAnnouncer: ");
			    } else if (t.equals("resumeAnnouncer")) {
				t1.resumeAnnouncer();
				System.out.println("/ resumeAnnouncer: ");
			    } else if (t.equals("getLocation")) {
				System.out.println("/ getLocation: ");
				System.out.println(t1.getLocation(1000));
			    } else if (t.equals("cleanUp")) {
				System.out.println("/ cleanUp: ");
				t1.cleanUp();			
				break;
			    } else if (t.equals("discoverNames")) {
				NameSpecifier n1 =new NameSpecifier(st.nextToken());
				boolean discoverFull = (st.hasMoreElements());
				System.out.println("/ discoverNames: "+n1.toString());
				NameSpecifier []nses=t1.discoverNames(n1,discoverFull);
				System.out.println(" results length: "+nses.length);
				for (int i=0; i<nses.length; i++) {
				    System.out.println(" "+i+") "+nses[i].toString());
				}
			    } else if (t.equals("getHostByiName")) {
				NameSpecifier n1 =new NameSpecifier(st.nextToken());
				boolean toall =  (st.nextToken().equals("all"));

				System.out.println("/ getHostByiName: "+n1.toString()+
						   (toall?"all":"any"));
				EarlyBindRecord [] eb = t1.getHostByiName(n1,toall);
				System.out.println(" results length: "+eb.length);
				for (int i=0; i<eb.length; i++) {
				    System.out.println(" "+i+") "+eb[i].toString());
				    HostRecord hr = eb[i].getHostRecord();
				    byte[] data = hr.getData();
				    if ( data != null)
					System.out.println("With application data " + new String(data));
				}
			    } else if (t.equals("sendMessage")) {
				NameSpecifier ns1 =new NameSpecifier(st.nextToken());
				NameSpecifier ns2 =new NameSpecifier(st.nextToken());
				boolean toall =  (st.nextToken().equals("all"));
				String data = st.nextToken();
				System.out.println("/ sending message: "+
						   ns1.toString()+" "+ns2.toString()+
						   " "+(toall?"true":"false")+" "+
						   data);
				t1.sendMessage(ns1,ns2,toall,data.getBytes());
			    } else if (t.equals("sm")) {
				NameSpecifier ns2 =new NameSpecifier(st.nextToken());
				System.out.println("/ sending message: "+
						   ns.toString()+" "+ns2.toString()+
						   " "+(false?"all":"any")+" "+
						   "hey there");
				t1.sendMessage(ns,ns2,false,("hey there").getBytes());
			    } else if (t.equals("smm")) {
				NameSpecifier ns2 =new NameSpecifier(st.nextToken());
				System.out.println("/ sending message: "+
						   ns.toString()+" "+ns2.toString()+
						   " "+(true?"all":"any")+" "+
						   "hey there");
				t1.sendMessage(ns,ns2,true,("hey there").getBytes());
			    }
			    else if (t.equals("smn")) {
				NameSpecifier ns2 =new NameSpecifier(st.nextToken());
				System.out.println("/ sending message: "+
						   ns.toString()+" "+ns2.toString()+
						   " "+(true?"all":"any")+" "+
						   ns.toString()+" "+ (++curnum));
				t1.sendMessage(ns,ns2,true,(ns.toString()+" "+curnum).getBytes());
			    }
		    
			}
		    } catch (Exception e) { 
			System.out.println("error: "+e.toString()); 
			e.printStackTrace();
		    }
		}
	    }
	}
	
    }

    /**
     * Send a test late-binding packet. The packet data contains this
     * application's UID, a packet sequence number, the current time
     * in ms and the sender's name specifier.
     * @param t an instance of the TwineTester
     * @param dNS destination name specifier
     * @param multicast false->anycast, true->multicast 
     */
    public void sendTestPacket(TwineTester t, NameSpecifier dNS, boolean multicast) {
	long time = System.currentTimeMillis();

	String data = myAppuid + " " + (++curnum) + " " + 
	    time + " " + ns.toString();
	t.sendMessage(ns,dNS,multicast,data.getBytes());
	log.printStatus("SND: "+ (multicast?"all":"any") + " " + data,
			TwineLogger.IMPORTANT_MSG);
    }

    /**
     * Inner thread for sending late-binding packets in various modes.
     * @author Kalpak Kothari
     */
    class TestTask extends Thread {
	TwineTester t;
	Random r;
	int pktseed, timeseed, timeleft, pktleft;
	long sleep_time;
	boolean inProgress = false;

	TestTask(TwineTester t) {
	    this.t = t;
	    r = new Random(myAppuid);
	}

	public void run() {

	    try {
		sleep(15000); // sleep some time to let the name announcement trickle into Twine
	    } catch (InterruptedException e) { }
	    
	    while(true) {
		long subtime = 0;

		if(inProgress) {
		    //   log.printStatus("Send multicast packet", TwineLogger.TRACE_MSG);
		    if(pktleft>0) {
			long time = System.currentTimeMillis();
			sendTestPacket(t, ns, true);
			pktleft--;
			subtime = System.currentTimeMillis()-time;
		    }
		    if(pktleft != 0) {
			try { 
			    long sleep_time_left = sleep_time-subtime;
			    if(sleep_time_left > 0) {
				sleep(sleep_time_left);
			    }
			} catch (InterruptedException e) { }
		    }
		    else { // no more packets left
			inProgress = false;
		    }
		}
		else { // initialize values
		    int rand = r.nextInt(99); 
		    if(rand<BURSTRATE) {	// 1% burst (default)
			pktseed = 51;
			timeseed = 11;
			pktleft = 100+r.nextInt(pktseed);
			timeleft = 10+r.nextInt(timeseed);
		    }
		    else {		// 99% normal
			pktseed = 10;  // number of packets
			timeseed = 30; // seconds
			pktleft = 1+r.nextInt(pktseed);
			timeleft = 60+r.nextInt(timeseed);
		    }
		    sleep_time = (long)((long)(timeleft*1000)/(long)pktleft); // in ms
		    inProgress = true;
		    log.printStatus("PKT: " + pktleft + 
				    " Time (s): " + timeleft +
				    " Sleep (ms): " + sleep_time,
				    TwineLogger.IMPORTANT_MSG);
		}
	    }
	}
    }
    
}







