package ins.dsr;

import java.io.*;
import java.net.*;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;
import java.util.Vector;


public class DSR
    implements Runnable
{
    public static final int DEFAULT_DSR_PORT = 5678;

    protected int dsrPort = DEFAULT_DSR_PORT;
    protected DatagramSocket socket;
    protected Vector Vspaces; // Vector of Vspace

    protected ResolverSet resolvers;

    long DEFAULT_TTL=Long.MAX_VALUE;

    // CONSTRUCTORS


    DSR() throws Exception
    {
	try {
	    socket = new DatagramSocket(DEFAULT_DSR_PORT);
	} catch (SocketException se) {
	    throw new Exception("Couldn't open socket");
	}

	Vspaces = new Vector();
	resolvers = new ResolverSet();

	new Thread(this).start();
    }

    // METHODS
    public void run()
    {
	byte[] buffer = new byte[16384];

	printStatus("Running...");

	while (true) {
	    try {
		DatagramPacket packet = 
		    new DatagramPacket(buffer, buffer.length);
		socket.receive(packet);
		processPacket(packet);
	    } catch (IOException e) {
		printStatus("Problem receiving packet: " + e.getMessage());
		continue;
	    }
	}
    }


    // Processes a received DSR packet
    final private void processPacket(DatagramPacket p)
    {
	DSRRequest request = 
	    DSRRequest.readWireform(p.getData(), 0, p.getLength());

	if (request==null) { 
	    printStatus("Could not parse incoming request packet");
	    return;
	}

	// why not print out the string equivalent of the message we received?
	printStatus(request.toString());

	switch (request.getType()) {

	case DSRRequest.GetVspaceResolvers:
	    {
		NodeSet nset = resolvers.getResolvers
		    (getLocalPortion(request.getVspace()));
		Enumeration e = null;
		if (nset!=null) { e = nset.getNodes(); } else 
		    { e = (new Vector(1)).elements(); }
		
		DSRResponse response = 
		    DSRResponse.makeGetVspaceResolvers
		    (e, request.getSequenceNo());
		
		sendResponse(response.toBytes(), p.getAddress(), p.getPort());
		
		break;
	    }

	case DSRRequest.DiscoverVspaces:
	    {
		DSRResponse response = DSRResponse.makeDiscoverVspaces
		    (resolvers.getVspaces(), request.getSequenceNo());
		
		sendResponse(response.toBytes(), p.getAddress(), p.getPort());
		
		break;
	    }

	case DSRRequest.Ping:
	    {
		DSRResponse response = 
		    DSRResponse.makePing(request.getSequenceNo());
		
		sendResponse(response.toBytes(), p.getAddress(), p.getPort());
	    
		break;
	    }

	case DSRRequest.AdvertiseVspaces:
	    // an advertise Vspaces message from the node implies that those
	    // are all the Vspaces known by the node (some multi-packet version
	    // could be supported if packet size becomes an issue there)
	    // They also all expire at the same time
	    
	    {
		// strip non-local part of vspaces from resolver for local ref
		Enumeration vspaces = request.getVspaces().elements();
		Vector localvspaces = new Vector(request.getVspaces().size());
		while (vspaces.hasMoreElements()) {		    
		    String vspace = (String)vspaces.nextElement();
		    localvspaces.addElement(getLocalPortion(vspace));
		}

		// put them in the resolver set
		resolvers.integrateVspaceAdvertisement
		    (localvspaces, 
		     request.getInetAddress(), request.getUDPport(),
		     request.getTCPport(), request.getTTL(),
		     request.getComplete());
		
		DSRResponse response;
		if (request.getWantFullResponse()) {
		    // send back full VspacesResolvers message
		    response = makeVspacesResolvers(request.getVspaces(),
						    request.getSequenceNo());
		} else {
		    // otherwise just send ack-type response back 
		    response = DSRResponse.makeAckAdvertiseVspaces
			(request.getSequenceNo());
		}
		
		sendResponse(response.toBytes(), p.getAddress(), p.getPort());
		
		break;
	    }

	case DSRRequest.VspacesResolvers:
	    {    
		// send back full VspacesResolvers message
		DSRResponse response = 
		    makeVspacesResolvers(request.getVspaces(),
					 request.getSequenceNo());
		
		sendResponse(response.toBytes(), p.getAddress(), p.getPort());
		
		break;
	    }

	} // case statement
    }

    // gets the local portion of a vspace name
    // e.g. "camera:wind.lcs.mit.edu" would be camera
    //   BUT -- the identical vspace must be
    private static String getLocalPortion(String vspace)
    {
	int index = vspace.indexOf(':');

	if (index > -1) {
	    ////// probably also should check that it indeed
	    ////// is our local vspace. oh well for now.
	    return vspace.substring(0, index);
	} else {
	    return vspace;
	}
    }


    /**
     * Produces a makeVspacesResolvers packet -- 
     *     a list of vspaces and their corresponding nodes
     *  to be sent back
     * @param vspaces List of vspace string for which to retrieve data
     * @param sequenceNo Sequence Number to put in the packet
     */
    protected DSRResponse makeVspacesResolvers(Vector vspaces,
					       int sequenceNo) 
    {
	Vector vsnodeset = new Vector(vspaces.size());
	Enumeration e = vspaces.elements();
	
	while (e.hasMoreElements()) {
	    String vspace = (String)e.nextElement();
	    String localvspace = getLocalPortion(vspace);
	    NodeSet rs = resolvers.getResolvers(localvspace);
	    
	    // if we don't know about it, send back a blank nodeset for it
	    if (rs == null) rs = new NodeSet();

	    // add to the list
	    vsnodeset.addElement(new VSNodes(vspace, rs));
	}

	return DSRResponse.makeVspacesResolvers
	    (vsnodeset.elements(), sequenceNo);
    }



    final private void sendResponse(byte[] response, InetAddress ia, int port)
    {
	DatagramPacket p =
	    new DatagramPacket(response, response.length, ia, port);
	
	try {
	    socket.send(p);
	} catch (IOException e) {
	    printStatus("Couldn't send response to INR: " + e.getMessage());
	}
    }

    final static void printStatus(String s)
    {
	System.out.println("DSR: " + s);
    }

    final static void turnOnFileWatcher()
    {
	// FileWatcher thread is used when DSR is run

	//  as a WinNT service. The C++ code for NT service

	//  will communicate with this java code through file.
	// new Thread(new FileWatcher("c:\\",
	//    "DSRexit.cmd", 10000)).start();
    }

    public final static void main(String[] args)
    {
	DSR dsr;

	try {
	    dsr = new DSR();

	    for (int i=0; i<args.length; i++)
	    {
		if (args[i].equals("-w"))
		    dsr.turnOnFileWatcher();
	    }
	} catch (Exception e) {
	    printStatus("Couldn't start DSR: " + e.getMessage());
	}
    }
}

