package ins.dsr;

import java.io.*;
import java.net.*;
import java.util.*;

/**
 * The DSRRequest class abstracts away the exact wireframe format
 * of a message being sent to a DSR, which will avoid multitudes of
 * changes being necessary any time a DSR message needs to be altered.
 *
 * This always refers to a message sent to the DSR. Every message 
 * contains a sequence number, and any response should have the 
 * same sequence number
 *
 * Due to the abstraction here, security with the DSR/certificates/etc.
 * may not be too difficult to add, since everything is in one 
 * master location.
 *
 * This could very well have been implemented using lots of
 * object-oriented subclasses and the like. However, encapsulating it
 * in one file sounded more pleasant than the thoughts of lots of
 * superfluous 10-line classes populating the directory.
 *
 * -jjlilley 
 */

public class DSRRequest {
    
    /* Possible */
    public static final int GetVspaceResolvers=2;
    public static final int DiscoverVspaces=3;
    public static final int Ping=4;
    public static final int AdvertiseVspaces=5;
    public static final int VspacesResolvers=6;
    private static final int MAXTYPE=6;

    private int type=0;
    private String vspace=null;
    private String hostname=null;
    private InetAddress host=null;
    private int udpport=0;
    private int tcpport=0;

    private long ttl=0;
    private boolean complete=true; // is an advertisement all-inclusive?
    private boolean wantFullResponse=true; // do we want resolvers from ad msg?

    private Vector vspaces;

    private int sequenceNo=0;

    
    private DSRRequest(int type, String vspace, String hostname, 
		       int udpport, int tcpport, int sequenceNo) 
    {
	this.type= type;
	this.vspace = vspace;
	this.hostname = hostname;
	this.udpport = udpport;
	this.tcpport = tcpport;
	this.sequenceNo = sequenceNo;
    }


    /** Makes an AdvertiseVspaces message
     * here
     * @param vspace Relevant vspace
     * @param host Requesting INR host 
     * @param udpport UDP port of INR host
     * @param tcpport TCP port of INR host
     * @param ttl Time to live for the message in ms
     * @param vspaces list of vspaces on the hose
     *       -- be careful -- this must not be modified until
     *          toBytes is called
     * @param complete Does this message list _all_ vspaces on the INR
     *        (answering true here is good)
     * @param sequenceNo sequence number for the request
     * @return DSRRequest that can be transmitted */
    public static DSRRequest 
	makeAdvertiseVspaces(InetAddress host, int udpport, int tcpport, 
			     long ttl, boolean complete, 
			     boolean wantFullResponse, Vector vspaces, 
			     int sequenceNo) 
    {
	return new DSRRequest(AdvertiseVspaces, host, udpport, tcpport, 
			      ttl, complete, wantFullResponse, vspaces, 
			      sequenceNo);
    }

    private DSRRequest(int type, InetAddress host, int udpport, int tcpport, 
		       long ttl, boolean complete, boolean wantFullResponse,
		       Vector vspaces, int sequenceNo) 
    {
	this.type= type;
	this.vspace = null;
	this.host = host;
	this.udpport = udpport;
	this.tcpport = tcpport;
	this.sequenceNo = sequenceNo;
	this.vspaces = vspaces;
	this.ttl = ttl;
	this.complete = complete;
	this.wantFullResponse = wantFullResponse;
    }

    

    /** Makes a GetVspaceResolvers request - need more thorough description 
     * @param vspace Relevant vspace
     * @return DSRRequest that can be transmitted
     */
    public static DSRRequest 
	makeGetVspaceResolvers(String vspace, int sequenceNo) 
    {
	return new DSRRequest(GetVspaceResolvers, vspace, sequenceNo);
    }

    private DSRRequest(int type, String vspace, int sequenceNo) 
    {
	this.type=type;
	this.vspace=vspace;
	this.sequenceNo=sequenceNo;
    }


    /** Makes a DiscoverVspaces request - discovers known vspaces
     * @return DSRRequest that can be transmitted 
     */
    public static DSRRequest makeDiscoverVspaces(int sequenceNo) 
    {
	return new DSRRequest(DiscoverVspaces, sequenceNo);
    }

    /** Makes a Ping request to the DSR
     */
    public static DSRRequest makePing(int sequenceNo) 
    {
	return new DSRRequest(Ping, sequenceNo);
    }

    private DSRRequest(int type, int sequenceNo) 
    {
	this.type=type;
	this.sequenceNo = sequenceNo;
    }

    /**
     * Requests a set of resolvers for each vspace in a set of vspaces
     * @param vspaces list of vspaces on the host
     * @param sequenceNo the sequence number
     */
    public static DSRRequest 
	makeVspacesResolvers(Enumeration vspaces, int sequenceNo) 
    {
	return new DSRRequest(VspacesResolvers, vspaces, sequenceNo);
    }

    private DSRRequest(int type, Enumeration vspaces, int sequenceNo)
    {
	this.type=type;
	this.sequenceNo = sequenceNo;
	this.vspaces = new Vector();
	while (vspaces.hasMoreElements()) 
	    this.vspaces.addElement(vspaces.nextElement());
    }



    /** From an input wireforms, produces an array of DSRRequests.
     * @param data Input array
     * @return DSRRequest[] Set of resulting requests 
     */
    public static DSRRequest readWireform(byte[] data) 
    {
	return readWireform(data, 0, data.length);
    }


    /** 
     * From an input wireforms, produces an array of DSRRequests.
     * @param data Input array
     * @param offset Where to start
     * @param length How much to read
     * @return DSRRequest[] Set of resulting requests 
     */
    public static DSRRequest readWireform
	(byte[] data, int offset, int length) 
    {
	if (offset+length>data.length || length<8) return null;
	if (data[0+offset]!='D' || data[1+offset]!='S' || 
	    data[2+offset]!='Q' || data[3+offset]>DSRRequest.MAXTYPE || 
	    data[3+offset]==0)
	    return null;

	int type=data[3+offset];
	int sequenceNo = Conversion.extract32toInt(data, 4+offset);
	DSRRequest result=null;

	switch (type) {
		
	case GetVspaceResolvers: 
	    {		
		int len = data[offset+8];
		if (len>length-8) return null;
		String vspace = new String(data, offset+9, len);
		
		result = new DSRRequest
		    (GetVspaceResolvers, vspace, sequenceNo);
		break;
	    }

	case DiscoverVspaces: 
	    {		
		result = new DSRRequest(DiscoverVspaces, sequenceNo);
		break;
	    }

	case Ping:
	    {
		result = new DSRRequest(Ping, sequenceNo);
		break;
	    }

	case AdvertiseVspaces:
	    {
		int ofs = offset+10;
		
		Node node = Node.readWireform(data, ofs);
		
		StringSet ss = StringSet.readWireform
		    (data, ofs+Node.WIRELENGTH, 
		     length-ofs+offset-Node.WIRELENGTH);
		
		if (node==null || ss==null) return null;
		
		result = new DSRRequest(AdvertiseVspaces, 
					node.getInetAddress(), 
					node.getUDPport(), node.getTCPport(), 
					node.getTTL(), data[offset+8]!=0,
					data[offset+9]!=0,
					ss.getVector(), sequenceNo);
		break;
	    }

	case VspacesResolvers:
	    {	    
		StringSet ss = StringSet.readWireform
		    (data, offset+8, length-8);
		
		if (ss==null) return null;
		
		result = new DSRRequest
		    (VspacesResolvers, ss.elements(), sequenceNo);
		
		break;
	    }
	} // switch
	    
	return result;
    }


    /** 
     * Produces a wire-form version of the request suitable for
     * transmission and reconstruction 
     * @return byte[] Resulting bytes 
     */
    public byte[] toBytes() {
	String result=null;

	// mega case statement
	switch (type) {
	case GetVspaceResolvers:
	    {
		byte [] vs = (vspace!=null) ? vspace.getBytes() : new byte[0];
		int len = (vs.length>255) ? 255 : vs.length;
		byte [] data = new byte[len+9];
		
		data[0]= (byte)'D';data[1]= (byte)'S';
		data[2]= (byte)'Q';data[3]=(byte)type;
		
		Conversion.insertInt(data, 4, this.sequenceNo);
		
		data[8] = (byte)len;
		System.arraycopy(vs, 0, data, 9, len);
		
		return data;
	    }

	case AdvertiseVspaces: 
	    {	    
		Node node = new Node(host, udpport, tcpport, ttl);
		
		byte [] ss = (new StringSet(vspaces)).toBytes();
		byte [] n = node.toBytes();
		
		// allocate the proper length
		byte [] data = new byte[10+ss.length+n.length];
		
		// header
		data[0]= (byte)'D';data[1]= (byte)'S';
		data[2]= (byte)'Q';data[3]=(byte)type;
		
		// sequence number
		Conversion.insertInt(data, 4, this.sequenceNo);
		// complete flag
		data[8] = (byte)(this.complete?1:0);
		data[9] = (byte)(this.wantFullResponse?1:0);
		
		// node information
		System.arraycopy(n, 0, data, 10, n.length);
		// vspaces information
		System.arraycopy(ss, 0, data, 10+n.length, ss.length);
		
		return data;
	    }

	case VspacesResolvers: 
	    {		
		byte [] ss = (new StringSet(vspaces)).toBytes();
		
		// allocate the proper length
		byte [] data = new byte[8+ss.length];
		
		// header
		data[0]= (byte)'D';data[1]= (byte)'S';
		data[2]= (byte)'Q';data[3]=(byte)type;
		
		// sequence number
		Conversion.insertInt(data, 4, this.sequenceNo);
		
		// vspaces information
		System.arraycopy(ss, 0, data, 8, ss.length);
		
		return data;
	    }
	    
	case DiscoverVspaces:
	case Ping: 
	    {
		byte [] data = new byte[8];
		
		data[0]= (byte)'D';data[1]= (byte)'S';
		data[2]= (byte)'Q';data[3]=(byte)type;
		
		Conversion.insertInt(data, 4, this.sequenceNo);
		
		return data;
	    } 
	} // switch
	
	return new byte[0];
    }


    // accessor methods galore

    public String getVspace() { return vspace; }
    public String getHostName() { return hostname; }
    public int getUDPport() { return udpport; }
    public int getTCPport() { return tcpport; }
    public boolean getComplete() { return complete; }
    public boolean getWantFullResponse() { return wantFullResponse; }
    public int getType() { return type; }


    public InetAddress getInetAddress() { 
	if (host!=null) return host;
	if (hostname!=null) { 
	    try { return InetAddress.getByName(hostname); }
	    catch (Exception e) { ; }
	}
	return null; 
    }

    public Vector getVspaces() { return vspaces; }
    public long getTTL() { return ttl; }

    public int getSequenceNo() { return this.sequenceNo; }
    public void setSequenceNo(int sequenceNo) { this.sequenceNo = sequenceNo; }



    /** 
     * Produces a human-readable representation of the request 
     */
    public  String toString() {

	switch (type) {
	case GetVspaceResolvers: 
	    {
		return ("GetVspaceResolvers "+Integer.toString(sequenceNo)+
			" { "+(vspace==null?"null":vspace)+
			" } ");
	    }

	case DiscoverVspaces: 
	    {
		return ("DiscoverVspaces "+
			Integer.toString(sequenceNo)+" { }");
	    }

	case Ping:
	    {
		return ("Ping "+Integer.toString(sequenceNo)+" { }");
	    }

	case AdvertiseVspaces: 
	    {
		return ("AdvertiseVspaces "+Integer.toString(sequenceNo)+" { "+
			(host==null?"null":host.getHostName())+" "+
			Integer.toString(udpport)+"/"+
			Integer.toString(tcpport)+"<"+
			(ttl==Long.MAX_VALUE?"INFINITE":
			 Long.toString(ttl)+"ms")
			+"> "+vspaces.toString()+" }");
	    }		

	case VspacesResolvers:
	    {
		return ("VspacesResolvers "+Integer.toString(sequenceNo)+" "+
			(vspaces==null?"[ ]":vspaces.toString()));
	    }

	default:
	    {
		return ("Request undefined");
	    }

	} // switch
    }


}
