package ins.dsr;

import java.io.*;
import java.net.*;
import java.util.*;

/**
 * The DSRResponse class will abstract away the exact wireform format
 * of a message being sent back from a DSR, which will avoid
 * multitudes of changes being necessary any time a DSR message needs
 * to be altered.
 *
 * This always refers to a message sent back from the DSR to the host. 
 * The sequence number on this message should be the same as the request.
 *
 * Due to the abstraction here, security with the DSR/certificates/etc.
 * may not be too difficult to add, since everything is in one 
 * master location.
 *
 * This could very well have been implemented using lots of 
 * object-oriented subclasses and the like. However, encapsulating 
 * it in one file sounded more pleasant than the thoughts of lots 
 * of superfluous 10-line classes populating the directory.
 *
 * -jjlilley 
 */

public class DSRResponse {
    
    /* Possible */
    public static final int GetVspaceResolvers=2;
    public static final int DiscoverVspaces=3;
    public static final int Ping=4;
    public static final int AckAdvertiseVspaces=5;
    public static final int VspacesResolvers=6;
    private static final int MAXTYPE=6;


    // real data
    private int type=0;
    private Vector set;
    private Vector set2;
    private int sequenceNo=0;


    /** Makes a GetVspaceResolvers response 
     * @param nodeset Set of nodes to be transmitted back
     * @return DSRResponse that can be transmitted
     */
    public static DSRResponse 
	makeGetVspaceResolvers(Enumeration nodeset, int sequenceNo) 
    {
	return new DSRResponse(GetVspaceResolvers, nodeset, sequenceNo);
    }


    /** Makes a DiscoverVspaces response 
     * @param nodeset Set of Vspace names to be transmitted back
     * @return DSRResponse that can be transmitted
     */
    public static DSRResponse 
	makeDiscoverVspaces(Enumeration vsnames, int sequenceNo) 
    {
	return new DSRResponse(DiscoverVspaces, vsnames, sequenceNo);
    }


    /** Makes a Ping response 
     * @return DSRResponse that can be transmitted
     */
    public static DSRResponse 
	makePing(int sequenceNo) 
    {
	return new DSRResponse(Ping, sequenceNo);
    }


    /** Makes a AdvertiseVspaces response 
     * @return DSRResponse that can be transmitted
     */
    public static DSRResponse 
	makeAckAdvertiseVspaces(int sequenceNo) 
    {
	return new DSRResponse(AckAdvertiseVspaces, sequenceNo);
    }

    /** 
     * A mesage consisting of virtual spaces and their accompanying
     * resolvers. e.g. "vspace1 has nodes n1, n2; vspace2 has node n3"
     * Useful for bulk transfer of vspace data back from resolver.
     * @param vsnodes Enumeration of VSNodes items -- which each
     *         contain a virtual space and set of corresponding nodes
     */
    public static DSRResponse
	makeVspacesResolvers(Enumeration vsnodes,
			     int sequenceNo) 
    {
	return new DSRResponse(VspacesResolvers, vsnodes, sequenceNo);
    }

    private DSRResponse(int type, Enumeration enum, int sequenceNo) 
    {
	this.type= type;
	this.set = new Vector();
	this.sequenceNo = sequenceNo;

	while (enum.hasMoreElements()) {

	    // make sure they're the right type
	    
	    switch (type) {
		
		//  GetVspaceResolvers all have type Node
	    case GetVspaceResolvers:
		{
		Object n = enum.nextElement();
		if (n instanceof Node)
		    this.set.addElement((Node)n);
		
		break;
		}

		// DiscoverVspaces requires String
	    case DiscoverVspaces: 
		{
		    Object n = enum.nextElement();
		    if (n instanceof String)
			this.set.addElement((String)n);
		    break;
		}

		// VspacesResolvers requires VSNodes
	    case VspacesResolvers:
		{
		    Object n = enum.nextElement();
		    if (n instanceof VSNodes)
			this.set.addElement(n);
		    break;
		}

	    } // switch
	}
    }


    private DSRResponse(int type, int sequenceNo) 
    {
	this.type= type;
	this.set = new Vector();
	this.sequenceNo = sequenceNo;
    }

    
    // for constructing from wireform
    private DSRResponse(int type, Vector set, int sequenceNo) 
    {
	this.type = type;
	this.set = set;
	this.sequenceNo = sequenceNo;
    }
    

    /** From an input wireforms, produces an array of DSRResponses.
     * @param data Input array
     * @return DSRResponse Resulting response, null if improper wireform
     */
    public static DSRResponse readWireform(byte[] data) 
    {
	return readWireform(data, 0, data.length);
    }


    /** From an input wireforms, produces an array of DSRResponses.
     * @param data Input array
     * @param offset Where to start
     * @param length How much to read
     * @return DSRResponse Resulting response, null if improper wireform
     */
    public static DSRResponse readWireform
	(byte[] data, int offset, int length) 
    {
	if (length+offset>data.length) return null;
	if (length<8) return null;
	if (!( (data[offset]==(byte)'D') && (data[offset+1]==(byte)'S') &&
	       (data[offset+2]==(byte)'R') ))
	    return null;

	int type=data[offset+3];
	if (type>MAXTYPE || type==0) return null;
	int sequenceNo = Conversion.extract32toInt(data, 4+offset);


	Vector set=new Vector();

	switch (type) {
	    
	case GetVspaceResolvers: 
	    {
		int count = Conversion.extract32toInt(data, 8+offset);
		int max =((length-8)/Node.WIRELENGTH);
		if (count<max) max=count;
		
		for (int i=0; i<max; i++) {
		    Node o = Node.readWireform(data,(12 + offset+ 
						     (Node.WIRELENGTH*i)));
		    if (o!=null) 
			set.addElement(o);
		}
		
		return new DSRResponse(type, set, sequenceNo);    
	    }
	    
	case DiscoverVspaces: 
	    {	    
		int pos=8;
		while (pos<length) {
		    int len = Conversion.extract16toIntLow16(data,offset+pos);
		    if (len==0) break;
		    
		    String o = new String(data,offset+pos+2,len);
		    if (o!=null)
			set.addElement(o);
		    
		    pos +=len+2;
		}

		return new DSRResponse(type, set, sequenceNo);
	    }

	case Ping:
	case AckAdvertiseVspaces:
	    {
		return new DSRResponse(type, sequenceNo);
	    }

	case VspacesResolvers: 
	    {
		int pos=8;
		while (pos<length) {
		    int len = Conversion.extract16toIntLow16(data,offset+pos);
		    if (len==0) break;
		    
		    VSNodes o = VSNodes.readWireform(data,offset+pos+2,len);
		    if (o!=null) set.addElement(o);
		    
		    pos +=len+2;
		}
		
		return new DSRResponse(type, set, sequenceNo);
	    }
	} // switch

	return null;
    }


    /** Produces a wire-form version of the response suitable for
     * transmission and reconstruction 
     * @return byte[] Resulting bytes 
     */
    public byte[] toBytes() 
    {
	switch (type) {

	case GetVspaceResolvers: 
	    {
		int size = this.set.size();

		byte[] data = new byte[Node.WIRELENGTH * size + 12];
		
		data[0]= (byte)'D';data[1]= (byte)'S';
		data[2]= (byte)'R';data[3]=(byte)type;
		
		Conversion.insertInt(data, 4, this.sequenceNo);
		Conversion.insertInt(data, 8, size);
		
		for (int i=0; i<size; i++) {
		    ((Node)this.set.elementAt(i))
			.copyBytes(data, (Node.WIRELENGTH * i) + 12);
		}
		
		return data;
	    }

	case DiscoverVspaces: 
	    {
		int size = this.set.size();
		int length=10;
		
		// this is fairly inefficient
		for (int i=0; i<size; i++) {
		    length +=
			(((String)this.set.elementAt(i)).getBytes().length+2);
		}
		
		byte[] data = new byte[length];
		
		data[0]= (byte)'D';data[1]= (byte)'S';
		data[2]= (byte)'R';data[3]=(byte)type;
		
		Conversion.insertInt(data, 4, this.sequenceNo);
		
		int pos=8;
		
		for (int i=0; i<size; i++) {
		    byte [] d=((String)this.set.elementAt(i)).getBytes();
		    int len = d.length;
		    Conversion.insertIntLow16(data, pos, len);
		    System.arraycopy(d, 0, data, pos+2, d.length);
		    pos += len+2;
		}

		Conversion.insertIntLow16(data, pos, 0);
		return data;
	    }

	case Ping:
	case AckAdvertiseVspaces: 
	    {
		byte[] data = new byte[8];
		
		data[0]= (byte)'D';data[1]= (byte)'S';
		data[2]= (byte)'R';data[3]=(byte)type;
		
		Conversion.insertInt(data, 4, this.sequenceNo);
		return data;
	    }

	case VspacesResolvers: 
	    {	    
		// first calculate the size to allocate
		int size = 10;
		for (int i=0; i<set.size(); i++) {
		    size += 2+((VSNodes)set.elementAt(i)).getByteLength();
		}
		
		byte[] data = new byte[size];
		
		// header
		data[0]= (byte)'D';data[1]= (byte)'S';
		data[2]= (byte)'R';data[3]=(byte)type;
		Conversion.insertInt(data, 4, this.sequenceNo);
		
		//now fill the VSNodes in
		int pos = 8;
		for (int i=0; i<set.size(); i++) {
		    int newpos = ((VSNodes)set.elementAt(i))
			.copyInto(data, pos+2);
		    Conversion.insertIntLow16(data, pos, (newpos-(pos+2)));
		    pos = newpos;
		}
		Conversion.insertIntLow16(data, pos, 0);
		
		return data;
	    }
	} // switch

	return null;
    }


    // accessor methods galore

    // Get list of dsr.Node(s) or Vspaces 
    public Enumeration getElements() { return set.elements(); }
    // Gives access to the underlying rep. Be careful!
    public Vector getVector() { return set; }
    // sequence number
    public void setSequenceNo(int sequenceNo) { this.sequenceNo = sequenceNo; }
    public int getSequenceNo() { return this.sequenceNo; }

    public int getType() { return this.type; } 


    /** Produces a human-readable representation of the response */
    public  String toString() {
	String result=null;
	
	switch (type) {
	    
	case GetVspaceResolvers: 
	    {
		result = "GetVspaceResolvers "+
		    Integer.toString(sequenceNo)+" { ";

		for (int i=0; i<this.set.size(); i++) {
		    result += "\"" +
			((Node)this.set.elementAt(i)).toString()+"\" ";
		}
		
		result +="}";

		break;
	    } 

	case DiscoverVspaces: 
	    {
		result = "DiscoverVspaces "+Integer.toString(sequenceNo)+" { ";
		
		for (int i=0; i<this.set.size(); i++) {
		    result += "\""+((String)this.set.elementAt(i))+"\" ";
		}
		
		result +="}";
		
		break;
	    }

	case Ping:
	    {
		result = "Ping "+Integer.toString(sequenceNo);
	    
		break;
	    }
	
	case AckAdvertiseVspaces:
	    {
		result = "AckAdvertiseVspaces "+Integer.toString(sequenceNo);

		break;
	    }

	case VspacesResolvers:
	    {
		result = "VspacesResolvers "+Integer.toString(sequenceNo)+" { ";
		
		for (int i=0; i<this.set.size(); i++) {
		    result += ((VSNodes)this.set.elementAt(i)).toString()+" ";
		}

		result += "}";

		break;
	    }
	    
	default:
	    result = "Response undefined";

	} // switch
	
	return result;
    }

}
