package ins.dsr;

import java.net.InetAddress;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.util.Vector;
import java.util.Enumeration;

public class Node
{
    // VARIABLES
    InetAddress ia;
    int UDPport;
    int TCPport;
    Vector vspaces;
    int rtt;

    long expireTime;

    Node (InetAddress i, int udp, int tcp)
    {
	ia = i;
	UDPport = udp;
	TCPport = tcp;
	vspaces = new Vector(2);
	rtt = Integer.MAX_VALUE;
	expireTime = Long.MAX_VALUE;
    }

    Node (InetAddress i, int udp, int tcp, long ttl)
    {
	ia = i;
	UDPport = udp;
	TCPport = tcp;
	vspaces = new Vector(2);
	rtt = Integer.MAX_VALUE;
	expireTime = (ttl==Long.MAX_VALUE)? 
	    ttl : ttl+System.currentTimeMillis();
    }


    Node (InetAddress i, int udp, int tcp, String vs)
    {
	ia = i;
	UDPport = udp;
	TCPport = tcp;
	vspaces = new Vector(2);
	vspaces.addElement(vs);
	rtt = Integer.MAX_VALUE;
	expireTime = Long.MAX_VALUE;
    }

    Node (InetAddress i, int udp, int tcp, String vs, long ttl)
    {
	ia = i;
	UDPport = udp;
	TCPport = tcp;
	vspaces = new Vector(2);
	vspaces.addElement(vs);
	rtt = Integer.MAX_VALUE;
	expireTime = (ttl==Long.MAX_VALUE)? 
	    ttl : ttl+System.currentTimeMillis();
    }

    Node (InetAddress i, int udp, int tcp, Vector vs)
    {
	ia = i;
	UDPport = udp;
	TCPport = tcp;
	rtt = Integer.MAX_VALUE;
	expireTime = Long.MAX_VALUE;

	vspaces = new Vector(vs.size());
	Enumeration e = vs.elements();
	while (e.hasMoreElements()) vspaces.addElement(e.nextElement());
    }

    Node (InetAddress i, int udp, int tcp, Vector vs, long ttl)
    {
	ia = i;
	UDPport = udp;
	TCPport = tcp;
	rtt = Integer.MAX_VALUE;
	expireTime = (ttl==Long.MAX_VALUE)? 
	    ttl : ttl+System.currentTimeMillis();

	vspaces = new Vector(vs.size());
	Enumeration e = vs.elements();
	while (e.hasMoreElements()) vspaces.addElement(e.nextElement());
    }

    void addVspace(String vs)
    {
	if (vspaces.indexOf(vs) == -1)
	    vspaces.addElement(vs);
    }

    void removeVspace(String vs)
    {
	vspaces.removeElement(vs);
    }

    Enumeration getVspaces() { return vspaces.elements(); }

    boolean containsVspace(String vs) 
    {
	return (vspaces.indexOf(vs) > -1);
    }

    int countVspace()
    {
	return vspaces.size();
    }

    public int hashCode() 
    {
	return ia.hashCode();
    }
    
    public boolean isExpired() 
    {
	return (System.currentTimeMillis()>expireTime);
    }

    public boolean isExpired(long curTime) 
    {
	return (curTime>expireTime);
    }


    /**
     * Returns true if the IP address and port number are
     * equal (note: no comparison of vspaces or times performed).
     */
    public boolean equals(Object obj)
    {
	if (obj instanceof Node)
	{
	    Node n = (Node)obj;

	    // Note: port# 0 matches everything
	    if (ia.equals(n.ia) &&
		((UDPport==n.UDPport) || (UDPport==0) || (n.UDPport==0)) &&
		((TCPport == n.TCPport) || (TCPport==0) || (n.TCPport==0)))
		return true;
	    else
		return false;
	}
	else return false;
    }

    /**
     * Returns a String representation of the Node.
     */
    public String toString()
    {
	return (ia.getHostName() + ":" + UDPport + ":" + TCPport +
		" <"+vspaces.toString()+">:ttl="+
		(expireTime==Long.MAX_VALUE?"INFINITE":
		 Long.toString(expireTime-System.currentTimeMillis())));
    }

    // for wireform stuff
    public static final int WIRELENGTH = 12;


    /** Constructs a node from a wireform */
    public static Node readWireform(byte [] data, int offset) 
    {
	try {
	    String stria = Integer.toString(data[offset] & 0xFF) +
		"." + Integer.toString(data[offset+1] & 0xFF) +
		"." + Integer.toString(data[offset+2] & 0xFF) +
		"." + Integer.toString(data[offset+3] & 0xFF);
	    
	    InetAddress ia = InetAddress.getByName(stria);
	    int UDPport = Conversion.extract16toIntLow16(data, offset+4);
	    int TCPport = Conversion.extract16toIntLow16(data, offset+6);
	    int intttl  = Conversion.extract32toInt(data, offset+8); 
	    
	    long ttl = (intttl==Integer.MAX_VALUE) ? 
		Long.MAX_VALUE : (intttl)*100;

	    return new Node(ia, UDPport, TCPport, ttl);
	}
	catch (Exception e) { return null; }
    }

    public int getTCPport() { return TCPport; }
    public int getUDPport() { return UDPport; }
    public InetAddress getInetAddress() { return ia; }
    
    /** 
     *  Gets the time to live, in ms. 
     *    Long.MAX_VALUE means that it never expires 
     */
    public long getTTL() 
    { 
	if (expireTime==Long.MAX_VALUE) 
	    return expireTime;
	else
	    return (expireTime-System.currentTimeMillis());
    }


    /** 
     *  Gets the time to live, in ms. 
     *    Long.MAX_VALUE means that it never expires 
     */
    public long getTTL(long currentTime) 
    { 
	if (expireTime==Long.MAX_VALUE) 
	    return expireTime;
	else
	    return (expireTime-currentTime);
    }
    

    /** 
     * Sets the ttl in ms, with Long.MAX_VALUE meaning that
     *      it never expires 
     */
    public void setTTL(long ttl) 
    {
	if (ttl==Long.MAX_VALUE) 
	    expireTime=ttl;
	else 
	    expireTime = System.currentTimeMillis()+ttl;
    }

    public void updateTTL(long ttl) 
    {
	long expireTime = (ttl==Long.MAX_VALUE? Long.MAX_VALUE :
			   System.currentTimeMillis()+ttl);

	if (expireTime > this.expireTime)
	    this.expireTime = expireTime;
    }

    public byte[] toBytes() 
    {
	byte[] response =new byte[WIRELENGTH];

	byte[] addr = this.ia.getAddress();
	response[0] = addr[0];
	response[1] = addr[1];
	response[2] = addr[2];
	response[3] = addr[3];
	
	Conversion.insertIntLow16(response, 4, UDPport);
	Conversion.insertIntLow16(response, 6, TCPport);

	int toInsert = (expireTime==Long.MAX_VALUE)? Integer.MAX_VALUE : 
	    (int)((expireTime-System.currentTimeMillis())/100);

	Conversion.insertInt(response, 8, toInsert);
	return response;
    }


    public int copyBytes(byte[] response, int offset) 
    {
	byte[] addr = this.ia.getAddress();
	response[offset] = addr[0];
	response[offset+1] = addr[1];
	response[offset+2] = addr[2];
	response[offset+3] = addr[3];
	
	Conversion.insertIntLow16(response, offset+4, UDPport);
	Conversion.insertIntLow16(response, offset+6, TCPport);

	int toInsert = (expireTime==Long.MAX_VALUE)? Integer.MAX_VALUE : 
	    (int)((expireTime-System.currentTimeMillis())/100);

	Conversion.insertInt(response, offset+8, toInsert);

	return offset+WIRELENGTH;
    }

    
}
