package ins.dsr;

import java.net.InetAddress;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Hashtable;

/**
 * ResolverSet consists of vspaces and the sets of nodes in them.
 *
 * Its underlying form is similar to ins.inr.VSNeighbors, in that
 * it has a Hashtable mapping vspaces (Strings) to resolvers (NodeSets).
 * In addition, it has a Vector with all of the known resolvers.
 * Only one copy of a resolver exists, and all the vspace NodeSets 
 * simply contain pointers to the single copy of each node.
 */

class ResolverSet
{
    // VARIABLES
    protected Hashtable rep; // maps vspace names (Strings) 
                             // to resolvers (NodeSets)

    NodeSet allResolvers;   // NodeSet for all Vspaces this
			   // INR belongs to
			   // AllResolvers is a Vector of Node(s)

    // CONSTRUCTORS
    ResolverSet()
    {
	rep = new Hashtable(47);
	allResolvers = new NodeSet();
    }

    /**
     * Integrates an AdvertiseVspace message, updating the representation
     *  with the new announcement.
     * If the message is "complete" the advertised vspaces effectively
     *  replace those in the old advertisement (this is recommended),
     *  but update the ttl.
     * If the message is not "complete" the new vspaces are merely added.
     * This works on the underlying rep by mutating the old Node
     *  rather than replacement.
     */
    public synchronized boolean integrateVspaceAdvertisement
	(Vector vspaces, InetAddress ia, int udp, int tcp, 
	 long TTL, boolean complete)
    {
	// print the status
	//System.out.println("- integrating AdvertiseVspaces message");
	//System.out.println(" "+ia.toString()+" "+udp+":"+tcp+" <"+TTL+"> "+
	//		   (complete?"true":"false")+" "+vspaces);

	Node node = new Node(ia, udp, tcp, vspaces, TTL);

	Node oldNode = allResolvers.get(node);
	
	if (oldNode == null) { // resolver never known before
	    allResolvers.addUnconditionally(node);
	    
	    // go through each vspace
	    Enumeration e = vspaces.elements();
	    while (e.hasMoreElements()) { 
		String vspace = (String)e.nextElement(); 
		
		NodeSet ns = (NodeSet)rep.get(vspace);
		if (ns == null)
		{
		    rep.put(vspace, new NodeSet(node));
		} else {  // resolver unknown, vspace known
		    ns.addUnconditionally(node);
		}
	    }
	} else { // resolver known before

	    // keep track of additions/removals between oldNode and node
	    Vector toAdd = new Vector();
	    Vector toRemove = new Vector();

	    // vspaces in oldNode but not in node get remove
	    //  if complete
	    if (complete) {
		Enumeration e = oldNode.getVspaces();
		
		while (e.hasMoreElements()) {
		    String vs = (String)e.nextElement();
 		    if (!node.containsVspace(vs)) {
			toRemove.addElement(vs);
			oldNode.removeVspace(vs);
		    }    
		}
	    }
	    
	    // vspaces in node but not in old node always
	    //   get added
	    Enumeration e = node.getVspaces();
	    
	    while (e.hasMoreElements()) {
		String vs = (String)e.nextElement();		
		if (!oldNode.containsVspace(vs)) {
		    toAdd.addElement(vs);
		    oldNode.addVspace(vs);
		}
	    }

	    // set TTL
	    oldNode.updateTTL(node.getTTL());

	    // add from toAdd vspace list
	    e = toAdd.elements();
	    while (e.hasMoreElements()) {
		String vs = (String)e.nextElement();
	
		// find vspace
		NodeSet ns = (NodeSet)rep.get(vs);
		if (ns == null)
		{
		    rep.put(vs, new NodeSet(oldNode));
		} else { // just add to vspace
		    ns.put(oldNode);
		}
	    }

	    // remove from toRemove vspace list
	    e = toRemove.elements();
	    while (e.hasMoreElements()) {
		String vs = (String)e.nextElement();

		// find vspace
		NodeSet ns = (NodeSet)rep.get(vs);
		if (ns == null)
		{
		    // bad, shouldn't happen
		} else { // just remove from vspace
		    ns.remove(oldNode);

		    // is it empty now?
		    if (ns.countNodes() == 0) {
			rep.remove(vs);
		    }
		}
	    }

	} // else resolver known before

	return true;
    }


    /**
     * Adds a Vspace to the list, with an initial NodeSet.
     * If it already exists, does not change anything and returns false.
     *
     * @param vspace The vspace to add
     * @param n NodeSet to use initially
     * @return Whether the vspace could be added.
     */
    public synchronized boolean addVspace(String vspace, NodeSet n)
    {
	System.out.println("adding vspace: "+vspace+", resolvers: "+n);

	if (rep.containsKey(vspace))
	    return false;

	rep.put(vspace, n);

	return true;
    }


    /**
     * Removes the vspace from the list and returns whether or not 
     * it was successful. Does not modify individual nodes -- not
     *  for external use
     *
     * @param vspace The vspace to remove
     * @return Whether the vspace was found to remove
     */
    protected synchronized boolean removeVspace(String vspace) 
    {
	return (rep.remove(vspace) != null);
    }


    /**
     * Returns the resolvers known for a particular vspace
     *
     * @param vspace The vspace to look up 
     * @return The associated resolvers, with expired ones pruned.
     */
    public synchronized NodeSet getResolvers(String vspace)
    {
	return pruneExpired((NodeSet)rep.get(vspace));
    }


    /**
     * Returns all the vspaces that we know about.
     */
    Enumeration getVspaces()
    {
	return rep.keys();
    }

    
    /** 
     * Prunes all the expired nodes from the list and also
     *  the allResolvers list and applicable individual vspace
     *  lists. Should be called on anything returned to the
     *  user so that expired entries are never returned. 
     *
     * @param ns The NodeSet to filter
     * @return The filtered NodeSet
     */
    public synchronized NodeSet pruneExpired(NodeSet ns) 
    {
	if (ns == null) return null;

	long time = System.currentTimeMillis(); // "cache" time

	for (int i=0; i<ns.countNodes(); i++)
	{
	    Node n = ns.nodeNumber(i);

	    // is the node expired?
	    if (time>n.expireTime) {
		i--;

		// chop it off this set and allResolvers
		ns.remove(n);

		if (ns!=allResolvers) 
		    allResolvers.remove(n);
		
		// remove it from the vspace lists that it's in
		for (Enumeration vses=n.getVspaces(); vses.hasMoreElements();)
		{
		    String vs = (String)vses.nextElement();

		    NodeSet nst = (NodeSet)rep.get(vs);
		    if (nst != null) { // it _should_ be the case
			nst.remove(n); // take it away!
			
			// if it's empty now, delete it!
			if (nst.countNodes() == 0) {
			    rep.remove(vs);
			}
		    }
		} // for (vses.hasMoreElements())
	    } // if expired
	} // has more elements

	return ns;
    }


    /**
     * Returns all the nodes that we know of
     *
     * @return NodeSet containing all known nodes, pruned for expiration times
     */
    public synchronized NodeSet getAllResolvers()
    {
	return pruneExpired(allResolvers);
    }

    /**
     * Adds a resolver with the specified characteristics
     *
     * @param vspace An initial vspace
     * @param ia     The INR's address
     * @param udp    The INR's udp port
     * @param tcp    The INR's tcp port
     * @param ttl    Time to live in ms, Long.MAX_VALUE for infinite
     */
    boolean addResolver
	(String vspace, InetAddress ia, int udp, int tcp, long TTL)
    {
	Node newNode = new Node(ia, udp, tcp, vspace, TTL);
	return addResolver(vspace, newNode);
    }


    /**
     * Adds a resolver with the specified characteristics
     *
     * @param vspace  An initial vspace
     * @param newNode The constructed node
     */
    synchronized boolean addResolver(String vspace, Node newNode)
    {
	newNode.addVspace(vspace);

	NodeSet neigh = (NodeSet)rep.get(vspace);

	if (neigh != null) // does the vspace already exist??
	{	    
	    Node n = neigh.get(newNode);

	    if (n == null)	// nope, then add it
	    {
		// Check if AllResolvers already contains it
		Node existingNode = allResolvers.get(newNode);
		if (existingNode != null)	// newNode exists AllResolvers
		{
		    existingNode.addVspace(vspace);
		    existingNode.updateTTL(newNode.getTTL());
		    neigh.put(existingNode);
		}
		else		// AllResolvers doesn't have newNode
		{
		    allResolvers.addUnconditionally(newNode);
		    neigh.put(newNode);
		}
	    }
	    else { // it's already there, update the expiration
				
		n.updateTTL(newNode.getTTL());

		return false;	// since ia:port already in vspace
	    }
	}
	else 	// create a new Vspace
	{
	    // Check if AllResolvers already contains it
	    Node existingNode = allResolvers.get(newNode);
	    if (existingNode != null)	// newNode exists AllResolvers
	    {
		    existingNode.addVspace(vspace);
		    existingNode.updateTTL(newNode.getTTL());
		    neigh = new NodeSet(existingNode);
	    }
	    else		// AllResolvers doesn't have newNode
	    {
		    allResolvers.addUnconditionally(newNode);
		    neigh = new NodeSet(newNode);
	    }

	    rep.put(vspace, neigh);
	}

	return true;
    }



    /** 
     * Accepts a string in the form of host:UDPport:TCPport:vspace:ttl. 
     *  Used more for NodeConsole/high-level interaction
     *  Returns whether or not it was able to parse the string 
     */
    boolean addResolver(String s)
    {
	java.util.StringTokenizer st = 
	    new java.util.StringTokenizer(s,":;\n\r\t ");
	
	if (st.countTokens()<5) return false;
	try { 
	    InetAddress ia = InetAddress.getByName(st.nextToken()); 
	    int udp = Integer.parseInt(st.nextToken());
	    int tcp = Integer.parseInt(st.nextToken());
	    String vspace = st.nextToken();
	    long ttl = Long.parseLong(st.nextToken());
	    addResolver(vspace, ia, udp, tcp, ttl);
	    return true;
	}
	catch (Exception e) { return false; }
    }


    /**
     * Gets a resolver with a set of characteristics.
     * Slightly more efficient that the following function, if
     *  the vspace name is known
     *
     * @param vspace An initial vspace
     * @param ia     The INR's address
     * @param udp    The INR's udp port
     * @param tcp    The INR's tcp port
     */
    synchronized Node getResolver
	(String vspace, InetAddress ia, int udp, int tcp)
    {
	NodeSet neigh = (NodeSet)rep.get(vspace);

	if (neigh == null)
	    return null;

	Node n = neigh.get(new Node(ia, udp, tcp, vspace));
	if (n.isExpired()) 
	{
	    return null;
	} else {
	    return n;
	}
    }


    /**
     * Gets a resolver with a set of characteristics.
     *
     * @param ia     The INR's address
     * @param udp    The INR's udp port
     * @param tcp    The INR's tcp port
     */
    synchronized Node getResolver(InetAddress ia, int udp, int tcp)
    {
	Node n = allResolvers.get(new Node(ia, udp, tcp));
	if (n.isExpired()) 
	    return null;
	else 
	    return n;
    }

    /**
     * Removes a resolver from a specific vspace
     *
     * @param vspace An initial vspace
     * @param ia     The INR's address
     * @param udp    The INR's udp port
     * @param tcp    The INR's tcp port
     */
    synchronized boolean removeResolver
	(String vspace, InetAddress ia, int udp, int tcp)
    {
	NodeSet neigh =  (NodeSet)rep.get(vspace);

	Node rmNode = new Node(ia, udp, tcp, vspace);

	if (!neigh.remove(rmNode))
	    return false;

	// Check allResolvers
	Node existingNode = allResolvers.get(rmNode);

	if (existingNode == null)	// should not be null
	    return false;

	existingNode.removeVspace(vspace);
	if (existingNode.countVspace() == 0)
	{
	    allResolvers.remove(existingNode);
	}

	return true;
    }

    /**
     * Returns a String representation of the ResolverSet.
     *
     * @return String representing the structure.
     */
    public String toString()
    {
	StringBuffer strbuf = new StringBuffer();

	pruneExpired(allResolvers);

	strbuf.append("ResolverSet has " + rep.size() + " vspaces.\n");
	for (Enumeration enum = rep.keys(); enum.hasMoreElements(); )
	{
	    Object vspace = enum.nextElement();
	    Object nodeset = rep.get(vspace);

	    strbuf.append("VSPACE " + vspace.toString() + "\n");
	    strbuf.append(nodeset.toString() + "\n");
	}
	return strbuf.toString();
    }




    // test driver
    public static void main(String args[]) throws Exception {
	ResolverSet rs = new ResolverSet();
	Vector v;
	InetAddress ia = InetAddress.getLocalHost();
	
	System.out.println(rs.toString());

	v = new Vector();v.addElement("vs1");v.addElement("vs2");

	rs.integrateVspaceAdvertisement(v, ia, 123,456, 3000, true);

	rs.integrateVspaceAdvertisement(v, ia, 321,643, 3000, true);

	try { Thread.sleep(500); } catch (Exception e) {;}

	System.out.println(rs.toString());

	v=new Vector();v.addElement("vs2");

	rs.integrateVspaceAdvertisement(v, ia, 123,456, 3000, true);

	try { Thread.sleep(500); } catch (Exception e) {;}

	System.out.println(rs.toString());
	try { Thread.sleep(2000); } catch (Exception e) {;}

	System.out.println(rs.toString());

    }
}	   










