package ins.dsr;

import java.util.*;
import java.net.InetAddress;

/** VSNodes is a container class which encapsulates a 
 *  virtual space name and a set of nodes.
 *  e.g.  vspace1: [ node1 node2 ]
 */
public class VSNodes 
{
    private String vspace;
    private Vector nodes;

    // CONSTRUCTORS
    public VSNodes(String vspace, NodeSet nodes) 
    {
	this.vspace = vspace;
	this.nodes = new Vector(nodes.countNodes());
	
	Enumeration e = nodes.getNodes();
	while (e.hasMoreElements()) 
	    this.nodes.addElement(e.nextElement());
    }

    public VSNodes(String vspace, Enumeration nodes) 
    {
	this.vspace = vspace;
	this.nodes = new Vector();
	
	while (nodes.hasMoreElements()) {
	    Object o = nodes.nextElement();
	    if (o instanceof Node)
		this.nodes.addElement(o);
	}
    }

    private VSNodes(String vspace, Vector nodes) 
    {
	this.vspace = vspace;
	this.nodes = nodes;
    }


    // METHODS
    public static VSNodes readWireform 
	(byte [] data) 
    {
	return readWireform(data, 0, data.length);
    }


    public static VSNodes readWireform
	(byte [] data, int offset, int length) 
    {
	if (offset+length>data.length) return null;
	if (length<4) return null;
	
	int sl = Conversion.extract16toIntLow16(data, offset);
	if (sl+4>length) return null;
	String vspace = new String(data, offset+2, sl);

	int entries = Conversion.extract16toIntLow16
	    (data, offset+2+sl);

	if ((sl+4+(entries*Node.WIRELENGTH))>length)
	    return null;
	Vector nodes = new Vector(entries);

	int pos = offset+sl+4;
	for (int i=0; i<entries; i++) {
	    Node n = Node.readWireform(data, pos);
	    if (n!=null) nodes.addElement(n);
	    pos+=Node.WIRELENGTH;
	}

	return new VSNodes(vspace, nodes);
    }

    public int getByteLength() 
    {
	byte [] s = vspace.getBytes();

	return (nodes.size()*Node.WIRELENGTH)+4+s.length;
    }

    public byte [] toBytes() 
    {
	byte [] s = vspace.getBytes();
	
	int size = nodes.size();
	byte [] result = new byte
	    [(size*Node.WIRELENGTH)+4+s.length];

	Conversion.insertIntLow16(result, 0, s.length);
	System.arraycopy(s, 0, result, 2, s.length);
	Conversion.insertIntLow16(result, 2+s.length, nodes.size());
	
	int pos = 4+s.length;
	for (int i=0; i<size; i++) 
	    pos = ((Node)nodes.elementAt(i))
		.copyBytes(result, pos);
	
	return result;
    }


    public int copyInto(byte [] data, int offset)
    {
	byte [] s = vspace.getBytes();
	
	int size = nodes.size();

	Conversion.insertIntLow16(data, offset, s.length);
	System.arraycopy(s, 0, data, 2+offset, s.length);
	Conversion.insertIntLow16
	    (data, 2+s.length+offset, nodes.size());
	
	int pos = 4+s.length+offset;
	for (int i=0; i<size; i++) 
	    pos = ((Node)nodes.elementAt(i))
		.copyBytes(data, pos);
	
	return pos;
    }

    public String toString() 
    {
	return (vspace==null?"null":vspace)+": "+
	    (nodes==null?"[ ]":nodes.toString());
    }

    public int hashCode()
    {
	return vspace.hashCode()+nodes.size();
    }

    public boolean equals(Object o) 
    {
	if (o==null || !(o instanceof VSNodes)) return false;
	
	VSNodes v = (VSNodes)o;

	if (vspace == null) {
	    if (v.vspace != null) return false;
	} else {
	    if (!vspace.equals(v.vspace))
		return false;
	}
	
	if (nodes == null) {
	    if (v.nodes != null) return false;
	} else {
	    if (!nodes.equals(v.nodes))
		return false;
	}
	
	return true;
    }

    public String getVspace()
    {
	return vspace;
    }

    public Enumeration getNodes()
    {
	return nodes.elements();
    }

    public boolean containsNode(InetAddress ia, int UDPport, int TCPport)
    {
	Node n = new Node(ia, UDPport, TCPport);

	return nodes.contains(n);
    }


    // test driver
    /*    public static void main(String args[]) throws Exception
    {
	InetAddress ia = InetAddress.getLocalHost();
	Node n1 = new Node(ia, 123, 456, 5000);
	Node n2 = new Node(ia, 321, 654, 5000);

	Vector v = new Vector();
	v.addElement(n1);v.addElement(n2);

	VSNodes vn = new VSNodes("vspace1", v);
	System.out.println(vn.toString());

	byte [] b = new byte[100];
	int sz = vn.copyInto(b, 0);
	System.out.println("to "+sz+" bytes");
	vn = VSNodes.readWireform(b,0,sz);
	System.out.println("now\n"+vn);

	v= new Vector();
	v.addElement(vn);
	DSRResponse dr = 
	    DSRResponse.makeVspacesResolvers(v.elements(), 12345678);

	System.out.println("dsrresponse ver "+dr.toString());
	b = dr.toBytes();
	System.out.println("converted "+b);
	dr = DSRResponse.readWireform(b);
	System.out.println("after "+dr.toString());
	}*/


}
