package ins.inr;

import java.util.*;
import java.io.*;
import java.net.InetAddress;
import ins.namespace.*;

/**
 *  AppManager deals with requests from the application other than routing.
 *  This includes 
 *    <li> listening to announcements 
 *    <li> discovery requests
 *    <li> early binding requests
 *
 * This class is divided into sections based on these three parts 
 */

class AppManager 
    implements IHandler
{
    // VARIABLES
    protected Resolver resolver;
    protected VSNameTree nameTrees;
    protected Communicator comm;
    protected VSRouteTable routeTables;
    protected RouteManager routeMn;
    protected DSRManager dsrMn;
    protected VSResolvers resolvers;
    protected Forwarder forwarder;

    protected NameSpecifier announceNS, discoverNS, earlybindNS;
    protected NameSpecifier sourceNS, discoverReplyNS, earlybindReplyNS;

    static Attribute controlmsgAttr = new Attribute("control-msg");
    static Value announceVal = new Value("announcement");
    static Value discoverVal = new Value("discovery");
    static Value earlybindVal = new Value("early-binding");
    static Value discoverReplyVal = new Value("discovery-reply");
    static Value earlybindReplyVal = new Value("early-binding-reply");

    static Attribute childAttr = new Attribute("child");
    static NameSpecifier childstar = new NameSpecifier("[child=*]");

    
    // for forwarding discovery/earlybinding messages back
    RespondRecord [] respondRecords;
    final int RR_SIZE = 16;
    int nextRR = 0;


    protected RouteEntry myuidRE;
    final static byte[] zerobyte = new byte[0];

    PrintWriter pw;
    boolean isLog;

    // CONSTRUCTORS
    AppManager()
    {
	if (Resolver.DEBUG>=2) {
	    isLog = true;
	    try{
		String logfile = "AppManager.log";
		pw = new PrintWriter(new FileWriter(logfile), true);
	    } catch (IOException e) {
		isLog = false;
	    }
	} 
	else
	   isLog = false;

	respondRecords = new RespondRecord[RR_SIZE];
	for (int i=0; i<RR_SIZE; i++) respondRecords[i] = null;
    }

    // METHODS

    /** 
     *  A callback function: will be called during initialization
     *  Provide an access to the VSNameTree and Forwarder objects
     *  (Interface implementation of IHandler)
     *  @param r the Resolver object
     */
    public void init(Resolver r)
    {
	resolver = r;
	nameTrees = r.nameTrees;
	comm = r.comm;
	routeTables = r.routeTables;
	routeMn = r.routeMn;
	dsrMn = r.dsrMn;
	resolvers = r.resolvers;
	forwarder = r.forwarder;

	myuidRE = routeTables.lookup(resolver.defaultVSpace, resolver.INRuid);
	if (myuidRE == null)	// fatal error, should not happen
	{
	    printStatus("Fatal Error: routeTables is not correctly" +
		"initialzed yet inside AppManager.init().");
	    System.exit(-1);
	}	

	announceNS = new NameSpecifier();
	announceNS.addAVelement(new AVelement(controlmsgAttr, announceVal));
	discoverNS = new NameSpecifier();
	discoverNS.addAVelement(new AVelement(controlmsgAttr, discoverVal));
	earlybindNS = new NameSpecifier();
	earlybindNS.addAVelement(new AVelement(controlmsgAttr, earlybindVal));
	discoverReplyNS = new NameSpecifier();
	discoverReplyNS.addAVelement
	    (new AVelement(controlmsgAttr, discoverReplyVal));
	earlybindReplyNS = new NameSpecifier();
	earlybindReplyNS.addAVelement
	    (new AVelement(controlmsgAttr, earlybindReplyVal));

	//announceNS = new NameSpecifier("[control-msg=announcement]");
	//discoverNS = new NameSpecifier("[control-msg=discovery]");
	//earlybindNS = new NameSpecifier("[control-msg=early-binding]");
	//discoverReplyNS = new NameSpecifier("[control-msg=discovery-reply]");
	//earlybindReplyNS = 
	//    new NameSpecifier("[control-msg=early-binding-reply]");


	// Note: getHostName() will use ip addr if DNS name doesn't exist
	// but better use getHostAddress() since Java for Windows doesn't
	// correctly get the name if the ip suddenly changes
	// (it may cache the name somewhere)
	sourceNS = new NameSpecifier(
	     "[host=" + comm.localhost.getHostAddress() +
	     "][UDPport=" + comm.UDPport + "]" +
	     "][TCPport=" + comm.TCPport + "]");

	// Announcement, discovery and early-binding messages for all vspaces
	// are destined to me
	Enumeration enum = nameTrees.getVspaces();
	// System.out.println(nameTrees.toString());
	while (enum.hasMoreElements())
	{
	    produceVspace((String)(enum.nextElement()));
	}

    }

    /** 
     * Performs actions necessary to add vspace for this module.
     * Specifically, it registers handlers in the vspace's nametree
     * for its functionality.
     */
    protected void produceVspace(String vspace)
    {
	NameRecord nr;

	announceNS.setVspace(vspace);
	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTrees.addNameRecord(announceNS, vspace, nr);
	
	discoverNS.setVspace(vspace);
	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTrees.addNameRecord(discoverNS, vspace, nr);

	earlybindNS.setVspace(vspace);
	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTrees.addNameRecord(earlybindNS, vspace, nr);

	discoverReplyNS.setVspace(vspace);
	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTrees.addNameRecord(discoverReplyNS, vspace, nr);

	earlybindReplyNS.setVspace(vspace);
	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTrees.addNameRecord(earlybindReplyNS, vspace, nr);
    }

    
    /***********************************************************/
    /* Announcements */

    /**
     * Process a name announcement coming from the Client Library,
     *  advertising a service.
     * @param msg NameAnnouncement from the application
     */
    protected void processAnnouncement(Message msg)
    {
	if (Resolver.DEBUG >= 3)
	    printStatus("received an announcement from app:");

	// ask RouteManager to disseminate the source name if necessary
	if (routeMn == null)
	{
	    printStatus("WARNING: Packet arrives before " +
		   "RouteManager is ready.. source-name is not disseminated!");
	    printStatus("The packet is \n" + msg.packet.toString());
	    return;
	}

	NameUpdate nu = new NameUpdate(msg.packet.data);
	// if (Resolver.DEBUG >= 4) printStatus(nu.toString());

	// if name announced has no vspace then use dest name's vspace
	// if dNS contains no vspace, ignore announcement
	String vspace = nu.ns.getVspace();
	if (vspace == null)
	{
	    vspace = msg.packet.dNS.getVspace();
	    if (vspace == null)
	    {
		printStatus("WARNING: no vspace info, packet dropped!");
		return;
	    }
	    nu.ns.setVspace(vspace);
	}
	
	System.out.println("received ad for: "+nu.ns);
	
	Vector v = null;
	v = nameTrees.lookupExactMatch(nu.ns, nu.announcer);

	switch (nu.action)
	{
	case NameUpdate.ADD_NAME:  // ***** ADD_NAME / UPDATE_NAME
	case NameUpdate.UPDATE_NAME:  // ***** ADD_NAME / UPDATE_NAME

	{	

	if (v == null)	// null = no vspace info
	{	// should not occur though since we have checked above
	    printStatus("Internal error: null vspace encountered!");
	    return;
	}

	if (v.size() == 0)	// name not exist yet
	{
	    // Use MAX_NAME_TTL since we are the first-hop INR (i.e., do soft 
	    // state)
	    // note: below set routeEntry to myuidRE to mean next hop
	    // is application node.
	    NameRecord nr = new NameRecord(resolver.INRuid, myuidRE,
		//new HostRecord(msg.ia, msg.port, 0),
		nu.hostRecord,
		nu.appMetric, RouteManager.MAX_NAME_TTL, 
		0, true, nu.announcer);

	    nameTrees.addNameRecord(nu.ns, null, nr);

	    // Disseminate name now since I'm first-hop INR
	    routeMn.sendNameUpdate(nu.ns, nr, NameUpdate.ADD_NAME, null);
	} 
	else
	{
	    // update name
	    NameRecord nr = (NameRecord)(v.elementAt(0));

	    boolean diffNameRecord = false;

	    // Update name record to reflect possibly new IP address
	    // and port of app if mobile app...
	    HostRecord hr = nr.getHostRecord();
	    HostRecord newhr = nu.hostRecord;

	    if (!Conversion.byteCompare(newhr.address, hr.address, 4))
	    {
		hr.address = newhr.address;
		diffNameRecord = true;
	    }

	    if (hr.UDPport != newhr.UDPport)
	    {
		hr.UDPport = newhr.UDPport;
		diffNameRecord = true;
	    }

	    if (hr.TCPport != newhr.TCPport)
	    {
		hr.TCPport = newhr.TCPport;
		diffNameRecord = true;
	    }

	    // this assumes that an add/delete do not happen simultaneously
	    if (hr.transport.size() !=  newhr.transport.size())
	    {
		hr.transport = newhr.transport;
		hr.port = newhr.port;
		diffNameRecord=true;
	    }

	    // the first-hop INR is responsible for soft-state
	    // needs to refresh entry's TTL
	    nr.setExpireTime(RouteManager.MAX_NAME_TTL);

	    // mobility case: if nr.getINRuid() is different from my uid
	    // (i.e., resolver.uid) that means app has attached to me 
	    // now from different INR
	    if (nr.getINRuid() != resolver.INRuid)
	    {
		nr.setINRuid(resolver.INRuid);
		nr.setRouteEntry(myuidRE);			
		diffNameRecord = true;
	    }
		    
	    if (diffNameRecord)  // if diff and 1st-hop INR
		routeMn.sendNameUpdate(nu.ns, nr, NameUpdate.UPDATE_NAME,
			 null);
	}   // else of if (v.size()==0)

	break;

	}  // ****** end of ADD_NAME / UPDATE_NAME

	case NameUpdate.REMOVE_NAME:  // ***** REMOVE_NAME
	{
	    if (v == null)	// null = no vspace info
		return;
	    else if (v.size() > 0)	// name exist
	    {
		NameRecord nr = (NameRecord)(v.elementAt(0));

		nameTrees.removeNameRecord(vspace, nr);
		routeMn.sendNameUpdate(nu.ns, nr, NameUpdate.REMOVE_NAME,
			 null);
	    }

	break;

	}  // ***** end of REMOVE_NAME

	}  // switch(nu.action)
    }




    /**************************************************************/
    /* Discovery part */

    /**
     * Processes a Discovery request from the application --
     *
     * Called whenever a [control-msg=discovery] request is received
     * Responsible for getting an appropriate response back.
     */
    protected void processDiscovery(Message msg)
    {
	if (Resolver.DEBUG >= 3)
	    printStatus("received a discovery request from app:");

	if (msg.packet.data.length<11) return;

	String defaultVspace = msg.packet.dNS.getVspace();

	// get info from the message
	int sequenceNo = Conversion.extract32toInt(msg.packet.data, 0);
	InetAddress addr = Conversion.extractInetAddress(msg.packet.data, 4);
	int udpport = Conversion.extract16toIntLow16(msg.packet.data, 8);
	int discoverSteps = msg.packet.data[10];

	String nsstr = 
	    new String(msg.packet.data, 11, msg.packet.data.length-11);
	NameSpecifier ns = new NameSpecifier(nsstr);

	// is this a wide area request? if so, make the names "wide area"
	boolean makeVspaceFull = false;

	// make sure there is a request vspace
	String vsp = ns.getVspace();
	if (vsp == null) vsp = defaultVspace;
	if (vsp == null) 
	{
	    printStatus("No vspace on request "+nsstr);
	    return;
	}

	// if we got a wide area request, change it to a local
	//  area request, but set makeVspaceFull flag to set back
	//  to full form on leaving
	String nvsp = dsrMn.standardizeVspace(vsp);
	if (vsp != nvsp) 
	{
	    ns.setVspace(nvsp);
	    vsp = nvsp;
	    makeVspaceFull = true;
	}

	String ourVspace = ns.getVspace();
	String returnVspace = msg.packet.sNS.getVspace();

	// printStatus("processing discovery "+ns.toString());

	// do the lookup
	Vector foundNSes = new Vector();
	Vector unfoundVSes = new Vector();
	recursiveLookup(ns, discoverSteps, 
			foundNSes, unfoundVSes,
			makeVspaceFull, true);

	// if we could find eveything locally, we're set!
	if (unfoundVSes.size()==0)
	{
	    sendDiscoveryResponse(foundNSes, sequenceNo, addr, udpport, 
				  ourVspace, returnVspace);
	    return;
	}

	// we didn't find everything locally - so we record what we know,
	//  then send requests for the unknown vspaces.
	// since we're single-threaded, we don't block, but rather
	//   once we receive all the appropriate responses,
	//   we ship the result off
	// if a response is never received, we never send anything,
	//   so some higher level process can timeout and retransmit
	//   the request. we have a limited number of simultaneous
	//   discovery slots to prevent info from staying around forever.

	// set of record so that we can deal with a response
	RespondRecord rr = new RespondRecord
	    (addr, udpport, foundNSes, ourVspace, returnVspace, sequenceNo,
	     makeVspaceFull, false /*nocare*/);

	addToRespondTable(rr);

	// send discovery request for each unknown vspace
	for (Enumeration e=unfoundVSes.elements(); e.hasMoreElements(); )
	{
	    String vspace = (String)e.nextElement();

	    ns.setVspace(vspace);
	 
	    // create sequenceNo, add it to respond table
	    int seqNo = dsrMn.getNextSequenceNo();

	    // construct data packet
	    byte [] nsbytes = ns.toString().getBytes();
	    byte [] bytes = new byte[nsbytes.length+11];

	    Conversion.insertInt(bytes, 0, seqNo);
	    System.arraycopy(comm.localhost.getAddress(), 0, bytes, 4, 4);
	    Conversion.insertIntLow16(bytes, 8, comm.UDPport);
	    bytes[10] = (byte)(4); //////////////////////// maxSteps -1!
	    System.arraycopy(nsbytes, 0, bytes, 11, nsbytes.length);
	    
	    // set vspaces on packet to send
	    sourceNS.setVspace(ourVspace);
	    discoverNS.setVspace(vspace);

	    // where do we send it to?
	    Node node = dsrMn.getNodeInVspace(vspace);
	    if (node == null) 
		continue;

	    // add it to the respond records
	    rr.sequenceNos.addElement(new Integer(seqNo));

	    // off it goes!
	    Packet packet = new Packet
		(sourceNS, discoverNS, 8, Packet.toAny, false, 
		 zerobyte, bytes);

	    comm.sendMessage(new Message(packet, node.ia, node.UDPport));
	}
	
	rr.haveAllSequenceNos = true;

	// we don't send anything back to the original host,
	//   at least not here -- the processDiscoveryResponse code does that.
	
    }

    /**
     * We want to allow some resolvers to send partial responses with slightly
     * different destination name-specifiers by calling directly the function
     * called within this method
     * @author Magdalena Balazinska
     */
    void sendDiscoveryResponse(Vector nset, int sequenceNo,
			       InetAddress addr, int udpport, 
			       String ourVspace, String returnVspace)
    {
	sendDiscoveryResponse(nset,sequenceNo,addr,udpport,ourVspace,returnVspace,discoverReplyNS);
    }

    /**
     * Sends the response to a discovery request.
     * This is a separate function since it can be called both
     *   from processDiscovery() and processDiscoveryReply() 
     *      (due to aggregation)
     */
    void sendDiscoveryResponse(Vector nset, int sequenceNo,
			       InetAddress addr, int udpport, 
			       String ourVspace, String returnVspace, 
			       NameSpecifier dest)
    {
	
	Packet packet = makeDiscoveryResponsePacket(nset,sequenceNo,
						    ourVspace,returnVspace,dest);
	if ( packet != null )
	    comm.sendMessage(new Message(packet, addr, udpport));
	

    }

    /**
     * Decoupled the creation of the response packet
     * from the act of sending the packet so we can reuse
     * this function even when sending in a different manner.
     */
    public Packet makeDiscoveryResponsePacket(Vector nset, int sequenceNo,
					      String ourVspace, String returnVspace, 
					      NameSpecifier dest) {

	if (nset == null) return null;
	
	int len=6; // basic reply has sequence no, and terminating null string
	int limit=nset.size();
	int addition;
	int i=0;
	
	Vector names = new Vector(nset.size());
	for (Enumeration e=nset.elements(); e.hasMoreElements(); ) {
	    String strcurNS = (String)e.nextElement();
	    names.addElement(strcurNS);
	    addition = 2 + strcurNS.length();
	    if (len + addition > 8192) {
		limit = i-1;
		break;
	    }
	    len = len + addition;
	    i++;
	}

	if (len>8192) {
	    printStatus("WARNING: early-binding info has length >8192");
	    printStatus(".. only " + len + " bytes will be sent");
	}
	
	byte[] bytes = new byte[len];
	Conversion.insertInt(bytes, 0, sequenceNo);
	int ct=4;
	
	for (i=0; i<limit; i++)
	    {
		String str = (String)(names.elementAt(i));
		byte[] strBytes = str.getBytes();
		Conversion.insertIntLow16(bytes, ct, strBytes.length);
		System.arraycopy(strBytes, 0, bytes, ct+2, strBytes.length);
		ct = ct+2+strBytes.length;
	    }
	
	Conversion.insertIntLow16(bytes, ct, 0); // send a final 0-length

	sourceNS.setVspace(ourVspace);
	//discoverReplyNS.setVspace(returnVspace);
	dest.setVspace(returnVspace);

	//Packet packet = new Packet
	//  (sourceNS, discoverReplyNS, 8, Packet.toAny, false, 
	//   zerobyte, bytes);

	Packet packet = new Packet
	    (sourceNS, dest, 8, Packet.toAny, false, 
	     zerobyte, bytes);

	return packet;
    }


    /**
     * Processes a reply from a child vspace
     * (For aggregate vspaces in which some of the children are not
     *  locally hosted, they need to be contacted - this deals with
     *  the reply coming back)
     */
    protected void processDiscoveryReply(Message msg)
    {
	byte[] data = msg.packet.data;
	if (data.length<6)
	    return;

	int sequenceNo = Conversion.extract32toInt(data, 0);

	RespondRecord rr = getRespondRecord(new Integer(sequenceNo));
	if (rr==null) return;

	Vector result = new Vector();
	
	// parse the response into a Vector of strings of
	//     normalized namespecifiers
	int pos = 4;
	while (pos < data.length)
	{
	    int len = Conversion.extract16toIntLow16(data, pos);
	    if (pos+len+2>data.length || len==0) break;

	    String nsstr = new String(data, pos+2, len);
	    NameSpecifier ns = new NameSpecifier(nsstr);

	    // adjust for the wide area
	    String vspace = ns.getVspace();
	    String nvspace = dsrMn.standardizeVspace(vspace);
	    if (vspace != nvspace)
		ns.setVspace(nvspace);
	    if (rr.makeVspaceFull)
		dsrMn.extendToWideArea(ns);

	    result.addElement(ns.toString());
	    
	    pos+=(len+2);
	}

	if (rr.checkInNumber(new Integer(sequenceNo), result.elements()))
	{
	    purgeRespondRecord(rr.theirSequenceNo); // yes this is safe
	    sendDiscoveryResponse(rr.result, rr.theirSequenceNo, rr.addr, 
				  rr.port, rr.ourvspace, rr.resultvspace);

	}
    }


    /*********************************************************************/
    /* Earlybinding Part */

    /**
     * Processes an EarlyBinding message from an application
     * @param msg Message from the application
     */
    protected void processEarlyBinding(Message msg)
    {
	if (Resolver.DEBUG >= 3)
	    printStatus("received an early-binding request from app:");

	if (msg.packet.data.length<11)  return;

	int sequenceNo = Conversion.extract32toInt(msg.packet.data, 0);
	InetAddress addr = Conversion.extractInetAddress(msg.packet.data, 4);
	int udpport = Conversion.extract16toIntLow16(msg.packet.data, 8);
	boolean all = (msg.packet.data[11]!=0);

	String s = new String(msg.packet.data, 11, msg.packet.data.length-11);
	NameSpecifier ns = new NameSpecifier(s);

	//System.out.println("processing early binding "+s);

	// make sure the request vspace is alright
	String vsp = ns.getVspace();
	String nvsp = dsrMn.standardizeVspace(vsp);
	if (vsp != nvsp) ns.setVspace(nvsp);

	String defaultVspace = msg.packet.dNS.getVspace();

	String ourVspace = ns.getVspace();
	String returnVspace = msg.packet.sNS.getVspace();

	// do the lookup
	Vector foundEBes = new Vector();
	Vector unfoundVSes = new Vector();
	recursiveLookup(ns, 4, 
			foundEBes, unfoundVSes,
			false /*don't care*/, false);

	// if we could find eveything locally, we're set!
	if (unfoundVSes.size()==0)
	{
	    sendEarlyBindingResponse(foundEBes, sequenceNo, addr, udpport, 
				     ourVspace, returnVspace, all);
	    return;
	}

	// we didn't find everything locally - so we record what we know,
	//  then send requests for the unknown vspaces.
	// since we're single-threaded, we don't block, but rather
	//   once we receive all the appropriate responses,
	//   we ship the result off
	// if a response is never received, we never send anything,
	//   so some higher level process can timeout and retransmit
	//   the request. we have a limited number of simultaneous
	//   discovery slots to prevent info from staying around forever.

	// set of record so that we can deal with a response
	RespondRecord rr = new RespondRecord
	    (addr, udpport, foundEBes, ourVspace, returnVspace, sequenceNo,
	     false /*don't care*/, all);

	addToRespondTable(rr);

	// send discovery request for each unknown vspace
	for (Enumeration e=unfoundVSes.elements(); e.hasMoreElements(); )
	{
	    String vspace = (String)e.nextElement();

	    ns.setVspace(vspace);
	 
	    // create sequenceNo, add it to respond table
	    int seqNo = dsrMn.getNextSequenceNo();

	    // construct data packet
	    byte [] nsbytes = ns.toString().getBytes();
	    byte [] bytes = new byte[nsbytes.length+11];

	    Conversion.insertInt(bytes, 0, seqNo);
	    System.arraycopy(comm.localhost.getAddress(), 0, bytes, 4, 4);
	    Conversion.insertIntLow16(bytes, 8, comm.UDPport);
	    bytes[10] = (byte)(all?1:0);
	    System.arraycopy(nsbytes, 0, bytes, 11, nsbytes.length);
	    
	    // set vspaces on packet to send
	    sourceNS.setVspace(ourVspace);
	    earlybindNS.setVspace(vspace);

	    // where do we send it to?
	    Node node = dsrMn.getNodeInVspace(vspace);
	    if (node == null) 
		continue;

	    // add it to the respond records
	    rr.sequenceNos.addElement(new Integer(seqNo));

	    // off it goes!
	    Packet packet = new Packet
		(sourceNS, earlybindNS, 8, Packet.toAny, false, 
		 zerobyte, bytes);

	    comm.sendMessage(new Message(packet, node.ia, node.UDPport));
	}
	
	rr.haveAllSequenceNos = true;

	// we don't send anything back to the original host,
	// at least not here -- the processEarlyBindingReply code does that.

    }

    /**
     * We want to allow some resolvers to send partial responses with slightly
     * different destination name-specifiers by calling directly the function
     * called within this method
     * @author Magdalena Balazinska
     */
    void sendEarlyBindingResponse(Vector rset, int sequenceNo,
				  InetAddress addr, int udpport, 
				  String ourVspace, String returnVspace,
				  boolean all)
    {
	NameSpecifier dest = new NameSpecifier(earlybindReplyNS);
	sendEarlyBindingResponse(rset,sequenceNo,addr,udpport,ourVspace,returnVspace,all,dest);
    }


    /**
     * Sends response for an early-binding request
     */
    void sendEarlyBindingResponse(Vector rset, int sequenceNo,
				  InetAddress addr, int udpport, 
				  String ourVspace, String returnVspace,
				  boolean all, NameSpecifier dest)
    {

	Packet packet = makeEarlyBindingResponsePacket(rset,sequenceNo,
						       ourVspace,returnVspace,all,dest);
	if ( packet != null )
	    comm.sendMessage(new Message(packet, addr, udpport));
    }


    /**
     * Decoupling the creation of the response packet
     * from the transmission itself. 
     */
    public Packet makeEarlyBindingResponsePacket(Vector rset, int sequenceNo,
						 String ourVspace, String returnVspace,
						 boolean all, NameSpecifier dest) {

	if (rset == null) return null;

	// if only best/anycast, then size the set down to one max
	if (!all) {
	    if (rset.size()>0) {
		int bestMetric=Integer.MIN_VALUE; EBRecord best = null;
		
		for (int i=0; i<rset.size(); i++) {
		    EBRecord thisItem = ((EBRecord)rset.elementAt(i));
		    int thisMetric = thisItem.appMetric;
		    if (thisMetric > bestMetric) 
		    {
			best = thisItem; bestMetric = thisMetric;
		    }
		}
		
		rset.removeAllElements();
		rset.addElement(best);
	    }
	}


	int len=10; // minimum length includes sequence number, final null
	int limit=rset.size();
	int addition;
	for (int i=0; i<rset.size(); i++)
	{
	    addition = 6 + ((EBRecord)rset.elementAt(i)).hostRecord.length;
	    if (len + addition > 8192)
	    {
		limit = i-1;
		break;
	    }
	    len = len + addition;
	}

	if (len>8192) {
	    printStatus("WARNING: early-binding info has length >8192");
	    printStatus(".. only " + len + " bytes will be sent");
	}

	byte[] bytes = new byte[len];
	Conversion.insertInt(bytes, 0, sequenceNo);
	int ct=4;

	EBRecord ebr;
	for (int i=0; i<limit; i++)
	{
	    ebr = (EBRecord)rset.elementAt(i);
	    byte [] hrbytes = ebr.hostRecord;
	    Conversion.insertInt(bytes, ct, ebr.appMetric);
	    Conversion.insertIntLow16(bytes, ct+4, hrbytes.length);
	    System.arraycopy(ebr.hostRecord, 0, bytes, ct+6, hrbytes.length);
	    ct = ct+6+hrbytes.length;
	}

	Conversion.insertInt(bytes, ct, 0);  // insert final 0's
	Conversion.insertIntLow16(bytes, ct+4, 0); 

	NameSpecifier src = new NameSpecifier(sourceNS);
	//NameSpecifier dest = new NameSpecifier(earlybindReplyNS);

	src.setVspace(ourVspace);
	dest.setVspace(returnVspace);

	Packet packet = new Packet
	  (src, dest, 8, Packet.toAny, false, zerobyte, bytes);
	return packet;
    }


    /**
     * Processes an early-binding reply from a child vspace
     * (For aggregate vspaces in which some of the children are not
     *  locally hosted, they need to be contacted - this deals with
     *  the reply coming back)
     */
    protected void processEarlyBindingReply(Message msg)
    {

	byte[] data = msg.packet.data;
	if (data.length<10)
	    return;

	int sequenceNo = Conversion.extract32toInt(data, 0);

	RespondRecord rr = getRespondRecord(new Integer(sequenceNo));
	if (rr==null) return;

	Vector result = new Vector();
	
	// parse the response into a Vector of strings of
	//     normalized namespecifiers
	int pos = 4;
	while (pos < data.length)
	{
	    int metric = Conversion.extract32toInt(data, pos);
	    int len = Conversion.extract16toIntLow16(data, pos+4);

	    if (pos+len+6>data.length || len==0) break;

	    byte [] hrbytes = new byte[len];
	    System.arraycopy(data, pos+6, hrbytes, 0, len);
	    
	    result.addElement(new EBRecord(metric, hrbytes));
	    
	    pos+=(len+6);
	}

	// we use early-binding both for regular early-binding discovery
	// and aggregate anycast
	if (!rr.isAnycast())
	{
	    // "normal" early-binding case

	    // check in, and if we're done, send the response!
	    if (rr.checkInNumber(new Integer(sequenceNo), result.elements()))
	    {
		purgeRespondRecord(rr.theirSequenceNo); // yes this is safe
		sendEarlyBindingResponse(rr.result, rr.theirSequenceNo, 
					 rr.addr, rr.port, rr.ourvspace, 
					 rr.resultvspace, rr.all);
		
	    }
	} else {
	    // aggregate anycast case

	    // check in and see if we're done. If so, send response
	    if (rr.checkInAnycast(new Integer(sequenceNo), result.elements()))
	    {
		purgeRespondRecord(rr.theirSequenceNo); // yes this is safe

		Object o = ((Object[])rr.result.elementAt(0))[0];
		Object[] pkg = ((Object[])rr.result.elementAt(1));
		Packet packet = (Packet)pkg[0];

		if (o instanceof NameRecord) {
		    packet.dNS.setVspace(rr.resultvspace);
		    forwarder.sendAnycastToNameRecord
			((NameRecord)o, packet,
			 /*from*/ (Node)pkg[1], 
			 /*isTCP*/ ((Boolean)pkg[2]).booleanValue());

		} else {
		    String vspace = (String)o;
		    Node node = dsrMn.getNodeInVspace(vspace);
		    if (node != null) {
			packet.dNS.setVspace(vspace);
			comm.sendMessage(new Message(packet, node.ia, 
						     node.UDPport));
		    }
		} // if (instanceof ...	
	    } // anycast fully checked in
	} // aggregate anycast

    }


    /***********************************************************/
    /* Aggregate anycast support code */


    void doAggregateAnycast(String bestVspace, NameRecord bestNR,
			    int bestMetric, Packet p, Vector unfoundVSes,
			    Node from, boolean isTCP) 
    {
	Object [] best = new Object[] { bestNR, 
					new Integer(bestMetric) };
	Object [] sendInfo = new Object[] { p, from, new Boolean(isTCP) };
	Hashtable seqNoMap = new Hashtable(13);

	Vector encodedInfo = new Vector(3);
	encodedInfo.addElement(best); encodedInfo.addElement(sendInfo);
	encodedInfo.addElement(seqNoMap);

	NameSpecifier ns = p.dNS;
	String ourVspace = ns.getVspace();

	// set of record so that we can deal with a response
	RespondRecord rr = new RespondRecord
	    (null, -1234, encodedInfo, ourVspace, bestVspace, 
	     dsrMn.getNextSequenceNo(),
	     false /*don't care*/, false /* anycast, not all */);
    
	addToRespondTable(rr);

	// create and send an early binding request!!
	for (Enumeration e=unfoundVSes.elements(); e.hasMoreElements(); )
	{
	    String vspace = (String)e.nextElement();

	    ns.setVspace(vspace);
	 
	    // create sequenceNo, add it to respond table
	    int seqNo = dsrMn.getNextSequenceNo();
	    Integer IseqNo = new Integer(seqNo);

	    seqNoMap.put(IseqNo, vspace);

	    // construct data packet
	    byte [] nsbytes = ns.toString().getBytes();
	    byte [] bytes = new byte[nsbytes.length+11];

	    Conversion.insertInt(bytes, 0, seqNo);
	    System.arraycopy(comm.localhost.getAddress(), 0, bytes, 4, 4);
	    Conversion.insertIntLow16(bytes, 8, comm.UDPport);
	    bytes[10] = (byte)(0); // anycast!
	    System.arraycopy(nsbytes, 0, bytes, 11, nsbytes.length);
	    
	    // set vspaces on packet to send
	    sourceNS.setVspace(ourVspace);
	    earlybindNS.setVspace(vspace);

	    // where do we send it to?
	    Node node = dsrMn.getNodeInVspace(vspace);
	    if (node == null) 
		continue;

	    // add it to the respond records
	    rr.sequenceNos.addElement(IseqNo);

	    // off it goes!
	    Packet packet = new Packet
		(sourceNS, earlybindNS, 8, Packet.toAny, false, 
		 zerobyte, bytes);

	    comm.sendMessage(new Message(packet, node.ia, node.UDPport));
	}
	
	rr.haveAllSequenceNos = true;
    }




    /***********************************************************/
    /* Common aggregate vspace code */

    /**
     * Given a nametree of an aggregate vspace, returns a vector of all the
     *  child vspace string names (i.e. all the [child=x] "services")
     */
    Vector getChildVspaces(NameStoreInterface nt)
    {
	NameRecord []nr = nt.lookup(childstar);  // look up all _children_
	
	Vector vspaces = new Vector(nr.length+1);
	
	for (int i=0; i<nr.length; i++) 
	{
	    NameSpecifier cNS = nt.extract(nr[i]);
	    AVelement cAVE = cNS.getAVelement(childAttr);
	    
	    if (cAVE != null) 
	    {
		Value cVal = cAVE.getValue();
		if (!cVal.isWildcard())
		    vspaces.addElement(cVal.toString());
	    }
	}

	return vspaces;
    }

    /**
     * Respond Table
     */
    void addToRespondTable(RespondRecord rr)
    {
	int writeAt;
	for (writeAt = 0; writeAt< RR_SIZE; writeAt++)
	    if (respondRecords[writeAt]==null) break;

	if (writeAt >= RR_SIZE) {
		writeAt = nextRR;
		nextRR = (nextRR+1) % RR_SIZE;
	}

	respondRecords[writeAt] = rr;
    }


    RespondRecord getRespondRecord(Integer sequenceNo)
    {
	for (int i = 0; i< RR_SIZE; i++) {
	    RespondRecord rr = respondRecords[i];

	    if (rr != null) {
		int ind = rr.sequenceNos.indexOf(sequenceNo);
		if (ind > -1)
		    return rr;
	    }
	}
	return null;
    }

    // cleans up the rr
    void purgeRespondRecord(int theirSequenceNo)
    {
	for (int i = 0; i< RR_SIZE; i++) {
	    RespondRecord rr = respondRecords[i];

	    if (rr!= null) {
		if (rr.theirSequenceNo == theirSequenceNo) {
		    respondRecords[i] = null;
		    return;
		}
	    }
	}
    }


    /**
     * Does a lookup for discovery or early binding on a NameSpecifier filter.
     * The recursive nature is necessary due to aggregation
     *
     * @param ns   Filter with top-level vspace set  [service=*][vspace=test]
     * @param maxSteps        Max discovery hops -- if set to 0, returns
     *                          names of aggregate vspaces at this level
     *                          rather than proceeding recursively
     * @param foundNSes       Vector to put all the discovered local 
     *                          NameSpecifier Strings (discovery) or
     *                          NameRecords (early binding).
     * @param unfoundVSes     Vector to put all the unhost vspaces
     * @param makeVspaceFull  Do we have to cannonicalize the resulting
     *                          vspace names for the wide area?
     * @param discovery       Are we doing discovery or early binding?
     */
    void recursiveLookup(NameSpecifier ns, int maxSteps,
			 Vector foundNSes, Vector unfoundVSes,
			 boolean makeVspaceFull, boolean discovery)
    {
	String vspace = ns.getVspace();  // assumed non-null

	// see if we host the nametree...
	NameStoreInterface nt = nameTrees.getNameTree(vspace);
	if (nt == null)  // if not, return it as a "to do" vspace
	{
	    unfoundVSes.addElement(vspace);
	    return;
	}
	
	if (nt.isAggregateVspace() && maxSteps>0)
	{
	    Vector vspaces = getChildVspaces(nt);
	 
	    for (Enumeration vs = vspaces.elements(); vs.hasMoreElements(); )
	    {
		String v = (String)vs.nextElement();

		ns.setVspace(v);
		recursiveLookup(ns, maxSteps-1, 
				foundNSes, unfoundVSes,
				makeVspaceFull, discovery);
	    }
	    // return; // controversial -- look on all levels or just leaves?
	}

	// otherwise do a normal name tree lookup
	NameRecord [] rset = nt.lookup(ns);

	// if we're doing early binding, just return namerecord
	if (!discovery) 
	{
	    for (int i=0; i<rset.length; i++) {
		EBRecord ebr = new EBRecord(rset[i].getAppMetric(),
					    rset[i].getHostRecord().toBytes());
		foundNSes.addElement(ebr);
	    }
	    return;
	}

	// otherwise, process for discovery	
	for (int i=0; i<rset.length; i++)
	{
	    NameSpecifier curNS = nt.extract(rset[i]);
	    if (makeVspaceFull)
		dsrMn.extendToWideArea(curNS);
	    foundNSes.addElement(curNS.toString());
	}
    }

    /** 
     * Process late binding message
     *
     * @param msg Message from the application
     * @author Kalpak Kothari
     */
    protected void processLateBinding(Message msg)
    {
	// Useless in AppManager (here only since this needs to be
	// called from INSreceive which is in this class). This method
	// is re-defined in TwineAppManager.
    }

    /************************************************************/
    /* Routing */

    /** 
     *  A callback function: will be called every time a new
     *  INS packet for this handler arrives by UDP.
     *  (Interface implementation of IHandler)
     *  @param msg Message from the application
     */
    public void INSreceive(Message msg)
    {
	//if (Resolver.DEBUG >= 5) printStatus("receiving a message by UDP...");
	//if (Resolver.DEBUG >= 5) printStatus(msg.toString());

	//System.out.println("msg: \nsrc: "+msg.packet.sNS.toString()+"dest: "+
	//		   msg.packet.dNS.toString());

	Packet packet = msg.packet;
	
	AVelement av = packet.dNS.getAVelement(controlmsgAttr);
	if(av != null) {
	    Value val = av.getValue();
	    if (val.equals(announceVal)) 
		{
		    processAnnouncement(msg);
		}
	    else if (val.equals(discoverVal)) 
		{
		    processDiscovery(msg);
		}
	    else if (val.equals(earlybindVal)) 
		{
		    processEarlyBinding(msg);
		}
	    else if (val.equals(discoverReplyVal)) 
		{
		    processDiscoveryReply(msg);
		}
	    else if (val.equals(earlybindReplyVal)) 
		{
		    processEarlyBindingReply(msg);
		}
	}
	else {
	    // late-binding packet (added by kalpak)
	    processLateBinding(msg);
	}
    }

    /** 
     *  A callback function: will be called every time a new
     *  INS packet for this handler arrives by TCP.
     *  (Interface implementation of IHandler)
     *  @param p the INS packet
     *  @param from the neighbor sending the packet p     
     */
    public void INSreceiveByTCP(Packet p, Node from)
    {
	printStatus("receiving a packet by TCP...");
	printStatus("WARNING: normally should not receive packets by TCP!");
	printStatus("... no processing done on the packet, i.e.");
	printStatus(p.toString());

    }

    public void printStatus(String s)
    {
	String text = "AM: "+s;
	System.out.println(text);
	if (isLog) pw.println(text);
    }
}


// class used to track who to respond to
class RespondRecord {

    Vector sequenceNos;
    InetAddress addr;
    int port;
    Vector result;
    String ourvspace;
    String resultvspace;
    int theirSequenceNo;
    boolean makeVspaceFull;
    boolean all;

    boolean haveAllSequenceNos;

    RespondRecord(InetAddress addr, int port, Vector result, 
		  String ourvspace, String resultvspace, int theirSequenceNo,
		  boolean makeVspaceFull, boolean all)
    {	
	this.sequenceNos = new Vector();
	this.addr = addr;
	this.port = port;
	this.result = result;
	this.ourvspace = ourvspace;
	this.resultvspace = resultvspace;
	this.theirSequenceNo = theirSequenceNo;
	this.makeVspaceFull = makeVspaceFull;
	this.all = all;
	
	this.haveAllSequenceNos = false;
    }

    /**
     * Checks a sequence number with its result into the list,
     *  if that was the last non-blank item, return true;
     */
    boolean checkInNumber(Integer sequenceNo, Enumeration res)
    {
	if (sequenceNos.removeElement(sequenceNo))
	{
	    while (res.hasMoreElements()) {
		Object o = res.nextElement();
		this.result.addElement(o);
	    }
	}

	return ((sequenceNos.size() == 0) && haveAllSequenceNos);
    }

    boolean isAnycast() {
	return (port == -1234);
    }

    boolean checkInAnycast(Integer sequenceNo, Enumeration res)
    {
	if (sequenceNos.removeElement(sequenceNo))
	{
	    int currentBest = ((Integer)(((Object[])result.elementAt(0))[1]))
		.intValue();

	    while (res.hasMoreElements()) {
		EBRecord thisRec = (EBRecord)res.nextElement();

		if (thisRec.appMetric < currentBest) {
		    String vspace = ((String)((Hashtable)result.elementAt(2))
				     .get(sequenceNo));

		    resultvspace = vspace;

		    currentBest = thisRec.appMetric;

		    result.setElementAt(new Object[] 
					{ vspace,new Integer(currentBest)}, 0);
		}
	    }
	}
	return ((sequenceNos.size() == 0) && haveAllSequenceNos);
    }
	

    public String toString()
    {
	return sequenceNos.toString()+" <"+addr.getHostName()+":"+
	    Integer.toString(port)+">";
    }
}

class EBRecord {
    int appMetric;
    byte [] hostRecord;

    EBRecord(int appMetric, byte[] hostRecord)
    {
	this.hostRecord = hostRecord;
	this.appMetric = appMetric;
    }
}
