package ins.inr;

import java.net.*;
import java.io.*;

public class Communicator
    implements Runnable
{
    // VARIABLES
    int UDPport;
    int TCPport;
    InetAddress localhost;
    protected MobilitySocket socket;		// UDP socket
    protected ServerSocket TCPsocket;

    Forwarder forwarder;
    VSNeighbors neighbors;
    RouteManager routeMn;
    DSRManager dsrMn;
    OverlayManager overlayMn;

    UDPForwardThread udpThread;
    private byte [] receiveBuffer = new byte[Packet.MAX_PACKET_SIZE]; // arbitrary choice

    boolean listening;
    //nikos
    private  long bytesSent;
    private  int packetsSent;

    // sort of but not really...
    static private final int DEFAULT_UDP_PORT = 5432; // UDP port number to use
    static private final int DEFAULT_TCP_PORT = 5432; // TCP port number to use


    // CONSTRUCTORS
    /**
     * Creates a new Communicator listening on the default port.
     * @exception Exception Something went wrong.
     */
    public Communicator()
	throws Exception
    {
	try {
	    initSocket((localhost=InetAddress.getLocalHost()), 
		       DEFAULT_UDP_PORT, DEFAULT_TCP_PORT);
	} catch (UnknownHostException e) {
	    printStatus("Couldn't get local host name");
	    throw new Exception("Couldn't get local host name " +
				e.getMessage());
	}
    }

    /**
     * Creates a new Communicator.
     * @param p The UDP port the node will listen on
     * @exception Exception Something went wrong.
     */
    public Communicator(int udp, int tcp)
	throws Exception
    {
	try {
	    initSocket((localhost=InetAddress.getLocalHost()), udp, tcp);
	} catch (UnknownHostException e) {
	    printStatus("Couldn't get local host name");
	    throw new Exception("Couldn't get local host name " +
				e.getMessage());
	}
    }

    // METHODS

    /**
     * Initializes Communicator's socket.
     * @param udp The UDP port the node will listen on.
     * @param tcp The TCP port the node will listen on.
     * @exception Exception Something when wrong.
     */
    private void initSocket(InetAddress ia, int udp, int tcp)
	throws Exception
    {
	// Open a socket on the port p, or any avail. port if p == 0

	try {
	    if (udp<=-1) udp = DEFAULT_UDP_PORT;
	    if (tcp<=-1) tcp = DEFAULT_TCP_PORT;
	
	    if (udp == 0) {
		socket = new MobilitySocket();
		UDPport = socket.getLocalPort();
	    } else {
		socket = new MobilitySocket(udp);
		UDPport = udp;
	    }

	    // comment line below if wants two different random ports for 
	    // UDP and TCP
	    if (tcp == 0) tcp=UDPport;	// same port as UDP

	    TCPsocket = new ServerSocket(tcp);   // tcp=0 means random
	    TCPport = (tcp==0)? TCPsocket.getLocalPort():tcp;

	    listening = true;

	} catch (SocketException se) {
	    se.printStackTrace();
	    throw new Exception("Couldn't open socket.");
	}

	printStatus("Listening on UDP port " + UDPport + ".");
    }


    void init(Resolver r)
    {
	forwarder = r.forwarder;
	neighbors = r.neighbors;
	dsrMn = r.dsrMn;
	overlayMn = r.overlayMn;
	routeMn = r.routeMn;
	//nikos
	bytesSent=0;
	packetsSent=0;
	new Thread(this).start();
    }


    /**
     * Receives a valid Message (blocks until one is received)
     * The only intended caller of this method is ReceiverThread,
     *   in its receiving loop. Otherwise, synchronization, or allocating
     *   new bytes every time would be necessary 
     * @return the Message received
     */
    Message receiveMessage()  
    {
	DatagramPacket packet;
	Packet p;
	    
	// Get packets until we've got one from a neighbor
	packet = new DatagramPacket(receiveBuffer, receiveBuffer.length);
	
	try {
	    socket.receive(packet);
	    if (Resolver.DEBUG >=2) printStatus("Received a packet (UDP).");
	} catch (IOException e) {
	    // Connection refused, but can't tell which one 
	    // it's coming from
	    printStatus ("Exception in receiving Datagram packet.");
	    e.printStackTrace();
	}    

	try { 
	    p = new Packet(receiveBuffer, packet.getLength());
	} catch (Exception e) {
	    printStatus("Received bad packet from "+
			packet.getAddress().toString()+
			":"+packet.getPort());
	    return null;
	}
	
	return(new Message(p, packet.getAddress(), packet.getPort()));
    }



    /**
     * Sends a message to a neighbor.
     * @param msg the Message to send
     */
    public void sendMessage(Message msg)
    {

	if ((msg.ia == null) || (msg.port <=0)) {
	    printStatus("Invalid destination for the following message (dropped):");
	    printStatus(msg.packet.toString());
	    return;
	}

	dsrMn.adjustOutgoingVspace(msg.packet);
	
	// Create the packet and send it
	byte [] data = msg.getPacketBytes();
	//nikos
	bytesSent+=data.length;
	packetsSent++;
	
	DatagramPacket packet = 
	    new DatagramPacket(data, data.length, msg.ia, msg.port);
	try {
	    socket.send(packet);
	    if (Resolver.DEBUG>=2) printStatus("Sent a packet (UDP).");
	    //printStatus(msg.packet.toString());
	} catch (IOException e) {
	    if (e.getMessage().equals("Connection refused")) {
		printStatus("Connection refused from IP address ");
		printStatus(msg.ia.toString() + " and UDP port " + msg.port);
		// removeNeighbor(msg.neighbor);
	    } else {
		printStatus("Error sending message: " + e.getMessage());
	    }
	}

    }
    
    /** 
     * Resets the number of data bytes sent (done by TwineMetricEstimator)
     *
     * @author Nikos Michalakis
     */
    public void resetBytesSent(){
	bytesSent=0;
	packetsSent=0;
    }
    
    /** 
     * Returns the bytesSent so far (to be accessed by TwineMetricEstimator)
     *
     * @return number of bytes sent
     * @author Nikos Michalakis
     */
    public long getBytesSent(){
	return this.bytesSent;
    }
  
    /** 
     * Returns the number of packets sent so far (to be accessed by TwineMetricEstimator)
     *
     * @return number of packets sent
     * @author Nikos Michalakis
     */
    public int getPacketsSent(){
	return this.packetsSent;
    }

    /**
     * Sends a message to a neighbor using TCP.
     * @param msg the Message to send
     */
    boolean sendByTCP(Packet p, Node n)
    {
	if ((p==null) || (n==null)) return false;

	DataOutputStream out = n.out;
	if (out == null) 
	    {
		printStatus("WARNING: output stream is null for node:");
		printStatus(n.toString());
		return false;
	    }

	byte[] data = p.toBytes();
	try {
	    out.writeInt(data.length);
	    out.write(data, 0, data.length);
	    out.flush();
	    if (Resolver.DEBUG>=2) printStatus("Sent a packet (TCP).");

	} catch (IOException e) {
	    e.printStackTrace();
	}

	return true;
    }

    void disconnectTCP(Node node)
    {
	node.thread.isReceiving = false;
	node.thread.interrupt();
    }

    boolean connectTCP(Node node)
    {
	try{

	    printStatus("Establishing TCP connection to node " + node.toString());

	    Socket s = new Socket(node.ia, node.TCPport);
	    DataInputStream in = new DataInputStream(new BufferedInputStream(s.getInputStream()));
	    DataOutputStream out = new DataOutputStream(new BufferedOutputStream(s.getOutputStream()));

	    s.setTcpNoDelay(true);	// disable Nagle's algrithm

	    printStatus("... connected (local port = " + s.getLocalPort() + ")");
	    printStatus("(remote port = " + s.getPort() + ")");

	    // Send out the TCP port I listen to so remote node can check that
	    // I am a permitted neighbor to connect
	    out.writeInt(TCPport);
	    out.flush();

	    // Create servicing thread
	    TCPForwardThread t = 
		new TCPForwardThread(s, in, this, forwarder, 
				     overlayMn, node, true);

	    node.thread = t;
	    node.in = in;
	    node.out = out;

	    t.start();

	    // if the connecting neighbor is new node (in its init phase
	    //  of MST contruction), then send all names and routes that
	    //  I know  (also used by healing)
	    if (node.sendAllKnown)
		{
		    if (routeMn == null)
			{
			    printStatus("ASSERT: RouteManager is still null" +
					"in the listening mode");
			}
		    else
			{
			    routeMn.sendAllRoutes(node);
			    routeMn.sendAllNames(node);
			}
		    node.sendAllKnown = false;
		}

	} catch (IOException e) {
	    printStatus(e.getMessage());
	    e.printStackTrace();
	    return false;
	}

	return true;
    }

    public void run()
    {
	// start the UDPForwardThread
	udpThread = new UDPForwardThread(this, forwarder);
	udpThread.start();
	printStatus("Started the UDPForwardThread.");

	Socket s;
	TCPForwardThread t;
	DataInputStream in;
	DataOutputStream out;
	InetAddress remoteAddr;
	int remotePort;
	int portListened;
	Node node;

	printStatus("TCP now listens on port " + TCPport + ".");
	// start listening for TCP connection
	while (listening)
	    {
		try {
		    s = TCPsocket.accept();

		    printStatus("Accepting a new TCP connection...");

		    remoteAddr = s.getInetAddress();
		    remotePort = s.getPort();
		    printStatus("from " + remoteAddr.toString() + ":" +remotePort);
		    printStatus("local port = " + s.getLocalPort());

		    if (Resolver.DEBUG >= 5) 
			printStatus("trying to create I/O streams for the connection...");

		    // Create I/O streams 
		    in = new DataInputStream(new BufferedInputStream (s.getInputStream()));
		    out = new DataOutputStream(new BufferedOutputStream(s.getOutputStream()));

		    if (Resolver.DEBUG >= 5) printStatus("I/O streams created.");
		    if (Resolver.DEBUG >= 5) printStatus("checking neighbor eligibility.");

		    portListened = in.readInt();

		    node = neighbors.getNeighbor(remoteAddr, 0, portListened);

		    printStatus("Matched node :" + node.toString());

		    int retry=0;
		    while ((node == null) && (retry<3))
			{
			    printStatus("re-try...");
			    try { Thread.sleep(200); }
			    catch (InterruptedException e) {;}

			    retry++;
			    node = neighbors.getNeighbor(remoteAddr, 0, portListened);
			}

		    if (node == null)	// not a neighbor
			{
			    s.close();	// only neighbors can connect
			    printStatus("WARNING: unknown neighbor...terminating connection!");
			    break;
			}

		    if (node.in != null)	// already connected
			{
			    s.close();	// only neighbors can connect
			    printStatus("WARNING: duplicate connection...terminating connection!");
			    break;
			}

		    if (Resolver.DEBUG >= 5) printStatus("eligibility satisfied...");

		    s.setTcpNoDelay(true);	// disable Nagle's algrithm

		    // Create servicing thread
		    t = new TCPForwardThread(s, in, this, forwarder, 
					     overlayMn, node, false);

		    // Set corresponding socket and streams info for the node
		    node.thread = t;
		    node.in = in;
		    node.out = out;

		    printStatus("connected to neighbor " + node.toString());

		    // Start servicing the connection
		    t.start();

		    // if the connecting neighbor is new node (in its init phase
		    //  of MST contruction), then send all names and routes that
		    //  I know  (also used by healing)
		    if (node.sendAllKnown)
			{
			    if (routeMn == null)
				{
				    printStatus("ASSERT: RouteManager is still null" +
						"in the listening mode");
				    continue;
				}

			    routeMn.sendAllRoutes(node);
			    routeMn.sendAllNames(node);

			    node.sendAllKnown = false;
			}

		} catch (IOException e) {
		    printStatus("IOException in acception new TCP connection");
		    e.printStackTrace();
		}
	    }

	try {
	    TCPsocket.close(); 
	    // Close TCP sockets to other INRs
	    // .. todo

	}
	catch (IOException e) {e.printStackTrace();}

    }


    public int getUDPPort() {
	return UDPport;
    }

    public int getTCPPort() {
	return TCPport;
    }

    public InetAddress getLocalhost() {
	return localhost;
    }
    protected void printStatus(String s) 
    {
	System.out.println("Communicator: " + s);
    }
}
