package ins.inr;

import ins.namespace.*;
import ins.dsr.DSRRequest;
import ins.dsr.DSRResponse;
import java.util.*;
import java.io.*;
import java.net.*;

/**
 * The DSRManager handles all the communication with the DSR.
 * Namely, it takes care of requesting necessary vspace information.
 * This includes:
 *  - periodically announcing the vspaces we support (AdvertiseVspaces message)
 *  - receiving VspacesResolvers responses after each AdvertiseVspaces message
 *  - explicitly requesting VspacesResolvers messages when a non-local
 *     vspace is needed
 *
 * This module allows multiple threads to use its functions and employs
 * a waiting list for response sequence numbers.
 *
 * Ports used: one UDP port shared for all DSR communication
 * Threads: - receiver thread for listening to the DSR
 *          - VSAnnouncer thread for periodically announcing all
 *            of our vspaces and getting updates on them from the dsr
 *
 * -jjlilley
 */

class DSRManager
    implements IHandler, Runnable
{
    static final int DEFAULT_DSR_PORT = 5678;

    protected InetAddressPort[] localdsrs; // set of local dsrs
    String localdomainprefix;

    protected int dsrPort;

    protected VSResolvers resolvers;
    protected VSNeighbors neighbors;
    protected Communicator comm;
    protected VSNameTree nameTrees;

    protected VspaceAnnouncer vannouncer;

    boolean running = false;

    DatagramSocket dsrSocket;
    byte [] dsrBuffer;

    // to wait for received packets of a given sequence number
    Vector waitingTable = new Vector(4);

    DSRResponse waitingResponse = null;

    long lastVspacesResolvers = 0;


    // CONSTRUCTORS
    DSRManager(String [] dsrs) throws Exception
    {
	dsrSocket = new DatagramSocket();
	// dsrSocket.setSoTimeout(0);
	dsrBuffer = new byte[16384];
	this.localdsrs = processDSRAddrs(dsrs);

	// make a prefix to adhere to
	if (localdsrs.length>0) {
	    localdomainprefix = localdsrs[0].addr.getHostName();
	    if (localdomainprefix.startsWith("dsr."))
		localdomainprefix = localdomainprefix.substring(4);
	} else { localdomainprefix = ""; }
    }


    public void init(Resolver r) 
    {
	resolvers = r.resolvers;
	neighbors = r.neighbors;
	comm = r.comm;
	nameTrees = r.nameTrees;

	(new Thread(this)).start();
	vannouncer = new VspaceAnnouncer(this, 1200000); //say every 20 minutes
    }


    /**
     * Refreshes the contents of the vspaces that are hosted locally.
     * Returns true if at least one dsr from each relevant domain
     *  could be contacted.
     */
    public boolean refreshDSRInfo()
    {
	// prevent redundant requests within a 15 second window
	if ((System.currentTimeMillis() - lastVspacesResolvers) < 15000) 
	{
	    return true;
	}

	boolean success = true;
		
	Vector [] vs = 
	    DSRManager.filterVspaceNames(nameTrees.getVspaces());
		
	Vector domainNames = vs[0];
	Vector domainVspaces = vs[1];

	for (int i = 0; i < domainNames.size(); i++) {    
	    String domain = (String)domainNames.elementAt(i);
	    Vector vspaceList = (Vector)domainVspaces.elementAt(i);

	    if (vspaceList.size() == 0) continue;

	    DSRRequest request = DSRRequest.makeVspacesResolvers
		(vspaceList.elements(), getNextSequenceNo());
	    
	    System.out.println("DM: Refreshing vspaces: "+ request.toString());

	    DSRResponse response = 
		useSingleDSR(request, (domain==null? localdsrs: 
				       getDSRForDomain(domain)));

	    if (response == null) {
		printStatus("Could not contact domain "+domain);
		success=false;
	    }
	}
		
	return (success);
    }


    /** 
     * Sends a VspacesResolvers message to the appropriate DSR
     * (a remote one if necessary) to refresh the VSResolvers data.
     * Returns whether or not the DSR could be contacted.
     */
    public boolean getNodesForVspace(String vspace)
    {
	InetAddressPort [] dsrs = getDSRForVspace(vspace);

	if (dsrs.length == 0) return false;

	Vector v = new Vector(1); v.addElement(vspace);
	
	DSRRequest request = DSRRequest.makeVspacesResolvers
	    (v.elements(), getNextSequenceNo());

	DSRResponse response = useSingleDSR(request, dsrs);

	return (response != null);
    }

    /**
     * If we have data in VSResolvers for a vspace, pick a Node
     *  from the list.
     * If not, ask the DSR. 
     * If that didn't work or there are none, return null.
     */
    public Node getNodeInVspace(String vspace)
    {
	// if we don't host its name tree, see if we know of
	// resolvers we can forward it to
	NodeSet nodeset = resolvers.getResolvers(vspace);

	// if we don't, contact that DSR to get info for that vspace    
	if (nodeset == null || nodeset.countNodes() ==0) 
	{
	    if (getNodesForVspace(vspace))
		nodeset = resolvers.getResolvers(vspace);
	}
	
	// see if that worked and we can get a node to forward to
	Node node = (nodeset==null? null: nodeset.pickNode());
	if (node == null) 
	{
	    printStatus("could not find info for vspace "+vspace);
	    return null;
	}

	return node;
    }


    /**
     * Should be called when a vspace is added beyond the initial set
     * so that another advertisement may be sent
     */
    public void triggerUpdate() { vannouncer.triggerUpdate(); }


    /** 
     * Gets the remote portion of a vspace name.
     *  e.g. cameras:wind.lcs.mit.edu returns wind.lcs.mit.edu
     */
    public static String getRemotePortion(String vspace)
    {
	int index = vspace.indexOf(':');
	
	if (index > -1) {
	    try { return vspace.substring(index+1); }
	    catch (IndexOutOfBoundsException e) { return null; }
	} else {
	    return null;
	}
    }


    /** 
     * Gets the local portion of a vspace name.
     *  e.g. cameras:wind.lcs.mit.edu returns cameras
     */
    public static String getLocalPortion(String vspace)
    {
	int index = vspace.indexOf(':');
	
	if (index > -1) {
	    return vspace.substring(0, index); 
	} else {
	    return vspace;
	}
    }

    /** Returns whether the vspace name is explicitly local,
     * i.e. it has a colon in its name. 
     * Note that this is not enough to say that it really is 
     * local -- for that, run on the result of standardizeVspace()
     */
    public static boolean isExplicitlyLocal(String vspace)
    {
	return (vspace.indexOf(':')>0);
    }


    /**
     * Returns an internal version of the vspace name standardized
     * for the wide area.
     *
     * Local vspace names get their local names returned 
     *      (e.g. cameras:wind.lcs.mit.edu => camera if wind.lcs is local)
     * Remote vspace names are left as is (on the assumption that
     *   it doesn't matter as much to have multiple entries for remote items,
     *     e.g. wind and wind.lcs.mit.edu, as it does for local items --
     *     though this might not be a good assumption ??)
     */
    public String standardizeVspace(String vspace)
    {
	if (vspace == null) return null;

	int index = vspace.indexOf(':');
	if (index == -1) return vspace;

	String local = vspace.substring(0, index);
	String remote;
	try { remote = vspace.substring(index+1); }
	catch (ArrayIndexOutOfBoundsException e) { return local; }

	InetAddressPort [] remotedsrs = getDSRForDomain(remote);
	for (int i=0; i<remotedsrs.length; i++) {
	    for (int j=0; j<localdsrs.length; j++) {
		if (remotedsrs[i].equals(localdsrs[j]))
		    return local;
	    }
	}

	return vspace;
    }


    /**
     * Basically, if the destination is a remote vspace, we should
     *  use the full source vspace syntax for our source.
     */
    protected void adjustOutgoingVspace(Packet packet)
    {
	String destVspace = packet.dNS.getVspace();
	String srcVspace = packet.sNS.getVspace();

	// to a remote location?
	if (destVspace!=null && isExplicitlyLocal(destVspace))
	{
	    // we're fine if source is also wide-area qualified
	    if (srcVspace==null || isExplicitlyLocal(srcVspace))
		return; 
	    
	    packet.sNS.setVspace(srcVspace+":"+localdomainprefix);
	}
    }


    protected void extendToWideArea(NameSpecifier ns)
    {
	String vspace = ns.getVspace();
	if (vspace != null && vspace.indexOf(':')==-1)
	    ns.setVspace(vspace+":"+localdomainprefix);
    }


    /**
     * Returns a set of dsrs that services a vspace
     */
    public InetAddressPort [] getDSRForVspace(String vspace) 
    {
	String remotePortion = getRemotePortion(vspace);

	if (remotePortion==null) return localdsrs;
	else return getDSRForDomain(remotePortion);
    }


    /** 
     * Gets a plausible dsr machine for a given domainname. 
     */
    static InetAddressPort [] getDSRForDomain(String domainname) 
    {
	if (domainname == null) return new InetAddressPort[0];

	InetAddress ia=null;

	try {
	    ia = InetAddress.getByName("dsr."+domainname);
	} catch (UnknownHostException e) { ; }

	if (ia == null) {
	    try { ia = InetAddress.getByName(domainname); }
	    catch (UnknownHostException e) { ; }
	}

	if (ia != null) {
	    return new InetAddressPort[] {
		new InetAddressPort(ia, DEFAULT_DSR_PORT)};
	}

	return new InetAddressPort[0];
    }



    public void run() 
    {
	running = true;

	while (true) {
	    try {
		DatagramPacket dp = 
		    new DatagramPacket(dsrBuffer, dsrBuffer.length);

		dsrSocket.receive(dp);
		DSRResponse resp = 
		    DSRResponse.readWireform(dp.getData(), 0, dp.getLength());

		if (resp != null) 
		    processResponse(resp);

	    } catch (Exception e) {
		printStatus("receive loop: "+e.toString());
	    }
	}
    }


    protected void processResponse(DSRResponse resp) 
    {
	System.out.println("got response: "+resp);

	IntegerObj i = new IntegerObj(resp.getSequenceNo());

	int ind = waitingTable.indexOf(i);	
	
	if (ind > -1) {
	    IntegerObj o = (IntegerObj)waitingTable.elementAt(ind);
	    o.obj = resp;

	    if (resp.getType() == DSRResponse.VspacesResolvers)
		processVspacesResolvers(resp);

	    synchronized (o) { o.triggered = true; o.notify(); }
	}
    }


    /**
     * Processes a DSRResponse of type VspacesResolvers,
     * which returns all the information about a set of 
     * Vspaces. 
     */
    protected void processVspacesResolvers
	(DSRResponse response)
    {
	long currentTime = System.currentTimeMillis();
	lastVspacesResolvers = currentTime;

	Vector toRemove = new Vector();

	// for each advertised vspace
	for (Enumeration vsnodes = response.getElements();
	     vsnodes.hasMoreElements(); ) 
	{
	    ins.dsr.VSNodes vsn = (ins.dsr.VSNodes)vsnodes.nextElement();
	    String vspace = vsn.getVspace();
	    
	    // add advertised resolvers that we may not have
	    for (Enumeration nodes = vsn.getNodes();
		 nodes.hasMoreElements(); ) 
	    {
		ins.dsr.Node node = (ins.dsr.Node)nodes.nextElement();

		InetAddress ia = node.getInetAddress();
		int UDPport = node.getUDPport();
		int TCPport = node.getTCPport();

		// don't add yourself :)
		if ((UDPport == comm.UDPport) && 
		    (TCPport == comm.TCPport) &&
		    (comm.localhost.equals(ia)))
		    continue;

		System.out.println(":: Adding resolver: "+vspace+":"+ia+":"+
				   UDPport+":"+TCPport);
		resolvers.addResolver(vspace, ia, UDPport, TCPport, 
				      node.getTTL(currentTime));
	    }

	    // remove nodes/vspaces that we know of but aren't in the
	    //     advertisement
	    // as an important note: we want to remove only resolvers
	    //   from vspaces that are _not_ neighbors... the overlaymanager
	    //   can deal with neighbors, and that way we don't interfere
	    //   with the spanning tree if the dsr gets restarted and doesn't
	    //   have full information
	    toRemove.removeAllElements(); // clear the vector

	    for (Enumeration ourresolvers = 
		     resolvers.getResolvers(vspace).getNodes();
		 ourresolvers.hasMoreElements(); )
	    {
		Node r = (Node)ourresolvers.nextElement();

		// if it's a neighbor, ignore it
		if (neighbors.getNeighbor(vspace, r) != null)
		    continue;
		
		// check if it's in our announcement
		if (!vsn.containsNode(r.ia, r.UDPport, r.TCPport))
		    toRemove.addElement(r);
	    }
	    
	    // actually remove them here
	    // (if we do it in the previous loop, it messes up the Enumeration)
	    for (Enumeration elems = toRemove.elements();
		 elems.hasMoreElements(); )
	    {
		Node r = (Node)elems.nextElement();
		System.out.println(":: REMOVING resolver: "+vspace+":"+
				   r.toString());

		resolvers.removeResolver(r, vspace);
	    }

	}  // while vsnodes
    }



    IntegerObj preWaitForResponse(int sequenceNo) 
    {
	IntegerObj i = new IntegerObj(sequenceNo);

	waitingTable.addElement(i);
	return i;
    }


    DSRResponse waitForResponse(IntegerObj i, int timeout) 
    {
	try { synchronized (i) { if (!i.triggered) i.wait(timeout); } }
	catch (Exception e) { ; }

	waitingTable.removeElement(i);

	return (DSRResponse)i.obj; 
    }




    /** Announces a request to all dsrs and expects a nominal
     * response. Retries up to 3 times per dsr.
     * Returns whether at least one DSR received/ACKed the message.
     */
    protected boolean announceToManyDSRs(DSRRequest request,
					 InetAddressPort [] iap)
    {
	boolean sentToAtLeastOne = false;

	// make sure receiver thread is running first!
	while (!running) { 
	    try { Thread.sleep(100); } catch (Exception e) { ; }
	}

	int i=0;int retries=0;
	int sequenceNo= getNextSequenceNo();

	while (i<iap.length) {

	    request.setSequenceNo(sequenceNo);
	    
	    byte [] data = request.toBytes();
	    
	    DatagramPacket inPacket = 
		new DatagramPacket(data, data.length,
				   iap[i].addr, iap[i].port);

	    DSRResponse resp = null;

	    try {
		IntegerObj sn = preWaitForResponse(sequenceNo);

		dsrSocket.send(inPacket);
		resp = waitForResponse(sn, 1000);
	    } catch (Exception e) { 
		printStatus("contacting DSR: "+e.toString());
	    }

	    if (resp == null) { retries++; }
	    if (resp != null || retries>2) { 
		i++; retries = 0; 
		sequenceNo = getNextSequenceNo(); 
	    }
	    if (resp != null) { sentToAtLeastOne = true; }
	}
	return sentToAtLeastOne;
    }



    /** 
     * Try to get a response for a dsr request --
     *  Contacts the dsrs, in order, max twice each,
     *  to get a DSRResonse for a DSRRequest, 
     *  with the same sequence number.
     */
    protected DSRResponse useSingleDSR(DSRRequest request, 
				       InetAddressPort [] iap)
    {
	// make sure receiver thread is running first!
	while (!running) { 
	    try { Thread.sleep(100); } catch (Exception e) { ; }
	}

	byte [] data = request.toBytes();

	int sequenceNo = request.getSequenceNo();

	for (int i=0; i<iap.length*2; i++) {

	    if ((i%2)==1) {
		try { Thread.sleep(1500); } catch (Exception e) {;}
	    }
	    
	    DatagramPacket inPacket = 
		new DatagramPacket(data, data.length,
				   iap[i/2].addr, iap[i/2].port);

	    
	    try {
		IntegerObj sn = preWaitForResponse(sequenceNo);

		dsrSocket.send(inPacket);
		DSRResponse resp = waitForResponse(sn, 1000);

		if (resp != null) return resp;		

	    } catch (Exception e) { 
		printStatus("contacting DSR: "+e.toString());
	    }

	}

	return null;
    }


    /* filters vspace names by domainname. 
     * The first vector in the returned array is a list of domainnames
     *   and the second is a list of corresponding Vectors of vspace names.
     * The first element of the first vector is null, to represent
     *   the set of local vspaces
     */
    protected static Vector [] filterVspaceNames(Enumeration vspaces)
    {
	Vector domainNames = new Vector();
	Vector domainVspaces = new Vector();
	
	// filter the vspaces -- local ones and remote ones
	// remote ones get lumped together by domain
	domainNames.addElement(null);
	domainVspaces.addElement(new Vector());
	
	while (vspaces.hasMoreElements()) {
	    String vspace = (String)vspaces.nextElement();
	    String domain = getRemotePortion(vspace);
	    
	    if (domain == null) {
		((Vector)domainVspaces.elementAt(0))
		    .addElement(vspace);
	    } else {
		int ind = domainNames.indexOf(domain);
		if (ind > -1) {
		    ((Vector)domainVspaces.elementAt(ind))
			.addElement(vspace);
		} else {
		    Vector v = new Vector();
		    v.addElement(vspace);
		    
		    domainNames.addElement(domain);
		    domainVspaces.addElement(v);
		}			   
	    }
	}
	return new Vector[] { domainNames, domainVspaces };
    }




    /*
     * From the array of strings given at initialization,
     * comes up with an array of InetAddressPorts suitable for
     * sending messages to.
     */
    InetAddressPort [] processDSRAddrs(String [] dsrs) 
    {
	Vector v = new Vector((dsrs==null?4:dsrs.length));
	
	if (dsrs!=null && dsrs.length>0) {
	    
	    for (int i=0; i<dsrs.length; i++) {
		InetAddress ia=null;
		try { ia = InetAddress.getByName(dsrs[i]); } 
		catch (UnknownHostException e) { ia =null; }

		if (ia!=null) 
		    v.addElement(new InetAddressPort(ia, DEFAULT_DSR_PORT));
	    }
	} else {
	    
	    try {
		String localhost;
		String dsrStr;

		localhost = InetAddress.getLocalHost().getHostName();
		int ind = localhost.indexOf('.');
		
		if (ind>-1) {
		    InetAddressPort [] iap = 
			getDSRForDomain(localhost.substring(ind+1));

		    for (int i =0; i< iap.length; i++)
			v.addElement(iap[i]);
		}

	    } catch (Exception e) { ; }

	}

	InetAddressPort[] result = new InetAddressPort[v.size()];
	v.copyInto(result);
	return result;
    }


    public void printStatus(String s) {
	System.out.println("DM: "+s);
    }

    public void INSreceive(Message m) {

    }

    public void INSreceiveByTCP(Packet p, Node n) {

    }

    static Object sequenceNoLock = new Object();
    static int sequenceNo=(int)System.currentTimeMillis();
    static int getNextSequenceNo() { 
	synchronized (sequenceNoLock) { sequenceNo++; } return sequenceNo; }

}


/* This auxiliary class announces the VSpaces supported by this
 * INR and requests updated Resolver info for all the members 
 * of those vspaces periodically.
 */
class VspaceAnnouncer extends Thread
{
    Communicator comm;
    DSRManager dm;
    VSNameTree nameTrees;
    boolean insleep;

    long ttl;

    // makes the advertiser and starts it 
    // uses vspaces in VSNameTree to advertise
    public VspaceAnnouncer(DSRManager dm, long ttl) 
    {
	this.dm = dm;
	this.comm = dm.comm;
	this.nameTrees = dm.nameTrees;
	this.ttl = ttl;
	
	this.start();
    }

    /* forces the sleep to break and another 
     * announcement to occur 
     */
    public void triggerUpdate() 
    {
	if (insleep) this.interrupt();
    }

    public void run() 
    {
	int announcementnumber = 0;

	while (true) {

	    boolean succeeded = false; 
	    announcementnumber++;
	    try { 
		Vector [] vs = 
		    DSRManager.filterVspaceNames(nameTrees.getVspaces());
		
		Vector domainNames = vs[0];
		Vector domainVspaces = vs[1];

		for (int i = 0; i < domainNames.size(); i++) {    
		    String domain = (String)domainNames.elementAt(i);
		    Vector vspaceList = (Vector)domainVspaces.elementAt(i);

		    if (vspaceList.size() == 0) continue;

		    // create advertisement message
		    // don't request VspacesResolvers message if this
		    //  is the first message.
		    DSRRequest rq = DSRRequest.makeAdvertiseVspaces
			(comm.localhost, comm.UDPport, comm.TCPport, ttl,
			 true, (announcementnumber!=1), 
			 vspaceList, dm.getNextSequenceNo());
		    
		    System.out.println("DM_VA: Announcing vspaces to DSRs: "+
				       rq.toString());

		    succeeded = dm.announceToManyDSRs
			(rq, (domain == null? dm.localdsrs:
			      dm.getDSRForDomain(domain)));
		}

	    } catch (Exception ex) { ex.printStackTrace(); }

	    try { 
		long sleeptime = succeeded ? (ttl/2) : (ttl / 10);
		System.out.println("DM_VA: sleeping "+sleeptime+" ms");
		insleep =true;
		Thread.sleep(sleeptime);
		insleep=false;
	    } catch (Exception ex) { 	    
		    insleep=false;
		    ex.printStackTrace(); 
	    }
	}
    }

}


// container class to handle the notification in the waiting list
class IntegerObj 
{
    int i;             // the sequence number
    Object obj;        // placeholder for the response
    boolean triggered; // has notify() already been called

    public IntegerObj(int i) 
    {
	this.i=i;
	this.obj=null;
	this.triggered=false;
    }

    public boolean equals(Object o) 
    {
	if (o==null || !(o instanceof IntegerObj)) return false;
	return ( ((IntegerObj)o).i == i);
    }

    public int hashCode() { return i; }

    public String toString() { return "[ "+Integer.toString(i)+" "+
				   (obj==null?"null":obj.toString())+" ]"; }

}


// container class to bundle an InetAddress with a port
class InetAddressPort
{
    public InetAddress addr;
    public int port;


    public InetAddressPort (InetAddress addr, int port) {
	this.addr = addr;
	this.port = port;
    }	


    public int hashCode () {
	return addr.hashCode()+port;
    }

    public boolean equals(Object obj) {
	if (obj==null || !(obj instanceof InetAddressPort)) return false;
	if ( ((InetAddressPort)obj).port != port ) return false;
	return addr.equals( ((InetAddressPort)obj).addr);
    }

    public String toString() 
    {
	return (addr.getHostName()+":"+
		Integer.toString(port));
    }
	
}




