package ins.inr;

import java.util.*;
import java.net.InetAddress;
import java.net.UnknownHostException;
import ins.namespace.*;


class Forwarder
{
    // VARIABLES
    protected Resolver resolver;
    protected Communicator comm;
    protected VSNameTree nameTrees;
    protected VSRouteTable routeTables;
    protected VSResolvers resolvers;
    protected RouteManager routeMn;
    protected DSRManager dsrMn;
    protected AppManager appMn;

    protected RouteEntry myuidRE;
    protected Vector nextHopList; 	// used to filter duplicate
	// packets sent to same neighbor (due to multiple names map to
	// that same neighbor);

    // for dealing with hierarchical vspace routing
    static Attribute childAttr = new Attribute("child");
    static NameSpecifier childstar = new NameSpecifier("[child=*]");
    static Attribute controlmsgAttr = new Attribute("control-msg");

    // for name lookup response
    static VspaceNRPair [] zeronr = new VspaceNRPair[0];


    Forwarder()
    {
    }

    void init(Resolver r)
    {
	resolver = r;
	comm = r.comm;
	nameTrees = r.nameTrees;
	routeTables = r.routeTables;
	routeMn = r.routeMn;
	dsrMn = r.dsrMn;
	resolvers = r.resolvers;
	appMn = r.appMn;

	myuidRE = routeTables.lookup(resolver.defaultVSpace, resolver.INRuid);
	if (myuidRE == null)	// fatal error, should not happen
	{
	    printStatus("Fatal Error: routeTables is not correctly" +
		"initialzed yet inside Forwarder.init().");
	    System.exit(-1);
	}	

	nextHopList = new Vector();
    }


    /**
     * This is called by UDPForwardThread when a UDP packet is received 
     *  to forward it to the proper location. The packet is given either to
     *  multicast() or anycast().
     */
    synchronized void forward(Message msg)
    {
	// Infer source name and disseminate it if necessary
	Packet packet = msg.packet;

	if (Resolver.DEBUG >= 2) printStatus("UDP receives a packet...\n");
	if (Resolver.DEBUG >= 5) printStatus(packet.toString());

	if (packet.decrementTTL() == 0)
	{
	    printStatus("WARNING: packet TTL reaches zero, dropped it");
	    return;
	}

	Node from = new Node(msg.ia, msg.port, 0);
	// Check anycast or multicast
	if (packet.all)
	{	// multicast
	    multicast(packet, from, false);
	}
	else
	{	// anycast
	    anycast(packet, from, false);
	}

    }


    /**
     * This is called whenever a TCP packet is received through an
     *   open connection. It is given to multicast() or anycast().
     * This isn't quite to specification currently, since packets
     *   intended for outside vspaces are sent by UDP instead of 
     *   TCP in order to avoid setting up a number of one-time 
     *   TCP connections.
     */
    synchronized void forwardByTCP(Packet packet, Node from)
    {
	if (Resolver.DEBUG >= 2) printStatus("TCP receives a packet...\n");
	if (Resolver.DEBUG >= 5) printStatus(packet.toString());

	if (packet.decrementTTL() == 0)
	{
	    printStatus("WARNING: packet TTL reaches zero, dropped it");
	    return;
	}

	// Check anycast or multicast
	if (packet.all)
	{	// multicast
	    multicast(packet, from, true);
	}
	else
	{	// anycast
	    anycast(packet, from, true);
	}
    }


    /**
     * Looks up the nametable information and produces LookupResponse.
     * This will be recursive in the case of aggregate vspaces, at
     *  least if the message is not a control-msg
     * (All sorts of ugly things happen if control-msgs are sent using
     *  the interpreted aggregate vspaces)
     *
     * The response is a set of NameRecord/vspace name pairs. These
     * are all the locally-known (but not necessarily last-hop, etc.)
     * entries. 
     * The unknownVspaces is also mutated as a response -- 
     *   it consists of all the external vspaces which may also 
     *   have the item. Typically, this will be just a single entry 
     *   with the original vspace name, signifying that it is not
     *   hosted locally and that it should be forwarded to an INR
     *   which knows about it (found out with dsrMn.getNodeInVspace()
     * In the case of aggregate vspaces, it will be a list with all
     *   the child vspaces which should be contacted.
     */
    protected VspaceNRPair[] lookup(NameSpecifier ns, String defaultTo,
				    Vector unknownVspaces)
    {
  String vspace = ns.getVspace();
	if (vspace == null) vspace = defaultTo;
	if (vspace == null) 
	{
	    printStatus("vspace on packet is null ==> bad");
	    return null;
	}

	VspaceNRPair [] result = zeronr;

	// do we host that nametree??
	NameStoreInterface nt = nameTrees.getNameTree(vspace);	

	// do we host it locally or not?
	if (nt == null) {
	    unknownVspaces.addElement(vspace);

	} else {

	    // get the namerecords from the local nametree
	    NameRecord [] n = nt.lookup(ns);

	    result = new VspaceNRPair[n.length];
	    for (int i=0; i<n.length; i++)
      {
		result[i] = (new VspaceNRPair(vspace, n[i]));
      }
	    
	    boolean recursive =(ns.getAVelement(controlmsgAttr)==null);

	    // if it's aggregate, create the applicable VspaceNodePairs
	    if (nt.isAggregateVspace() && recursive) {
		
		// loop through each child vspace
		for (Enumeration e = getChildVspaces(nt).elements();
		     e.hasMoreElements(); ) 
		{
		    String vsp = (String) e.nextElement();
		    
		    NameSpecifier newns = new NameSpecifier(ns);
		    newns.setVspace(vsp);
		    VspaceNRPair [] nrs= lookup(newns, null, unknownVspaces);

		    if (nrs.length > 0) {
			VspaceNRPair [] newnrs = 
			    new VspaceNRPair[nrs.length+result.length];
			System.arraycopy(result, 0, newnrs, 0, result.length);
			System.arraycopy(nrs, 0, newnrs, result.length,
					 nrs.length);
			result = newnrs;
		    }		    
		} // for
		
	    } // nt.isAggregateVspace
	} // nt != null
	return result;
    }


    /** 
     * Gets the child vspaces for a nametree with aggregate vspaces
     *  Used for the recursive lookup above 
     */
    Vector getChildVspaces(NameStoreInterface nt)
    {
	NameRecord []nr = nt.lookup(childstar);  // look up all _children_
	
	Vector vspaces = new Vector(nr.length+1);
	
	for (int i=0; i<nr.length; i++) 
	{
	    NameSpecifier cNS = nt.extract(nr[i]);
	    AVelement cAVE = cNS.getAVelement(childAttr);
	    
	    if (cAVE != null) 
	    {
		Value cVal = cAVE.getValue();
		if (!cVal.isWildcard())
		    vspaces.addElement(cVal.toString());
	    }
	}

	return vspaces;
    }


    /**
     * Anycasts a packet to best destination.
     */
    protected void anycast(Packet packet, Node from, boolean isTCP)
    {
	adjustLocalVspaces(packet);
	// add source name info to NameTree if necessary
	// use the INRuid from the option field of the packet if exist
	checkSourceName(packet, from, isTCP);
	// Lookup destination name
	if (Resolver.DEBUG >= 5)
	{
	    printStatus(nameTrees.toString());
	    printStatus("(ANYCAST) looking up " + packet.dNS.toString());
	}

	Vector unknownVspaces = new Vector(1);

	// new way using our augmented internal lookup
	VspaceNRPair [] result = lookup(packet.dNS, null, unknownVspaces);

	if (result.length == 0 && unknownVspaces.size() == 0) 
	{
	    printStatus("no anycast match: dropping packet");
	    return;
	}

	// find the "best" locally-hosted name
	VspaceNRPair bestName = null;
	int bestMetric = Integer.MAX_VALUE;
	    
	for (int i = 0; i < result.length; i++) 
	{
	    NameRecord thisNR = result[i].nr;

	    int currentMetric = thisNR.getAppMetric();
	    if (currentMetric <= bestMetric) 
	    {
		// to avoid round_about going of a last-hop INR
		// handing off packet to another better last-hop INR
		// due to rapid changes of anycast metric, if I'm last-hop
		// INR then pick bestName only from "my" apps...
		if (!packet.fromApp && thisNR.isNeighbor())
		   if (thisNR.getINRuid() != resolver.INRuid)
			continue;

		bestName = result[i];
		bestMetric = currentMetric;
	    }
	}
	    
	// For debugging
	if ((Resolver.DEBUG >= 5) && (result.length>1)) {
	    printStatus("**ANYCAST packet, picked one among " + 
			result.length + " : **");
	    //printStatus(nameTrees.extract(bestName.nr, null).toString());
	}
    
	
	// if there are no remotely-hosted possibilities,
	//  just send the packet
	if (unknownVspaces.size() == 0 /*|| bestMetric ==0*/) { 
	    packet.fromApp = false;
	    packet.dNS.setVspace(bestName.vspace);
	    sendAnycastToNameRecord(bestName.nr, packet, from, isTCP);
	    return;
	}

	// if there are no local and 1 remote, we can just sent to that
	// one remote vspace
	if (unknownVspaces.size() == 1 && bestName == null) {
	    String vspace = (String)unknownVspaces.elementAt(0);
	    Node node = dsrMn.getNodeInVspace(vspace);
	    if (node != null) {
		packet.dNS.setVspace(vspace);
		comm.sendMessage(new Message(packet, node.ia, node.UDPport));
	    }
	    return;
	}


	// the more difficult case is when there are either
	//  both remote and the local-best location to look at OR
	// multiple remote locations...
	// We need to sent to the single best, which we can only know
	// by sending a message to all the subvspaces -- this routine
	// in appManager
	appMn.doAggregateAnycast(bestName.vspace, bestName.nr, bestMetric,
				 new Packet(packet), unknownVspaces, 
				 from, isTCP);
    
    }


    /**
     * Multicasts a packet to the proper matches
     */
    protected void multicast(Packet packet, Node from, boolean isTCP)
    {
	if (packet.fromApp)
	{
	    adjustLocalVspaces(packet);
	    checkSourceName(packet, from, isTCP);
	} 

	// Lookup destination name
	if (Resolver.DEBUG >= 5)
	{
	    printStatus(nameTrees.toString());
	    printStatus("(MULTICAST) looking up " + packet.dNS.toString());
	}

	// new way using our augmented internal lookup
	Vector unknownVspaces = new Vector(1);

	VspaceNRPair [] result = lookup(packet.dNS, null, unknownVspaces);
        //printStatus("multicast() looks up and matches " + result.length + " records!");

	// do we need to forward it to a remotely-hosted vspace(s)?
	if (unknownVspaces.size()>0)
	{
	    for (int i=0; i<unknownVspaces.size(); i++) 
	    {
		String vspace = (String)unknownVspaces.elementAt(i);
		Node node = dsrMn.getNodeInVspace(vspace);
		if (node != null) {
		    packet.dNS.setVspace(vspace);
		    // we're sort of punting the isTCP here, oh well for now
		    comm.sendMessage(new Message(packet,node.ia,node.UDPport));
		}
	    }
	}


	// If we didn't find any matches, drop the packet.
	if (result.length == 0) 
	{
	    if (unknownVspaces.size() == 0) printStatus("dropped packet");
	    return;
	}

	packet.fromApp = false;
	    
	//nextHopList = new Vector();
	nextHopList.removeAllElements();
	
	for (int i = 0; i < result.length; i++) {
	    packet.dNS.setVspace(result[i].vspace);
	    sendMulticastToNameRecord(result[i].nr, packet, from, isTCP);
	}		
    }


    /* 
     * Takes off unnecessary local vspace suffixes for domain-qualified
     * vspaces
     *  e.g. if wind.lcs.mit.edu is local to us, 
     *          then cameras:wind.lcs.mit.edu will become cameras
     */
    protected void adjustLocalVspaces(Packet packet)
    {
	String origvspace, finalvspace;

	origvspace = packet.dNS.getVspace();
	finalvspace = dsrMn.standardizeVspace(origvspace);
	if (finalvspace != origvspace)
	    packet.dNS.setVspace(finalvspace);

	origvspace = packet.sNS.getVspace();
	finalvspace = dsrMn.standardizeVspace(origvspace);
	if (finalvspace != origvspace)
	    packet.sNS.setVspace(finalvspace);
    }


    /**
     * For Route Inference, checks if the source name is known
     */
    protected void checkSourceName(Packet packet, Node from, boolean isTCP)
    {
	// printStatus("checkSourceName() *********");

	if (packet.getAppUID() == 0)	   // app doesn't want to announce
	{
	    if (Resolver.DEBUG >=4)
	        printStatus("**Sender doesn't want to announce source name!**");
	    return;
	}

	// ask RouteManager to disseminate the source name if necessary
	if (routeMn == null)
	{
	    printStatus("WARNING: Packet arrives before " +
		   "RouteManager is ready.. source-name is not disseminated!");
	    printStatus("The packet is \n" + packet.toString());
	    return;
	}

	// if source name has no vspace then use dest name's vspace
	// if dNS contains no vspace, source-name is ignored
	String vspace = packet.sNS.getVspace();
	if (vspace == null)
	{
	    vspace = packet.dNS.getVspace();
	    if (vspace == null)
	    {
		printStatus("WARNING: no vspace info, packet dropped!");
		return;
	    }
	    packet.sNS.setVspace(vspace);
	}
	
	Vector v = nameTrees.lookupExactMatch(packet.sNS, packet.getAppUID());
	
	if (v == null)	// null = no vspace info
	{	// should not occur though since we have checked above
	    printStatus("Internal error: null vspace encountered!");
	    return;
	}

	if (v.size() == 0)	// name not exist yet
	{
	    // Use MAX_NAME_TTL since we are the first-hop INR (i.e., do soft 
	    // state)
	    // note: below set routeEntry to myuidRE to mean next hop
	    // is application node.
	    NameRecord nr = new NameRecord(resolver.INRuid, myuidRE,
		new HostRecord(from.ia, from.UDPport, from.TCPport),
		RouteManager.UNDEFINED_APP_METRIC, RouteManager.MAX_NAME_TTL, 
		0, true, packet.getAppUID());

	    System.out.println("******* INFERING "+packet.sNS);

	    nameTrees.addNameRecord(packet.sNS, null, nr);

	    // Disseminate name now if I'm first-hop INR
	    if (packet.fromApp)
	        routeMn.sendNameUpdate(packet.sNS, nr, 
		    NameUpdate.ADD_NAME, null);
	} 
	else
	{
	    // update name
	    NameRecord nr = (NameRecord)(v.elementAt(0));

	    if (!packet.fromApp)
	    {	// if I'm not first-hop INR, only need to check/set INRuid
		long INRuid;
		INRuid = (long)(Conversion.extract32toInt(from.ia.getAddress(), 0));
		INRuid = (INRuid << 16) | ((long)from.UDPport & 0xFFFFl);

		//if (nr.getINRuid() != INRuid)
		nr.setINRuid(INRuid);
		return;
	    }

	    // otherwise...
	    boolean diffNameRecord = false;

	    // Update name record to reflect possibly new IP address
	    // and port of app if mobile app...
	    HostRecord hr = nr.getHostRecord();

	    byte[] newAddr = from.ia.getAddress();
	    if (!Conversion.byteCompare(newAddr, hr.address, 4))
	    {
		hr.address = newAddr;
		diffNameRecord = true;
		printStatus("App is from new IP address");
	    }

	    if ((!isTCP) && (from.UDPport != 0))
	    {
	        if (hr.UDPport != from.UDPport)
		{
		    hr.UDPport = from.UDPport;
		    diffNameRecord = true;
		    printStatus("App is from new UDPp ort");
		}
	    }

	    if ((isTCP) && (from.TCPport != 0))
	    {
		if (hr.TCPport != from.TCPport)
		{
		    hr.TCPport = from.TCPport;
		    diffNameRecord = true;
		    printStatus("App is from new TCP port");
		}
	    }

	    // if not fromApp then I'm not first-hop INR, and only
	    // the first-hop INR is responsible for soft-state
	    // also refresh entry's TTL
	    nr.setExpireTime(RouteManager.MAX_NAME_TTL);

	    // mobility case: if nr.getINRuid() is different my uid
	    // (i.e., resolver.uid) that means app has attached to me 
	    // from different INR
	    if (nr.getINRuid() != resolver.INRuid)
	    {
		nr.setINRuid(resolver.INRuid);
		nr.setRouteEntry(myuidRE);			
		diffNameRecord = true;
		printStatus("...different INRuid...");
	    }
		    
	    if (diffNameRecord)  // if different and 1st-hop INR
		routeMn.sendNameUpdate(packet.sNS, nr, 
		    NameUpdate.UPDATE_NAME, null);
	}   // else of if (v.size()==0)
    }



    /**
     * Sends a packet to a NameRecord.
     * I'm not _exactly_ sure why this is different than the anycast
     * version (dealing with last hop? William?), I'm just the one
     * modularizing it :) -jjlilley
     */
    protected void sendMulticastToNameRecord(NameRecord nr, Packet packet,
	Node from, boolean isTCP)
    {
	if (Resolver.DEBUG >= 5) 
	    printStatus("sendMulticastToNameRecord()");

	synchronized (nr) 
	{
	    if (nr.isNeighbor())
	    {
		// if for a neighbor
		// Send out packets to next hops

		Node n = nr.getNeighbor();

		//if (n==null)	// meaning send to app node
		if (nr.getINRuid() == resolver.INRuid)
		{
		    HostRecord hr = nr.getHostRecord();

		    InetAddress ia = hr.getInetAddress();

		    if (ia != null) {
			// for now, ignore isTCP since don't want to set up TCP to app
			comm.sendMessage(new Message(packet, ia, hr.UDPport));
			printStatus("********** sent to app node ");
		    }
		    else
		        printStatus("********** INVALID InetAddress of application");
		        
		    return;
		}

		if (n==null)
		{
		    RouteEntry curRE = routeTables.lookup(packet.dNS.getVspace(), nr.getINRuid());
		    if (curRE!=null)
		    {
			n = curRE.nextHop;
			nr.setRouteEntry(curRE);
		    }
		    else
		    {
			printStatus("WARNING:  multicast dropped packet " + packet.toString());
			return;
		    }
		}

		if (n==null) 
		{
		    printStatus("WARNING: multicast has null next hop - dropped packet");
		    printStatus(packet.toString());
		    return;
		}

		// Prevent sending back to where we receive it from
		if (n.equals(from)) {
		    //printStatus("prevent sending back to from!");
		    return;
		}

		// if already sent to n before, don't send again
		if (nextHopList.indexOf(n) >= 0)
		{
		    //printStatus("aggregation prevents sending it again!");
		    return;
		}
		else
		    nextHopList.addElement(n);
		
		if (isTCP) comm.sendByTCP(packet, n);
		else comm.sendMessage(new Message(packet, n.ia, n.UDPport));

		printStatus("********** sent to NEXT hop INR " + n.toString());

		Resolver.logSendPacket(packet);
	    } 
	    else 
	    { 
		// else for handler
		IHandler h = nr.getHandler();
		if (isTCP) h.INSreceiveByTCP(packet, from);
		else h.INSreceive(new Message(packet, from.ia, from.UDPport));
	    }			
	}  // END synchronized(nr)		
    }


    /**
     * Sends a packet to a NameRecord.
     * I'm not _exactly_ sure why this is different than the multicast
     * version (dealing with last hop? William?), I'm just the one
     * modularizing it :) -jjlilley
     */
    void sendAnycastToNameRecord(NameRecord nr, Packet packet,
				 Node from, boolean isTCP)
    {
	if (nr.isNeighbor())
	{    
	    // if for a neighbor
	    // Send out packets to next hop

	    if (nr.getINRuid()==resolver.INRuid)  // meaning send to app node
	    {
		// Just double checking one more time
		if (nr.getNeighbor() != null)
		{
		    printStatus("BAD BAD BAD: Internal error of "+
				"inconsistent NameRecord!");
		    printStatus(nr.toString());
		}

		HostRecord hr = nr.getHostRecord();
	
		InetAddress ia = hr.getInetAddress();

		if (ia != null) {
		    // for now, ignore isTCP since don't want to set up TCP
		    comm.sendMessage(new Message(packet, ia, hr.UDPport));
		    printStatus("********** sent to app node ");
		}
		return;
	    }

	    // else, send to last hop INR 
	    // extract IP addr and UDP port from INRuid
	    InetAddress lastHopIa = 
		extractAddrFromINRuid(nr.getINRuid());

	    if (lastHopIa == null)	// invalid IP address
	    {
		printStatus("WARNING: Invalid IP address of last-hop INR\n"+
			    "***** Drop the packet: \n"+packet.toString());
		return;
	    }

	    int lastHopPort = extractPortFromINRuid(nr.getINRuid());

	    // for now, ignore isTCP since don't want to set up TCP to last hop
	    // INR
	    comm.sendMessage(new Message(packet, lastHopIa, lastHopPort));
	    printStatus("********** sent to last hop INR ");
	}
	else 
	{ 
	    // else for handler
	    IHandler h = nr.getHandler();
	    if (isTCP) h.INSreceiveByTCP(packet, from);
	    else h.INSreceive(new Message(packet, from.ia, from.UDPport));
	}
    }

    /**
     * The long INRuid has info embedded in it, including the
     * IP address of the advertising node.
     */
    protected InetAddress extractAddrFromINRuid(long INRuid)
    {
	int intia = (int)((INRuid >> 16) & 0xFFFFFFFFl);
	byte[] addr = new byte[4];
	Conversion.insertInt(addr, 0, intia);
	InetAddress ia = null;

	// convert addr to InetAddress type		
	try{
	    String stria = Integer.toString(addr[0] & 0xFF) +
	      "." + Integer.toString(addr[1] & 0xFF) +
	      "." + Integer.toString(addr[2] & 0xFF) +
	      "." + Integer.toString(addr[3] & 0xFF);

	    ia = InetAddress.getByName(stria);
	} catch (UnknownHostException e) {
	    e.printStackTrace();
	}

	return ia;
    }
	
    protected int extractPortFromINRuid(long INRuid)
    {
	return (int)(INRuid & 0xFFFFl);
    }

    static void printStatus(String s)
    {
	String text = /*new Date(System.currentTimeMillis()).toString() + */
			"FWD: "+s;
	System.out.println(text);
    }


    /** Inner classes to return result from lookup */
    class VspaceNRPair {
	String vspace;
	NameRecord nr;

	VspaceNRPair(String vspace, NameRecord nr)
	{
	    this.vspace = vspace;
	    this.nr = nr;
	}
    }

}
