package ins.inr;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.*;

/** 
 * Represents an early-binding record for a host
 * Byte representation is as following:
 *    4 bytes (IP address)
 *    2 bytes (UDP port)
 *    2 bytes (TCP port)
 *    1 byte  (length of transport name)
 *    n bytes (transport name)
 *    2 bytes (the corresponding port # for this transport)
 *    ..... next transport-port pairs
 *    1 byte  (making app data appear like another transport-port pair)
 *    m bytes (app data option labeled APP_DATA)
 *    1 byte  (app data length)
 *    n bytes (app data)
 * example:
 *   early binding record describing a host at 18.31.0.100
 *   that supports TCP connection on port 7000 and HTTP on
 *   port 8000: 
 *    18.31.0.100  (IP address - 4 bytes)
 *    3            ('TCP' has 3 characters - 1 byte)
 *    TCP          (tranport name - 3 bytes)
 *    7000         (port # for this TCP - 2 bytes)
 *    4            ('HTTP' has 4 characters - 1 byte)
 *    HTTP         (tranport name - 3 bytes)
 *    8000         (port # for this HTTP - 2 bytes)
 */
public class HostRecord
{
    // VARIABLES
    public final static byte[] INVALID_ADDRESS = {0,0,0,0};
    public final static short DEFAULT_PORT = 0;

    public byte[] address;
    public int UDPport;
    public int TCPport;		// not currently used yet for application node
    public Vector transport;	// vector of String
    public Vector port;	// vector of short

    // Allowing some app dependant data in HostRecords
    public static String APP_DATA = "APP";
    protected byte[] data;
    protected int sizeData;


    // CONSTRUCTORS
    public HostRecord()
    {
	try {
	    address = InetAddress.getLocalHost().getAddress();
	} catch (UnknownHostException e) {
	    address = new byte[4];
	    address[0] = 0;
	    address[1] = 0;
	    address[2] = 0;
	    address[3] = 0;
	}

	UDPport = 0;
	TCPport = 0;
	transport = new Vector();
	port = new Vector();
	data = null;
	sizeData = 0;


    }

    public HostRecord(int udp, int tcp, Vector t, Vector p)
    {
	try {
	    address = InetAddress.getLocalHost().getAddress();
	} catch (UnknownHostException e) {
	    address = new byte[4];
	    address[0] = 0;
	    address[1] = 0;
	    address[2] = 0;
	    address[3] = 0;
	}

	UDPport = udp;
	TCPport = tcp;
	transport = t;
	port = p;
    }

    /* 	//Can't declare this function, declaration duplicates with 
    	//public HostRecord(byte[] bytes, int offset, int len)
    public HostRecord(byte[] a, int udp, int tcp)
    {
	address = a;
	UDPport = udp;
	TCPport = tcp;
	transport = new Vector();
	port = new Vector();
    }*/

    public HostRecord(InetAddress a, int udp, int tcp)
    {
	address = a.getAddress();
	UDPport = udp;
	TCPport = tcp;
	transport = new Vector();
	port = new Vector();
	data = null;
	sizeData = 0;

    }

    public HostRecord(byte[] a, int udp, int tcp, Vector t, Vector p)
    {
	address = a;
	UDPport = udp;
	TCPport = tcp;
	transport = t;
	port = p;
	data = null;
	sizeData = 0;

    }

    public void HostRecord(byte[] bytes)
	throws Exception
    {

	if (bytes.length < 8)
	    throw new Exception("Invalid HostRecord bytes");

	address = new byte[4];
	address[0] = bytes[0];
	address[1] = bytes[1];
	address[2] = bytes[2];
	address[3] = bytes[3];
	
	// extract UDPport and TCPport
	UDPport = Conversion.extract16toIntLow16(bytes, 4);
	TCPport = Conversion.extract16toIntLow16(bytes, 6);

	transport = new Vector();
	port = new Vector();

	int pt = 8;
	while (pt < bytes.length)
	{
	    // extract the transport type
	    int len = bytes[pt];
	    String strTransport = new String(bytes, pt+1, len);
	    
	    // Adding support for app level data:
	    if ( strTransport.equals(APP_DATA)) {
		extractAppData(bytes,pt+1+len);
		pt = pt + 1 + len + 1 + sizeData;
	    }
	    else {
		transport.addElement(strTransport);
		pt = pt + len + 1;

		// calculate the (short) port from 2 bytes
		short p;
		p =  (short)( bytes[pt]         & 0xFF);
		p |= (short)((bytes[pt+1] << 8) & 0xFF00);
		port.addElement(new Short(p));
		pt = pt + 2;
	    }
	}
    }

    /*
     * Method extracting application level data from HostRecord.
     * This data is not interpreted. It's kept as an array of bytes.
     */
    protected void extractAppData(byte[] bytes,int offset) {

	sizeData = bytes[offset++];
	if ( sizeData != 0 ) {
	    data = new byte[sizeData];
	    System.arraycopy(bytes,offset,data,0,sizeData);
	}
	
    }
	
    public HostRecord(byte[] bytes, int offset, int len)
	throws Exception
    {

	if (bytes.length <4)
	    throw new Exception("Invalid HostRecord bytes");

	address = new byte[4];
	address[0] = bytes[offset];
	address[1] = bytes[offset+1];
	address[2] = bytes[offset+2];	
	address[3] = bytes[offset+3];

	// extract UDPport and TCPport
	UDPport = Conversion.extract16toIntLow16(bytes, offset+4);
	TCPport = Conversion.extract16toIntLow16(bytes, offset+6);

	transport = new Vector();
	port = new Vector();

	int pt = offset+8;
	while (pt < (offset+len))
	{
	    // extract the transport type
	    int strlen = bytes[pt];
	    String strTransport = new String(bytes, pt+1, strlen);

	    // Adding support for app level data:
	    if ( strTransport.equals(APP_DATA)) {
		extractAppData(bytes,pt+1+strlen);
		pt = pt + 1 + strlen + 1 + sizeData;
	    }
	    else {
		transport.addElement(strTransport);
		pt = pt + strlen + 1;

		// calculate the (short) port from 2 bytes
		short p;
		p =  (short)( bytes[pt]         & 0xFF);
		p |= (short)((bytes[pt+1] << 8) & 0xFF00);
		port.addElement(new Short(p));
		pt = pt + 2;
	    }
	}

    }

    public byte[] toBytes()
    {
	int byteLength = getByteLength();
	if (byteLength == 0) return new byte[0];
	byte[] bytes = new byte[byteLength];

	bytes[0] = address[0];
	bytes[1] = address[1];
	bytes[2] = address[2];
	bytes[3] = address[3];

	// insert UDPport and TCPport
	Conversion.insertIntLow16(bytes, 4, UDPport);
	Conversion.insertIntLow16(bytes, 6, TCPport);

	int pt = 8;
	for (int i=0; i<transport.size(); i++)
	{
	    // Insert transport type
	    String strTransport = (String)transport.elementAt(i);
	    int len = strTransport.length();
	    bytes[pt] = (byte)(len & 0xFF);
	    pt++;
	    System.arraycopy(strTransport.getBytes(), 0, bytes, pt, len);

	    // Insert associated port #
	    short p = ((Short)(port.elementAt(i))).shortValue();
	    bytes[pt+len] = (byte)(p & 0xFF);
	    bytes[pt+len+1] = (byte)(p >> 8);
	    pt = pt + len + 2;
	}

	// Adding support for app level data:
	if ( sizeData > 0 ) {

	    // Length of option name
	    int appOptionLength = APP_DATA.length();
	    bytes[pt++] = (byte)(appOptionLength & 0xFF);

	    // Option name
	    System.arraycopy(APP_DATA.getBytes(), 0, bytes, pt, appOptionLength);
	    pt += appOptionLength;

	    // Length of app data
	    bytes[pt++] = (byte)(sizeData & 0xFF);

	    // App data itself
	    System.arraycopy(data,0,bytes,pt,sizeData);
	}

	return bytes;
    }

    public int getByteLength() 
    {
	if (transport.size() != port.size()) 
	    return 0;

	int len = 8;
	for (int i=0; i<transport.size(); i++)
	{
	    String strTransport = (String)transport.elementAt(i);
	    len = len + 1 + strTransport.length() + 2;
	}
	
	// Adding sizeData allows to support app level data
	return len + bytesForAppDataOption();
    }

    protected int bytesForAppDataOption() {
	if ( sizeData > 0 )
	    return 1 + APP_DATA.length() + 1 + sizeData;
	else return 0;
    }

    public void addTransport(String t, short p)
    {
	transport.addElement(t);
	port.addElement(new Short(p));
    }

    /** Returns the port for a transport type, or -1 if not found */
    public short getPortForTransport(String t) 
    {
	int index = transport.indexOf(t);
	if (index == -1)
	    return (short) -1;
	else
	    return ((Short)port.elementAt(index)).shortValue();
    }

    public String toString()
    {
	StringBuffer strBuf = new StringBuffer(
		      "Address = " + Integer.toString(address[0] & 0xFF) +
		      "." + Integer.toString(address[1] & 0xFF) +
		      "." + Integer.toString(address[2] & 0xFF) +
		      "." + Integer.toString(address[3] & 0xFF) );

	strBuf.append("\nUDPport = " + UDPport);
	strBuf.append("\nTCPport = " + TCPport + "\nTransport Types:\n\t");

	for (int i=0; i<transport.size(); i++)
	{
	    strBuf.append((String)transport.elementAt(i));
	    strBuf.append(" port # ");
	    strBuf.append(((Short)port.elementAt(i)).toString());
	    strBuf.append("\n\t");
	}

	strBuf.append("\nWith " + sizeData + " bytes of app data");

	return strBuf.toString();
    }    

    public InetAddress getInetAddress() {
	// convert hr.address to InetAddress type	
	try {
	    String stria = Integer.toString(address[0] & 0xFF) +
		"." + Integer.toString(address[1] & 0xFF) +
		"." + Integer.toString(address[2] & 0xFF) +
		"." + Integer.toString(address[3] & 0xFF);
	    InetAddress ia = InetAddress.getByName(stria);
	    return ia;
	} catch (UnknownHostException e) { return null; }
    }
    

    public boolean equals(Object obj)
    {
	if (!(obj instanceof HostRecord)) return false;
	HostRecord hr = (HostRecord)obj;

	if (address[0] == hr.address[0] &&
	    address[1] == hr.address[1] &&
	    address[2] == hr.address[2] &&
	    address[3] == hr.address[3] &&
	    transport.equals(hr.transport) &&
	    port.equals(hr.port)) {

	    // For all else equal compare app level data
	    if ( data == null)
		return true;
	    else if ( hr.data == null)
		return false;
	    else if ( Arrays.equals(data,hr.data))
		return true;
	    else return false;
	}
	else { 
	    return false;
	}
    }

    // To support application level data in HostRecord
    public void addData(byte[] appData) {

	data = appData;
	if ( data != null)
	    sizeData = appData.length;
	
    }

    public byte[] getData() {
	return data;
    }
}
