package ins.inr;

import ins.namespace.*;
import java.util.Vector;

/** 
 * Represents ONE update information, i.e. based on one RouteEntry
 * There can be many RouteUpdate objects in one RoutePacket.
 * (current implementation only uses one RoutePacket)  
 *
 * the length of RouteUpdate (in bytes) = 
 * 2 (describing ns length)
 * length of ns (in bytes) + 
 * 4 (int appMetric) +
 * 4 (int ttl) +
 * 8 (long announcer) +
 * 8 (long INRuid) +
 * 8 (long cacheLifetime) +
 * 2 (describing host-record length) +
 * length of host-record (see HostRecord class)
 */
public class NameUpdate
{
    // VARIABLES	
    public NameSpecifier ns;
    public int appMetric;
    public long INRuid;        // the UID of INR announcing this name
    public long cacheLifetime;
    public HostRecord hostRecord;
    public long announcer;
    public byte action;
    public int ttl;

    // these should be here to be accessible to ins.api package
    public static final byte ADD_NAME = 0;
    public static final byte REMOVE_NAME = 1;
    public static final byte UPDATE_NAME = 2;

    // CONSTRUCTORS

    private NameUpdate() 
    {
	ns=null;
	appMetric=0;
	INRuid=0;
	cacheLifetime=0;
	hostRecord=null;
	announcer=0;
	ttl = (int)RouteManager.MAX_NAME_TTL;
    }

    /** 
     * Create a new RouteUpdate 
     * @param n the NameSpecifier the route update is for
     * @param m the appMetric of the route update
     * @param uid the UID of INR announcing the name
     * @param ct the time that objects associated with this route
     *           should be cached, or 0 for no caching
     * @param hr the early-binding host record
     * @param ann the announcer of the update
     * @param act the associated action for this update (see RouteManager)
     */
    /*    public NameUpdate(NameSpecifier n, int m, long uid, long ct, 
	HostRecord hr, long ann, byte act)
    {
	ns = n;
	appMetric = m;
	INRuid = uid;
	cacheLifetime = ct;
	hostRecord = hr;
	announcer = ann;
	action = act;
	//this.ttl = (int)RouteManager.MAX_NAME_TTL;
    }
    */
    /** 
     * Create a new RouteUpdate 
     * @param n the NameSpecifier the route update is for
     * @param m the appMetric of the route update
     * @param uid the UID of INR announcing the name
     * @param ct the time that objects associated with this route
     *           should be cached, or 0 for no caching
     * @param hr the early-binding host record
     * @param ann the announcer of the update
     * @param act the associated action for this update (see RouteManager)
     * @param ttl time to live in milliseconds
     */
    public NameUpdate(NameSpecifier n, int m, long uid, long ct, 
	HostRecord hr, long ann, byte act, int ttl)
    {
	ns = n;
	appMetric = m;
	INRuid = uid;
	cacheLifetime = ct;
	hostRecord = hr;
	announcer = ann;
	action = act;
	this.ttl = ttl;
    }


    /** 
     * Create a new RouteUpdate 
     * @param n the NameSpecifier the route update is for
     * @param nr the NameRecord for NameSpecifier n
     * @param act the associated action for this update (see RouteManager)
     */
    NameUpdate(NameSpecifier n, NameRecord nr, byte act)
    {	// NOTE: don't make this constructor 'public' such that it is visible
	// to applications (this constructor is only for use by RouteManager)
	ns = n;
	appMetric = nr.getAppMetric();
	INRuid = nr.getINRuid();
	cacheLifetime = nr.getCacheLifetime();
	hostRecord = nr.getHostRecord();
	announcer = nr.getAnnouncer();
	action = act;
	//ttl = (int)nr.getTimeLeft();
	ttl = -1;
    }

    /** 
     * Create a new RouteUpdate 
     * @param bytes the wire format of a route update 
     * @exception Exception something went wrong
     */
    public NameUpdate(byte[] bytes)
    {
	fillInFromBytes(bytes, 0);
    }

    public static NameUpdate [] getNameUpdatesFromBytes(byte [] bytes)
    {
	Vector v=new Vector(16);
	int start = 0;

	while (start<bytes.length) {
	    NameUpdate nu = new NameUpdate();
	    start = nu.fillInFromBytes(bytes, start);
	    v.addElement(nu);
	}
	
	NameUpdate [] nuArr = new NameUpdate[v.size()];
	v.copyInto(nuArr);

	return nuArr;
    }

    private int fillInFromBytes(byte [] bytes, int offset)
    {
	short nsLength;

	action = bytes[offset];

	nsLength = (short)(Conversion.extract16toIntLow16(bytes, offset+1));

	ns = new NameSpecifier(new String(bytes, offset+3, nsLength));

	int ct = offset+3+nsLength;   // 0-based index

	// calculate the (int) appMetric from 4 bytes
	appMetric = Conversion.extract32toInt(bytes, ct);
	ct = ct + 4;

	// calculate the (int) ttl from 4 bytes
	ttl = Conversion.extract32toInt(bytes, ct);
	ct = ct + 4;

	// calculate the (long) announcer from 8 bytes
	announcer |= Conversion.extract64toLong(bytes, ct);
	ct = ct + 8;

	// calculate the (long) INRuid from 8 bytes
	INRuid = Conversion.extract64toLong(bytes, ct);
	ct = ct + 8;

	// calculate the (long) cacheLifetime from 8 bytes
	cacheLifetime = Conversion.extract64toLong(bytes, ct);
	ct = ct + 8;

	//  	System.err.println(" ## " +
	//  			   bytes[ct] + "." + 
	//  			   bytes[ct+1] + "." + 
	//  			   bytes[ct+2] + "." + 
	//  			   bytes[ct+3] + "." + 
	//  			   bytes[ct+4] + "." + 
	//  			   bytes[ct+5] + "." + 
	//  			   bytes[ct+6] + "." + 
	//  			   bytes[ct+7]);

	short hrLength;
	hrLength = (short)Conversion.extract16toIntLow16(bytes, ct);

	try {
	    hostRecord = new HostRecord(bytes, ct+2, hrLength);
	} catch (Exception e) {
	    System.out.println("WARNING: received invalid HostRecord in update!!");
	    hostRecord = new HostRecord(HostRecord.INVALID_ADDRESS,0,0,
		new Vector(), new Vector());
	}

	return (ct+2+hrLength);
    }

    /** 
     * Get number of bytes needed to represent this RouteUpdate
     * @return the number of bytes needed to represent this RouteUpdate
     */
    /* public int getByteLength() 
    {
	return (ns.toString().length() + hostRecord.getByteLength() + 28);
	} */

    public static void mutateMetricInWireform
	(byte [] data, int offset, int metric)
    {
	int nslen = Conversion.extract16toIntLow16(data, offset+1);
	Conversion.insertInt(data, nslen+3+offset, metric);
    }

    public static int getMetricInWireform(byte [] data, int offset)
    {
	int nslen = Conversion.extract16toIntLow16(data, offset+1);
	return Conversion.extract32toInt(data, nslen+3+offset);
    }

    /**
     * Returns the RouteUpdate represented as an array of bytes.
     * @return the RouteUpdate represented as an array of bytes
     */
    public byte[] toBytes() 
    {
	byte[] nsBytes = ns.toString().getBytes();
	short nsLength = (short)nsBytes.length;
	byte [] hrArray = hostRecord.toBytes();
	int hrLength = hrArray.length;

	byte[] bytes = new byte[1+ nsLength + hrLength + 36]; 

	bytes[0] = action;
	Conversion.insertIntLow16(bytes, 1, (int)nsLength);
	
	System.arraycopy(nsBytes, 0, bytes, 3, nsLength);
	int pt = nsLength+3;
	Conversion.insertInt(bytes, pt, appMetric);
	pt += 4;

	Conversion.insertInt(bytes, pt, ttl);
	pt += 4;

	Conversion.insertLong(bytes, pt, announcer);
	pt += 8;

	Conversion.insertLong(bytes, pt, INRuid);
	pt += 8;

	Conversion.insertLong(bytes, pt, cacheLifetime);
	pt +=8;

	//  	System.err.println(" ** " +
	//  			   bytes[nsLength+18] + "." + 
	//  			   bytes[nsLength+19] + "." + 
	//  			   bytes[nsLength+20] + "." + 
	//  			   bytes[nsLength+21] + "." + 
	//  			   bytes[nsLength+22] + "." + 
	//  			   bytes[nsLength+23] + "." + 
	//  			   bytes[nsLength+24] + "." + 
	//  			   bytes[nsLength+25]);

	Conversion.insertIntLow16(bytes, pt, (int)hrLength);
	System.arraycopy(hrArray, 0, bytes, pt+2, hrLength);

	return bytes;
    }

    /**
     * Returns a String representation of the NameUpdate.
     */
    public String toString()
    {
	String output = "action: " + action +
	    "\n  NameSpecifier: " + ns +
	    "\n  appMetric: " + appMetric +
	    "\n  Announcer: " + announcer +
	    "\n  INRuid: " + INRuid +
	    "\n  cacheLifetime: " + cacheLifetime +
	    //"\n  ttl: " + ttl +
	    "\n  HostRecord: " + hostRecord.toString();

	return(output);
    }	
}
