package ins.inr;

import java.util.Date;
import java.util.Enumeration;
import java.util.Vector;
import java.io.*;
import ins.namespace.*;

class NetProber 
    implements IHandler, Runnable
{
    // VARIABLES
    public final static int PING_INTERVAL = 15000;	// 15s
    
    protected Resolver resolver;
    protected VSNameTree nameTrees;
    protected VSResolvers resolvers;
    protected Communicator comm;

    protected NameSpecifier pingNS, pingackNS, setNS, setackNS;
    protected NameSpecifier sourceNS;

    Attribute controlmsgAttr = new Attribute("control-msg");
    Value pingVal = new Value("ping");
    Value pingackVal = new Value("ping-ack");
    Value setVal = new Value("setrtt");
    Value setackVal = new Value("setrtt-ack");

    final static byte[] zerobyte = new byte[0];
    final long DEFAULT_NEW_NODE_LIFE = 600000; // 10 minutes 

    long pingTime;
    Node pingNode;
    int rtt;
    boolean isWaitForPingAck, isWaitForSetAck;
    Object waitForPingAck, waitForSetAck;

    Thread periodicThread;	// to check expired names

    PrintWriter pw;
    boolean isLog;
    boolean probing;

    // CONSTRUCTORS
    NetProber()
    {
  waitForPingAck = new Object();
	waitForSetAck = new Object();

	if (Resolver.DEBUG>=2) {
	    isLog = true;
	    try{
		String logfile = "NetProber.log";
		pw = new PrintWriter(new FileWriter(logfile), true);
	    } catch (IOException e) {
		isLog = false;
	    }
	} 
	else
	   isLog = false;

    }

    // METHODS

    /** 
     *  A callback function: will be called during initialization
     *  Provide an access to the VSNameTree and Forwarder objects
     *  (Interface implementation of IHandler)
     *  @param r the Resolver object
     */
    public void init(Resolver r)
    {
	resolver = r;
	nameTrees = r.nameTrees;
	comm = r.comm;
	resolvers = r.resolvers;

	pingNS = new NameSpecifier("[control-msg=ping]");
	pingackNS = new NameSpecifier("[control-msg=ping-ack]");
	setNS = new NameSpecifier("[control-msg=setrtt]");
	setackNS = new NameSpecifier("[control-msg=setrtt-ack]");

	// Note: getHostName() will use ip addr if DNS name doesn't exist
	// but better use getHostAddress() since Java for Windows doesn't
	// correctly get the name if the ip suddenly changes
	// (it may cache the name somewhere)
	sourceNS = new NameSpecifier(
	     "[host = " + comm.localhost.getHostAddress() +
	     "][UDPport = " + comm.UDPport + "]" +
	     "][TCPport = " + comm.TCPport + "]");

  NameRecord nr;
	// ping, pingack, set, and setack messages for all vspaces
	// are destined to me
	Enumeration enum = nameTrees.getVspaces();
	// System.out.println(nameTrees.toString());
	while (enum.hasMoreElements())
	{
	    String vspace = (String)(enum.nextElement());
	    produceVspace(vspace);
	}


  // Create a thread for periodic ping
	periodicThread = new Thread(this);
    }


    /** 
     *  A callback function: will be called every time a new
     *  INS packet for this handler arrives by UDP.
     *  (Interface implementation of IHandler)
     *  @param msg
     */
    public void INSreceive(Message msg)
    {
	printStatus("receiving a message by UDP...");

	Packet packet = msg.packet;
	Packet p;
	String vspace = packet.dNS.getVspace();

	if (Resolver.DEBUG >= 5) 
	    printStatus(packet.toString());
	
	AVelement av = packet.dNS.getAVelement(controlmsgAttr);
	if (av.getValue().equals(pingVal)) 
	{
	    pingackNS.setVspace(vspace);
	    p = new Packet(sourceNS, pingackNS, 2, 
		Packet.toAny, false, zerobyte, zerobyte);
	    comm.sendMessage(new Message(p, msg.ia, msg.port));
	}
	else if (av.getValue().equals(pingackVal)) 
	{
	    if (!isWaitForPingAck) return;
	    if (!msg.ia.equals(pingNode.ia) || 
		(msg.port != pingNode.UDPport))
	    	return;	 // invalid or duplicate due to retransmit

	    long curTime = System.currentTimeMillis();
	    rtt = (int)(curTime - pingTime);
	    
	    isWaitForPingAck = false;
	    synchronized (waitForPingAck) {
		waitForPingAck.notifyAll();
	    }
	}
	else if (av.getValue().equals(setVal)) 
	{
	    Node n;

	    // make sure that resolvers has Node n for vspace
	    n = new Node(msg.ia, msg.port, 0, vspace, 
			 DEFAULT_NEW_NODE_LIFE);

	    if (resolvers.addResolver(vspace, n))
	    {
		if (Resolver.DEBUG >=5)
		   printStatus("added resolver: " + n.toString());
	    }
	    else
	    {
		if (Resolver.DEBUG >=5)
		    printStatus("already exist: resolver " + n.toString());
	    }

	    // lookup INRuid whose RTT to be set
	    n = resolvers.getResolver(msg.ia, msg.port, 0);

	    n.rtt = Conversion.extract32toInt(packet.data, 0);

	    // then send ACK
	    setackNS.setVspace(vspace);
	    p = new Packet(sourceNS, setackNS, 2, 
		Packet.toAny, false, zerobyte, zerobyte);
	    comm.sendMessage(new Message(p, msg.ia, msg.port));

	}
	else if (av.getValue().equals(setackVal)) 
	{
	    if (!isWaitForSetAck) return;
	    if (!msg.ia.equals(pingNode.ia) || 
		(msg.port != pingNode.UDPport))
	    	return;	 // invalid or duplicate due to retransmit
	    
	    isWaitForSetAck = false;
	    synchronized (waitForSetAck) {
		waitForSetAck.notifyAll();
	    }
	}

    }

    /** 
     *  A callback function: will be called every time a new
     *  INS packet for this handler arrives by TCP.
     *  (Interface implementation of IHandler)
     *  @param p the INS packet
     *  @param from the neighbor sending the packet p     
     */
    public void INSreceiveByTCP(Packet p, Node from)
    {
	printStatus("receiving a message by TCP...");
    }

    protected void determineRTT(Node n, String vspace)
    {
	pingNS.setVspace(vspace);
	Packet p = new Packet(sourceNS, pingNS, 2, 
		Packet.toAny, false, zerobyte, zerobyte);
	Message msg = new Message(p, n.ia, n.UDPport);

	pingNode = n;
	isWaitForPingAck = true;
	pingTime = System.currentTimeMillis();
	comm.sendMessage(msg);

	try {
	synchronized(waitForPingAck) {
	    waitForPingAck.wait();
	}
	} catch (InterruptedException e) {
	    e.printStackTrace();
	}

	// then send setrtt control-msg
	// should have time out and re-transmit (todo)

	setNS.setVspace(vspace);
	msg.packet.dNS = setNS;
	msg.packet.data = new byte[4];
	Conversion.insertInt(msg.packet.data, 0, rtt);

	// printStatus("*** rtt is " + rtt);

	isWaitForSetAck = true;	
	comm.sendMessage(msg);

	try {
	synchronized(waitForSetAck) {
	    waitForSetAck.wait();
	}
	} catch (InterruptedException e) {
	    e.printStackTrace();
	}

	n.rtt = rtt;

	printStatus(n.toString());
    }

    void pingAll()
    {
	// Find RTT for all vspaces
	Enumeration enum;
	enum = nameTrees.getVspaces();
	while (enum.hasMoreElements())
	{
	    String vspace = (String)(enum.nextElement());
	    NodeSet nset = resolvers.getResolvers(vspace);
	    for (Enumeration e=nset.getNodes(); e.hasMoreElements(); )
	    {
		determineRTT((Node)(e.nextElement()), vspace);
	    }
	}
    }
    

    /** 
     * Performs actions necessary to add vspace for this module.
     * Specifically, it registers handlers in the vspace's nametree
     * for its functionality.
     */
    void produceVspace(String vspace)
    {
	NameRecord nr;

	pingNS.setVspace(vspace);
	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTrees.addNameRecord(pingNS, vspace, nr);
	
	pingackNS.setVspace(vspace);
	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTrees.addNameRecord(pingackNS, vspace, nr);
	
	setNS.setVspace(vspace);
	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTrees.addNameRecord(setNS, vspace, nr);
	
	setackNS.setVspace(vspace);
	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTrees.addNameRecord(setackNS, vspace, nr);
    }


    void startPeriodicPing()
    {
	probing = true;
	periodicThread.start();
    }

    void stopPeriodicPing()
    {
	probing = false;
    }

    public void run()
    {
	while (probing)
	{	
	    pingAll();

    	    try {Thread.sleep(PING_INTERVAL);}
	    catch (InterruptedException e) {e.printStackTrace();}
	}
    }

    public void printStatus(String s)
    {
	String text = /*new Date(System.currentTimeMillis()).toString() + */"Prober: "+s;
	System.out.println(text);
	if (isLog) pw.println(text);
    }
}
