package ins.inr;

import java.util.Date;
import java.util.Vector;
import java.util.Enumeration;
import java.util.StringTokenizer;
import java.io.*;
import java.net.*;
import ins.namespace.*;
import ins.dsr.DSRRequest;
import ins.dsr.DSRResponse;

class OverlayManager 
    implements IHandler, Runnable
{
    // VARIABLES
    public final static int DEFAULT_DSR_PORT = 5678;
    public final static int PROBE_INTERVAL = 15000;	// 120s
    public final static long DEFAULT_RETRANSMIT_TIMER = 1000;	// 1s
						// not usually used
    public final static int RTT_MULTIPLIER = 8;   // was 6 before
    public final static int OWT_MULTIPLIER = 6;   // was 4 before

    public final static int PROBE_MODE = 0;
    public final static int LINK_REPLACE_MODE = 1;

    final static int THRESHOLD = 2;
    
    protected Resolver resolver;
    protected VSNameTree nameTrees;
    protected VSResolvers resolvers;
    protected VSNeighbors neighbors;
    protected VSRouteTable routeTables;
    protected Communicator comm;
    protected DSRManager dsrMn;

    protected NameSpecifier overlayNS, sourceNS;
    protected AVelement overlayAVE;
    final static byte[] zerobyte = new byte[0];


    MutableBoolean isWaitForNeighborAck, isWaitForReplaceAck, 
	isWaitForCRAck, isWaitForDone, isWaitForConnectAck;
    Object waitForNeighborAck, waitForReplaceAck,
	waitForCRAck, waitForDone, waitForConnectAck;
    Node neighborNode, connectNode;


    int expMetric, expBtneckMetric;
    long expSource, expFirstEnd, expSecondEnd;
    Node expSourceNode;
    String expVSpace;

    Thread periodicThread;
    boolean sendingProbe;

    int mode;

    PrintWriter pw;
    boolean isLog;

    final long DEFAULT_NEW_NODE_LIFE = 600000; // 10 minutes 


    // CONSTRUCTORS
    OverlayManager()
    {
    	isWaitForNeighborAck = new MutableBoolean();
	waitForNeighborAck = new Object();
	neighborNode = null;

	isWaitForReplaceAck = new MutableBoolean();
	waitForReplaceAck = new Object();

	isWaitForCRAck = new MutableBoolean();
	waitForCRAck = new Object();

	isWaitForDone = new MutableBoolean();
	waitForDone = new Object();

    	isWaitForConnectAck = new MutableBoolean();
	waitForConnectAck = new Object();
	connectNode = null;

	if (Resolver.DEBUG>=2) {
	    isLog = true;
	    try{
		String logfile = "OverlayManager.log";
		pw = new PrintWriter(new FileWriter(logfile), true);
	    } catch (IOException e) {
		isLog = false;
	    }
	} 
	else
	   isLog = false;

    }

    // METHODS

    /** 
     *  A callback function: will be called during initialization
     *  Provide an access to the VSNameTree and Forwarder objects
     *  (Interface implementation of IHandler)
     *  @param r the Resolver object
     */
    public void init(Resolver r)
    {
	resolver = r;
	nameTrees = r.nameTrees;
	comm = r.comm;
	resolvers = r.resolvers;
	neighbors = r.neighbors;
	routeTables = r.routeTables;
	dsrMn = r.dsrMn;

	// create overlayNS = [control-msg=overlay]
	overlayAVE = new AVelement(new Attribute("control-msg"),
				   new Value("overlay"));
	overlayNS = new NameSpecifier();
	overlayNS.addAVelement(overlayAVE);

	// Note: getHostName() will use ip addr if DNS name doesn't exist
	// but better use getHostAddress() since Java for Windows doesn't
	// correctly get the name if the ip suddenly changes
	// (it may cache the name somewhere)
	sourceNS = new NameSpecifier(
	     "[host = " + comm.localhost.getHostAddress() +
	     "][UDPport = " + comm.UDPport + "]" +
	     "][TCPport = " + comm.TCPport + "]");

	// Contacting DSR to "register" itself and find other INRs
	if (!dsrMn.refreshDSRInfo())
	    System.exit(-1);

	Enumeration enum;
	enum = nameTrees.getVspaces();
	while (enum.hasMoreElements())
	{
	    String vspace = (String)(enum.nextElement());
	    produceVspace(vspace);
	}

	// Ask NetProber to determine RTTs to all other INRs
	r.prober.pingAll();

	// Init phase of MST
	enum = nameTrees.getVspaces();
	while (enum.hasMoreElements())
	{
	    initPhase((String)(enum.nextElement()));
	}

	// Create a thread for sending periodic PROBE
	periodicThread = new Thread(this);
    }


    void initPhase(String vspace)
    {
	NodeSet nset = resolvers.getResolvers(vspace);

	if ((nset==null) || (nset.countNodes()==0))
	    return;

	printStatus("Resolvers are: \n" + nset.toString());

	// Pick a resolver to which I have the best connection
	int bestRTT = Integer.MAX_VALUE;
	Node bestNode = null;

	for (Enumeration e=nset.getNodes(); e.hasMoreElements(); )
	{
	    Node n = (Node)(e.nextElement());
	    if (n.rtt < bestRTT)
	    {
		bestRTT = n.rtt;
		bestNode = n;
	    }
	}

	// Send NEIGHBOR-RQ to bestNode
	// a string in the form of vspace:host:UDPport:TCPport
	String message = "neighbor-rq\n" + 
	    vspace + ":" + comm.localhost.getHostAddress() + ":" +
	    comm.UDPport + ":" + comm.TCPport + "\n";
	
	// create a namespecifier [control-msg=overlay][vspace=therightvspace]
	NameSpecifier thisoverlayNS = new NameSpecifier();
	thisoverlayNS.addAVelement(overlayAVE);
	thisoverlayNS.setVspace(vspace);

	Packet neighborRqPacket = new Packet(sourceNS, thisoverlayNS, 2, 
	    Packet.toAny, false, zerobyte, message.getBytes());

	isWaitForNeighborAck.value = true;
	neighborNode = bestNode;

	int retransmitTimer = (RTT_MULTIPLIER * bestNode.rtt)/2;
	
	int numtries=0;

	// do the wait in here (rather than creating new Thread
	//   of SendWithRetransmit) since this is Init Phase of MST
	while (isWaitForNeighborAck.value)
	{

	    // allow a maximum number of tries to do a neighbor request
	    numtries++;
	    if (numtries > 3) {
		resolvers.removeResolver(bestNode, vspace);
		initPhase(vspace);
		return;
	    }

	comm.sendMessage(new Message(neighborRqPacket, bestNode.ia, 
	    bestNode.UDPport));

	// Wait for NEIGHBOR-OK or NEIGHBOR-FAIL
	try {
	synchronized(waitForNeighborAck) {
	    waitForNeighborAck.wait(retransmitTimer);
	}
	} catch (InterruptedException e) {
	    e.printStackTrace();
	}

	}  // while(isWaitForNeighborAck)

    }

    /** 
     *  A callback function: will be called every time a new
     *  INS packet for this handler arrives by UDP.
     *  (Interface implementation of IHandler)
     *  @param msg
     */
    public void INSreceive(Message msg)
    {
	printStatus("receiving a message by UDP...");

	String data = new String(msg.packet.data);
	//printStatus(msg.toString());
	printStatus(data);
	
	StringTokenizer st = new StringTokenizer(data, "\n \t\r,:()");
	
	String name=null;
	InetAddress ia=null;
	String stria=null;
	String strport=null;
	int UDPport=0;
	int TCPport=0;

	while (st.hasMoreTokens()) 
	{
	String command = st.nextToken();
	    
	if (command.equals("neighbor-rq")) 
	{
	    if (st.countTokens()>=4) 
	    {
		name = st.nextToken();

		try 
		{
			stria = trim(st.nextToken());
			ia = InetAddress.getByName(stria);
		        strport = trim(st.nextToken());
			UDPport = Integer.parseInt(strport);
		        strport = trim(st.nextToken());
			TCPport = Integer.parseInt(strport);
		} catch (UnknownHostException e) { 
			printStatus
			    ("Invalid INR address in produce_vspace request");
			continue;
		} catch (Exception e) { 
			printStatus
			    ("Malformed number in produce_vspace request");
			continue;
		}

		System.out.println("neighbor-rq("+name+":"+stria+ 
			":"+UDPport+":"+TCPport+") from " + 
			msg.ia.toString() + ":" + msg.port);

		Node n = new Node(ia, UDPport, TCPport, name, 
				  DEFAULT_NEW_NODE_LIFE);

		// if already exists, addResolver will return false
		resolvers.addResolver(name, n);
		//System.out.println("before getResolver = "+n.toString());
		n = resolvers.getResolver(n);
		//System.out.println("from getResolver = "+n.toString());
		n.UDPport = UDPport;
		n.TCPport = TCPport;
		n.sendAllKnown = true;
		n.state = Node.ACTIVE;
		
		neighbors.addNeighbor(name, n);

		// Creating and sending a NEIGHBOR-OK packet
		// a string in the form of vspace:host:UDPport:TCPport
		String message = "neighbor-ok\n" + 
			name + ":" + comm.localhost.getHostAddress() + ":" +
			comm.UDPport + ":" + comm.TCPport + "\n";
	
		// create a namespecifier [control-msg=overlay]
		//                        [vspace=therightvspace]
		NameSpecifier thisoverlayNS = new NameSpecifier();
		thisoverlayNS.addAVelement(overlayAVE);
		thisoverlayNS.setVspace(name);

		Packet neighborOkPacket = new Packet(sourceNS, thisoverlayNS, 
			2, Packet.toAny, false, zerobyte, 
			message.getBytes());

		comm.sendMessage(new Message(neighborOkPacket, msg.ia, msg.port));

	    } else 
	    {
		printStatus("Too few parameters in neighbor-rq");
		continue;
	    }   
	} // command.equals("neighbor-rq")

	else if (command.equals("neighbor-ok")) 
	{
	    if (!isWaitForNeighborAck.value) return;
	    if (!msg.ia.equals(neighborNode.ia) || 
		(msg.port != neighborNode.UDPport))
	    	return;	 // invalid or duplicate due to retransmit

	    neighborNode.state = Node.ACTIVE;
	    neighbors.addNeighbor(st.nextToken(), neighborNode);

	    isWaitForNeighborAck.value = false;

	    // Setup TCP connection to bestNode
	    if (neighborNode.in == null)  // !null because this INR belongs to
				// >1 vspace, prev.vspace has caused TCP conn.
	    {
///***test this 
		neighborNode.sendAllKnown = true;
	    	comm.connectTCP(neighborNode);
	    }

	    synchronized (waitForNeighborAck) {
		waitForNeighborAck.notifyAll();
	    }
	}

	else if (command.equals("neighbor-fail")) 
	{
	    if (!isWaitForNeighborAck.value) return;
	    if (!msg.ia.equals(neighborNode.ia) || 
		(msg.port != neighborNode.UDPport))
	    	return;	 // invalid or duplicate due to retransmit

	    isWaitForNeighborAck.value = false;

	    synchronized (waitForNeighborAck) {
		waitForNeighborAck.notifyAll();
	    }
	}

	else if (command.equals("connect-rq")) 
	{
	    processConnectRq(st, msg);
	}

	else if (command.equals("connect-ack")) 
	{
	    processConnectAck(st, msg);
	}

	} //while
    }

    /** 
     *  A callback function: will be called every time a new
     *  INS packet for this handler arrives by TCP.
     *  (Interface implementation of IHandler)
     *  @param p the INS packet
     *  @param from the neighbor sending the packet p     
     */
    public void INSreceiveByTCP(Packet p, Node from)
    {
	printStatus("receiving a message by TCP...");

	String data = new String(p.data);
	//printStatus(msg.toString());
	printStatus(data);
	
	StringTokenizer st = new StringTokenizer(data, "\n \t\r,:()");
	
	String name=null;

	while (st.hasMoreTokens()) 
	{
	    String command = st.nextToken();
	    
	    if (command.equals("probe")) 
	    {
		processProbe(st, from);
	    }

	    else if (command.equals("replace-rq")) 
	    {
		processReplaceRq(st, from);
	    }

	    else if (command.equals("replace-ok")) 
	    {
		processReplaceOk(st, from);
	    }

	    else if (command.equals("replace-fail")) 
	    {
		processReplaceFail(st, from);
	    }

	    else if (command.equals("replace-done")) 
	    {
		processReplaceDone(st, from);
	    }

	    else if (command.equals("dormant-rq")) 
	    {
		processDormantRq(st, from);
	    }

	    else if (command.equals("confirm-replace")) 
	    {
		processCR(st, from);
	    }

	    else if (command.equals("confirm-ok")) 
	    {
		processConfirmOk(st, from);
	    }

	    else if (command.equals("confirm-fail")) 
	    {
		processConfirmFail(st, from);
	    }

	    else if (command.equals("dormant-ok")) 
	    {
		processDormantOk(st, from);
	    }

	    else if (command.equals("dormant-fail")) 
	    {
		processDormantFail(st, from);
	    }

	    else if (command.equals("disconnect")) 
	    {
		processDisconnect(st, from);
	    }

	    else
	    {
		printStatus("WARNING: unknown message received:");
		printStatus(command);
	    }

	} // while
    }


    protected void processProbe(StringTokenizer st, Node from)
    {
	printStatus("processing probe ...");

	if (st.countTokens()>=5) 
	{
	    String vspace = st.nextToken();
	    long source = Long.parseLong(st.nextToken());
	    int metric = Integer.parseInt(st.nextToken());
	    long firstEnd = Long.parseLong(st.nextToken());
	    long secondEnd = Long.parseLong(st.nextToken());

	    printStatus("*** " + source + ",(" + firstEnd + "," +
		secondEnd + "):" + metric);

	    long fromINRuid = from.getINRuid();

	    if (from.rtt > metric)	// new bottleneck found
	    {
		metric = from.rtt;
		firstEnd = fromINRuid;
		secondEnd = resolver.INRuid;
	    }
	    else if (from.rtt == metric)
	    {
		if (fromINRuid > firstEnd)
		{
		    firstEnd = fromINRuid;
		    secondEnd = resolver.INRuid;
		}
		else if (fromINRuid == firstEnd)
		{
		    printStatus("****** WARNING: should not get here ****");
		    printStatus("****** ..... " + fromINRuid);

		    if (resolver.INRuid > secondEnd)
		    {
		        firstEnd = fromINRuid;
		        secondEnd = resolver.INRuid;
		    }
		}
	    }

	    InetAddress ia = extractAddrFromINRuid(source);
	    int UDPport = extractPortFromINRuid(source);
	    Node srcNode = resolvers.getResolver(vspace, ia, UDPport, 0);

	    if (srcNode == null)
	    {
		srcNode = new Node(ia, UDPport, 0, vspace,
				   DEFAULT_NEW_NODE_LIFE);
		resolvers.addResolver(vspace, srcNode);

		printStatus("Added to resolvers a newly found INR with uid " + source);
		printStatus(srcNode.toString());
	    }

	    printStatus("*** ... metric to source is " + srcNode.rtt);

	    if ((THRESHOLD*srcNode.rtt) < metric && (mode == PROBE_MODE))	
		// found bottleneck replacement and not in the middle of another
		// experiment
	    {
		printStatus("*** found link replacement for bottleneck ***");
		printStatus("... (" + firstEnd + "," + secondEnd + "):" + metric);
		printStatus("... with new link to " + srcNode.toString());

		mode = LINK_REPLACE_MODE;
		if (secondEnd == resolver.INRuid)
		    quickLinkReplace(vspace, source, srcNode, metric, firstEnd, 
		    secondEnd, from);
		else
		    linkReplace(vspace, source, srcNode, metric, firstEnd, 
		    secondEnd, from);
	    }
	    else
	    {		// propagate PROBE to neighbors
		sendProbe(vspace, source, metric, firstEnd, secondEnd,
		    from);
	    }
	}
	else
	{
	    printStatus("Too few parameters in PROBE message");
	}   
    }

    protected void processReplaceRq(StringTokenizer st, Node from)
    {
	if (st.countTokens() < 7)
	{
	    printStatus("Too few parameters in Replace-rq message");
	    return;
	}

	// format is ReplaceRq(vspace, Cab:a,b, Cis:i,s) to next hop of b
	String vspace = st.nextToken();
	int btneckMetric = Integer.parseInt(st.nextToken());
	long firstEnd = Long.parseLong(st.nextToken());
	long secondEnd = Long.parseLong(st.nextToken());

	int replacingMetric = Integer.parseInt(st.nextToken());
	long expInit = Long.parseLong(st.nextToken());
	long source = Long.parseLong(st.nextToken());

	Node firstEndNode = neighbors.getNeighbor(vspace,
		extractAddrFromINRuid(firstEnd),
		extractPortFromINRuid(firstEnd), 0);

	// check if it is fine to forward the ReplaceRq
	// or new bottleneck for the path has just appeared ..
/*	if (from.rtt > btneckMetric)	
	{
	    sendReplaceFail(vspace, btneckMetric, firstEnd, secondEnd,
		replacingMetric, expInit, source, from, false);
	    return;
	}
*/

	if (firstEnd == resolver.INRuid)
	{
	    // path-verification fails 
	    sendReplaceFail(vspace, btneckMetric, firstEnd, secondEnd,
		    replacingMetric, expInit, source,  from, false);
	    return;
	}
	
	if (secondEnd == resolver.INRuid)
	{
	    if (firstEndNode == null)
	    {
		printStatus("**** WARNING:  Replace-rq message contains " +
		    "INVALID firstEnd INRuid");
	        sendReplaceFail(vspace, btneckMetric, firstEnd, secondEnd,
		replacingMetric, expInit, source, from, false);
	        return;
	    }		
		
	    if ((firstEndNode.state == Node.PROGRESS) || 
		(firstEndNode.state == Node.DORMANT) ||
		(firstEndNode.accessByI.size()>0) ||	// someone
			// already tries to disconnect this link or
			// is already accessed by another exp
		(firstEndNode.rtt <= THRESHOLD*replacingMetric)) // no need
			// to replace
	    {
		if ((firstEndNode.expInit == expInit) && 
		    (firstEndNode.expSource == source))
		// the re-transmit thread is already handling this
		{
		    printStatus("Another thread has already handled this.");
		    printStatus(".. or link is already disconnected!");
		}
		else
		{	// different expInit tries to disconnect this link
	            sendReplaceFail(vspace, btneckMetric, firstEnd, secondEnd,
		    replacingMetric, expInit, source,  from, false);
		}   

	        return;
	    }

	    // Lock the metric of incoming and outgoing links
/*	    from.rttLocked = true;
	    firstEndNode.rttLocked = true;
*/

	    // add to list of exp accessing link
	    from.accessByI.addElement(new Long(expInit));
	    from.accessByS.addElement(new Long(source));
	    from.accessByC.addElement(new Integer(replacingMetric));

	    // exclusive access is set inside sendDormantRq()
	    // Send DORMANT-RQ to firstEnd
	    sendDormantRq(firstEndNode, vspace, replacingMetric, 
	    	expInit, source);
	    return;
	}

	// Forward the message otherwise
	RouteEntry re = routeTables.lookup(vspace, secondEnd);
	
	if (re == null)	
	{
	    printStatus("WARNING:  Replace-rq contains secondEnd INRuid" +
		" that my route table has no entry for!");
	    sendReplaceFail(vspace, btneckMetric, firstEnd, secondEnd,
		replacingMetric, expInit, source, from, false);
	    return;
	}

	// Lock the metric of incoming and outgoing links
/*	from.rttLocked = true;
	re.nextHop.rttLocked = true;
*/	
	if ((from.state == Node.ACTIVE) && 
	    (re.nextHop.state == Node.ACTIVE))
	{
	    // add to list of exp accessing link
	    from.accessByI.addElement(new Long(expInit));
	    from.accessByS.addElement(new Long(source));
	    from.accessByC.addElement(new Integer(replacingMetric));

	    re.nextHop.accessByI.addElement(new Long(expInit));
	    re.nextHop.accessByS.addElement(new Long(source));
	    re.nextHop.accessByC.addElement(new Integer(replacingMetric));

	    sendReplaceRq(vspace, btneckMetric, firstEnd, secondEnd,
	    replacingMetric, expInit, source, re.nextHop, false);
	}
	else
	    sendReplaceFail(vspace, btneckMetric, firstEnd, secondEnd,
		replacingMetric, expInit, source,  from, false);
	
    }

    protected void processDormantRq(StringTokenizer st, Node from)
    {
	if (st.countTokens() < 4)
	{
	    printStatus("Too few parameters in DORMANT-RQ message");
	    return;
	}

	String vspace = st.nextToken();
	int ISMetric = Integer.parseInt(st.nextToken());
	long expInit = Long.parseLong(st.nextToken());
	long source = Long.parseLong(st.nextToken());

	boolean success = true;
	
	if ((from.state == Node.DORMANT) || (from.state == Node.INACTIVE))
	{
	    printStatus("ASSERT: Invalid state of link, receiving " +
	    	"dormant-rq for an already-dormant link");
	    success = false;
	}
	
	else if (from.state == Node.PROGRESS)
	{
	    // This means both from and I tried to put the link to dormant
	    // pick either one as winner

	    // Send dormant-ok or dormant-fail depending who wins
	    if (from.expMetric < ISMetric) // if I win
	    {
		    success = false;
	    }
	    else if (from.expMetric == ISMetric)
	    {	// compare based on the uid of exp-initiator
		if (from.expInit < expInit)	// if I win
		    success = false;
	    }
	}
    
	if (success)
	{		
	    //from.state = Node.DORMANT;
	    from.state = Node.PROGRESS;
	    from.expVSpace = vspace;
	    from.expInit = expInit;
	    from.expSource = source;
	    from.expMetric = ISMetric;


	    // from.rttLocked = true;
	    
	    // send CONFIRM-REPLACE to source
	    RouteEntry re = routeTables.lookup(vspace, source);

	    if (re == null)	
	    {
		printStatus("WARNING:  processDormantRq() finds source INRuid" +
		    " that my route table has no entry for!");
		printStatus("Dropped the Replace-Rq! *******");
		return;
	    }

	    if (re.nextHop.state == Node.ACTIVE)
	    {
	        // add to list of exp accessing link
	        re.nextHop.accessByI.addElement(new Long(expInit));
	        re.nextHop.accessByS.addElement(new Long(source));
	        re.nextHop.accessByC.addElement(new Integer(ISMetric));
	    
	    	sendConfirmReplace(vspace, from.rtt, from.getINRuid(), resolver.INRuid,
	    	ISMetric, expInit, source, re.nextHop, true);

	    }
	    else
	    {

	    // send dormant-fail to from
	    String message = "dormant-fail\n";

	    printStatus("send (" + message +")");
	    
	    // create a namespecifier [control-msg=overlay]
	    //                        [vspace=therightvspace]
	    NameSpecifier thisoverlayNS = new NameSpecifier();
	    thisoverlayNS.addAVelement(overlayAVE);
	    thisoverlayNS.setVspace(vspace);

	    Packet packet = new Packet(sourceNS, thisoverlayNS, 2, 
		Packet.toAny, false, zerobyte, message.getBytes());

	    comm.sendByTCP(packet, from);
	    	
	    }
	}
	else
	{
	    // send dormant-fail to from
	    String message = "dormant-fail\n";

	    printStatus("send (" + message +")");
	    
	    // create a namespecifier [control-msg=overlay]
	    //                        [vspace=therightvspace]
	    NameSpecifier thisoverlayNS = new NameSpecifier();
	    thisoverlayNS.addAVelement(overlayAVE);
	    thisoverlayNS.setVspace(vspace);

	    Packet packet = new Packet(sourceNS, thisoverlayNS, 2, 
		Packet.toAny, false, zerobyte, message.getBytes());

	    comm.sendByTCP(packet, from);
	}
    }


    protected void processCR(StringTokenizer st, Node from)
    {
	if (st.countTokens() < 7)
	{
	    printStatus("Too few parameters in confirm-replace message");
	    return;
	}

	// format is ConfirmReplace(vspace, Cab:a,b, Cis:i,s) to next hop of b
	String vspace = st.nextToken();
	int btneckMetric = Integer.parseInt(st.nextToken());
	long firstEnd = Long.parseLong(st.nextToken());
	long secondEnd = Long.parseLong(st.nextToken());

	int replacingMetric = Integer.parseInt(st.nextToken());
	long expInit = Long.parseLong(st.nextToken());
	long source = Long.parseLong(st.nextToken());

	if (secondEnd == resolver.INRuid)
	{
	    // path-verification fails 
	    sendConfirmFail(vspace, btneckMetric, firstEnd, secondEnd,
		    replacingMetric, expInit, source,  from);
	    return;
	}
	
	if (source == resolver.INRuid)
	{
	    sendConfirmOk(vspace, btneckMetric, firstEnd, secondEnd,
		replacingMetric, expInit, source,  from);

	    return;
	}
	
	// Forward the message otherwise
	RouteEntry re = routeTables.lookup(vspace, source);
	
	if (re == null)	
	{
	    printStatus("WARNING:  Replace-rq contains source INRuid" +
		" that my route table has no entry for!");
	    sendReplaceFail(vspace, btneckMetric, firstEnd, secondEnd,
		replacingMetric, expInit, source, from, false);
	    return;
	}

	if ((from.state == Node.ACTIVE) && 
	    (re.nextHop.state == Node.ACTIVE))
	{
	    // add to list of exp accessing link
	    from.accessByI.addElement(new Long(expInit));
	    from.accessByS.addElement(new Long(source));
	    from.accessByC.addElement(new Integer(replacingMetric));

	    re.nextHop.accessByI.addElement(new Long(expInit));
	    re.nextHop.accessByS.addElement(new Long(source));
	    re.nextHop.accessByC.addElement(new Integer(replacingMetric));

	    sendConfirmReplace(vspace, btneckMetric, firstEnd, secondEnd,
	    replacingMetric, expInit, source, re.nextHop, false);
	}
	else
	{
	    sendConfirmFail(vspace, btneckMetric, firstEnd, secondEnd,
		replacingMetric, expInit, source,  from);	
	}
    }


    protected void processConfirmOk(StringTokenizer st, Node from)
    {
	if (st.countTokens() < 7)
	{
	    printStatus("Too few parameters in confirm-ok message");
	    return;
	}

	// format is ConfirmOk(vspace, Cab:a,b, Cis:i,s) to next hop of b
	String vspace = st.nextToken();
	int btneckMetric = Integer.parseInt(st.nextToken());
	long firstEnd = Long.parseLong(st.nextToken());
	long secondEnd = Long.parseLong(st.nextToken());

	int replacingMetric = Integer.parseInt(st.nextToken());
	long expInit = Long.parseLong(st.nextToken());
	long source = Long.parseLong(st.nextToken());

	if (firstEnd == resolver.INRuid)
	{
	    // verify message
	    Node secondEndNode = neighbors.getNeighbor(vspace,
		extractAddrFromINRuid(secondEnd),
		extractPortFromINRuid(secondEnd), 0);

	    if (isWaitForCRAck.value &&
	        (secondEndNode != null) &&
	        (secondEndNode.expInit == expInit) &&
	        (secondEndNode.expSource == source))
	    {
	        printStatus("... the waited confirm-ok/fail message arrived!");
	    }
	    else
	    {
		printStatus("WARNING: unexpected confirm-ok message." +
		  " Message is ignored!");
	    	return;
	    }
	    
	    isWaitForCRAck.value = false;
	    	    
	    synchronized (waitForCRAck) {
		waitForCRAck.notify();
	    }

	    secondEndNode.state = Node.DORMANT;

	    // send dormant-ok to secondEnd
	    String message = "dormant-ok\n";

	    printStatus("send (" + message +")");
	    
	    // create a namespecifier [control-msg=overlay]
	    //                        [vspace=therightvspace]
	    NameSpecifier thisoverlayNS = new NameSpecifier();
	    thisoverlayNS.addAVelement(overlayAVE);
	    thisoverlayNS.setVspace(vspace);

	    Packet packet = new Packet(sourceNS, thisoverlayNS, 2, 
		Packet.toAny, false, zerobyte, message.getBytes());

	    comm.sendByTCP(packet, secondEndNode);
	    
	    // remove from list of exp accessing link
	    int ndx1 = from.accessByI.indexOf(new Long(expInit));
	    from.accessByI.removeElementAt(ndx1);
	    from.accessByS.removeElementAt(ndx1);
	    from.accessByC.removeElementAt(ndx1);

	    return;
	}

	// Forward the message otherwise
	RouteEntry re = routeTables.lookup(vspace, firstEnd);
	
	if (re == null)	
	{
	    printStatus("WARNING:  Confirm-ok contains firstEnd INRuid" +
		" that my route table has no entry for!");
	    printStatus("Dropped the Confirm-ok! *******");
	    return;
	}

	// remove from list of exp accessing link
	int ndx1 = from.accessByI.indexOf(new Long(expInit));
	from.accessByI.removeElementAt(ndx1);
	from.accessByS.removeElementAt(ndx1);
	from.accessByC.removeElementAt(ndx1);

	int ndx2 = re.nextHop.accessByI.indexOf(new Long(expInit));
	re.nextHop.accessByI.removeElementAt(ndx2);
	re.nextHop.accessByS.removeElementAt(ndx2);
	re.nextHop.accessByC.removeElementAt(ndx2);

	if ((from.state == Node.ACTIVE) && 
	    (re.nextHop.state == Node.ACTIVE))
	{
	    sendConfirmOk(vspace, btneckMetric, firstEnd, secondEnd,
		replacingMetric, expInit, source, re.nextHop);
	}
	else
	{
	    printStatus("WARNING: invalid state(s) in processing ConfirmOk");
	    printStatus("Dropped the ConfirmOk message!");
	}
    }

    protected void processConfirmFail(StringTokenizer st, Node from)
    {
	if (st.countTokens() < 7)
	{
	    printStatus("Too few parameters in confirm-ok message");
	    return;
	}

	// format is ConfirmFail(vspace, Cab:a,b, Cis:i,s) to next hop of b
	String vspace = st.nextToken();
	int btneckMetric = Integer.parseInt(st.nextToken());
	long firstEnd = Long.parseLong(st.nextToken());
	long secondEnd = Long.parseLong(st.nextToken());

	int replacingMetric = Integer.parseInt(st.nextToken());
	long expInit = Long.parseLong(st.nextToken());
	long source = Long.parseLong(st.nextToken());

	if (firstEnd == resolver.INRuid)
	{
	    // verify message
	    Node secondEndNode = neighbors.getNeighbor(vspace,
		extractAddrFromINRuid(secondEnd),
		extractPortFromINRuid(secondEnd), 0);

	    if (isWaitForCRAck.value &&
	        (secondEndNode != null) &&
	        (secondEndNode.expInit == expInit) &&
	        (secondEndNode.expSource == source))
	    {
	        printStatus("... the waited confirm-ok/fail message arrived!");
	    }
	    else
	    {
		printStatus("WARNING: unexpected confirm-fail message." +
		  " Message is ignored!");
	    	return;
	    }
	    
	    isWaitForCRAck.value = false;
	    	    
	    synchronized (waitForCRAck) {
		waitForCRAck.notify();
	    }

	    // make link back up again
	    secondEndNode.state = Node.ACTIVE;
	    secondEndNode.expVSpace = null;
	    secondEndNode.expInit = 0;
	    secondEndNode.expSource = 0;
	    secondEndNode.expMetric = 0;

	    // send dormant-fail to secondEnd
	    String message = "dormant-fail\n";

	    printStatus("send (" + message +")");
	    
	    // create a namespecifier [control-msg=overlay]
	    //                        [vspace=therightvspace]
	    NameSpecifier thisoverlayNS = new NameSpecifier();
	    thisoverlayNS.addAVelement(overlayAVE);
	    thisoverlayNS.setVspace(vspace);

	    Packet packet = new Packet(sourceNS, thisoverlayNS, 2, 
		Packet.toAny, false, zerobyte, message.getBytes());

	    comm.sendByTCP(packet, secondEndNode);

	    
	    // remove from list of exp accessing link
	    int ndx1 = from.accessByI.indexOf(new Long(expInit));
	    from.accessByI.removeElementAt(ndx1);
	    from.accessByS.removeElementAt(ndx1);
	    from.accessByC.removeElementAt(ndx1);

	    return;
	}
	
	// Forward the message otherwise
	RouteEntry re = routeTables.lookup(vspace, firstEnd);
	
	if (re == null)	
	{
	    printStatus("WARNING:  Confirm-fail contains firstEnd INRuid" +
		" that my route table has no entry for!");
	    printStatus("Dropped the Confirm-fail! *******");
	    return;
	}

	// remove from list of exp accessing link
	int ndx1 = from.accessByI.indexOf(new Long(expInit));
	from.accessByI.removeElementAt(ndx1);
	from.accessByS.removeElementAt(ndx1);
	from.accessByC.removeElementAt(ndx1);

	int ndx2 = re.nextHop.accessByI.indexOf(new Long(expInit));
	re.nextHop.accessByI.removeElementAt(ndx2);
	re.nextHop.accessByS.removeElementAt(ndx2);
	re.nextHop.accessByC.removeElementAt(ndx2);

	if ((from.state == Node.ACTIVE) && 
	    (re.nextHop.state == Node.ACTIVE))
	{
	    sendConfirmFail(vspace, btneckMetric, firstEnd, secondEnd,
		replacingMetric, expInit, source, re.nextHop);
	}
	else
	{
	    printStatus("WARNING: invalid state(s) in processing ConfirmFail");
	    printStatus("Dropped the ConfirmFail message!");
	}
    }

//UPTOHERE

    protected void processDormantOk(StringTokenizer st, Node from)
    {
	// if (!isWaitForDormantAck) return;

	// if (!from.equals(dormantNode)) return;

	if (from.state != Node.PROGRESS)
	{
	    printStatus("ASSERT: received dormant-ok for invalid link state");
	    printStatus("link state is " + from.state);
	    return;
	}
	   
	from.state = Node.DORMANT;
	
	if (from.expInit == resolver.INRuid)
	{

	    if (mode==LINK_REPLACE_MODE)
	    {
	    			
	    printStatus("... the waited dormant-ok/fail message arrived!");

	    Node exist = neighbors.getNeighbor(expVSpace, expSourceNode.ia,
		expSourceNode.UDPport, expSourceNode.TCPport);
	    if (exist == null)	// Otherwise, establish a new link to expSource
	    {
		// send CONNECT-RQ
		printStatus("sending CONNECT-RQ to " + expSourceNode.toString());

		sendConnectRq(expVSpace, expSourceNode);
		return;		
	    }

	    printStatus("WARNING: received replace-ok to establish a link " +
		" that has been in state " + expSourceNode.state);

	    } 

	    else
	    {
	        printStatus("WARNING: unexpected dormant-ok message.");
	        printStatus("current mode is " + mode);
	    }
	}

	else
	{
	    // send REPLACE-OK message to expInitiator
	    RouteEntry re = routeTables.lookup(from.expVSpace, from.expInit);

	    // unlock the metrics of incoming and outgoing links
	    // from.rttLocked = false;
	    if (re == null)	
	    {
		printStatus("WARNING:  processDormantOk() finds expInit INRuid" +
		    " that my route table has no entry for!");
		printStatus("... the expVSpace INRuid is " + from.expVSpace);
		printStatus("... the expInit INRuid is " + from.expInit);
		printStatus("... the routing table is " + routeTables.toString());
		printStatus("Dropped the dormant-ok! *******");
		return;
	    }
	    //re.nextHop.rttLocked = false;

	    sendReplaceOk(from.expVSpace, from.rtt, from.getINRuid(), 
		resolver.INRuid, from.expMetric, from.expInit, 
		from.expSource, re.nextHop, true);	
	}

	//isWaitForDormantAck = false;
	//dormantNode = null;
    }

    protected void processDormantFail(StringTokenizer st, Node from)
    {
	// if (!isWaitForDormantAck) return;

	// if (!from.equals(dormantNode)) return;

	// make link back up again
	from.state = Node.ACTIVE;
	from.expVSpace = null;
	from.expInit = 0;
	from.expSource = 0;
	from.expMetric = 0;
	
	//from.rttLocked = false;

	// if I'm initiating the replacement experiment
	if (from.expInit == resolver.INRuid)
	{
	    if (mode==LINK_REPLACE_MODE)
	    {
		printStatus("... the waited dormant-ok/fail message arrived!");
		mode = PROBE_MODE;
		expMetric = expBtneckMetric = 0;
		expSource = expFirstEnd = expSecondEnd = 0;
		expSourceNode = null;
		expVSpace = null;
	    }
	    else
	    {
	        printStatus("WARNING: unexpected dormant-fail message!");
	        printStatus("current mode is " + mode);
	    }   
	}
	else
	{   
	    // send REPLACE-FAIL message to expInitiator
	    RouteEntry re = routeTables.lookup(from.expVSpace, from.expInit);

	    // unlock the metrics of incoming and outgoing links
	    //from.rttLocked = false;
	    if (re == null)	
	    {
		printStatus("WARNING:  processDormantFail() finds expInit INRuid" +
		    " that my route table has no entry for!");
		printStatus("Dropped the Replace-ok! *******");
		return;
	    }
	    //re.nextHop.rttLocked = false;

	    sendReplaceFail(from.expVSpace, from.rtt, from.getINRuid(), 
		resolver.INRuid, from.expMetric, from.expInit, 
		from.expSource, re.nextHop, false);	

	}

	//isWaitForDormantAck = false;
	//dormantNode = null;
    }



    protected void processReplaceOk(StringTokenizer st, Node from)
    {
	if (st.countTokens() < 7)
	{
	    printStatus("Too few parameters in Replace-ok message");
	    return;
	}

	// format is ReplaceOk(vspace, Cab:a,b, Cis:i,s) to next hop of b
	String vspace = st.nextToken();
	int btneckMetric = Integer.parseInt(st.nextToken());
	long firstEnd = Long.parseLong(st.nextToken());
	long secondEnd = Long.parseLong(st.nextToken());

	int replacingMetric = Integer.parseInt(st.nextToken());
	long expInit = Long.parseLong(st.nextToken());
	long source = Long.parseLong(st.nextToken());

	if (expInit == resolver.INRuid)	   // for me
	{
	    if ((mode==LINK_REPLACE_MODE) && isWaitForReplaceAck.value)
	    {
	    			
	    printStatus("... the waited replace-ok/fail message arrived!");

	    // verify message
	    if (expSource != source)
	    {
		printStatus("WARNING: unexpected source INRuid in the" +
		    " replace-ok message. Message is ignored!");
	    	return;
	    }
	    
	    isWaitForReplaceAck.value = false;
	    	    
	    synchronized (waitForReplaceAck) {
		waitForReplaceAck.notify();
	    }

	    // If new link to source has already been there,
	    // this means, secondEnd didn't get my REPLACE-DONE
	    // message before, and this replace-ok is a retransmission
	    // of it.. Hence send REPLACE-DONE right away.
	    Node exist = neighbors.getNeighbor(vspace, expSourceNode.ia,
		expSourceNode.UDPport, expSourceNode.TCPport);
	    if (exist == null)
	    {
		// Otherwise, establish a new link to expSource
		// send CONNECT-RQ
		printStatus("sending CONNECT-RQ to " + expSourceNode.toString());

		sendConnectRq(vspace, expSourceNode);
		return;		
	    }

	    printStatus("WARNING: received replace-ok to establish a link " +
		" that has been in state " + expSourceNode.state);
	    
	    } // if (isWaitForReplaceAck)

	    else
	    {
	        printStatus("WARNING: unexpected replace-ok message, perhaps ");
		printStatus(" due to re-transmission from secondEnd.");
	        printStatus("current mode is " + mode);

		// to avoid keep receiving this message, send replace-done		
	    }

	    // Forward the message
	    RouteEntry re = routeTables.lookup(vspace, secondEnd);
	
	    if (re == null)	
	    {
		printStatus("WARNING:  Replace-rq contains secondEnd INRuid" +
		    " that my route table has no entry for!");
		return;
	    }

	    sendReplaceDone(vspace, btneckMetric, firstEnd, secondEnd,
		replacingMetric, expInit, source, re.nextHop);
	    
	}
	else
	{
	    printStatus("... forward replace-ok message");
	    printStatus(expInit + " vs " + resolver.INRuid);
	    
	    // Forward the message
	    RouteEntry re = routeTables.lookup(vspace, expInit);

	    // unlock the metrics of incoming and outgoing links
	    //from.rttLocked = false;
	    if (re == null)	
	    {
		printStatus("WARNING:  Replace-ok contains expInit INRuid" +
		    " that my route table has no entry for!");
		printStatus("Dropped the Replace-ok! *******");
		return;
	    }
	    //re.nextHop.rttLocked = false;

	    sendReplaceOk(vspace, btneckMetric, firstEnd, secondEnd,
		replacingMetric, expInit, source, re.nextHop, false);
	}  
    }

    protected void processReplaceFail(StringTokenizer st, Node from)
    {
	if (st.countTokens() < 7)
	{
	    printStatus("Too few parameters in Replace-fail message");
	    return;
	}

	// format is ReplaceFail(vspace, Cab:a,b, Cis:i,s) to next hop of b
	String vspace = st.nextToken();
	int btneckMetric = Integer.parseInt(st.nextToken());
	long firstEnd = Long.parseLong(st.nextToken());
	long secondEnd = Long.parseLong(st.nextToken());

	int replacingMetric = Integer.parseInt(st.nextToken());
	long expInit = Long.parseLong(st.nextToken());
	long source = Long.parseLong(st.nextToken());

	if (expInit == resolver.INRuid)	   // for me
	{
	    if ((mode==LINK_REPLACE_MODE) && isWaitForReplaceAck.value)
	    {
	    			
	    printStatus("... the waited replace-ok/fail message arrived!");

	    isWaitForReplaceAck.value = false;
	    synchronized (waitForReplaceAck) {
		waitForReplaceAck.notify();
	    }

	    mode = PROBE_MODE;
	    expMetric = expBtneckMetric = 0;
	    expSource = expFirstEnd = expSecondEnd = 0;
	    expSourceNode = null;
	    expVSpace = null;

	    }	// if (isWaitForReplaceAck)
	    else
	    {
	        printStatus("WARNING: unexpected replace-fail message, perhaps");
		printStatus(" due to re-transmission from secondEnd.");
	        printStatus("current mode is " + mode);
	    }   

	}
	else
	{

	    // Forward the message
	    RouteEntry re = routeTables.lookup(vspace, expInit);
	
	    // unlock the metrics of incoming and outgoing links
	    //from.rttLocked = false;
	    if (re == null)	
	    {
		printStatus("WARNING:  Replace-fail contains expInit INRuid" +
		    " that my route table has no entry for!");
		printStatus("Dropped the Replace-ok! *******");
		return;
	    }
	    //re.nextHop.rttLocked = false;

	    sendReplaceFail(vspace, btneckMetric, firstEnd, secondEnd,
		replacingMetric, expInit, source, re.nextHop, false);
	}  
    }

    protected void processReplaceDone(StringTokenizer st, Node from)
    {
	if (st.countTokens() < 7)
	{
	    printStatus("Too few parameters in Replace-fail message");
	    return;
	}

	// format is ReplaceFail(vspace, Cab:a,b, Cis:i,s) to next hop of b
	String vspace = st.nextToken();
	int btneckMetric = Integer.parseInt(st.nextToken());
	long firstEnd = Long.parseLong(st.nextToken());
	long secondEnd = Long.parseLong(st.nextToken());

	int replacingMetric = Integer.parseInt(st.nextToken());
	long expInit = Long.parseLong(st.nextToken());
	long source = Long.parseLong(st.nextToken());

	if (secondEnd == resolver.INRuid)	   // for me
	{
	    if (isWaitForDone.value)
	    {
	    			
	    printStatus("... the waited replace-done message arrived!");

	    isWaitForDone.value = false;
	    synchronized (waitForDone) {
		waitForDone.notify();
	    }

	    InetAddress ia = extractAddrFromINRuid(firstEnd);
	    int UDPport = extractPortFromINRuid(firstEnd);

	    Node firstEndNode = neighbors.getNeighbor(vspace,
		ia, UDPport, 0);

	    if (firstEndNode == null)
	    {
		printStatus("Cannot disconnect link to secondEnd due to "+
		    "invalid secondEnd INRuid");
		return;
	    }
	    
	    // Disconnect link
	    sendDisconnect(vspace, firstEndNode);
	    
	    neighbors.removeNeighbor(vspace, ia, UDPport, 0);
	    printStatus("neighbor " + ia.toString() + ":" + UDPport + 
	    	" removed!");

	    Node exist = neighbors.getNeighbor(ia, UDPport, 0);
	    if (exist == null)
	    {   
	    	firstEndNode.state = Node.INACTIVE;
	    	if (firstEndNode.thread != null) 
	    	{
		    printStatus("Disconnecting the TCP connection to " +
		        firstEndNode.toString());
		     
		    firstEndNode.thread.isReceiving = false;
		    try {
		    firstEndNode.thread.socket.close();

		    // No need for below, TCPForwardThread will do it itself
		    /*printStatus("... socket was closed!");
		    firstEndNode.thread.socket = null;
		    firstEndNode.in.close();
		    printStatus("... in stream was closed!");
		    firstEndNode.out.close();
		    printStatus("... out stream was closed!");
		    firstEndNode.in = null;
		    firstEndNode.out = null;
		    firstEndNode.thread.interrupt();
		    printStatus("... TCPForwardThread was interrupted!");
		    */
		    }
		    catch (IOException e) {
		    	printStatus("Cannot disconnect the TCP connection!");
		    	e.printStackTrace();
		    }
		    
		}
	    }
	    
	    }	// if (isWaitForDone)
	    else
	    {
	        printStatus("WARNING: unexpected replace-done message");
	        printStatus("current mode is " + mode);
	    }   

	}
	else
	{
	    // Forward the message
	    RouteEntry re = routeTables.lookup(vspace, secondEnd);
	
	    if (re == null)	
	    {
		printStatus("WARNING:  Replace-done contains secondEnd INRuid" +
		    " that my route table has no entry for!");
		printStatus("Dropped the Replace-done! *******");
		return;
	    }
	    
	    sendReplaceDone(vspace, btneckMetric, firstEnd, secondEnd,
		replacingMetric, expInit, source, re.nextHop);
	}  
    }


    protected void processConnectRq(StringTokenizer st, Message msg)
    {
	printStatus("processing connect-rq ...");

	String name=null;
	InetAddress ia=null;
	String stria=null;
	String strport=null;
	int UDPport=0;
	int TCPport=0;
	
	if (st.countTokens()>=4) 
	{
	    name = st.nextToken();

	    try 
	    {
		stria = trim(st.nextToken());
		ia = InetAddress.getByName(stria);
		strport = trim(st.nextToken());
		UDPport = Integer.parseInt(strport);
		strport = trim(st.nextToken());
		TCPport = Integer.parseInt(strport);
	    } catch (UnknownHostException e) { 
		printStatus
		    ("Invalid INR address in produce_vspace request");
		return;
	    } catch (Exception e) { 
		printStatus
		    ("Malformed number in produce_vspace request");
		return;
	    }

	    System.out.println("connect-rq("+name+":"+stria+ 
		":"+UDPport+":"+TCPport+") from " + 
		msg.ia.toString() + ":" + msg.port);

	    Node n = new Node(ia, UDPport, TCPport, name,
			      DEFAULT_NEW_NODE_LIFE);

	    // if already exists, addResolver will return false
	    // normally should return false at this point
	    // below is just for double-checking
	    if (resolvers.addResolver(name, n))
	        printStatus("WARNING: at this point VSResolver should "+
	            "already have " + n.toString());
	            
	    n = resolvers.getResolver(ia, UDPport, TCPport);
	    n.UDPport = UDPport;
	    n.TCPport = TCPport;

	    n.state = Node.ACTIVE;
		
	    neighbors.addNeighbor(name, n);

	    if (n.in == null)
	        printStatus("streams are null, ok to establish TCP connection");
	    else
		printStatus("WARNING: streams are NOT null!");

	    // Creating and sending a CONNECT-ACK packet
	    // a string in the form of vspace:host:UDPport:TCPport
	    String message = "connect-ack\n" + 
		name + ":" + comm.localhost.getHostAddress() + ":" +
		comm.UDPport + ":" + comm.TCPport + "\n";
	
	    // create a namespecifier [control-msg=overlay]
	    //                        [vspace=therightvspace]
	    NameSpecifier thisoverlayNS = new NameSpecifier();
	    thisoverlayNS.addAVelement(overlayAVE);
	    thisoverlayNS.setVspace(name);

	    Packet packet = new Packet(sourceNS, thisoverlayNS, 
			2, Packet.toAny, false, zerobyte, 
			message.getBytes());

	    comm.sendMessage(new Message(packet, msg.ia, msg.port));
	}	
    }

    protected void processConnectAck(StringTokenizer st, Message msg)
    {
	printStatus("processing connect-ack ...");

	String vspace=null;
	InetAddress nia=null;
	String strnia=null;
	String strnport=null;
	int nUDPport=0;
	int nTCPport=0;
	
	if (!isWaitForConnectAck.value) {
	    printStatus("WARNING: unexpected connect-ack from " + 
		msg.ia.getHostAddress() + ":" + msg.port);
	    printStatus("... connect-ack is ignored");
	    return;
	}

	if (st.countTokens()<4) 
	{
	    printStatus("Too few parameters in CONNECT-ACK message");
	    return;
	}

	vspace = st.nextToken();
	try 
	{
		strnia = trim(st.nextToken());
		nia = InetAddress.getByName(strnia);
		strnport = trim(st.nextToken());
		nUDPport = Integer.parseInt(strnport);
		strnport = trim(st.nextToken());
		nTCPport = Integer.parseInt(strnport);
	} catch (UnknownHostException e) { 
		printStatus
		    ("Invalid INR address in produce_vspace request");
		return;
	} catch (Exception e) { 
		printStatus
		    ("Malformed number in produce_vspace request");
		return;
	}

	System.out.println("connect-ack("+vspace+":"+strnia+ 
		":"+nUDPport+":"+nTCPport+") from " + 
		msg.ia.toString() + ":" + msg.port);
	
	if ((!connectNode.ia.equals(nia)) || (connectNode.UDPport != nUDPport))
	{
	    printStatus("WARNING: connect-ack contains different addr/port " +
		nia.getHostAddress() + "/" + nUDPport);
	    printStatus("... connect-ack is ignored");
	    return;	 // invalid or duplicate due to retransmit
	}
	
	connectNode.TCPport = nTCPport;

	isWaitForConnectAck.value = false;
	synchronized (waitForConnectAck) {
	    waitForConnectAck.notifyAll();
	}

	printStatus("Establishing TCP connection to source.");
	
	// Establish the TCP connection
	// Note: set the connectNode.sendAllKnown to false still
	if (connectNode.in == null)  // !null because this INR belongs to
				// >1 vspace, prev.vspace has caused TCP conn.
	{
	    // double check
	    if (connectNode.state != Node.INACTIVE)
	        printStatus("WARNING: processConnectAck() is adding a new " +
	            "link that has been in state " + connectNode.state);

	    connectNode.state = Node.ACTIVE;
	    neighbors.addNeighbor(vspace, connectNode);
	    comm.connectTCP(connectNode);
	}
        else
            neighbors.addNeighbor(vspace, connectNode);

	if (expSecondEnd == resolver.INRuid)
	{
	    InetAddress ia = extractAddrFromINRuid(expFirstEnd);
	    int UDPport = extractPortFromINRuid(expFirstEnd);

	    Node firstEndNode = neighbors.getNeighbor(expVSpace,
		ia, UDPport, 0);

	    if (firstEndNode == null)
	    {
		printStatus("Cannot disconnect link to expFirstEnd due to "+
		    "invalid expFirstEnd INRuid");
		return;
	    }
	    
	    // Disconnect link
	    sendDisconnect(expVSpace, firstEndNode);
	    
	    neighbors.removeNeighbor(expVSpace, ia, UDPport, 0);
	    printStatus("neighbor " + ia.toString() + ":" + UDPport + 
	    	" removed!");

	    Node exist = neighbors.getNeighbor(ia, UDPport, 0);
	    if (exist == null)
	    {   
	    	firstEndNode.state = Node.INACTIVE;
	    	if (firstEndNode.thread != null) 
	    	{
		    printStatus("Disconnecting the TCP connection to " +
		        firstEndNode.toString());
		     
		    firstEndNode.thread.isReceiving = false;
		    try {
		    firstEndNode.thread.socket.close();

		    // No need for below, TCPForwardThread will do it itself
		    /*printStatus("... socket was closed!");
		    firstEndNode.thread.socket = null;
		    firstEndNode.in.close();
		    printStatus("... in stream was closed!");
		    firstEndNode.out.close();
		    printStatus("... out stream was closed!");
		    firstEndNode.in = null;
		    firstEndNode.out = null;
		    firstEndNode.thread.interrupt();
		    printStatus("... TCPForwardThread was interrupted!");
		    */
		    }
		    catch (IOException e) {
		    	printStatus("Cannot disconnect the TCP connection!");
		    	e.printStackTrace();
		    }
		    
		}
	    } // if (exist == null)

	}
	else
	{
	    // send REPLACE-DONE(Cab:a,b, Cis:i,s) to secondEnd
	    RouteEntry re = routeTables.lookup(expVSpace, expSecondEnd);
	
	    if (re == null)
	    {
		printStatus("WARNING: secondEnd INRuid is not in the" +
	        "routing table! Cancel sending replace-done to secondEnd.");
		return;
	    }

	    sendReplaceDone(expVSpace, expBtneckMetric, expFirstEnd,
		expSecondEnd, expMetric, resolver.INRuid, expSource, re.nextHop);
	}  // else

	connectNode = null;

	// back to PROBE_MODE   
	mode = PROBE_MODE;
	expMetric = expBtneckMetric = 0;
	expSource = expFirstEnd = expSecondEnd = 0;
	expSourceNode = null;
	expVSpace = null;
    }
    
    protected void processDisconnect(StringTokenizer st, Node from)
    {
	printStatus("processing disconnect ...");

	String name=null;
	InetAddress ia=null;
	String stria=null;
	String strport=null;
	int UDPport=0;
	int TCPport=0;
	
	if (st.countTokens()>=4) 
	{
	    name = st.nextToken();

	    try 
	    {
		stria = trim(st.nextToken());
		ia = InetAddress.getByName(stria);
		strport = trim(st.nextToken());
		UDPport = Integer.parseInt(strport);
		strport = trim(st.nextToken());
		TCPport = Integer.parseInt(strport);
	    } catch (UnknownHostException e) { 
		printStatus
		    ("Invalid INR address in produce_vspace request");
		return;
	    } catch (Exception e) { 
		printStatus
		    ("Malformed number in produce_vspace request");
		return;
	    }

	    System.out.println("disconnect("+name+":"+stria+ 
		":"+UDPport+":"+TCPport+") from " + 
		from.toString());

	    neighbors.removeNeighbor(name, ia, UDPport, TCPport);

    	}
    }
    
    protected void sendReplaceRq(String vspace, int btneckMetric, 
	long firstEnd, long secondEnd, int replacingMetric,
	long expInit, long source, Node to, boolean retransmit)
    {	
	// send ReplaceRq(Cab:a,b, Cis:i,s) to next hop of b
	String message = "replace-rq\n" +
	    vspace + "," +
	    btneckMetric + ":" + firstEnd + "," + secondEnd + "," +
	    replacingMetric + ":" + expInit + "," + source + "\n";

	printStatus("send (" + message +")");

	// create a namespecifier [control-msg=overlay]
	//                        [vspace=therightvspace]
	NameSpecifier thisoverlayNS = new NameSpecifier();
	thisoverlayNS.addAVelement(overlayAVE);
	thisoverlayNS.setVspace(vspace);

	Packet packet = new Packet(sourceNS, thisoverlayNS, 2, 
	    Packet.toAny, false, zerobyte, message.getBytes());

	if (!retransmit)
	    comm.sendByTCP(packet, to); // send without retransmission
	else
	{
	    RouteEntry re = routeTables.lookup(vspace, secondEnd);
	    long retransmitTimer;
	    if (re == null)
	    {
	    	printStatus("WARNING: secondEnd INRuid is not in the " +
	    	    "routing table! Use one-way time to secondEnd instead.");
		Node secondEndNode = resolvers.getResolver(vspace, 
		    extractAddrFromINRuid(secondEnd), 
		    extractPortFromINRuid(secondEnd), 0);
		if (secondEndNode == null)
		{
		    printStatus("WARNING: second INRuid is not in the" +
	    	    "VSResolver! Use DEFAULT_RETRANSMIT_TIMER instead.");		    retransmitTimer = DEFAULT_RETRANSMIT_TIMER;
		}
		else
		    retransmitTimer = RTT_MULTIPLIER*secondEndNode.rtt; 
					// rtt is NOT thru overlay
	    }
	    else
	        retransmitTimer = OWT_MULTIPLIER*re.metricRTT; // metricRTT is one-way
						  // thru overlay
	    isWaitForReplaceAck.value = true;

	    //*** create a new thread for below
 	    new Thread( new SendWithRetransmit(comm, packet, to, 
	        isWaitForReplaceAck, waitForReplaceAck, 
	        retransmitTimer, true) ). start();
    	}
    }


    protected void sendConfirmReplace(String vspace, int btneckMetric, 
	long firstEnd, long secondEnd, int replacingMetric,
	long expInit, long source, Node to, boolean retransmit)
    {	
	// send ReplaceRq(Cab:a,b, Cis:i,s) to next hop of b
	String message = "confirm-replace\n" +
	    vspace + "," +
	    btneckMetric + ":" + firstEnd + "," + secondEnd + "," +
	    replacingMetric + ":" + expInit + "," + source + "\n";

	printStatus("send (" + message +")");

	// create a namespecifier [control-msg=overlay]
	//                        [vspace=therightvspace]
	NameSpecifier thisoverlayNS = new NameSpecifier();
	thisoverlayNS.addAVelement(overlayAVE);
	thisoverlayNS.setVspace(vspace);

	Packet packet = new Packet(sourceNS, thisoverlayNS, 2, 
	    Packet.toAny, false, zerobyte, message.getBytes());

	if (!retransmit)
	    comm.sendByTCP(packet, to); // send without retransmission
	else
	{
	    RouteEntry re = routeTables.lookup(vspace, source);
	    long retransmitTimer;
	    if (re == null)
	    {
	    	printStatus("WARNING: source INRuid is not in the " +
	    	    "routing table! Use one-way time to source instead.");
		Node srcNode = resolvers.getResolver(vspace, 
		    extractAddrFromINRuid(source), 
		    extractPortFromINRuid(source), 0);
		if (srcNode == null)
		{
		    printStatus("WARNING: source INRuid is not in the" +
	    	    "VSResolver! Use DEFAULT_RETRANSMIT_TIMER instead.");		    retransmitTimer = DEFAULT_RETRANSMIT_TIMER;
		}
		else
		    retransmitTimer = RTT_MULTIPLIER*srcNode.rtt; 
					// rtt is NOT thru overlay
	    }
	    else
	        retransmitTimer = OWT_MULTIPLIER*re.metricRTT; // metricRTT is one-way
						  // thru overlay
	    isWaitForCRAck.value = true;

	    //*** create a new thread for below
 	    new Thread( new SendWithRetransmit(comm, packet, to, 
	        isWaitForCRAck, waitForCRAck, 
	        retransmitTimer, true) ). start();
    	}
    }

    protected void sendConfirmOk(String vspace, int btneckMetric, 
	long firstEnd, long secondEnd, int replacingMetric,
	long expInit, long source, Node to)
    {
	String message = "confirm-ok\n" +
	    vspace + "," +
	    btneckMetric + ":" + firstEnd + "," + secondEnd + "," +
	    replacingMetric + ":" + expInit + "," + source + "\n";

	printStatus("send (" + message +")");

	// create a namespecifier [control-msg=overlay]
	//                        [vspace=therightvspace]
	NameSpecifier thisoverlayNS = new NameSpecifier();
	thisoverlayNS.addAVelement(overlayAVE);
	thisoverlayNS.setVspace(vspace);
	
	Packet packet = new Packet(sourceNS, thisoverlayNS, 2, 
	    Packet.toAny, false, zerobyte, message.getBytes());

	comm.sendByTCP(packet, to); // send without retransmission
    }
    
    protected void sendConfirmFail(String vspace, int btneckMetric, 
	long firstEnd, long secondEnd, int replacingMetric, 
	long expInit, long source, Node to)
    {
	// fail the experiment	
	String message = "confirm-fail\n" +
	    vspace + "," +
	    btneckMetric + ":" + firstEnd + "," + secondEnd + "," +
	    replacingMetric + ":" + expInit + "," + source + "\n";

	printStatus("send (" + message +")");
	
	// create a namespecifier [control-msg=overlay]
	//                        [vspace=therightvspace]
	NameSpecifier thisoverlayNS = new NameSpecifier();
	thisoverlayNS.addAVelement(overlayAVE);
	thisoverlayNS.setVspace(vspace);
	
	Packet packet = new Packet(sourceNS, thisoverlayNS, 2, 
	    Packet.toAny, false, zerobyte, message.getBytes());

	comm.sendByTCP(packet, to); // send without retransmission
    }    


    protected void sendReplaceOk(String vspace, int btneckMetric, 
	long firstEnd, long secondEnd, int replacingMetric,
	long expInit, long source, Node to, boolean retransmit)
    {
	String message = "replace-ok\n" +
	    vspace + "," +
	    btneckMetric + ":" + firstEnd + "," + secondEnd + "," +
	    replacingMetric + ":" + expInit + "," + source + "\n";

	printStatus("send (" + message +")");

	// create a namespecifier [control-msg=overlay]
	//                        [vspace=therightvspace]
	NameSpecifier thisoverlayNS = new NameSpecifier();
	thisoverlayNS.addAVelement(overlayAVE);
	thisoverlayNS.setVspace(vspace);
	
	Packet packet = new Packet(sourceNS, thisoverlayNS, 2, 
	    Packet.toAny, false, zerobyte, message.getBytes());

	if (!retransmit)
	    comm.sendByTCP(packet, to); // send without retransmission
	else
	{
	    RouteEntry re = routeTables.lookup(vspace, expInit);
	    long retransmitTimer;
	    if (re == null)
	    {
	    	printStatus("WARNING: expInit INRuid is not in the" +
	    	    "routing table! Use one-way time to expInit instead.");
		Node expInitNode = resolvers.getResolver(vspace, 
		    extractAddrFromINRuid(expInit), 
		    extractPortFromINRuid(expInit), 0);
		if (expInitNode == null)
		{
		    printStatus("WARNING: expInit INRuid is not in the" +
	    	    "VSResolver! Use DEFAULT_RETRANSMIT_TIMER instead.");
		    retransmitTimer = DEFAULT_RETRANSMIT_TIMER;
		}
		else
		    retransmitTimer = RTT_MULTIPLIER*expInitNode.rtt; 
					// rtt is NOT thru overlay
	    }
	    else
		retransmitTimer = OWT_MULTIPLIER*re.metricRTT;	// metricRTT is one-way
							// thru overlay
	    isWaitForDone.value = true;

	    //*** create a new thread for below
	    new Thread( new SendWithRetransmit(comm, packet, to, 
		isWaitForDone, waitForDone, 
		retransmitTimer, true) ). start();
	} // else
    }

    protected void sendReplaceFail(String vspace, int btneckMetric, 
	long firstEnd, long secondEnd, int replacingMetric, 
	long expInit, long source, Node to, boolean retransmit)
    {
	// fail the experiment	
	String message = "replace-fail\n" +
	    vspace + "," +
	    btneckMetric + ":" + firstEnd + "," + secondEnd + "," +
	    replacingMetric + ":" + expInit + "," + source + "\n";

	printStatus("send (" + message +")");
	
	// create a namespecifier [control-msg=overlay]
	//                        [vspace=therightvspace]
	NameSpecifier thisoverlayNS = new NameSpecifier();
	thisoverlayNS.addAVelement(overlayAVE);
	thisoverlayNS.setVspace(vspace);
	
	Packet packet = new Packet(sourceNS, thisoverlayNS, 2, 
	    Packet.toAny, false, zerobyte, message.getBytes());

	if (!retransmit)	
	    comm.sendByTCP(packet, to); // send without retransmission
	else
	{

	    RouteEntry re = routeTables.lookup(vspace, expInit);
	    long retransmitTimer;
	    if (re == null)
	    {
	    	printStatus("WARNING: expInit INRuid is not in the" +
	    	    "routing table! Use one-way time to expInit instead.");
		Node expInitNode = resolvers.getResolver(vspace, 
		    extractAddrFromINRuid(expInit), 
		    extractPortFromINRuid(expInit), 0);
		if (expInitNode == null)
		{
		    printStatus("WARNING: expInit INRuid is not in the" +
	    	    "VSResolver! Use DEFAULT_RETRANSMIT_TIMER instead.");
		    retransmitTimer = DEFAULT_RETRANSMIT_TIMER;
		}
		else
		    retransmitTimer = RTT_MULTIPLIER*expInitNode.rtt; 
					// rtt is NOT thru overlay
	    }
	    else
		retransmitTimer = OWT_MULTIPLIER*re.metricRTT;	// metricRTT is one-way
							// thru overlay

	    isWaitForDone.value = true;

	    //*** create a new thread for below
	    new Thread( new SendWithRetransmit(comm, packet, to, 
		isWaitForDone, waitForDone, 
		retransmitTimer, true) ). start();
	}
    }    


    protected void sendReplaceDone(String vspace, int btneckMetric, 
	long firstEnd, long secondEnd, int replacingMetric,
	long expInit, long source, Node to)
    {
	String message = "replace-done\n" +
	    vspace + "," +
	    btneckMetric + ":" + firstEnd + "," + secondEnd + "," +
	    replacingMetric + ":" + expInit + "," + source + "\n";

	printStatus("send (" + message +")");

	// create a namespecifier [control-msg=overlay]
	//                        [vspace=therightvspace]
	NameSpecifier thisoverlayNS = new NameSpecifier();
	thisoverlayNS.addAVelement(overlayAVE);
	thisoverlayNS.setVspace(vspace);
	
	Packet packet = new Packet(sourceNS, overlayNS, 2, 
	    Packet.toAny, false, zerobyte, message.getBytes());

	//if (!retransmit)
	    comm.sendByTCP(packet, to); // send without retransmission
	/*else
	{
	    RouteEntry re = routeTables.lookup(vspace, secondEnd);
	    long retransmitTimer;
	    if (re == null)
	    {
	    	printStatus("WARNING: secondEnd INRuid is not in the" +
	    	    "routing table! Use one-way time to secondEnd instead.");
	    	    
		Node secondEndNode = resolvers.getResolver(vspace, 
		    extractAddrFromINRuid(secondEnd), 
		    extractPortFromINRuid(secondEnd), 0);
		if (secondEndNode == null)
		{
		    retransmitTimer = DEFAULT_RETRANSMIT_TIMER;
		    printStatus("WARNING: secondEnd INRuid is not in the" +
	    	    "VSResolver! Use DEFAULT_RETRANSMIT_TIMER instead.");
		}   
		else
		    retransmitTimer = RTT_MULTIPLIER*secondEndNode.rtt; 
					// rtt is NOT thru overlay
	    }
	    else
		retransmitTimer = OWT_MULTIPLIER*re.metricRTT;	// metricRTT is one-way
							// thru overlay
	    isWaitForDoneAck.value = true;

	    //*** create a new thread for below
	    new Thread( new SendWithRetransmit(comm, packet, to, 
		isWaitForDoneAck, waitForDoneAck, 
		retransmitTimer, true) ). start();
	} // else*/
    }


    protected void sendDormantRq(Node to, String vspace, 
    	int replacingMetric, long expInit, long source)
    {
	// send DORMANT-RQ(Cis, i, s) to other end of link
	// format is replacingMetric, exp-initiator, source
	// isWaitForDormantAck = true;
	// dormantNode = to;

	// mark the link as in progress first
	to.state = Node.PROGRESS;
	//to.rttLocked = true;
	
	to.expInit = expInit;
	to.expSource = source;
	to.expMetric = replacingMetric;
	to.expVSpace = vspace;
		
	String message = "dormant-rq\n" +
	    vspace + "," +
	    replacingMetric + "," + expInit + "," + source + "\n";

	printStatus("send (" + message +")");
	

	// create a namespecifier [control-msg=overlay]
	//                        [vspace=therightvspace]
	NameSpecifier thisoverlayNS = new NameSpecifier();
	thisoverlayNS.addAVelement(overlayAVE);
	thisoverlayNS.setVspace(vspace);

	Packet packet = new Packet(sourceNS, thisoverlayNS, 2, 
	    Packet.toAny, false, zerobyte, message.getBytes());

	comm.sendByTCP(packet, to);

    }


    protected void sendConnectRq(String vspace, Node to)
    {
	// a string in the form of vspace:host:UDPport:TCPport
	String message = "connect-rq\n" + 
	    vspace + ":" + comm.localhost.getHostAddress() + ":" +
	    comm.UDPport + ":" + comm.TCPport + "\n";
	
	// create a namespecifier [control-msg=overlay]
	//                        [vspace=therightvspace]
	NameSpecifier thisoverlayNS = new NameSpecifier();
	thisoverlayNS.addAVelement(overlayAVE);
	thisoverlayNS.setVspace(vspace);

	Packet neighborRqPacket = new Packet(sourceNS, thisoverlayNS, 2, 
	    Packet.toAny, false, zerobyte, message.getBytes());

	isWaitForConnectAck.value = true;
	connectNode = to;

	int retransmitTimer = (RTT_MULTIPLIER * to.rtt)/2;
	
	//*** create a new thread for below
	new Thread( new SendWithRetransmit(comm, neighborRqPacket,
	    to, isWaitForConnectAck, waitForConnectAck, 
	    retransmitTimer, false) ). start();
		
    }

    protected void sendDisconnect(String vspace, Node to)
    {
	// a string in the form of vspace:host:UDPport:TCPport
	String message = "disconnect\n" + 
	    vspace + ":" + comm.localhost.getHostAddress() + ":" +
	    comm.UDPport + ":" + comm.TCPport + "\n";

	// create a namespecifier [control-msg=overlay]
	//                        [vspace=therightvspace]
	NameSpecifier thisoverlayNS = new NameSpecifier();
	thisoverlayNS.addAVelement(overlayAVE);
	thisoverlayNS.setVspace(vspace);

	Packet packet = new Packet(sourceNS, thisoverlayNS, 2, 
	    Packet.toAny, false, zerobyte, message.getBytes());

	comm.sendByTCP(packet, to);	// without retransmission		
    }

    protected void quickLinkReplace(String vspace, long source, Node srcNode, 
	int btneckMetric, long firstEnd, long secondEnd, Node from)
    {
	expVSpace = vspace;
	expSource = source;
    	expSourceNode = srcNode;
    	expBtneckMetric = btneckMetric;
	expMetric = srcNode.rtt;
	expFirstEnd = firstEnd;
	expSecondEnd = secondEnd;

	if ((from.state == Node.PROGRESS) || 
		(from.state == Node.DORMANT) )	// someone
			// already tries to disconnect this link
	    return;

	// lock metric of link toward the source
	//from.rttLocked = true;

	sendDormantRq(from, vspace, expMetric, resolver.INRuid, source);
    }


    protected void linkReplace(String vspace, long source, Node srcNode,
	int btneckMetric, long firstEnd, long secondEnd, Node from)
    {
	expVSpace = vspace;
	expSource = source;
    	expSourceNode = srcNode;
    	expBtneckMetric = btneckMetric;
	expMetric = srcNode.rtt;
	expFirstEnd = firstEnd;
	expSecondEnd = secondEnd;

	// lock metric of link toward the source
/*	from.rttLocked = true;
*/

	// add to list of exp accessing link
	from.accessByI.addElement(new Long(resolver.INRuid));
	from.accessByS.addElement(new Long(source));
	from.accessByC.addElement(new Integer(expMetric));
	
	sendReplaceRq(vspace, btneckMetric, firstEnd, secondEnd, 
	    expMetric, resolver.INRuid, source, from, true);
    }


    protected void sendProbe(String vspace, long source, int metric,
	long firstEnd, long secondEnd, Node except)
    {
	if (vspace == null) {
	    printStatus("ASSERT: in sendRouteUpdate(): no vspace specified");
	    return; 
	}

	// Send a PROBE message to all neighbors
	// a string in the form of vspace:source, bottleneck_metric, 
	//   first end point of bottleneck, second end point
	String message = "probe\n" + 
	    vspace + ":" + source + "," + metric + "," +
	    firstEnd + "," + secondEnd + "\n";
	
	// create a namespecifier [control-msg=overlay]
	//                        [vspace=therightvspace]
	NameSpecifier thisoverlayNS = new NameSpecifier();
	thisoverlayNS.addAVelement(overlayAVE);
	thisoverlayNS.setVspace(vspace);
	
	Packet probePacket = new Packet(sourceNS, thisoverlayNS, 2, 
	    Packet.toAny, false, zerobyte, message.getBytes());

	Enumeration e = neighbors.getNeighbors(vspace).getNodes();
	
	while (e.hasMoreElements()) 
	{
	    Node n = (Node)(e.nextElement());
	    if (! n.equals(except)) // usually except == re.nextHop
				    // except when there is a loop in topology
		comm.sendByTCP(probePacket, n);
	}

    }

    void removeNeighbor(Node node)
    {
 	for (Enumeration e=node.vspaces.elements(); e.hasMoreElements(); )
	{
	    String vspace = (String)(e.nextElement());
	    heal(vspace, node);
	}
	neighbors.totallyRemoveNeighbor(node);
    	resolvers.totallyRemoveNeighbor(node);
    }

    void heal(String vspace, Node node)
    {
	printStatus("...Quick-healing vspace " + vspace);

    	Vector twoHopNodes = new Vector();

 	for (Enumeration e=routeTables.getRouteEntries(vspace); e.hasMoreElements(); )
	{
	    RouteEntry re = (RouteEntry)(e.nextElement());
	    if ((re.nextHop != null) && (re.nextHop.equals(node)))
	    {
		if (re.metricHop == 2)
		{
	    	    twoHopNodes.addElement(new Long(re.INRuid));
		    printStatus("...found two-hop node " + re.INRuid);
		}
		routeTables.remove(vspace, re.INRuid);
		printStatus("...removing RouteEntry for INR " + re.INRuid);

		re.nextHop = null;
		re.metricHop = 0;
		re.metricRTT = 0;
	    }	
	}
	
	if (twoHopNodes.size() > 0)
	{
	    // Heal the spanning tree using Quick-Healing
	    // select an INR with INRuid < my INRuid
	    long peerINRuid = 0;
	    for (Enumeration enum=twoHopNodes.elements(); enum.hasMoreElements(); )
	    {
	    	long cur = ((Long)enum.nextElement()).longValue();
	    	if ((cur > peerINRuid) && (cur < resolver.INRuid))
	    	{
	    	    peerINRuid = cur;
	    	}
	    }
    
	    if  (peerINRuid != 0)
	    {
		printStatus("QuickHealing picks INRuid to peer: " + peerINRuid);

		// Send handshake to peerINRuid
		
		InetAddress ia = extractAddrFromINRuid(peerINRuid);
		int UDPport = extractPortFromINRuid(peerINRuid);
		Node peerNode = resolvers.getResolver(vspace, ia, UDPport, 0);

		printStatus("sending NEIGHBOR-RQ to " + peerNode.toString());

		// Send NEIGHBOR-RQ to peerNode
		// a string in the form of vspace:host:UDPport:TCPport
		String message = "neighbor-rq\n" + 
		    vspace + ":" + comm.localhost.getHostAddress() + ":" +
		    comm.UDPport + ":" + comm.TCPport + "\n";
	
		// create a namespecifier [control-msg=overlay][vspace=therightvspace]
	NameSpecifier thisoverlayNS = new NameSpecifier();
	thisoverlayNS.addAVelement(overlayAVE);
	thisoverlayNS.setVspace(vspace);

	Packet neighborRqPacket = new Packet(sourceNS, thisoverlayNS, 2, 
	    Packet.toAny, false, zerobyte, message.getBytes());

	isWaitForNeighborAck.value = true;
	neighborNode = peerNode;

	int retransmitTimer = (RTT_MULTIPLIER * peerNode.rtt)/2;
	
	int numtries=0;

	//*** create a new thread for below
	new Thread( new SendWithRetransmit(comm, neighborRqPacket,
	    peerNode, isWaitForNeighborAck, waitForNeighborAck, 
	    retransmitTimer, false) ). start();

		// Establish TCP connection to it -> performed by processConnectAck()
	    }
	}

	// do incremental healing
    }    	
    
    void startRelaxationProbe()
    {
	printStatus("Starting periodic relaxation probes...");
	sendingProbe = true;
	periodicThread.start();    	
    }

    void stopRelaxationProbe()
    {
	printStatus("Stopping periodic relaxation probes...");
	sendingProbe = false;
    }
    
    public void run()
    {
	mode = PROBE_MODE;

	while (sendingProbe)
	{
	    for (Enumeration e = neighbors.getVspaces(); e.hasMoreElements();)
	    {
		String vspace = (String)e.nextElement();
		sendProbe(vspace, resolver.INRuid, 0, 0l, 0l, null);
	    }

	    try {Thread.sleep(PROBE_INTERVAL);}
	    catch (InterruptedException e) {e.printStackTrace();}	    
	}
    }

    /** 
     * Performs actions necessary to add vspace for this module.
     * Specifically, it registers handlers in the vspace's nametree
     * for its functionality.
     */
    protected void produceVspace(String vspace)
    {
	NameRecord nr; 

	overlayNS.setVspace(vspace);
	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTrees.addNameRecord(overlayNS, vspace, nr);
    }


    private String trim(String s) 
    {
	int l = s.length();
	
	for (int i = l-1; i > 0; i--) {
	    if (s.charAt(i) > ' '){
		return s.substring(0,i+1);	
	    }
	    
	}
	return s;
    }


    protected InetAddress extractAddrFromINRuid(long INRuid)
    {
	int intia = (int)((INRuid >> 16) & 0xFFFFFFFFl);
	byte[] addr = new byte[4];
	Conversion.insertInt(addr, 0, intia);
	InetAddress ia = null;

	// convert addr to InetAddress type		
	try{
	    String stria = Integer.toString(addr[0] & 0xFF) +
	      "." + Integer.toString(addr[1] & 0xFF) +
	      "." + Integer.toString(addr[2] & 0xFF) +
	      "." + Integer.toString(addr[3] & 0xFF);

	    ia = InetAddress.getByName(stria);
	} catch (UnknownHostException e) {
	    e.printStackTrace();
	}

	return ia;
    }
	
    protected int extractPortFromINRuid(long INRuid)
    {
	return (int)(INRuid & 0xFFFFl);
    }

    public void printStatus(String s)
    {
	String text = /*new Date(System.currentTimeMillis()).toString() + */"OM: "+s;
	System.out.println(text);
	if (isLog) pw.println(text);
    }
}






