package ins.inr;

import ins.namespace.NameSpecifier;

/**
 * Represents an INS packet.
 * Wire representation:
 *     BYTE MEANING
 *     0    Version
 *     1    Hop limit (TTL)
 *     2    Pointer to the start of the source name-specifier
 *     3    ""
 *     4    Pointer to the start of the destination name-specifier
 *     5    ""
 *     6    Pointer to the start of the option field
 *     7    ""
 *     8    Pointer to the start of the data
 *     9    ""
 *    10    Bit field:  7     6     5     4     3     2     1     0
 *                      Any   App     -     -     -     -     -     -
 *    11    Reserved for future fields   
 *    12    Source name-specifier
 *   ...
 *   [i]    Destination name-specifier
 *   ...
 *   [j]    Option, sequence of [kind][length][option data]
 *   ...
 *   [k]    Data
 *   ...
 */
public class Packet
{
    // VARIABLES
    final static int MAX_PACKET_SIZE = 65535-40;	
		// UDP and IP headers are at least 40 bytes;

    // Constants for "all" variable
    public final static boolean toAll = true;
    public final static boolean toAny = false;
    public final static byte AppUID_kind = 1;

    final static int DEFAULT_TTL = 64;
    final static boolean DEFAULT_DELIVERY = toAny;
    final static boolean DEFAULT_APP = true;
    final static byte DEFAULT_VERSION = 1; 


    public byte version;	// the version of the Packet format
    public int TTL;           	// the Time-to-live for the packet (hops)
    public NameSpecifier sNS;  	// the source name-specifier
    public NameSpecifier dNS;  	// the destination name-specifier
    public byte[] option;	// the option field
    public byte[] data;        	// the packet data
    public boolean all;        	// true if packet should go to ALL recipients
                               	// false if packet should to go ANY recipient
    public boolean fromApp;    	// packet is sent out by an app or INR


    // CONSTRUCTORS

    /**
     * Creates a Packet (with DEFAULT_VERSION)
     * @param sNS the source NameSpecifier of the Packet
     * @param dNS the destination NameSpecifier of the Packet
     * @param TTL the time-to-live of the Packet (number of hops)
     * @param all true if the packet should go to ALL recipients
     * @param fromApp true if the packet is 
     * @param option the option field of the Packet
     * @param data the data of the Packet
     */
    public Packet(NameSpecifier sNS, NameSpecifier dNS, int TTL,
		  boolean all, boolean fromApp, byte[] option, byte[] data)
    {
	this.sNS = sNS;
	this.dNS = dNS;
	this.TTL = TTL;
	this.all = all;
	this.fromApp = fromApp;
	this.option = option;
	this.data = data;
	version = DEFAULT_VERSION;
    }


    /**
     * Creates a Packet
     * @param sNS the source NameSpecifier of the Packet
     * @param dNS the destination NameSpecifier of the Packet
     * @param TTL the time-to-live of the Packet (number of hops)
     * @param all true if the packet should go to ALL recipients
     * @param fromApp true if the packet is 
     * @param option the option field of the Packet
     * @param data the data of the Packet
     * @param version the version field of the Packet
     */
    public Packet(NameSpecifier sNS, NameSpecifier dNS, int TTL,
	boolean all, boolean fromApp, byte[] option, byte[] data, 
	byte version)
    {
	this.sNS = sNS;
	this.dNS = dNS;
	this.TTL = TTL;
	this.all = all;
	this.fromApp = fromApp;
	this.option = option;
	this.data = data;
	this.version = version;
    }
    
    /** 
     * Packet copy-constructor 
     */
    public Packet(Packet p)
    {
	this.sNS = new NameSpecifier(p.sNS);
	this.dNS = new NameSpecifier(p.dNS);
	this.TTL = p.TTL;
	this.all = p.all;
	this.fromApp = p.fromApp;
	this.option = new byte[p.option.length];
	System.arraycopy(p.option, 0, this.option, 0, p.option.length);
	this.data = new byte[p.data.length];
	System.arraycopy(p.data, 0, this.data, 0, p.data.length);
	this.version = p.version;
    }


    /**
     * Creates a Packet with TTL 64, fromApp flag set, option 
     * containing uid and version 1
     * @param sNS the source NameSpecifier of the Packet
     * @param dNS the destination NameSpecifier of the Packet
     * @param uid the UID of the application associated with sNS
     * @param data the data of the Packet
     */
    public Packet(NameSpecifier sNS, NameSpecifier dNS, long uid,
		  boolean all, byte[] data)
    {
	this.sNS = sNS;
	this.dNS = dNS;
	this.TTL = DEFAULT_TTL;
	this.all = all;
	this.fromApp = DEFAULT_APP;

	option = new byte[10];
	option[0] = AppUID_kind;
	option[1] = 10;
	Conversion.insertLong(option, 2, uid);

	this.data = data;
	this.version = DEFAULT_VERSION;
    }

    /**
     * Creates a Packet with TTL 64, fromApp flag set, zero option 
     * and version 1
     * @param sNS the source NameSpecifier of the Packet
     * @param dNS the destination NameSpecifier of the Packet
     * @param uid the UID of the application associated with sNS
     * @param data the data of the Packet
     */
    public Packet(NameSpecifier sNS, NameSpecifier dNS,
		  boolean all, byte[] data)
    {
	this.sNS = sNS;
	this.dNS = dNS;
	this.TTL = DEFAULT_TTL;
	this.all = all;
	this.fromApp = DEFAULT_APP;
	this.option = new byte[0];
	this.data = data;
	this.version = DEFAULT_VERSION;
    }

    /**
     * Creates a Packet with TTL 64, intentional anycast delivery,
     *  fromApp flag set, option containing AppUID and version 1
     * @param sNS the source NameSpecifier of the Packet
     * @param dNS the destination NameSpecifier of the Packet
     * @param uid the UID of the application associated with sNS
     * @param data the data of the Packet
     */
    public Packet(NameSpecifier sNS, NameSpecifier dNS, long uid,
	byte[] data)
    {
	this.sNS = sNS;
	this.dNS = dNS;
	this.TTL = DEFAULT_TTL;
	this.all = DEFAULT_DELIVERY;
	this.fromApp = DEFAULT_APP;

	option = new byte[10];
	option[0] = AppUID_kind;
	option[1] = 10;
	Conversion.insertLong(option, 2, uid);

	this.data = data;
	this.version = DEFAULT_VERSION;
    }

    /**
     * Creates a Packet with TTL 64, intentional anycast delivery,
     *  fromApp flag set, zero option and version 1
     * @param sNS the source NameSpecifier of the Packet
     * @param dNS the destination NameSpecifier of the Packet
     * @param uid the UID of the application associated with sNS
     * @param data the data of the Packet
     */
    public Packet(NameSpecifier sNS, NameSpecifier dNS,
	byte[] data)
    {
	this.sNS = sNS;
	this.dNS = dNS;
	this.TTL = DEFAULT_TTL;
	this.all = DEFAULT_DELIVERY;
	this.fromApp = DEFAULT_APP;
	this.option = new byte[0];
	this.data = data;
	this.version = DEFAULT_VERSION;
	//hardenbuf = null;
    }

    /**
     * Creates a Packet, allocating new memory and copying the bytes
     * @param bytes the wire representation of the Packet
     * @param size is the length of the array that is good
     * @exception Exception The wire representation was invalid.
     */
    public Packet(byte[] bytes, int length) throws Exception
    {
	if ((length > bytes.length) || (length < 20))
	    throw new Exception("Invalid wire representation size");

	version = bytes[0];
	TTL = (int)bytes[1];

	int pt1 = Conversion.extract16toIntLow16(bytes, 2);
	int pt2 = Conversion.extract16toIntLow16(bytes, 4);
	int pt3 = Conversion.extract16toIntLow16(bytes, 6);
	int pt4 = Conversion.extract16toIntLow16(bytes, 8);

	all = Conversion.getBooleanFromByte(bytes[10], 7);
	fromApp = Conversion.getBooleanFromByte(bytes[10], 6);

	// check if pointers are valid first
	if (pt1<=pt2 && pt2<=pt3 && pt3<=pt4 && pt4<=length) {
	    sNS = new NameSpecifier(new String(bytes, pt1, pt2-pt1));
	    dNS = new NameSpecifier(new String(bytes, pt2, pt3-pt2));
	    option = new byte[pt4-pt3];
	    data = new byte[length-pt4];

	    // Note: if 0 byte then pt3=pt4 and pt4==length
	    System.arraycopy(bytes, pt3, option, 0, option.length);
	    System.arraycopy(bytes, pt4, data, 0, data.length);

	} else {
	    throw new Exception("Invalid wire representation");
	}
    }

    /**
     * Creates a Packet, allocating new memory and copying the bytes
     * @param bytes the wire representation of the Packet
     * @exception Exception The wire representation was invalid.
     */
    public Packet(byte[] bytes) throws Exception
    {
	int length = bytes.length;

	if (length < 20)
	    throw new Exception("Invalid wire representation size");

	version = bytes[0];
	TTL = (int)bytes[1];

	int pt1 = Conversion.extract16toIntLow16(bytes, 2);
	int pt2 = Conversion.extract16toIntLow16(bytes, 4);
	int pt3 = Conversion.extract16toIntLow16(bytes, 6);
	int pt4 = Conversion.extract16toIntLow16(bytes, 8);

	all = Conversion.getBooleanFromByte(bytes[10], 7);
	fromApp = Conversion.getBooleanFromByte(bytes[10], 6);

	// check if pointers are valid first
	if (pt1<=pt2 && pt2<=pt3 && pt3<=pt4 && pt4<=length) {
	    sNS = new NameSpecifier(new String(bytes, pt1, pt2-pt1));
	    dNS = new NameSpecifier(new String(bytes, pt2, pt3-pt2));
	    option = new byte[pt4-pt3];
	    data = new byte[length-pt4];

	    // Note: if 0 byte then pt3=pt4 and pt4==length

	    System.arraycopy(bytes, pt3, option, 0, option.length);
	    System.arraycopy(bytes, pt4, data, 0, data.length);

	} else {
	    throw new Exception("Invalid wire representation");
	}
    }

    /**
     * Decrements the TTL of the Packet
     * @return the decremented value of TTL
     */
    int decrementTTL()
    {
	TTL--;
	return TTL;
    }

    /**
     * Make this packet immutable from any changes that occur to
     * the sNS, dNS, data, etc. pointers.
     * (i.e. if the buffer or dNS were mutated after the packet was
     *  inserted in the buffer but before it was transmitted, this
     *  would not affect the delivery, unless the hardenbuf were 
     *  changed or the duplicated dNS were changed).
     * - This may be appropriate to execute automatically from
     *   the RateLimitedBuffer.
     */
/*    public void harden()
    {
	if (hardenbuf == null) {
	    hardenbuf = this.toBytes();
	    this.sNS = new NameSpecifier(this.sNS);
	    this.dNS = new NameSpecifier(this.dNS);	    
	}
	return;
    }

    public int hardenedlength() 
    {
	if (hardenbuf !=null) return hardenbuf.length;
	else return (data.length + sNS.toString().length()
		     +dNS.toString().length()+8);
    }
*/

    // METHODS


    /**
     * Returns the Packet represented as an array of bytes.
     */
    public byte[] toBytes()	
    {
	//if (hardenbuf != null) return hardenbuf;

	byte[] sNSBytes = sNS.toString().getBytes();
	byte[] dNSBytes = dNS.toString().getBytes();
	byte[] bytes = new byte[20 + 
			       sNSBytes.length + 
			       dNSBytes.length + 
			       option.length +
			       data.length];
	int pt1 = 20; // pointer to sNS
	int pt2 = pt1 + sNSBytes.length; // pointer to dNS
	int pt3 = pt2 + dNSBytes.length; // pointer to data
	int pt4 = pt3 + option.length;

	bytes[0] = version;
	bytes[1] = (byte)(TTL & 0xFF);
	Conversion.insertIntLow16(bytes, 2, pt1); // bytes[2], bytes[3]
	Conversion.insertIntLow16(bytes, 4, pt2); // bytes[4], bytes[5]
	Conversion.insertIntLow16(bytes, 6, pt3); // bytes[6], bytes[7]
	Conversion.insertIntLow16(bytes, 8, pt4); // bytes[8], bytes[9]

	bytes[10] = Conversion.setByteFromBoolean((byte)0, 7, all);
	bytes[10] = Conversion.setByteFromBoolean(bytes[10], 6, fromApp);

	// bytes[12] ...
	System.arraycopy(sNSBytes, 0, bytes, pt1, sNSBytes.length);
	System.arraycopy(dNSBytes, 0, bytes, pt2, dNSBytes.length);
	System.arraycopy(option, 0, bytes, pt3, option.length);
	System.arraycopy(data, 0, bytes, pt4, data.length);

	// CacheInfo not used

	return bytes;
    }


    /**
     * Returns the Packet represented as an array of bytes minus options.
     */
    public byte[] toMACBytes()	
    {
	//if (hardenbuf != null) return hardenbuf;

	byte[] sNSBytes = sNS.toString().getBytes();
	byte[] dNSBytes = dNS.toString().getBytes();
	byte[] bytes = new byte[20 + 
			       sNSBytes.length + 
			       dNSBytes.length + 
			       data.length];
	int pt1 = 20; // pointer to sNS
	int pt2 = pt1 + sNSBytes.length; // pointer to dNS
	int pt3 = pt2 + dNSBytes.length; // pointer to data
//  	int pt4 = pt3 + option.length;   // don't include MAC in check
	int pt4 = pt3;

	bytes[0] = version;

//  	bytes[1] = (byte)(TTL & 0xFF);
	bytes[1] = (byte)(0xFF);

	Conversion.insertIntLow16(bytes, 2, pt1); // bytes[2], bytes[3]
	Conversion.insertIntLow16(bytes, 4, pt2); // bytes[4], bytes[5]
	Conversion.insertIntLow16(bytes, 6, pt3); // bytes[6], bytes[7]
	Conversion.insertIntLow16(bytes, 8, pt4); // bytes[8], bytes[9]

//  	bytes[10] = Conversion.setByteFromBoolean((byte)0, 7, all);
//  	bytes[10] = Conversion.setByteFromBoolean(bytes[10], 6, fromApp);
	bytes[10] = (byte)(0xFF);


	// bytes[12] ...
	System.arraycopy(sNSBytes, 0, bytes, pt1, sNSBytes.length);
	System.arraycopy(dNSBytes, 0, bytes, pt2, dNSBytes.length);
//  	System.arraycopy(option, 0, bytes, pt3, option.length);
	System.arraycopy(data, 0, bytes, pt4, data.length);

	// CacheInfo not used

	return bytes;
    }



    /** 
     * Gives the offset of the data into the packet. 
     * (so that it may be potentially mutated)
     */
    public static int getDataOffset(byte [] data)
    {
	return Conversion.extract16toIntLow16(data, 8);
    }

    // keep commented -- packet should use Object.equals, which
    // only does a == address comparison
    /*    public boolean equals(Object anObject) 
    {
	Packet cmp = (Packet)anObject;

	if ( (sNS.toString().equals(cmp.sNS.toString())) &&
	     (dNS.toString().equals(cmp.dNS.toString())) &&
	     (data.equals(cmp.data)) )
	    return true;
	else
	    return false;
    }
    */

    public long getAppUID()
    {
	if ((option.length == 10) && (option[0]==AppUID_kind))
	{
	    long uid = Conversion.extract64toLong(option, 2);
	    return uid;
	}
	else
	    return 0;
    }

    public String toString()
    {
	StringBuffer str = new StringBuffer();

	str.append("\nTTL:    \t" + TTL);
	if (all) {
	    str.append("\nTo:         \tALL");
	} else {
	    str.append("\nTo:         \tANY");
	}

	if (fromApp) {
	    str.append("\nFrom App:      \tYes");
	} else {
	    str.append("\nFrom App:      \tNo");
	}

	str.append("\nSource:        \t");
	str.append(sNS.toString());
	str.append("\nDestination:\t");
	str.append(dNS.toString());

	str.append("\nOption Length:     \t");
	str.append(option.length);
	str.append(" bytes");

	if ((option.length == 10) && (option[0]==AppUID_kind))
	{
	    long uid = Conversion.extract64toLong(option, 2);
	    str.append("\n\t AppUID is " + uid);
	}

	str.append("\nData Length:     \t");
	str.append(data.length);
	str.append(" bytes");

	return str.toString();
    }


    public Object clone() {
	return new Packet(this);
    }


    
}


