package ins.inr;

import org.acplt.oncrpc.*;
import org.acplt.oncrpc.server.*;
import ins.inr.interResolverSvc.*;
import java.net.InetAddress;
import java.io.*;
import ins.inr.*;

/**
 * RPCCommunicator.java
 *
 * Inter-resolver communication with RPC.
 * Small messages are sent via RPC over UDP.
 * Large messages are sent via RPC over TCP.
 * <br> 
 * All resolver modules may queue-up messages for 
 * transfer by invoking sendMessage()
 * <br>
 * Incoming calls are dispatched with to the Forwarder
 * as for ordinary UDP datagrams.
 * 
 * Created: Mon Feb 25 <br>
 * Modified: $Id: RPCCommunicator.java,v 1.5 2002/03/21 00:01:55 mbalazin Exp $
 * @author Magdalena Balazinska
 *
 */
public class RPCCommunicator implements OncRpcDispatchable {

    protected Forwarder forwarder;
    protected TwineLogger log;
    protected OncRpcUdpServerTransport udpTrans;    
    protected OncRpcTcpServerTransport tcpTrans;    
    protected InetAddress localhost;
    protected int rpcPort;
    protected SimpleThreadPool pool;
    protected static int NB_THREADS = 6;
    protected static int MAX_RPC_PACKET = 14000;
 
    //-----------------------------------------------------
    /**
     * Starts an RPC server listening on the given
     * port.
     */
    public RPCCommunicator(int port) {
	
	log = new TwineLogger("RPCCOmmunicator","RPCCommunicator.log");
	rpcPort = port;

	try {

	    localhost=InetAddress.getLocalHost();
	    // For UDP calls
	    udpTrans = new OncRpcUdpServerTransport(this, port,
	    				    interresolver_prot.INTERRESOLVER_PROGRAM,
	    				    interresolver_prot.INTERRESOLVER_VERSION,
	    				    Packet.MAX_PACKET_SIZE); 
	    udpTrans.register();
	    udpTrans.listen();

	    // For TCP connections
	    tcpTrans = new OncRpcTcpServerTransport(this, port,
						    interresolver_prot.INTERRESOLVER_PROGRAM,
						    interresolver_prot.INTERRESOLVER_VERSION,
						    Packet.MAX_PACKET_SIZE); 
	    tcpTrans.register();
	    tcpTrans.listen();
	    log.printStatus("Server started...",TwineLogger.IMPORTANT_MSG);
	} catch (Exception e) {
	    log.printStatus("Starting rpc server failed",TwineLogger.KEY_MSG);
	}
    }

    /**
     * De-registers listeners on garbage collection
     */
    public void finalize() {

	try {
	    tcpTrans.unregister();
	    tcpTrans.close();
	    
	    udpTrans.unregister();
	    udpTrans.close();

	    log.printStatus("Server shut down.",TwineLogger.IMPORTANT_MSG);
	} catch ( Exception e) {
	    log.printStatus("Failed while un-registering",TwineLogger.IMPORTANT_MSG);
	}

    }

    //-----------------------------------------------------
    /** 
     * Uses parameter Resolver to get a reference
     * to the "forwarder" that dispatches incoming
     * messages based on their meaning.
     */
    public void init(Resolver r) {
	forwarder = r.forwarder;
	pool = new SimpleThreadPool(log,NB_THREADS);
    }


    //-----------------------------------------------------
    /**
     * Dispatches incoming RPC calls
     * <ul>
     * <li> Extracts argument (Packet) 
     * <li> Return "void" to client
     * <li> Transfer control to the "forwarder" for appropriate handling
     * </ul>
     */
    public void dispatchOncRpcCall(OncRpcCallInformation call,
				   int program, int version, int procedure)
	throws OncRpcException, IOException {
		
	log.printStatus("Incomming call for program "
			+ Integer.toHexString(program)
			+ "; version " + version
			+ "; procedure " + Integer.toHexString(procedure),
			TwineLogger.IMPORTANT_MSG);

	switch ( procedure ) {
	    
	case 0:
	case 1:
	    call.retrieveCall(XdrVoid.XDR_VOID);
	    call.reply(XdrVoid.XDR_VOID);
	    break;
	case 2:

	    try {
		// Get the packet transmitted
		interresolver_packet packet = new interresolver_packet();
		call.retrieveCall(packet);
	    
		// Call into the resolver
		Packet p = new Packet(packet.value, packet.length);
		log.printStatus("Receive message from host: " + packet.hostaddr + ":" + packet.port,
				TwineLogger.IMPORTANT_MSG);

		InetAddress addr = InetAddress.getByName(packet.hostaddr);
		int port = packet.port;
		Message msg = new Message(p,addr,port);
		call.reply(XdrVoid.XDR_VOID);
		forwarder.forward(msg);
	    } catch ( Exception e) {
		log.printStatus("Fatal error receiving RPC",TwineLogger.KEY_MSG);
	    }
	    
	    break;
	default:
	    call.failProcedureUnavailable();
	}
    }

    public InetAddress getLocalhost() {
	return localhost;
    }

    public int getRPCPort() {
	return rpcPort;
    }
    
    //-----------------------------------------------------
    /**
     * Adds one messages to the queue of  
     * pending RPC calls.
     */
    public void sendMessage(Message msg) {
	RPCCall call = new RPCCall(log,msg);
	pool.addOneTask((Task)call);
    }
}

//-----------------------------------------------------
/**
 *
 * This class is used to perform RPC calls asynchronously
 * using a thread pool.
 * @author Magdalena Balazinska
 */
class RPCCall extends Task {
 
    Message msg;
    TwineLogger log;

    public RPCCall(TwineLogger l, Message m) {
	log = l;
	msg = m;
    }

    public void run() {

	InetAddress ia = msg.ia;
	int port = msg.port;
	if ((ia == null) || (port <=0)) {
	    log.printStatus("Invalid destination for the following message (dropped):",
			    TwineLogger.KEY_MSG);
	    log.printStatus(msg.packet.toString(),TwineLogger.KEY_MSG);
	    return;
	}

	byte [] data = msg.getPacketBytes();

	interresolver_protClient shortClient = null;
	
	try {

	    // Parameter to the RPC call
	    interresolver_packet rpc_packet = new interresolver_packet();
	    rpc_packet.value = new byte[data.length];
	    System.arraycopy(data,0,rpc_packet.value,0,data.length);
	    rpc_packet.length = data.length;
	    rpc_packet.hostaddr = ia.getHostAddress();
	    rpc_packet.port = port;
		
	    log.printStatus("Sending RPC to " + rpc_packet.hostaddr + ":" + rpc_packet.port,
			    TwineLogger.IMPORTANT_MSG);

	    // Determine whether to send with UDP or TCP
	    if ( data.length < RPCCommunicator.MAX_RPC_PACKET) {
		log.printStatus("Using UDP RPC",TwineLogger.IMPORTANT_MSG);
		log.printStatus("Setting buffer size to: " + Packet.MAX_PACKET_SIZE,TwineLogger.IMPORTANT_MSG);
		OncRpcClient temp = new OncRpcUdpClient(ia,interresolver_prot.INTERRESOLVER_PROGRAM,1,port,
						Packet.MAX_PACKET_SIZE);
		shortClient = new interresolver_protClient(temp);
		//shortClient = new interresolver_protClient(ia,port,OncRpcProtocols.ONCRPC_UDP);
	    } else {
		log.printStatus("Using TCP RPC",TwineLogger.IMPORTANT_MSG);
		shortClient = new interresolver_protClient(ia,port,OncRpcProtocols.ONCRPC_TCP);
	    }
	    shortClient.INTERRESOLVER_SENDMESSAGE_1(rpc_packet);
	    return;

	} catch ( Exception e ) { 
	    e.printStackTrace(System.out); 
	    log.printStatus("RPC to " + ia.toString() + ":" + port + " failed!",
			    TwineLogger.KEY_MSG);
	    return;
	}

    }

}
