package ins.inr;

import java.util.Date;
import java.util.Enumeration;
import java.util.Vector;
import java.util.StringTokenizer;
import java.net.*;
import java.io.*;
import ins.namespace.*;

public class Resolver
    implements Runnable
{
    // VARIABLES

    // Switch for turning on/off debugging messages level (0-5)
    public static final int DEBUG = 5;

    // Possible values for mode
    public static final int SERVER_MODE = 0;
    public static final int INTERACTIVE_MODE = 1;

    long INRuid; // Stores a unique ID for us to use when
              // announcing names (4 bytes for addr + 2 for port)

    // long uid;

    // Modules (resource variables)
    protected Communicator comm;
    protected VSNameTree nameTrees;
    protected VSRouteTable routeTables;
    protected VSNeighbors neighbors;
    protected VSResolvers resolvers;
    protected Forwarder forwarder;
    protected DSRManager dsrMn;


    // Handlers
    protected OverlayManager overlayMn;
    protected RouteManager routeMn;
    protected AppManager appMn;
    protected NetProber prober;

    // this stuff is bad bad... use info in VSNameTree as authoritative
    //    Vector vspaces;
    String defaultVSpace;		// default VSpace

    static PrintWriter logWriter=null;
    LoggedPrintStream loggedStream;


    // CONSTRUCTORS
    public Resolver()
	throws Exception
    {
	Vector vs = new Vector();
	vs.addElement("default");

	init(vs, new Vector(1), SERVER_MODE, 0,0, null, null);	
			// on random available ports
    }

    public Resolver(/*String[]*/ Vector vspaces, Vector aggregateVspaces,
		    int mode, int udp, int tcp,
		    String[] dsrs, String [] neighs)
	throws Exception
    {
	init(vspaces, aggregateVspaces, mode, udp, tcp, dsrs, neighs);
    }

    void init(/*String[]*/ Vector vspaces, Vector aggregateVspaces, 
	      int mode, int udp, int tcp,
	      String[] dsrs, String [] neighs)
	throws Exception
    {
    	// Set system output stream to LoggedPrintStream
	loggedStream = new LoggedPrintStream(System.out);	
    	try {System.setOut(loggedStream); System.setErr(loggedStream);}
    	catch (SecurityException e) {e.printStackTrace();}
    	
	// Create sockets for communications
	comm = new Communicator(udp, tcp);	

	// Create uid
	byte[] addr = comm.localhost.getAddress();
	Date now = new Date();

	INRuid = (long)(Conversion.extract32toInt(addr, 0));
	INRuid = (INRuid << 16) | ((long)comm.UDPport & 0xFFFFl);

	/*
	uid = (long)(addr[0] & 0xFF);
	uid |= ((long)addr[1] <<  8) &                  0xFF00l;
	uid |= ((long)addr[2] << 16) &                0xFF0000l;
	uid |= ((long)addr[3] << 24) &              0xFF000000l;
	uid |= (now.getTime() << 32) &      0x0000FFFF00000000l;
	uid |= ((long)comm.UDPport << 48) & 0xFFFF000000000000l;
	*/

	printStatus("INR uid is " + INRuid);

	// Create other resources
	nameTrees = new VSNameTree();
	routeTables = new VSRouteTable();
	neighbors = new VSNeighbors();
	resolvers = new VSResolvers();
	forwarder = new Forwarder();

	printStatus("Resources created...");

	// Create handlers
	overlayMn = new OverlayManager();
	routeMn = new RouteManager();
	appMn = new AppManager();
	prober = new NetProber();
	dsrMn = new DSRManager(dsrs);

	printStatus("Handlers created...");

	// Init resources
	
	// DSRManager now does search for unknown dsr in domain
	defaultVSpace = (String)vspaces.elementAt(0);

	for (Enumeration e=vspaces.elements(); e.hasMoreElements(); )
	{
	    String vs = (String)e.nextElement();
	    nameTrees.produceVspace(vs);
	    neighbors.produceVspace(vs);
	    resolvers.produceVspace(vs);
	    routeTables.add(vs, INRuid, 
		new RouteEntry(INRuid, null, 0, 0, -1));
	}

	// set up data for each aggregate vspace
	Attribute childAttr = new Attribute("child");

	for (Enumeration e=aggregateVspaces.elements(); e.hasMoreElements(); )
	{
	    Vector v = (Vector)e.nextElement();
	    Enumeration subvses = v.elements();

	    String name = (String)subvses.nextElement();

	    // find its nametree, make it an aggregate one
	    NameStoreInterface nt = nameTrees.getNameTree(name);
	    nt.makeAggregateVspace();

	    //now add "[child=504][vspace=floor5]","[child=503][vspace=floor5]"
	    while (subvses.hasMoreElements())
	    {
		String subvs = (String)subvses.nextElement();
		NameRecord nr = new NameRecord((IHandler)dsrMn, 0, -1, 0, 
					       true, this.INRuid);
		NameSpecifier ns = new NameSpecifier();		
		ns.addAVelement(new AVelement(childAttr, new Value(subvs)));
		ns.setVspace(name);

		System.out.println("adding "+ns.toString()+" to "+name);

		nt.addNameRecord(ns, nr);
	    }
	}

	// add neighbors in list
	if (neighs != null) 
	    for (int i = 0; i<neighs.length; i++) {		
		Node n = neighbors.addNeighbor(neighs[i]);
		
		// this is gross, but necessary to prevent neighbor/resolver
		//   inconsistency
		if (n != null) {
		    Enumeration e = n.getVspaces();
		    String vsp = null;
		    if (e.hasMoreElements()) vsp=(String)(e.nextElement());

		    if (vsp!=null)
			resolvers.addResolver(vsp, n);
		}
	    }

	// Make sure callig forwarder.init() after routeTables.add(INRuid..) 
	// above
	comm.init(this);
	forwarder.init(this);
	neighbors.init(this);
	resolvers.init(this);

	printStatus("Resources initialized...");

	// Init handlers
	dsrMn.init(this);
	prober.init(this);
	overlayMn.init(this);
	routeMn.init(this);
	appMn.init(this);

	printStatus("Handlers initialized...");

	printStatus("*********************************************");

	printStatus("Neighbor is ");
	printStatus(neighbors.toString());

	printStatus("Other INRs in the system are ");
	printStatus(resolvers.toString());

	printStatus("*********************************************");

	new Thread(this).start();
    }


    public void run()
    {
    	//overlayMn.startRelaxationProbe();
    	    	
        java.io.BufferedReader br = new BufferedReader (new InputStreamReader(System.in));

	while (true) 
	{
	    try { 
		String str = br.readLine(); 
		if (str==null || str=="") continue;

		StringTokenizer st = new StringTokenizer(str);
		if (!st.hasMoreElements()) continue;

		String cmd = st.nextToken();

		if (cmd.equals("getNameTree")) {
		    System.out.println(nameTrees.toString());
		} else if (cmd.equals("lookup")) {
		    NameRecord []nr= nameTrees.lookup
			(new NameSpecifier(st.nextToken()), null);
		    System.out.println("Result: ("+nr.length+")");
		    for (int i=0; i<nr.length; i++)
			System.out.println(" - "+nr[i]);
		}

	    } catch (Exception e) { 
		e.printStackTrace();
		continue; 
	    }
	}

    }


    /**
     * Adds a set of vspaces to the resolver after it has been initialized.
     * This includes adding a name tree and advertising the new data
     *  to the DSR.
     * All vspaces in the list should be valid, localized names
     *  (i.e. cameras:wind.lcs.mit.edu is bad even if our dsr is wind.)
     *
     * If they're already present in the system, they will be ignored.
     */
    public void addVspaces(String [] vspaces) 
    {
	for (int i=0; i<vspaces.length; i++)
	{
	    String vspace = vspaces[i];

	    if (nameTrees.produceVspace(vspace)) {
		neighbors.produceVspace(vspace);
		resolvers.produceVspace(vspace);
		routeTables.add(vspace, INRuid, 
				new RouteEntry(INRuid, null, 0, 0, -1));
		
		prober.produceVspace(vspace);
		routeMn.produceVspace(vspace);
		overlayMn.produceVspace(vspace);
		appMn.produceVspace(vspace);
	    }
	}
	
	dsrMn.triggerUpdate();
    }


    /**
     * Removes a set of vspaces to the resolver after it has been initialized.
     * This includes removing the name tree and contacting the DSR.
     * All vspaces in the list should be valid, localized names
     *  (i.e. cameras:wind.lcs.mit.edu is bad even if our dsr is wind.)
     *
     * If they're not in the system, they will be ignored.
     */
    public void removeVspaces(String [] vspaces)
    {
	for (int i=0; i<vspaces.length; i++)
	{
	    String vspace = vspaces[i];

	    if (nameTrees.removeVspace(vspace)) {
		
		NodeSet tonotify = neighbors.getNeighbors(vspace);

		neighbors.removeVspace(vspace);	   
		routeTables.removeVspace(vspace);

		// don't remove from resolvers -- that serves as a cache
		// of INRs that we can route to instead of ourself

		// also don't need to remove for AppManager, etc. since
		// that was just putting things in the nametree

		// really need to tell all the tonotify neighbors
		// that we're going down
	    }
	}
	
	dsrMn.triggerUpdate();
    }

    static void printStatus(String s)
    {
	System.out.println("Resolver: " + s);
    }

    public String toString() 
    {
	return "Resolver at UDP port# "+ comm.UDPport + " ad TCP port#" +
	    comm.TCPport + " default VSpace:" + defaultVSpace;
    }



    /*  ---------- Other functions below: -------------------- */

    static void usage()
    {
	System.err.println("Usage: java " +
			   "ins.inr.Resolver vspace1[,vspace2,..] " +
			   "[-udp portnum] [-tcp portnum] [-d dsr] [n IP:udp:tcp:vspace] [-l] [-i]");
	System.err.println("  -udp  picks portnum as the port to use for UDP (0=random)");
	System.err.println("  -tcp  picks portnum as the port to use for TCP (0=random)");
	System.err.println("  -d  specifies dsr as the DSR");
	System.err.println("  -n  adds a neighbor");
	System.err.println("  -l  turns on packet-logging mode");
	System.err.println("  -i  uses interactive mode (not implemented yet)");
	System.err.println("  app are application classes to be started");

	System.exit(1);
    }

    public static boolean turnOnLogging() {
	try { 
	    logWriter = new PrintWriter
		(new FileWriter("packets.log"), true);
	} catch (IOException e) { return false; }
	return true;
    }
    
    public static boolean turnOffLogging() {
	if (logWriter != null)
	    try { logWriter.close(); } catch (Exception e) { ; }
	logWriter = null;
	return true;
    }

    public static void logSendPacket(Packet p) {
	// log the packet if applicable
	if (logWriter!=null) 
	    logWriter.println("SENDING "+
			      (new Date(System.currentTimeMillis()))+
			      ":\n"+p);
    }

    public static void logReceivePacket(Packet p) {
	// log the packet if applicable
	if (logWriter!=null) 
	    logWriter.println("RECEIVING "+
			      (new Date(System.currentTimeMillis()))+
			      ":\n"+p);
    }


    //  ****** MAIN function ******
    public static void main(String[] args)
    {
	
	IResolverFactory res;
	try {
	    res = (IResolverFactory)new ResolverFactory();
	    mainMethod(args,res);
	} catch (Exception e) {
	    System.out.println("The INR could not be started " +
			       "for the following reason:");
	    System.out.println(e.getMessage());
	    e.printStackTrace();
	}


    }

    //  ****** The actual main will be inherited by subclasses ******
    public static void mainMethod(String[] args, IResolverFactory res)
    {
	Vector dsrs = new Vector(); // Vector of String
	Vector neighbors = new Vector(); // Vector of String- host:port
	int UDPport = 0;	// any available (random) port
	int TCPport = 0;	// any available (random) port
	int mode = Resolver.SERVER_MODE;
	
	// Process the arguments
		
	int argnum = 0;

	// The first argument is the vspace

	String vspace = null;
	if (args.length <= argnum) {
	    usage();
	} else {
	    vspace = args[argnum];
	}
	argnum++;

	// tokenize, and return delimiters
	StringTokenizer st = new StringTokenizer(vspace, ",;(){}", true);
	Vector vspaces = new Vector(st.countTokens()/2);
	Vector aggregateVspaces = new Vector();


	while (st.hasMoreElements())
	{
	    String name = st.nextToken();
	    
	    // if we encounter a token, ignore it and move on
	    if (name.length()==1)
		if (name.equals(",") || name.equals(";") ||
		    name.equals("(") || name.equals(")") ||
		    name.equals("{") || name.equals("}"))
		    continue;

	    vspaces.addElement(name);

	    // if this is the last element (no more delimiters, etc.) bye!
	    if (!st.hasMoreElements()) break; 

	    // get the delimiter
	    String delim = st.nextToken();
	    if (delim.length()>1) continue; // weird case
	    
	    // if , or ; it's just a normal vspace
	    if (delim.equals(",") || delim.equals(";"))
		continue;
	    
	    if (delim.equals("(") || delim.equals("{"))
	    {
		boolean defineThem = delim.equals("(");
		
		Vector v = new Vector();
		v.addElement(name);
		
		while (st.hasMoreElements() && !delim.equals(")")
		       && !delim.equals("}"))
		{
		    String child = st.nextToken();
		    
		    // in case we have {}
		    if ((child.length()==1) &&
			(child.equals("}") || child.equals(")")))
		    {
			delim = child; break; 
		    }

		    v.addElement(child);
		    if (defineThem) vspaces.addElement(child);

		    if (st.hasMoreElements()) 
			delim = st.nextToken();
		}
		aggregateVspaces.addElement(v);
	    }
	}

	// The remaining arguments are options
	for ( ; argnum < args.length; argnum++) 
	{
	    String arg = args[argnum];
		
	    // Modified by Magda: this created problems for adding new options
	    // Check if we're done
	    //if (! arg.startsWith("-")) {
	    //break;

	    //} else 
	    if (arg.equals("-l")) {
		turnOnLogging();

	    } else if (arg.equals("-i")) {
		mode = Resolver.INTERACTIVE_MODE;

	    } else if (arg.equals("-udp")) {
		// OPTION: the -p option takes the next argument as a port

		// Capture the next argument
		argnum++;

		if (argnum == args.length) {
		    usage();
		}

		UDPport = Integer.parseInt(args[argnum]);


	    } else if (arg.equals("-tcp")) {
		// OPTION: the -p option takes the next argument as a port

		// Capture the next argument
		argnum++;

		if (argnum == args.length) {
		    usage();
		}

		TCPport = Integer.parseInt(args[argnum]);

	    } else if (arg.equals("-d")) {
		// OPTION: the -d option takes the next argument as a DSR 

		// Capture the next argument
		argnum++;
		
		if (argnum == args.length) {
		    usage();
		}

		dsrs.addElement(args[argnum]);

	    } else if (arg.equals("-n")) {
		// OPTION: the -n option takes the next argument as a neighbor
		// specified as a hostname, colon, then port number
		// i.e. fenway.lcs.mit.edu:1234

		// Capture the next argument
		argnum++;
		
		if (argnum == args.length) {
		    usage();
		}

		neighbors.addElement(args[argnum]);

	    } else if (arg.equals("-h")) {
		// OPTION: the -h option returns help

		usage();
	    }
	}

	String[] dsrsArray = new String[dsrs.size()];
	dsrs.copyInto(dsrsArray);

	String [] neighborsArray = new String[neighbors.size()];
	neighbors.copyInto(neighborsArray);

	try {
	    res.makeResolver(vspaces, aggregateVspaces,mode, UDPport, TCPport,
			     dsrsArray, neighborsArray);
	    //this.init(vspaces, aggregateVspaces,mode, UDPport, TCPport,
	    //	      dsrsArray, neighborsArray);
	    //new Resolver(vspaces, aggregateVspaces, mode, UDPport, TCPport,
	    //dsrsArray, neighborsArray);
	} catch (Exception e) {
	    System.out.println("The INR could not be started " +
			       "for the following reason:");
	    System.out.println(e.getMessage());
	    e.printStackTrace();
	}

    }

}

/** 
 * -----------------------------------------------
 * Interface to factories creating different types of
 * resolvers
 * -----------------------------------------------
 */
interface IResolverFactory {

    public Resolver makeResolver(/*String[]*/ Vector vspaces, 
				 Vector aggregateVspaces, 
				 int mode, int udp, int tcp,
				 String[] dsrs, String [] neighs) 
	throws Exception;

}

/** 
 *
 * This is an actual resolver factory that produces
 * an original kind of resolvers.
 *
 */ 
class ResolverFactory implements IResolverFactory {

    public ResolverFactory() {
    }
    
    public Resolver makeResolver(/*String[]*/ Vector vspaces, 
				 Vector aggregateVspaces, 
				 int mode, int udp, int tcp,
				 String[] dsrs, String [] neighs)
	throws Exception {

	return new Resolver(vspaces, aggregateVspaces,mode,udp,tcp,
			    dsrs, neighs);
    
    }

}
