package ins.inr;

import java.util.Date;
import java.util.Enumeration;
import java.util.Vector;
import java.io.*;

import ins.namespace.*;

class RouteManager 
    implements IHandler, Runnable
{
    // VARIABLES

    // Leaving on global TTL for old version of resolver
    public static final long MAX_NAME_TTL = 50000;  // 50s 
    // Modified by Magda: there is now a different refresh rate at the edge and in the core.
    public static final long MAX_NAME_EDGE_TTL = 40000;  // 20s refresh rate at the edge 
    public static final long MAX_NAME_CORE_TTL = 3600000;  // 1hr refresh rate in the core
    //public static final long MAX_NAME_EDGE_TTL = 18000000;  
    //public static final long MAX_NAME_CORE_TTL = 18000000;  

    public static final long MAX_ROUTE_TTL = 50000;  // 50s 
    public static final long ROUTE_UPDATE_INTERVAL = 15000;    // 15s
    public static final long EXPIRE_CHECK_INTERVAL = 15000;    // 15s

    public static final int UNDEFINED_APP_METRIC = Short.MAX_VALUE;

    protected Resolver resolver;
    protected VSNeighbors neighbors;
    protected VSNameTree nameTrees;
    protected Communicator comm;
    protected VSRouteTable routeTables;

    protected RouteEntry myuidRE;

    protected String defaultVSpace;

    protected Thread periodicThread; // thread that sends out periodic update
                                     // to neighbors
    protected Thread routeUpdateThread; // thread for periodic route-updates

    protected NameSpecifier routeNS, nameNS;
    protected NameSpecifier sourceNS;

    final static byte[] zerobyte = new byte[0];
    Attribute controlmsgAttr = new Attribute("control-msg");
    Value nameUpdateVal = new Value("name-update");
    Value routeUpdateVal = new Value("route-update");


    //Attribute hostAttr = new Attribute("host");
    //Attribute portAttr = new Attribute("port");

    PrintWriter pw;
    boolean isLog;

    boolean checkingExpireNames;


    // CONSTRUCTORS
    RouteManager()
    {
	checkingExpireNames = true;
    }

    // METHODS

    /** 
     *  A callback function: will be called during initialization
     *  Provide an access to the Resolver's resources and other handlers
     *  (Interface implementation of IHandler)
     *  @param r the Resolver object
     */
    public void init(Resolver r)
    {
	resolver = r;
	neighbors = r.neighbors;	
	nameTrees = r.nameTrees;
	comm = r.comm;
	routeTables = r.routeTables;
	defaultVSpace = r.defaultVSpace;

	myuidRE = routeTables.lookup(resolver.defaultVSpace, resolver.INRuid);
	if (myuidRE == null)	// fatal error, should not happen
	{
	    printStatus("Fatal Error: routeTables is not correctly" +
		"initialzed yet inside Forwarder.init().");
	    System.exit(-1);
	}	

	// Create the name specifier indicating routing and name update
	routeNS = new NameSpecifier("[control-msg=route-update]");
	nameNS = new NameSpecifier("[control-msg=name-update]");

	// Note: getHostName() will use ip addr if DNS name doesn't exist
	// but better use getHostAddress() since Java for Windows doesn't
	// correctly get the name if the ip suddenly changes
	// (it may cache the name somewhere)
	sourceNS = new NameSpecifier(
	     "[host = " + comm.localhost.getHostAddress() +
	     "][UDPport = " + comm.UDPport + "]" +
	     "][TCPport = " + comm.TCPport + "]");

	if (Resolver.DEBUG>=2) {
	    isLog = true;
	    try{
		String logfile = "RouteManager.log";
		pw = new PrintWriter(new FileWriter(logfile), true);
	    } catch (IOException e) {
		isLog = false;
	    }
	} 
	else
	   isLog = false;


	// Name-update and route-update messages for all vspaces 
	// are destined to me
	Enumeration enum = nameTrees.getVspaces();
	// System.out.println(nameTrees.toString());
	while (enum.hasMoreElements())
	{
	    String vspace = (String)(enum.nextElement());
	    produceVspace(vspace);
	}

	// Evoke periodic thread to remove expired NameRecords
	periodicThread = new Thread(this);
	periodicThread.start();

	// Send the first route update to neighbors
	for (Enumeration e = neighbors.getVspaces(); e.hasMoreElements();)
	{ 
	    String vspace = (String)e.nextElement();
	    sendRouteUpdate(vspace, myuidRE, myuidRE.nextHop /*==null*/);
	}
	
	// Evoke periodic Route-Update thread
	routeUpdateThread = new RouteUpdateThread(this);
	routeUpdateThread.start();
    }


    void sendNameUpdate(NameSpecifier ns, NameRecord nr, byte action, Node except)
    {
	// use Communicator directly
	String vspace = ns.getVspace();
	if (vspace == null) {
	    printStatus("ASSERT: in sendNameUpdate(): no vspace in name");
	    return; 
	}
	
	NameUpdate nu = new NameUpdate(ns, nr, action);

	nameNS.setVspace(vspace);

	Packet packet = new Packet(sourceNS, nameNS, 2,
			Packet.toAll, false, zerobyte,
			nu.toBytes());

	NodeSet nodeset = neighbors.getNeighbors(vspace);
	if (nodeset == null) return;
	Enumeration e = nodeset.getNodes();
	
	while (e.hasMoreElements()) {
	    Node n = (Node)(e.nextElement());
	    if (! n.equals(except)) // usually except == nr.getNeighbor()
				    // except when there is a loop in topology
	    {
		comm.sendByTCP(packet, n);
		if (Resolver.DEBUG >= 2)
		    printStatus("sent name update...");
		//if (Resolver.DEBUG >= 5)
		//    printStatus(nu.toString());
	    }
	}
    }

    void sendRouteUpdate(String vspace, RouteEntry re, Node except)
    {
	// use Communicator directly
	if (vspace == null) {
	    printStatus("ASSERT: in sendRouteUpdate(): no vspace specified");
	    return; 
	}
	
	routeNS.setVspace(vspace);

	Packet packet = new Packet(sourceNS, routeNS, 2,
			Packet.toAll, false, zerobyte,
			new RouteUpdate(re).toBytes());

	Enumeration e = neighbors.getNeighbors(vspace).getNodes();
	
	while (e.hasMoreElements()) 
	{
	    Node n = (Node)(e.nextElement());
	    if ((!n.equals(except)) // usually except == re.nextHop
				    // except when there is a loop in topology
		&& (n.state != Node.DORMANT)
		&& (n.state != Node.INACTIVE))
	    {
		comm.sendByTCP(packet, n);
		if (Resolver.DEBUG >= 3)
		    printStatus("sent route update...");
	    }
	}
    }

    void sendAllNames(Node to)
    {
	if (nameTrees == null) return;

	if (Resolver.DEBUG >= 3)
	{
	    printStatus("***** Send All Names *****");
	    printStatus(to.toString());
	}

	//long touid = (long)(Conversion.extract32toInt(to.ia.getAddress(), 0));
	//touid = (touid << 16) | ((long)to.UDPport & 0xFFFFl);

	for (Enumeration enum=to.vspaces.elements(); enum.hasMoreElements();)
	{
	    String vspace = (String)enum.nextElement();

	    printStatus(vspace);

	    NameStoreInterface nt = nameTrees.getNameTree(vspace);
	    for (Enumeration eNR = nt.getNameRecords(); eNR.hasMoreElements(); )
	    {
	        NameRecord nr = (NameRecord)eNR.nextElement();
		NameSpecifier ns = nt.extract(nr);

		if (!nr.isNeighbor()) continue;

		if (!nr.getAnnounceStatus()) continue;

		//if (nr.getINRuid() == touid) continue;
		if (to.equals(nr.getNeighbor())) continue;

		printStatus(ns.toString());

		nameNS.setVspace(vspace);

		Packet packet = new Packet(sourceNS, nameNS, 2,
		    Packet.toAll, false, zerobyte,
		    new NameUpdate(ns, nr, NameUpdate.ADD_NAME).toBytes());

		comm.sendByTCP(packet, to);
	    }
	}

    }

    void sendAllRoutes(Node to)
    {
	if (routeTables == null) return;

	if (Resolver.DEBUG >= 3)
	    printStatus("***** Send All Routes *****");

	for (Enumeration enum=to.vspaces.elements(); enum.hasMoreElements();)
	{
	    String vspace = (String)enum.nextElement();

	    printStatus(vspace);


	    for (Enumeration eRE = routeTables.getRouteEntries(vspace);
		((eRE!=null) && (eRE.hasMoreElements())); )
	    {
		RouteEntry re = (RouteEntry)eRE.nextElement();

		if (!to.equals(re.nextHop))
		{
		    printStatus(re.toString());
		    routeNS.setVspace(vspace);
		
		    Packet packet = new Packet(sourceNS, routeNS, 2,
			Packet.toAll, false, zerobyte,
			new RouteUpdate(re).toBytes());

		    comm.sendByTCP(packet, to);
		}
	    }
	}
    }

    /** 
     *  A callback function: will be called every time a new
     *  INS packet for this handler arrives by UDP.
     *  (Interface implementation of IHandler)
     *  @param msg
     */
    public void INSreceive(Message msg)
    {
	printStatus("WARNING: Received a message through UDP!");
	printStatus(".... NameUpdate and RouteUpdate should normally " +
		"be sent through TCP!");
	printStatus(".... The message is \n" + msg.toString());
    }

    /** 
     *  A callback function: will be called every time a new
     *  INS packet for this handler arrives by TCP.
     *  (Interface implementation of IHandler)
     *  @param packet the INS packet
     *  @param from the neighbor sending the Packet packet     
     */
    public void INSreceiveByTCP(Packet packet, Node from)
    {
	if (Resolver.DEBUG >= 2)
	    printStatus("receiving a message by TCP...");
	//if (Resolver.DEBUG >= 5)
	//    printStatus(packet.toString());
	
	AVelement av = packet.dNS.getAVelement(controlmsgAttr);
	if (av.getValue().equals(nameUpdateVal)) 
	{
	    processNameUpdate(packet, from);
	}
	else if (av.getValue().equals(routeUpdateVal)) 
	{
	    processRouteUpdate(packet, from);
	}
    }


    public void processNameUpdate(Packet packet, Node from)
    {
	if (Resolver.DEBUG >=3)
	    printStatus("processing name update...");

	NameUpdate nu = new NameUpdate(packet.data);
	// if (Resolver.DEBUG >= 4) printStatus(nu.toString());

	if (nu.INRuid == resolver.INRuid) 
	{
	    printStatus("WARNING: name update has my own INRuid, ignore it!");
	    printStatus(nu.toString());
	    return;
	}

	// if name in NameUpdate has no vspace then use dest name's vspace
	// if dNS contains no vspace, ignore update
	String vspace = nu.ns.getVspace();
	if (vspace == null)
	{
	    vspace = packet.dNS.getVspace();
	    if (vspace == null)
	    {
		printStatus("WARNING: no vspace info, NameUpdate ignored!");
	        printStatus(nu.toString());
		return;
	    }
	    nu.ns.setVspace(vspace);
	}


	switch (nu.action)
	{
	case NameUpdate.ADD_NAME:
	case NameUpdate.UPDATE_NAME:
	{

	    Vector v = nameTrees.lookupExactMatch(nu.ns, nu.announcer);
	    if (v == null)	// null = no vspace info
		return;
	    else if (v.size() == 0)	// name not exist yet
	    {
		if (nu.action == NameUpdate.UPDATE_NAME)
		{
		    printStatus("ASSERT for UPDATE_NAME: NameTree does not contain");
		    printStatus(nu.ns.toString() + " with INRuid " + nu.INRuid);
		    printStatus(".... adding the name instead!");
		}

		RouteEntry re = null;
		re = routeTables.lookup(defaultVSpace, nu.INRuid);
		if (re == null)		// unknown INRuid
		{
		    printStatus("WARNING: routeTables does not contain INRuid:");
		    printStatus("\t" + nu.INRuid);
		    printStatus("creating a new entry, with metric 0 for nextHop..");
		    re = new RouteEntry(nu.INRuid, from, 0, 0, MAX_ROUTE_TTL);
		    routeTables.add(defaultVSpace, nu.INRuid, re);
		}

		NameRecord nr = new NameRecord(nu, re, -1);
		nameTrees.addNameRecord(nu.ns, null, nr);

		sendNameUpdate(nu.ns, nr, NameUpdate.ADD_NAME, from);
	    }
	    else
	    {
	    	if (v.size() > 1)
		    printStatus("ASSERT: NameTree should only have ONE exact match, but"+
			v.size() + "were found!");

		if (nu.action == NameUpdate.ADD_NAME)
		{
		    printStatus("ASSERT: NameTree already has " + nu.ns.toString());
		    //printStatus(((NameRecord)(v.elementAt(0))).toString());
		    //printStatus(nu.toString());
		    printStatus("... updating the name instead!");
		}

		if (Resolver.DEBUG >=5)
		    printStatus("Updating NameTree entry " + nu.ns.toString());    		    

		// Since we have MST topology, we don't need Bellman Ford
		// algorithm to determine whether we need to update... if topology is
		// not a tree, we need to change the following code to calculate
		// the shortest path first...

		NameRecord nr = (NameRecord)(v.elementAt(0));
		RouteEntry re = null;
		if (nu.INRuid == nr.getINRuid())   
		    re = nr.getRouteEntry();
		else
		{			// app mobility occurs since 
					// new INR advertises the name now
		    re = routeTables.lookup(defaultVSpace, nu.INRuid);
		}

		nr.update(nu, re, -1);

		sendNameUpdate(nu.ns, nr, NameUpdate.UPDATE_NAME, from);
	    }

	    break;
	}

	case NameUpdate.REMOVE_NAME:
	{
	    //printStatus("RECEIVED REMOVE_NAME update ************");
	    Vector v = nameTrees.lookupExactMatch(nu.ns, nu.announcer);
	    if (v == null)	// null = no vspace info
		return;
	    else if (v.size() > 0)	// name exist
	    {
		printStatus("REMOVE_NAME matches " + v.size() + " record(s)");

		NameRecord nr = (NameRecord)(v.elementAt(0));
		if (vspace == null)
		{
		    if (Resolver.DEBUG >= 3)
		    {
			printStatus("Name to be removed has null vspace!");
			printStatus("... thus, can't remove the name!");
		    }
		    break;
		}

		//nameTrees.removeNameRecord(vspace, nr);
		NameStoreInterface nt = nameTrees.getNameTree(vspace);
		nt.removeNameRecord(nr);

		// Only to check if it actually removes
		/*v = nameTrees.lookupExactMatch(nu.ns, nu.announcer);
		if (v.size()>0) 
		{
		   printStatus("**** Internal error: This name cannot be removed" +
			" from name-tree (data structure problem?)" + nu.ns.toString() +
			" announced by " + nu.announcer);
		   printStatus("*************************************************");
		}*/

		sendNameUpdate(nu.ns, nr, NameUpdate.REMOVE_NAME, from);
	    }

	    // if reaches here, it means name doesn't exist, so no need
	    // to remove anything...

	    break;
	}

	default: 
	{
	    printStatus("WARNING: Unknown action in NameUpdate:");
	    printStatus(nu.toString());
	}

	} // switch

	// printStatus(nameTrees.toString());
    }


    public void processRouteUpdate(Packet packet, Node from)
    {
	if (Resolver.DEBUG >=3)
	    printStatus("processing route update...");

	RouteUpdate ru = new RouteUpdate(packet.data);
	if (Resolver.DEBUG >= 4) printStatus(ru.toString());

	// if none of dNS and sNS contains vspace, ignore update
	String vspace = packet.dNS.getVspace();
	if (vspace == null)
	{
	    vspace = packet.sNS.getVspace();
	    if (vspace == null)
	    {
		printStatus("WARNING: no vspace info, RouteUpdate ignored!");
	        printStatus(ru.toString());
		return;
	    }
	}

	RouteEntry re = routeTables.lookup(vspace, ru.INRuid);
	if (re == null)
	{
	    //if (Resolver.DEBUG >= 4)
		//printStatus("Adding a new RouteEntry based on above RouteUpdate");

	    re = new RouteEntry(ru, 1, from.rtt, from, MAX_ROUTE_TTL);
	    routeTables.add(vspace, ru.INRuid, re);
	}
	else
	{
	    //if (Resolver.DEBUG >= 4)
		//printStatus("Updating RouteEntry for above RouteUpdate");

	    // double check
	    if (re.getINRuid() != ru.INRuid)
	    {
		printStatus("Internal Error: VSRouteTable.lookup() doesn't "+
		    "work correctly, please check the class!");
		System.exit(-1);
	    }

	    re.update(from, ru.metricHop+1, ru.metricRTT + from.rtt,
		MAX_ROUTE_TTL);
	    
	}

	// propagate this route update to neighbor
	sendRouteUpdate(vspace, re, from);
    }

        
    public void run()
    {
	while (checkingExpireNames)
	{
	    for (Enumeration e = nameTrees.getVspaces(); e.hasMoreElements();)
	    {
		String vspace = (String)e.nextElement();
	        // printStatus("\tcheck expire names for vspace: " + vspace);

		// Check expired entries from RouteTable first
		routeTables.getRouteEntries(vspace);

		// Check expired name entries
		NameStoreInterface nt = nameTrees.getNameTree(vspace);
		for (Enumeration eNR = nt.getNameRecords(); eNR.hasMoreElements(); )
		{
		    NameRecord nr = (NameRecord)eNR.nextElement();
		    NameSpecifier ns = nt.extract(nr);

		    if (nr.testExpiration())
		    {
			// remove name and send name update to remove
			//if (Resolver.DEBUG >= 4)
			//{
			printStatus("************** Removing expire name:");
		        printStatus(ns.toString() + " with AppUID " +
			    nr.getAnnouncer());
			//}

			nt.removeNameRecord(nr);

			if (nr.getAnnounceStatus())
			    sendNameUpdate(ns, nr, NameUpdate.REMOVE_NAME, null);
		    }
		    else if (nr.isNeighbor())
		    {
			if (nr.getINRuid() == resolver.INRuid) continue;

			// correlate with RouteTable

			// check for new RouteEntry first
			if (nr.getNeighbor()==null)
			{
		    	    RouteEntry curRE = routeTables.lookup(vspace, nr.getINRuid());
			    nr.setRouteEntry(curRE);
			}

			RouteEntry aRE = nr.getRouteEntry();
			if ((aRE==null) || (aRE.testExpiration()))
			{
			    // actually keep deleted name for a while, well later

			    printStatus("************** Removing unreachable name:");
		            printStatus(ns.toString() + " with AppUID " +
	    			nr.getAnnouncer());
	    			
			    nt.removeNameRecord(nr);
			    if (nr.getAnnounceStatus())
				sendNameUpdate(ns, nr, NameUpdate.REMOVE_NAME, null);
			}
		    }
	        }
	    }

	    try {Thread.sleep(EXPIRE_CHECK_INTERVAL);}
	    catch (InterruptedException e) {e.printStackTrace();}
	}
    }


    /** 
     * Performs actions necessary to add vspace for this module.
     * Specifically, it registers handlers in the vspace's nametree
     * for its functionality.
     */
    protected void produceVspace(String vspace)
    {
	NameRecord nr;

	routeNS.setVspace(vspace);
	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTrees.addNameRecord(routeNS, vspace, nr);
	
	nameNS.setVspace(vspace);
	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTrees.addNameRecord(nameNS, vspace, nr);
    }


    public void printStatus(String s)
    {
	String text = /*new Date(System.currentTimeMillis()).toString() +*/
		 "RM: " + s;
	System.out.println(text);
	if (isLog) pw.println(text);
    }

}
