package ins.inr;

import ins.namespace.*;
import java.util.*;
import java.net.*;

/**
 * TwineAdvertisement.java
 * 
 * Represents one resource advertisement message.
 * A resource has several keys. Each one is used to identify
 * a set of resolvers that should receive the advertisement.
 * This class maps advertisements to keys and resolvers.
 * <br>
 * Created: Mon Oct 15 2001 <br>
 * Modified: $Id: TwineAdvertisement.java,v 1.6 2002/03/21 00:01:55 mbalazin Exp $
 * @author Magdalena Balazinska
 */
public class TwineAdvertisement implements TwineMessage {
 
    int seqNum;       // Sequence number of latest advertisement for this resource
    Hashtable indexByHostAddr;     // We map advertised keys, to IP addresses + ports
                                   // that are supposed to host them
    Hashtable oldIndexByHostAddr;  // List of hosts from previous advertisement
                                   // Used to determine the difference in the list of hosts
                                   // between two advertisements
    Hashtable indexByKeys;      // We keep all keys computed for an advertisement
    Packet adPacket;            // Packet containing latest advertisement
    int nbKeys;                 // Total number of keys computed for this advertisement
    int nbKeysDone;             // Number of keys processed for this advertisement
    boolean shouldBeRefreshed;  // Indicates whether this advertisement needs to be refreshed

    public void setPacket(Packet p) { adPacket = p;}
    public void updateSeqNum(int seqNb) { seqNum = seqNb;}

    /**
     * Class constructor
     * @param seqNb   The sequence number of the packet corresponding
     *                to this advertisement
     * @param p       The packet corresponding to this advertisement
     * @param numKeys The total number of keys computed from the
     *                description accompanying this advertisement
     */
    public TwineAdvertisement(int seqNb, Packet p, int numKeys) {
	nbKeys = numKeys;
	nbKeysDone = 0;
	shouldBeRefreshed = false;
	seqNum = seqNb;
	adPacket = p;
	indexByHostAddr = new Hashtable();
	indexByKeys = new Hashtable();
    }

    /**
     * Resets the attributes of the advertisement.
     * Most importantly, resets the list of hosts that
     * should receive the advertisement and the set of keys
     * <br>
     * This method should be used instead of the constructor
     * on all subsequence advertisements of a same resource
     * @param seqNb   The sequence number of the new packet 
     *                corresponding to this advertisement
     * @param p       The new packet corresponding to this 
     *                advertisement
     * @param numKeys The total number of keys computed from the
     *                description accompanying this advertisement 
     */
    public void reset(int seqNb, Packet p, int numKeys) {

	nbKeys = numKeys;
	nbKeysDone = 0;
	shouldBeRefreshed = false;
	seqNum = seqNb;
	adPacket = p;
	oldIndexByHostAddr = indexByHostAddr;
	indexByHostAddr = new Hashtable(oldIndexByHostAddr.size());
	indexByKeys.clear();
    }

    //-----------------------------------------------------
    // Methods from TwineMessage interface
    //-----------------------------------------------------

    public Packet getPacket() { return adPacket; }
    public int getSeqNo () { return seqNum;}

     /**
      * Determines if this advertisement should be sent to
      * the host specified by <code>addr</code> and <code>port</code>
      * @param byteKey Key used to determine the host.
      * @param addr The IP address of the host.
      * @param port The port of the host.
      * @return <code>true</code> if this advertisement has not yet been sent
      * to this host. <code>false</code> if this advertisement has alreay been
      * sent to this host using another key.
      */
    public boolean shouldSend(byte[] byteKey, InetAddress addr, int port) {
	
	synchronized(this) {
	    boolean shouldSend = !alreadySentTo(addr,port);
	    KeyEntry ke = getKeyEntry(byteKey);  
	    addKeyForHost(addr,port,ke);
	    return shouldSend;
	}

    }

    //-----------------------------------------------------
    // End methods from TwineMessage interface
    //-----------------------------------------------------

    /**
     * Increases the number of keys that have been processed
     * @return <code>true</code> if all keys have been processed.
     */
    public synchronized boolean incNbKeysAndTest() {
	nbKeysDone++;
	return (nbKeysDone == nbKeys);	
    }

    /**
     * Returns the list of resolvers that used to be in charge
     * of some key for this resource but that are not in charge
     * anymore. These resolvers should receive remove messages.
     */
    public Enumeration getSetGoneResolvers() {

	Vector v = new Vector();
	if ( oldIndexByHostAddr != null ) {
	    for ( Enumeration enum = oldIndexByHostAddr.keys(); enum.hasMoreElements();) {
		String host = (String)enum.nextElement();
		if ( indexByHostAddr.get(host) == null )
		    v.addElement(host);
	    }
	}
	return v.elements();

    }

    /**
     * Returns the list of resolvers that received
     * this advertisement
     */
    public Enumeration getSetAllResolvers() {
	return indexByHostAddr.keys();
    }

    /**
     * All advertisements need to be refreshed periodically
     * within the network of resolvers.
     * @return <code>true</code> if this advertisement should
     * be refreshed and <code>false</code> otherwise.
     */
    public synchronized boolean getShouldBeRefreshed() {
	return shouldBeRefreshed;
    }

    public synchronized void setShouldBeRefreshed() {
	shouldBeRefreshed = true;
    }

    /**
     * Retrieves the <code>KeyEntry</code> mapped 
     * to this key (byte array). The <code>KeyEntry</code>
     * is a tuple {<code>TwineAdvertisement</code>,key}
     * No adequate support for collisions at the moment
     */
    public synchronized KeyEntry getKeyEntry(byte[] byteKey) {

	int id = Conversion.extract32toInt(byteKey,byteKey.length-OFFSET);
	Integer integerID = new Integer(id);
	KeyEntry key = (KeyEntry)indexByKeys.get(integerID);
	if ( key == null ) {
	    key = new KeyEntry(this,byteKey);
	    indexByKeys.put(integerID,key);
	}
	return key;
    } 

    /**
     * Checks if this advertisement was already sent to the
     * host specified in parameters.
     */
    public synchronized boolean alreadySentTo(InetAddress addr, int port) {
	String temp = hashKeyFromHostAddr(addr,port);
	LinkedList keys = (LinkedList)indexByHostAddr.get(temp);
	if ( (keys != null) && (keys.size() >= 0)  ) {
	    return true;
	}
	else {
	    return false;
	}
    }

    /**
     * When an advertisement is mapped to the set of hosts that
     * should receive it, the same host may come up more than once.
     * This method adds a new key to the list of keys that map this 
     * advertisement to the host specified in parameters.
     */
    public synchronized void addKeyForHost(InetAddress addr, int port,
					   KeyEntry key)  { 

	String temp = hashKeyFromHostAddr(addr,port);
	LinkedList keys = (LinkedList)indexByHostAddr.get(temp);
	if ( keys == null) {
	    keys = new LinkedList();
	    indexByHostAddr.put(temp,keys);
	}
	
	keys.addLast(key);
    }

    /**
     * Returns the key most recently used to transmit this advertisement
     * to the host specified in parameters.
     */
    public synchronized byte[] getByteKeyForHost(InetAddress addr, int port) { 
	byte[] result = getByteKeyForHost(indexByHostAddr,addr,port);
	if ( result == null)
	    result = getByteKeyForHost(oldIndexByHostAddr,addr,port);
	return result;
    }

    /**
     * Looks-up the key most recently used to transmit this advertisement
     * to the host specified in parameters using only the hashtable given
     * in parameters.
     */
    public synchronized byte[] getByteKeyForHost(Hashtable byHostAddr, 
						 InetAddress addr, int port) { 
	
	byte[] result = null;
	if ( byHostAddr != null ) {
	    String temp = hashKeyFromHostAddr(addr,port);
	    LinkedList keys = (LinkedList)byHostAddr.get(temp);
	    try {
		result = ((KeyEntry)keys.getFirst()).getKey();
	    } catch (NoSuchElementException e) {
		result = null;
	    }
	}
	return result;
    }

    /**
     * When an advertisement is mapped to the set of hosts that
     * should receive it, the same host may come up more than once.
     * Each host therefore maps to a list of keys. The first one is
     * used to transmit the advertisement to the host. 
     * <br>
     * This method deletes the first key, and returns the second key
     * in the list, or <code>null</code> if no other key is in the list.
     */
    public synchronized KeyEntry getNextKeyForHost(InetAddress addr, int port) { 
						   
	KeyEntry result = null;
	String temp = hashKeyFromHostAddr(addr,port);
	LinkedList keys = (LinkedList)indexByHostAddr.get(temp);
	try {
	    KeyEntry old = (KeyEntry)keys.removeFirst();
	    result = (KeyEntry)keys.getFirst();
	} catch (NoSuchElementException e) {
	    result = null;
	}
	return result;

    }

    /*
     * The following three methods transform between 
     * <code>{host addresses,ports}</code>
     * and string keys to use in a hashtable.
     */
    public static String hashKeyFromHostAddr(InetAddress ia, int port) {
	return ia.getHostAddress() + "-" + port;
    }

    public static String hostAddrFromHashKey(String hashKey) {
	int index = hashKey.indexOf("-");
	return hashKey.substring(0,index);
    }

    public static int portFromHashKey(String hashKey) {
	int index = hashKey.indexOf("-");
	String portString = hashKey.substring(index+1,hashKey.length());
	return Integer.decode(portString).intValue();
    }


}

/**
 * Tuple <code>{advertisement,key}</code>
 * @author Magdalena Balazinska
 */
class KeyEntry {

    TwineAdvertisement ad;
    byte[] key;

    public KeyEntry(TwineAdvertisement myAd, byte[] byteKey) {
	ad = myAd;
	key = byteKey;
    }

    public byte[] getKey() { return key; }

}
    




