package ins.inr;

import ins.namespace.*;
import java.util.*;
import java.net.*;

/**
 * TwineAppManager.java <br>
 *
 * This class is responsible for all communications
 * between the resolver and client applications.
 * In contrary to the classical resolver for example, it
 * doesn't process query reply messages from other
 * resolvers.
 * <p>
 * Twine resolvers support both early-binding and late-binding.
 * <p>
 * Created: Wed Jun 20 16:37:24 2001<br>
 * Modified: $Id: TwineAppManager.java,v 1.6 2002/03/21 00:01:55 mbalazin Exp $
 *
 * @author Magdalena Balazinska
 * @author Kalpak Kothari
 *
 */

public class TwineAppManager extends AppManager {


    protected NameStoreInterface nameTree;  
    protected TwineStrandSplitter advStrandSplitter;
    protected TwineStrandSplitter queryStrandSplitter;
    protected TwineInterResolver resComm;
    protected TwineQueryManager queryManager;
    protected TwineNameTreeManipulator nameTreeManip;
    protected TwineMetricManager tmm;	// added by kalpak
    protected TwineLogger log;
    protected RPCCommunicator rpcComm; // added by kalpak

    // added by kalpak
    Long myINRuid; 
    protected NameSpecifier latebindReplyNS;
    
    static Value latebindReplyVal = new Value("late-binding-reply");
    

    //-----------------------------------------------------
    /** 
     * Class constructor
     * @author Magdalena Balazinska
     */
    TwineAppManager() {

	super();
	advStrandSplitter = (TwineStrandSplitter) new TwinePrefixes();
	queryStrandSplitter = (TwineStrandSplitter) new TwineFullStrands();
	log = new TwineLogger("AM",pw);
    }


    //-----------------------------------------------------
    /** 
     * By default, status printing is used for tracing
     * Overrids parent method to use the <code>TwineLogger</code>
     * @author Magdalena Balazinska
     */
    public void printStatus(String s)
    {
	log.printStatus(s,TwineLogger.TRACE_MSG);
    }

    /** -----------------------------------------------------
     *  A callback function: will be called during initialization
     *  Provides an access to the VSNameTree and Forwarder objects
     *  (Interface implementation of IHandler)
     *  Very similar to parent init.
     *  @param r the Resolver object
     *  @author Magdalena Balazinska
     *  -----------------------------------------------------
     */
    public void init(Resolver r)
    {
	resolver = r;
	nameTrees = r.nameTrees;
	comm = r.comm;
	rpcComm = ((TwineResolver)r).rpcComm; // added by kalpak
	routeTables = null;
	routeMn = r.routeMn;
	resComm = (TwineInterResolver)routeMn;
	dsrMn = r.dsrMn;
	resolvers = null;
	forwarder = r.forwarder;
	queryManager = ((TwineResolver)r).queryManager;
	nameTreeManip = ((TwineResolver)r).nameTreeManip;
	tmm = ((TwineResolver)r).tmm;	// added by kalpak
	myINRuid = new Long(r.INRuid);

	// Twine specific resource
	nameTree = ((TwineResolver)r).nameTree;

	announceNS = new NameSpecifier();
	announceNS.addAVelement(new AVelement(controlmsgAttr, announceVal));
	discoverNS = new NameSpecifier();
	discoverNS.addAVelement(new AVelement(controlmsgAttr, discoverVal));
	earlybindNS = new NameSpecifier();
	earlybindNS.addAVelement(new AVelement(controlmsgAttr, earlybindVal));
	discoverReplyNS = new NameSpecifier();
	discoverReplyNS.addAVelement(new AVelement(controlmsgAttr, discoverReplyVal));
	earlybindReplyNS = new NameSpecifier();
	earlybindReplyNS.addAVelement(new AVelement(controlmsgAttr, earlybindReplyVal));
	latebindReplyNS = new NameSpecifier();
	latebindReplyNS.addAVelement(new AVelement(controlmsgAttr, latebindReplyVal));

	sourceNS = new NameSpecifier(
	     "[host=" + comm.localhost.getHostAddress() +
	     "][UDPport=" + comm.UDPport + "]" +
	     "][TCPport = " + comm.TCPport + "]" +
	     "][RPCport = " + rpcComm.getRPCPort() + "]");

	// Adding everything to the name tree
	initNameTree();
	
    }


    //-----------------------------------------------------
    /**
     * Very similar to parent method.
     * @author Magdalena Balazinska
     */
    protected void initNameTree() {

	NameRecord nr;
	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTree.addNameRecord(announceNS, nr);
	
	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTree.addNameRecord(discoverNS, nr);

	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTree.addNameRecord(earlybindNS, nr);

	// Warning: All replies will be received in the TwineInterResolver

    }

    //-----------------------------------------------------
    /**
     *  Hides a useless AppManager method
     *  @author Magdalena Balazinska
     */
    protected void produceVspace(String vspace)
    {
    }


    /** -----------------------------------------------------
     *  Process a name announcement coming from the Client Library,
     *  advertising a service.
     *  Similar to parent method.
     *  @param msg NameAnnouncement from the application
     *  @author Magdalena Balazinska
     *  -----------------------------------------------------
     */
    protected void processAnnouncement(Message msg)
    {

	NameUpdate nu = new NameUpdate(msg.packet.data);
	NameRecord nr;

	log.printStatus("Received advertisement for: " + nu.ns.toString(),
			TwineLogger.INFO_MSG);

	switch (nu.action) {

	case NameUpdate.ADD_NAME: 
	case NameUpdate.UPDATE_NAME:

	    nr = nameTreeManip.getExactMatch(nu);
	    byte addOrUpdate = (nr == null) ? NameUpdate.ADD_NAME : NameUpdate.UPDATE_NAME;
	    boolean announce = false;

	    // Create new name-record for new advertisements
	    if ( nr == null ) {
		nr = nameTreeManip.createNameRecord(nu);
		announce = true;
		nameTree.addNameRecord(nu.ns,nr);
		log.printStatus("Added new service to tree",TwineLogger.TRACE_MSG);
	    }
	    // Advertise modifications too
	    else if ( nameTreeManip.updatedResource(nu,nr) ) {
		announce = true;
		log.printStatus("Resource info has changed.",TwineLogger.TRACE_MSG);
	    }

	    // Send update to other resolvers if it's a new advertisement
	    // if something has changed, or if it's time for a refresh
	    if ( announce || resComm.timeToRefresh(nr) ) {		
		sendNameUpdate(nu.ns,nr,addOrUpdate);
		log.printStatus("Sending resource info update.",TwineLogger.TRACE_MSG);
	    }
	    
	    break;

	case  NameUpdate.REMOVE_NAME: 

	    nr = nameTreeManip.getExactMatch(nu);
	    if ( nr == null ) {
		log.printStatus("Error! Trying to remove an unknown resource: dropping"
				,TwineLogger.IMPORTANT_MSG);
	    }
	    else {
		nameTree.removeNameRecord(nr);	
		int mySeqNum = dsrMn.getNextSequenceNo();
		resComm.sendRemove(mySeqNum,nu.ns,nr,NameUpdate.REMOVE_NAME);
		log.printStatus("Sending remove messages.",TwineLogger.TRACE_MSG);	 
	    }

	}
	log.printStatus("Name tree is: " +nameTree.toPrettyString(),TwineLogger.TRACE_MSG);

    }

    /**
     * Send name update to other resolvers
     * @author Magdalena Balazinska
     */
    public void sendNameUpdate(NameSpecifier ns, NameRecord nr, byte action) {

	// Split announced [name-specifier into strands
	Vector nameSpecs = advStrandSplitter.splitToStrands(ns);
	int mySeqNum = dsrMn.getNextSequenceNo();

	// Send remove messages to other resolvers
	resComm.sendNameUpdate(mySeqNum,ns,nr,action,nameSpecs);
	
    }

    /**
     * Processes a Discovery request from the application
     * Called whenever a [control-msg=discovery] request is received
     * @author Magdalena Balazinska
     */
    protected void processDiscovery(Message msg)
    {
	if (msg.packet.data.length<11)
	    log.printStatus("\n Weird condition???",TwineLogger.IMPORTANT_MSG);
	else {	

	    // Get info from the message
	    int queryType = TwineQueryManager.DISCOVERY; // Type of query
	    TwineQuery query = queryManager.makeQueryFromPacket(msg.packet,queryType);
	    NameSpecifier ns = query.getNameSpec();
	    log.printStatus("\nReceived discovery for: " + ns.toString(),
			    TwineLogger.INFO_MSG);  
	    
	    // Do local lookup.
	    Vector foundNSes = nameTreeManip.findNameSpecStrings(ns);

	    // Store in local state temporarily => expiration time from command line
	    int mySeqNo = dsrMn.getNextSequenceNo();
	    queryManager.addQuery(mySeqNo,foundNSes,query);

	    // Split to strands
	    Vector nameSpecs = queryStrandSplitter.splitToStrands(ns);
	    
	    // Send discovery request to other resolvers
	    log.printStatus("Asking other resolvers.",TwineLogger.TRACE_MSG);
	    resComm.discoverNames(mySeqNo,ns,nameSpecs);
 
	}
    }


    //  -----------------------------------------------------
    /**
     * Processes an Early binding request from the application
     * Called whenever a [control-msg=early-binding] request is received
     * @author Magdalena Balazinska 
     */
    protected void processEarlyBinding(Message msg)
    {

	log.printStatus("received an early-binding request from app:",
			TwineLogger.INFO_MSG);
	
	if (msg.packet.data.length<11)  
	    log.printStatus("\n Weird condition???",TwineLogger.IMPORTANT_MSG);
	else {

	    // Getting info from packet
	    int queryType = TwineQueryManager.EARLY_BINDING; 
	    TwineQuery query = queryManager.makeQueryFromPacket(msg.packet,queryType);
	    NameSpecifier ns = query.getNameSpec();

	    // Do local lookup.
	    Vector foundEBes = nameTreeManip.findHostRecords(ns);

	    // create our sequenceNo for the query
	    int mySeqNo = dsrMn.getNextSequenceNo();
	    queryManager.addQuery(mySeqNo,foundEBes,query);
	    
	    // Split to strands
	    Vector nameSpecs = queryStrandSplitter.splitToStrands(ns);
	    
	    // Send update to other resolvers
	    log.printStatus("Asking other resolvers.",TwineLogger.TRACE_MSG);
	    resComm.earlyBinding(mySeqNo,ns,query.allResults(),nameSpecs);
	    
	}
 
   }

    //  -----------------------------------------------------
    /**
     * Processes a late-binding request from the application
     * 
     * @param msg Message from application
     * @author Kalpak Kothari
     */
    protected void processLateBinding(Message msg)
    {

	log.printStatus("Received a late-binding message from app",
			TwineLogger.INFO_MSG);
	
	// Getting info from packet
	int queryType = TwineQueryManager.LATE_BINDING;
	// make special search query to find destinations
	TwineQuery query = queryManager.makeSearchQueryFromPacket(msg.packet,queryType);
	NameSpecifier ns = query.getNameSpec();
	log.printStatus("\nmessage for: " + ns.toString(),
			TwineLogger.INFO_MSG);  

	// Do local lookup.
	Vector foundEBes = nameTreeManip.findNameRecords(ns);
	log.printStatus("\nlocal lookup: " + foundEBes,
			TwineLogger.TRACE_MSG);  
	Vector foundLBes = new Vector();
	if(!msg.packet.all) {	// anycast case. keep only one (best) result.
	    int bestMetric=Integer.MIN_VALUE; NameRecord best = null;

	    for(Enumeration e = foundEBes.elements(); e.hasMoreElements(); ) {
		NameRecord nr = (NameRecord)e.nextElement();
		int thisMetric = nr.getAppMetric();
		if (thisMetric > bestMetric) {
		    best = nr; bestMetric = thisMetric;
		}
	    }
	    if(best != null) {
		TwineLBRecord tlbr = new TwineLBRecord(best.getAppMetric(), 
						       best.getINRuid(), 
						       tmm.getMetricNS(best.getINRuid()));
		foundLBes.addElement(tlbr);
	    }
	}
	else {	// multicast case. keep unique receivers.
	    Hashtable uniqueINRs = new Hashtable();
	    for(Enumeration e = foundEBes.elements(); e.hasMoreElements(); ) {
		NameRecord nr = (NameRecord)e.nextElement();
		long thisINRuid = nr.getINRuid();
		Long thisINRuidL = new Long(thisINRuid);
		if(!uniqueINRs.containsKey(thisINRuidL)) {
		    TwineLBRecord tlbr = new TwineLBRecord(nr.getAppMetric(), 
							   thisINRuid,
							   tmm.getMetricNS(thisINRuid));
		    uniqueINRs.put(thisINRuidL, tlbr);
		}
	    }

	    for(Enumeration e = uniqueINRs.elements(); e.hasMoreElements(); ) {
		foundLBes.addElement(e.nextElement());
	    }
	}

	// create our sequenceNo for the query
	int mySeqNo = dsrMn.getNextSequenceNo();
	queryManager.addQuery(mySeqNo,foundLBes,query,msg);
	    
	// Split to strands
	Vector nameSpecs = queryStrandSplitter.splitToStrands(ns);
	    
	// Send update to other resolvers
	log.printStatus("Asking other resolvers.",TwineLogger.TRACE_MSG);
	resComm.lateBinding(mySeqNo,ns,query.allResults(),nameSpecs);
    }

    /**
     * Sends response for a late-binding request
     * 
     * @param rset Result Set
     * @param sequenceNo unique sequence number of request
     * @param all Multicast/Anycast
     * @param dest Destination name specifier
     * @param msg Payload message (if non-intermediate resolver)
     * @author Kalpak Kothari
     */
    void sendLateBindingResponse(Vector rset, int sequenceNo,
				 InetAddress addr, int port, 
				 String ourVspace, String returnVspace,
				 boolean all, NameSpecifier dest, Message msg)
    {

	log.printStatus("TwineAppManager::sendLateBindingResponse - entering method.",
			TwineLogger.INFO_MSG);
	log.printStatus("destNS=" + dest,
			TwineLogger.INFO_MSG);

	log.printStatus("payload:\n" + msg,
			TwineLogger.INFO_MSG);
	
	if (rset == null) return;

	if(msg != null) {	
	    // we are the original resolver that has payload,
	    // so we are now supposed to deliver payload to destinations.
	    
	    if (!all) {
		// anycast case, size the set down to one max
		if (rset.size()>0) {
		    int bestMetric=Integer.MIN_VALUE; TwineLBRecord best = null;
		    
		    for (int i=0; i<rset.size(); i++) {
			TwineLBRecord thisItem = ((TwineLBRecord)rset.elementAt(i));
			int thisMetric = thisItem.getAppMetric();
			if (thisMetric > bestMetric) {
			    best = thisItem; bestMetric = thisMetric;
			}
		    }

		    // send payload off to corresponding resolver
		    long bestINRuid = best.getINRuid();
		    InetAddress destaddr = forwarder.extractAddrFromINRuid(bestINRuid);
		    int destport = forwarder.extractPortFromINRuid(bestINRuid);

		    // get the payload
		    Packet packet = msg.packet;
		    // indicate packet being sent from resolver and not application
		    packet.fromApp = false;
		    
		    log.printStatus("ANYCAST: destination resolver ->" +
				    destaddr + " : " + destport,
				    TwineLogger.IMPORTANT_MSG);
		    
		    rpcComm.sendMessage(new Message(packet, destaddr, destport));
		    
		}

	    }
	    else {
		// multicast case

		/** TODO **/
		// call TwineClusterer to cluster results
		// send payload off to parents (obtained from TwineClusterer)
		// attaching list of children with the payload.

		Hashtable h1 = new Hashtable();
		Hashtable receivers = new Hashtable();
		boolean localDest = false;
		
		for (int i=0; i<rset.size(); i++) {
		    TwineLBRecord thisItem = ((TwineLBRecord)rset.elementAt(i));
		    if(thisItem.getINRuid() == resolver.INRuid) {
			// skip local inruid here. but note that we
			// have local destinations
			localDest = true;
			continue;
		    }

		    NameSpecifier thisINRMetric = thisItem.getINRMetricNS();
		    if(thisINRMetric == null) {
			// we might find metrics in TMM
			long thisINRuid = thisItem.getINRuid();
			Long thisINRuidL = new Long(thisINRuid);
			thisINRMetric = tmm.getMetricNS(thisINRuid);
		    }
		    Vector thisINRMetricsHT = tmm.metricNStoHashtable(thisINRMetric);
		    // add to receivers: key=inruid, val=metrics
		    receivers.put(thisINRMetricsHT.elementAt(0), 
				  thisINRMetricsHT.elementAt(1)); 
		}
		
		h1.put(myINRuid, receivers);

		sequenceNo += (int)(resolver.INRuid & 0xFFFFFFFFl); // force unique number
		int clusterLevel = 1;
		log.printStatus("MCAST I " + sequenceNo + 
				" " + clusterLevel +
				" " + h1, TwineLogger.IMPORTANT_MSG);

		// perform clustering of receivers
		TwineClusterer tc = new TwineClusterer((TwineResolver)resolver, 
						       h1, true, 1);
		Hashtable clusters = tc.formClusters(true);

		log.printStatus("MCAST O " + sequenceNo + 
				" " + clusterLevel +
				" " + clusters, TwineLogger.IMPORTANT_MSG);

		// fix this!! add sequenceNo (unique) in options and
		// cluster level in options... fixed length.  other
		// resolvers can then increment the level... log this
		// to file.

		// for each parent, construct packet and send
		for(Enumeration e = clusters.keys(); e.hasMoreElements(); ) {

		    // get the payload
		    Packet packet = new Packet(msg.packet); // make new packet
		    byte[] packetbytes = packet.toBytes();
		    int optionmaxlen = (Packet.MAX_PACKET_SIZE - 20 -
					packet.sNS.toString().length() - 
					packet.dNS.toString().length() -
					packet.data.length);
		    byte[] tempoptions = new byte[optionmaxlen];
		    byte multicast_forward_kind = 2;
		    tempoptions[0] = multicast_forward_kind;
		    /// remember to fill optionlen in tempoptions[1,2]
		    Conversion.insertInt(tempoptions, 3, sequenceNo); // fix this! testing only
		    Conversion.insertIntLow16(tempoptions, 7, clusterLevel); // fix this! testing only
		    // current index in option byte array
		    int ct=9; 	// fix this! was 3 (remove sequenceNo,
		    		// and level after testing )

		    Long parentINRuid = (Long)e.nextElement();
		    
		    // for testing only. fix this!!
		    // MCAST OUT seqNum myINRuid myClusterLevel nextParentuid
		    log.printStatus("MCAST OUT " + sequenceNo + 
				    " " + myINRuid +
				    " " + clusterLevel +
				    " " + parentINRuid, TwineLogger.IMPORTANT_MSG);
		    

		    Hashtable children = (Hashtable)clusters.get(parentINRuid);
		    for(Enumeration f = children.keys(); f.hasMoreElements(); ) {
			Long childINRuid = (Long)f.nextElement();
			Hashtable childMetrics = (Hashtable)children.get(childINRuid);
			NameSpecifier childMetricNS = tmm.metricHTtoNS(childINRuid, childMetrics);
			byte[] childMetricbytes = childMetricNS.toString().getBytes();
			int metriclen = childMetricbytes.length;
			
			if(ct+metriclen > optionmaxlen) {
			    log.printStatus("WARNING: option field for multicast packet "+
					    "has exceeded its max length",
					    TwineLogger.IMPORTANT_MSG);
			    log.printStatus(".. only " + ct + " option bytes will be sent",
					    TwineLogger.IMPORTANT_MSG);
			    break;
			}
			else {
			    Conversion.insertIntLow16(tempoptions, ct, metriclen);
			    System.arraycopy(childMetricbytes, 0, tempoptions, ct+2, metriclen);
			    ct += 2+metriclen; // space for len + actual length
			}
		    }
		    // insert total option length
		    Conversion.insertIntLow16(tempoptions, 1, ct);

		    // construct final options byte array
		    byte[] option = new byte[ct];
		    System.arraycopy(tempoptions, 0, option, 0, ct);
		    // set new options in packet
		    packet.option = option;

		    // send payload off to corresponding resolver
		    long parentuid = parentINRuid.longValue();
		    InetAddress destaddr = forwarder.extractAddrFromINRuid(parentuid);
		    int destport = forwarder.extractPortFromINRuid(parentuid);
		    
		    /// fix problem with options. should maintain all
		    // other options from original packet.
		    // currently, it is creating a completely new option header.

		    // indicate packet being sent from resolver and not application
		    packet.fromApp = false;
		    
		    log.printStatus("MULTICAST: destination resolver ->" +
				    destaddr + " : " + destport,
				    TwineLogger.IMPORTANT_MSG);
		    
		    rpcComm.sendMessage(new Message(packet, destaddr, destport));
		    
		}

		// if local destinations, directly forward payload to them.
		if(localDest) {
		    // remove any options of kind=multicast_forward from packet
		    // should i make new packet?
		    msg.packet.option = new byte[0]; // fix this!!
		    resComm.processLateBindingPayload(msg);
		}

		log.printStatus("MULTICAST: done.",
				TwineLogger.INFO_MSG);
		
	    }
	}
	else {
	    // we are some intermediate resolver, and our job is to just return
	    // all results to the sender (need to fix a format for the packet data).

	    int len=10; // minimum length includes sequence number, final null
	    int numres=0; // current number of results
	    TwineLBRecord thisItem;
	    Hashtable uniqueINRs = new Hashtable(); // used for multicast

	    // figure out total length of data
	    int addition;

	    if(!all) { // anycast case. we only send one result.
		addition = 12; // fixed length(appmetric+inruid)
		numres++;
		len += addition;
	    }
	    else { // multicast case
		for (int i=0; i<rset.size(); i++) {
		    thisItem = ((TwineLBRecord)rset.elementAt(i));
		    long thisINRuid = thisItem.getINRuid();
		    Long thisINRuidL = new Long(thisINRuid);
		    NameSpecifier thisINRMetric = thisItem.getINRMetricNS();
		    if(thisINRMetric == null)
			thisINRMetric = tmm.getMetricNS(thisINRuid);
		    String thisINRMetricStr = thisINRMetric.toString();
		    if(!uniqueINRs.containsKey(thisINRuidL)) {
			uniqueINRs.put(thisINRuidL, thisINRMetricStr);
			addition = 2 + thisINRMetricStr.length(); // space for len + actual length
			numres++;
		    }
		    else addition = 0;
		    
		    if (len + addition > 8192) {	// exceeded limit
			numres--;
			log.printStatus("WARNING: late-binding-reply info has length >8192",
					TwineLogger.IMPORTANT_MSG);
			log.printStatus(".. only " + len + " bytes will be sent",
					TwineLogger.IMPORTANT_MSG);
			break;
		    }
		    len += addition;
		}
	    }

	    // construct actual data
	    byte[] bytes = new byte[len];
	    Conversion.insertInt(bytes, 0, sequenceNo);
	    int ct=4;

	    if(!all) { // anycast case
		int bestMetric=Integer.MIN_VALUE; TwineLBRecord best = null;

		// pick best result
		for (int i=0; i<rset.size(); i++) {
		    thisItem = ((TwineLBRecord)rset.elementAt(i));
		    int thisMetric = thisItem.getAppMetric();
		    if (thisMetric > bestMetric) {
			best = thisItem; bestMetric = thisMetric;
		    }
		}
		
		if(best != null) {
		    // insert best result
		    int thisMetric = best.getAppMetric();
		    long thisINRuid = best.getINRuid();
		    Conversion.insertInt(bytes, ct, thisMetric);
		    Conversion.insertLong(bytes, ct+4, thisINRuid);
		    ct += 12;
		}
	    }
	    else { // multicast case
		// add metric info for INRs
		for(Enumeration e = uniqueINRs.elements(); e.hasMoreElements(); ) {
		    String metricstr = (String)e.nextElement();
		    int metriclen = metricstr.length();
		    Conversion.insertIntLow16(bytes, ct, metriclen);
		    System.arraycopy(metricstr.getBytes(), 0, bytes, ct+2, metriclen);
		    ct += 2+metriclen;
		}
	    }

	    Conversion.insertInt(bytes, ct, 0);  	// insert final 0's, 
	    Conversion.insertIntLow16(bytes, ct+4, 0); 	// not quite sure why.
	    
	    NameSpecifier src = new NameSpecifier(sourceNS);
	    src.setVspace(ourVspace);
	    dest.setVspace(returnVspace);

	    Packet packet = new Packet
		(src, dest, 8, all, false, zerobyte, bytes);
	    
	    log.printStatus("Late-binding-response Packet: " + packet, 
			    TwineLogger.TRACE_MSG);

	    // send off packet to earlier resolver
	    rpcComm.sendMessage(new Message(packet, addr, port));
	}

	log.printStatus("TwineAppManager::sendLateBindingResponse - leaving method.",
			TwineLogger.INFO_MSG);
    }

    /**
     * We want to allow some resolvers to send partial responses with slightly
     * different destination name-specifiers by calling directly the function
     * called within this method
     * (based on sendEarlyBindingResponse method)
     * 
     * @author Kalpak Kothari
     */
    void sendLateBindingResponse(Vector rset, int sequenceNo,
				 InetAddress addr, int port, 
				 String ourVspace, String returnVspace,
				 boolean all, Message msg)
    {
	NameSpecifier dest = new NameSpecifier(latebindReplyNS);
	sendLateBindingResponse(rset,sequenceNo,addr,port,ourVspace,returnVspace,all,dest, msg);
    }

}


/**
 * Extending the EBRecord class just to override the equals
 * method.
 */
class TwineEBRecord extends EBRecord {


    TwineEBRecord(int appMetric, byte[] hostRecord)
    {
	super(appMetric,hostRecord);
    }

    // Two TwineEBRecords are the same if the host records are the same
    public boolean equals(Object o) {

	if ( ! (o instanceof TwineEBRecord ))
	    return false;

	byte[] hr = ((TwineEBRecord)o).hostRecord;
	if ( hr.length != hostRecord.length)
	    return false;

	for ( int i = 0; i < hostRecord.length; i++ ) {

	    if ( hr[i] != hostRecord[i])
		return false;

	}
	return true;
    }
}

/**
 * Maintains a response record for late-binding queries/replies
 *
 * @author Kalpak Kothari
 */
class TwineLBRecord {

    int appMetric;
    long inruid;
    NameSpecifier inrMetricNS;

    TwineLBRecord(int metric, long uid, NameSpecifier ns)
    {
	appMetric = metric;
	inruid = uid;
	inrMetricNS = ns;
    }
    
    // Two TwineLBRecords are the same if the appMetric, inruid, and
    // metricNS are equal
    public boolean equals(Object o) {
	if(!(o instanceof TwineEBRecord))
	    return false;

	int oMetric = ((TwineLBRecord)o).appMetric;
	if(oMetric != appMetric)
	    return false;
	
	long oinruid = ((TwineLBRecord)o).inruid;
	if(oinruid != inruid)
	    return false;
	
	NameSpecifier oNS = ((TwineLBRecord)o).inrMetricNS;
	if((oNS == null) && (inrMetricNS == null))
	    return true;
	else if(oNS != null)
	    if(!(oNS.equals(inrMetricNS)))
		return false;
	
	return true;
    }

    public int getAppMetric() {
	return appMetric;
    }

    public long getINRuid() {
	return inruid;
    }

    public NameSpecifier getINRMetricNS() {
	return inrMetricNS;
    }

}
