package ins.inr;

import java.util.*;
import ins.inr.*;

/**
 * TwineClusterer.java
 *
 * This class groups receivers into clusters 
 * using various metrics. Each cluster is
 * assigned a parent that the sender decides.
 *
 * @author Nikos Michalakis
 * @author Kalpak Kothari
 * @version 1.0
 */


class TwineClusterer{
    
    protected TwineResolver resolver;
    protected Hashtable receivers; //total list of receivers
    protected Hashtable clusterParents;  // the parents of clusters
    protected int maxNumClusterParents; //how many we can handle
    protected Hashtable remaining;  //{receivers} - {clusterParents}
    protected int defaultNumClusterParents; //a value the user can set
    protected Long myUID; //the current resolver's UID
    protected static TwineLogger log = new TwineLogger("TC","TwineClusterer.log");  // logger added by kalpak
    public float ALPHA;
    public static double W1=0.5;
    public static double W2=0.5;

    //=======

    // for testing only
    protected boolean TESTING = false;
    protected boolean EUCKLID = false;
    protected static int myTestCap = 10*1024*1024;
    protected static int myTestLoad = 200*1024;
    protected static int myTestPktSize = 1024;
    protected static int myTestMSN = 10;
    protected static int myTestBranchNumber = 10;
    protected static Vector myTestPosition = new Vector();
    protected static Vector myCoords = new Vector();

    //constructor:we need the list of 
    //receivers (Resolvers) and their corresponding attributes.
    public TwineClusterer(TwineResolver r,Hashtable receivers){
	this.resolver=r;
	Long key=(Long) receivers.keys().nextElement();
	this.myUID=key;
	this.receivers=(Hashtable) receivers.get(key);
	this.remaining=new Hashtable();
	this.maxNumClusterParents=findMaxNumClusterParents(false);
	ALPHA=resolver.tme.ALPHA;
    }
  
    public TwineClusterer(TwineResolver r,Hashtable receivers, boolean useMetric){
	this.resolver=r;
	Long key=(Long) receivers.keys().nextElement();
	this.myUID=key;
	this.receivers=(Hashtable) receivers.get(key);
	this.remaining=new Hashtable();
	this.maxNumClusterParents=findMaxNumClusterParents(useMetric);
	ALPHA=resolver.tme.ALPHA;
    }

    public TwineClusterer(TwineResolver r,Hashtable receivers, boolean useMetric, int
			  numClusterParents){
	this.resolver=r;
	Long key=(Long) receivers.keys().nextElement();
	this.myUID=key;
	this.receivers=(Hashtable) receivers.get(key);
	this.remaining=new Hashtable();
	setDefaultNumClusterParents(numClusterParents);
	this.maxNumClusterParents=findMaxNumClusterParents(useMetric);
	ALPHA=resolver.tme.ALPHA;
    }
  
    public TwineClusterer(){
	this.receivers=new Hashtable();
	this.remaining=new Hashtable();
	ALPHA=resolver.tme.ALPHA;
    }

    public TwineClusterer(boolean test, boolean euck){
	this.TESTING = true;
	this.EUCKLID=euck;
	this.receivers=new Hashtable();
	this.remaining=new Hashtable();
	ALPHA=resolver.tme.ALPHA;
    }

    //////end constructors///////    

    public void setReceivers(Hashtable receivers){
	Long key=(Long) receivers.keys().nextElement();
	this.myUID=key;
	this.receivers=(Hashtable) receivers.get(key);
    }

    /** Set a default number of cluster parents for this resolver
	@param num The number of desired default parents
    */
    private void setDefaultNumClusterParents(int num){
	defaultNumClusterParents=num;
    }

    /** Finds the maximum number of cluster parents this resolver affords.
	@param useMetric "false" will return the default number
	"true" will calculate the maximum number from the metrics
	@return the maximum number of cluster parents
    */
    private int findMaxNumClusterParents(boolean useMetric){
	if(TESTING) {
	    return myTestBranchNumber;
	}
	if(useMetric){
	    int maxnum=0;
	    //use the Resolver's metrics to figure out how
	    //many parents we can afford so that our load is ok.
	    maxnum=resolver.tmm.getBranchNumber(resolver.INRuid);
	    return maxnum;
	}else return defaultNumClusterParents;
    }
  
    /**  Checks whether we need to form clusters or not	
	 @return true if the resolvers sees more receivers than it can handle
    */
    private boolean clustersNeeded(){
	return (receivers.size()>this.maxNumClusterParents);
    }
  
  
    /** Selects which of the receivers will be the cluster parents
	@param useMetric "true" if we want to pick according to metrics
	@return A hashtable that contains the cluster parents
    */
    public Hashtable pickClusterParents(boolean useMetric){
	Hashtable h=new Hashtable();
	if(useMetric){
	    Hashtable entries=new Hashtable();
	    //first fill in the entries table with the entries and the metrics
	    //we want to use for comparison.
	    Vector myPosition;
	    Vector myCs=new Vector();
	    if(TESTING){
		myPosition = myTestPosition;
		myCs=myCoords;
	    }else 
		myPosition=resolver.tmm.getPosition(myUID.longValue());
	    //for testing
	    //      Vector myPosition=new Vector();
	    //myPosition.addElement(new Integer(50000));
	    //myPosition.addElement(new Integer(50000));
	    //myPosition.addElement(new Integer(50000));
  

	    //fill in the entries hashtable
	    for(Enumeration keys=receivers.keys();keys.hasMoreElements();){
		Long uid=((Long) keys.nextElement());
		//the hashtable that contains all the metrics
		Hashtable metrics=(Hashtable) receivers.get(uid);
		//the hashtable to put the metrics we want to use
		Hashtable comparisonMetrics=new Hashtable();
		int distance=0;
		if(TESTING && EUCKLID){
		    int x1=((Integer)myCs.elementAt(0)).intValue();
		    int y1=((Integer)myCs.elementAt(1)).intValue();
		    int x2=((Integer)((Vector) metrics.get("coords")).elementAt(0)).intValue();
		    int y2=((Integer)((Vector) metrics.get("coords")).elementAt(1)).intValue();
		    distance=calcEuclidianDist(x1,y1,x2,y2);
		}else
		    distance=calculateDistance(myPosition,(Vector)metrics.get("position"));
		log.printStatus("distance="+distance, TwineLogger.TRACE_MSG);
		int c=((Integer)metrics.get("capacity")).intValue();
		int l=((Integer)metrics.get("load")).intValue();
		int p=((Integer)metrics.get("ave_pkt_size")).intValue();
		int m=((Integer)metrics.get("multicast_source_number")).intValue();
		comparisonMetrics.put("capacity",new Integer((int)(ALPHA*(float)c)-l-m*p));
			     
		//pick the element at 0 which is the max of the distances.
		comparisonMetrics.put("distance",new Integer(distance));
		entries.put(uid,comparisonMetrics);
	    }
	    //create the Vector that holds the sequence of the metrics
	    Vector oMetrics=new Vector();
	    oMetrics.addElement("capacity");
	    oMetrics.addElement("distance");

	    int min=Math.min(maxNumClusterParents, receivers.size());
      
	    //pick the the receivers closest to this resolver with high capacities
	    h=pickBestSet(entries,oMetrics,min);
	}else{
	    int min=Math.min(maxNumClusterParents, receivers.size());
	    // System.out.println("min="+min);
	    // System.out.println("receivers size="+receivers.size());
	    Enumeration keys=receivers.keys();
	    for(int i=0;i<min;i++){
		Long key=(Long) keys.nextElement();
		h.put(key, receivers.get(key));
	    }
	}
	return h;
    }
  
    public Hashtable pickParentsWeighted(){
	return pickBestWeightedSet(Math.min(maxNumClusterParents,receivers.size()));
    }

    /** Groups the receivers into clusters.
	@param useMetric "true" to use metrics in forming clusters
    */
    public Hashtable formClusters(boolean useMetric){
	Hashtable clusters=new Hashtable();
	if(clustersNeeded()){
	    if(useMetric){
		//form the list of parents
		//	  this.clusterParents=pickClusterParents(useMetric);
		this.clusterParents= this.pickParentsWeighted();
		//find the remaining receivers. 
		Enumeration rkeys=receivers.keys();
		for(int i=0;i<receivers.size();i++){
		    Long rkey=(Long)rkeys.nextElement();
		    if(!clusterParents.containsKey(rkey))
			remaining.put(rkey,receivers.get(rkey));
		}
	  
	
		//use algorithm to group them...
		//  clusters=findClustersUsingDistance();
		clusters=findClustersUsingWeightedAverage();
	    }else{
		//no metric is taken into account, use default values
		//first, find the remaining receivers.
		this.clusterParents=pickClusterParents(false);
	  
		Enumeration rkeys=receivers.keys();
		for(int i=0;i<receivers.size();i++){
		    Long rkey=(Long) rkeys.nextElement();
		    if(!clusterParents.containsKey(rkey))
			remaining.put(rkey,receivers.get(rkey));
		}
	  
		//now assign the remaining receivers to clusters if there are any
	  
		//initialize the cluster with 0 remaining receivers
		Enumeration ckeys=clusterParents.keys();
		for(int i=0;i<clusterParents.size();i++){
		    //we add the parents first
		    Long ckey=(Long)ckeys.nextElement();
		    //this hashtable will contain the cluster.
		    Hashtable ch=new Hashtable();
		    //add the parent first
		    //ch.put(ckey,clusterParents.get(ckey));
		    clusters.put(ckey,ch);
		}
	  
		//now add the remaining to the cluster
		Enumeration remkeys=remaining.keys();
		//this will be used to distribute the remaining evenly
		Enumeration e=clusterParents.keys();
		while(remkeys.hasMoreElements()){
		    //if we went through all the clusters go back to the beginning
		    if(!e.hasMoreElements()){
			e=clusterParents.keys();
		    }
		    //pick the hashtable under the next parent
		    log.printStatus("cluster parents="+clusterParents.size(), TwineLogger.TRACE_MSG);
		    Hashtable h=(Hashtable) clusters.get(e.nextElement());
		    //add the child to this hashtable
		    Long member=(Long) remkeys.nextElement();
		    h.put(member,remaining.get(member));
		}
	    }
	}else{
	    //not many receivers so we have an empty cluster
	    this.clusterParents=pickClusterParents(useMetric);
	    Enumeration keys=clusterParents.keys();
	    for(int i=0;i<clusterParents.size();i++){
		//pick the key of the next element
		Long key=(Long) keys.nextElement();
		Hashtable h=new Hashtable();
		//make a hashtable and add itself in it
		// h.put(key,clusterParents.get(key));
		//put the hashtable as a value to this hashtable
		clusters.put(key,h);
	    }
	
	}
	return clusters;
    }

    /** Calculates the distance between two resolvers, namely
	the parent and the child
	@param pposition The position vector of the parent
	@param cposition The position vector of the child
	@return The vector of the differences sorted from max to min.
    */
    private int calculateDistance(Vector pposition,Vector cposition){
	Vector v=new Vector();  //will contain the differences
	//check whether the coordinate vector size matches.
	if((pposition.size()!=cposition.size()) || pposition.size()==0
	   || cposition.size()==0){
	    for(int i=0;i<resolver.tme.getPingStationsNumber();i++)
		v.addElement(new Integer(Integer.MAX_VALUE));
	    log.printStatus("calculateDistance: vector = " + v, TwineLogger.TRACE_MSG);
	    return ((Integer)v.elementAt(0)).intValue();
	}else{
      
	    for(int i=0;i<pposition.size();i++){
		int pcoordinate=((Integer) pposition.elementAt(i)).intValue();
		int ccoordinate=((Integer) cposition.elementAt(i)).intValue();
		v.addElement(new Integer(Math.abs(pcoordinate-ccoordinate)));
	    }
	}
	//sort the distances from max to min
	Vector v2=new Vector();
	while(v.size()>0){
	    int max=((Integer) v.elementAt(0)).intValue();
	    int max_index=0;
	    for (int i=0;i<v.size();i++){
		if(max<=((Integer) v.elementAt(i)).intValue()){
		    max=((Integer) v.elementAt(i)).intValue();
		    max_index=i;
		}
		v2.addElement(v.elementAt(i));
		v.removeElementAt(i);
	
	    }
	}
	/**
	   int prod=1;
	   for (int i=0;i<v2.size();i++){
	   prod *= ((Integer)v2.elementAt(i)).intValue();
	   }
	**/
	return ((Integer)v2.elementAt(0)).intValue();
	// return prod;
    }

    /** Used by pickBestSet method:
	Compares lexicographically two entries and returns the one that has
	larger metric vector. It requires a hashtable to find the metrics for
	each entry and a list of integer metrics in the order they are to be compared.
	The hashtable has entries of the format [long uid,Hashtable [metrics]].
	@param e1 Entry 1
	@param e2 Entry 2
	@param entries The hashtable that contains the metrics of these entries
	@param oMetrics The ordered metrics
	@return The max of the two entries
    */
    public Long lexicographicCompare(Long e1, Long e2, Hashtable entries,Vector oMetrics){
	// System.out.println("e1="+e1+" e2="+e2);
	if(e1==null && e2==null) return null;
	if(e1==null)return e2;
	if(e2==null)return e1;

	if (oMetrics.size()==0) return e1;
	else{
	    // System.out.println("else loop");
	    String key=(String)oMetrics.firstElement();
	    // System.out.println("key is "+key);
	    Hashtable m1=(Hashtable)entries.get(e1);
	    Hashtable m2=(Hashtable)entries.get(e2);
	    int v1=((Integer) m1.get(key)).intValue();
	    int v2=((Integer) m2.get(key)).intValue();
	    //     System.out.println("v1="+v1+" v2="+v2);
	    //oMetrics.removeElement(oMetrics.firstElement());
	    if(v1>v2) return e1;
	    else if(v1<v2) return e2;
	    else{
		Vector om=new Vector();
		if(oMetrics.size()>1){
		    for(int i=1;i<oMetrics.size();i++)
			om.addElement(oMetrics.elementAt(i));
		}
		return lexicographicCompare(e1,e2,entries,om);
	    }
	}
    }
    /** Compares two resolver entries using a weighted average of the
	available capacity ranking and the distance ranking. The 
	entries table contains the rankings of each resolver in both
	available capacity and distance from the desired resolver.
	@param e1 Is the first entry
	@param e2 Is the second entry
	@param entries The hashtable that contains the rankings of all the
	resolvers.
	@return The INRuid of the resolver that is the smallest weighted
	average.
    */
    public Long weightedCompare(Long e1, Long e2, Hashtable entries){
	if(e1==null && e2==null) return null;
	if(e1==null)return e2;
	if(e2==null)return e1;

	int d_rank1=((Integer)((Hashtable) entries.get(e1)).get("d_rank")).intValue();
	int d_rank2=((Integer)((Hashtable) entries.get(e2)).get("d_rank")).intValue();
	int l_rank1=((Integer)((Hashtable) entries.get(e1)).get("l_rank")).intValue();
	int l_rank2=((Integer)((Hashtable) entries.get(e2)).get("l_rank")).intValue();
    
	double ave1=W1*d_rank1+W2*l_rank1;
	double ave2=W1*d_rank2+W2*l_rank2;

	if(ave1<ave2) return e1;
	else return e2;
    }
    /** Creates a hashtable with entries the list of receivers and with
	two attributes. The first is the ranking of the receiver in terms
	of available capacity and the second is the ranking of the receiver
	in terms of distance from the sender. This table can be used
	to pick Parents and assign children to parents. 
	@param sender The resolver that we calculate the distances from.
	@param receivers The hashtable of the receivers that are being ranked.
	@return The hashtable that contains the ranks for each receiver.
    */
    public Hashtable rankReceivers(Long sender, Hashtable receivers){
	Hashtable h=new Hashtable();
	// for(int i=0;i<receivers.size();i++)
	// h.put(new Integer(i+1),new Hashtable());
	for(Enumeration e=receivers.keys();e.hasMoreElements();)
	    h.put((Long) e.nextElement(),new Hashtable());

	Hashtable entries1=new Hashtable(); //to rank with available capacity
	Hashtable entries2=new Hashtable(); //to rank with distance
	Vector myPosition=new Vector();
	Vector myCs=new Vector();
	if(sender.longValue()==myUID.longValue()) {
	    if(TESTING){
		myPosition = myTestPosition;
		myCs= myCoords;
	    }else
		myPosition=resolver.tmm.getPosition(myUID.longValue());
	}
	else{
	    //System.out.println("inside else");
	    if(TESTING && EUCKLID) myCs=(Vector)((Hashtable)this.remaining.get(sender)).get("coords");
	    else
		myPosition=(Vector)((Hashtable)this.remaining.get(sender)).get("position");
	}
	//    System.out.println("my position="+myPosition);
 
	for(Enumeration rkeys=receivers.keys();rkeys.hasMoreElements();){
	    Long rUID=(Long)rkeys.nextElement();
	    Hashtable metrics=(Hashtable) receivers.get(rUID);
	    //the hashtable to put the metrics we want to use
	    //	Hashtable comparisonMetrics=new Hashtable();
	    int distance=0;	
	    if(TESTING && EUCKLID){
		int x1=((Integer)myCs.elementAt(0)).intValue();
		int y1=((Integer)myCs.elementAt(1)).intValue();
		int x2=((Integer)((Vector) metrics.get("coords")).elementAt(0)).intValue();
		int y2=((Integer)((Vector) metrics.get("coords")).elementAt(1)).intValue();
		distance=calcEuclidianDist(x1,y1,x2,y2);
	    }else
		distance=calculateDistance(myPosition,(Vector)metrics.get("position"));
	    //log.printStatus("distance="+distance, TwineLogger.TRACE_MSG);
	    int c=((Integer)metrics.get("capacity")).intValue();
	    int l=((Integer)metrics.get("load")).intValue();
	    int p=((Integer)metrics.get("ave_pkt_size")).intValue();
	    int m=((Integer)metrics.get("multicast_source_number")).intValue();
      
	    int availableCapacity=((int)(ALPHA*(float)c))-(l+m*p);
	    if(availableCapacity<0) availableCapacity=0;

	    //  int distance=((Integer)(distanceVector.elementAt(0))).intValue();
	    //      System.out.println("AC="+availableCapacity+" D="+distance);
	    Hashtable rankMetrics1=new Hashtable();
	    Hashtable rankMetrics2=new Hashtable();
	    rankMetrics1.put("availableCapacity",new Integer(availableCapacity));
	    rankMetrics1.put("distance", new Integer(distance));
	    rankMetrics2.put("availableCapacity",new Integer(availableCapacity));
	    rankMetrics2.put("distance", new Integer(distance));
	    entries1.put(rUID,rankMetrics1);
	    entries2.put(rUID, rankMetrics2);

	}
	//      System.out.println("entries1"+entries1);
	//      System.out.println("entries2"+entries2);    
	int size=entries1.size();
	for(int i=0;i<size;i++){
	    Long maxID=null;
	    int ac=0;
      
	    for(Enumeration keys=entries1.keys();keys.hasMoreElements();){
		Long rUID=(Long)keys.nextElement();
		Hashtable rankMetrics=(Hashtable)entries1.get(rUID);
	
		int availableCapacity=((Integer)rankMetrics.get("availableCapacity")).intValue();
		if(availableCapacity>=ac){
		    maxID=rUID;
		    ac=availableCapacity;
		}
	    }
	    // System.out.print(" maxID="+maxID);
	    ((Hashtable) h.get(maxID)).put("l_rank",new Integer(i+1));
	    entries1.remove(maxID);
	}

	for(int i=0;i<size;i++){
	    Long maxID=null;
	    int d=Integer.MAX_VALUE;
      
	    for(Enumeration keys=entries2.keys();keys.hasMoreElements();){
		Long rUID=(Long)keys.nextElement();
		Hashtable rankMetrics=(Hashtable)entries2.get(rUID);
	
		int distance=((Integer)rankMetrics.get("distance")).intValue();
		if(distance<=d){
		    maxID=rUID;
		    d=distance;
		}
	    }
	    ((Hashtable) h.get(maxID)).put("d_rank",new Integer(i+1));
	    entries2.remove(maxID);
	}
	return h;
    }
    
    /** 
     * Will pick the best set of entries from the receiver set of
     * given size and given a vector that contains the order of the
     * metrics to be used.
     * @param entries The entries to be sorted and their metric values
     * @param oMetrics The order of the metrics to be used
     * @param howMany The size of the set we want to pick (greater than 0)
     * @return A Hashtable that contains the set of best entries picked
     */
    public Hashtable pickBestSet(Hashtable entries, Vector oMetrics,int howMany){
	Hashtable h=new Hashtable();
	for(int i=0;i<howMany;i++){
	    Long maxID=null;
     
	    for(Enumeration keys=entries.keys();keys.hasMoreElements();){
		maxID=lexicographicCompare(maxID,(Long) keys.nextElement(),entries,oMetrics);
	    }
	    h.put(maxID,receivers.get(maxID));
	    entries.remove(maxID);
	}
	return h;
	
    }

    public Hashtable pickBestWeightedSet(int howMany){
	Hashtable rh=rankReceivers(myUID,receivers);
	//System.out.println("rh=" +rh);
	Hashtable h=new Hashtable();
	for(int i=0;i<howMany;i++){
	    Long maxID=null;
	    for(Enumeration keys=rh.keys();keys.hasMoreElements();){
		maxID=weightedCompare(maxID, (Long) keys.nextElement(),rh);
	    }
	    h.put(maxID, receivers.get(maxID));
	    rh.remove(maxID);
	}
	return h;
    }

    /** Used by formClusters method.
	Finds the clusters such that each of the remaining receivers
	is to its closest parent.
	@return A Hashtable that of the form:
	Hashtable1=[parent(i)UID,Hashtable2]
	Hashtable2=[remaining(j)UID,Hashtable3]
	Hashtable3=[metric(k),val(k)]
    */
    public Hashtable findClustersUsingDistance(){
	Hashtable h=new Hashtable();
	//initialize h
	for(Enumeration pkeys=clusterParents.keys();pkeys.hasMoreElements();)
	    h.put(pkeys.nextElement(),new Hashtable());

	//first fill in the entries table with the uids and the metrics
	//we want to use for comparison.
	for(Enumeration ckeys=remaining.keys();ckeys.hasMoreElements();){
	    Hashtable entries=new Hashtable();
	    //pick a parent and its position
	    Long cuid=(Long) ckeys.nextElement();
	    Hashtable cmetrics=(Hashtable)remaining.get(cuid);
	    Vector cposition=(Vector) cmetrics.get("position");
	    Vector cCs=new Vector();
	    if(TESTING) cCs=(Vector)cmetrics.get("coords");
	    //fill in the entries hashtable
	    for(Enumeration pkeys=clusterParents.keys();pkeys.hasMoreElements();){
		Long puid=((Long) pkeys.nextElement());
		//the hashtable that contains all the metrics
		Hashtable pmetrics=(Hashtable) clusterParents.get(puid);
		//the hashtable to put the metrics we want to use
		Hashtable comparisonMetrics=new Hashtable();
		int distance=0;
		if(TESTING && EUCKLID){
		    int x1=((Integer)cCs.elementAt(0)).intValue();
		    int y1=((Integer)cCs.elementAt(1)).intValue();
		    int x2=((Integer)((Vector) pmetrics.get("coords")).elementAt(0)).intValue();
		    int y2=((Integer)((Vector) pmetrics.get("coords")).elementAt(1)).intValue();
		    distance=calcEuclidianDist(x1,y1,x2,y2);
		}else
		    distance=calculateDistance(cposition,(Vector)pmetrics.get("position"));
		comparisonMetrics.put("distance", new Integer(distance));
		entries.put(puid,comparisonMetrics);
	    }
	    //create the Vector that holds the sequence of the metrics
	    Vector oMetrics=new Vector();
	    oMetrics.addElement("distance");
      
	    //pick the parent closest to this receiver
	    Long chosenParent=(Long) pickBestSet(entries,oMetrics,1).keys().nextElement();
	    //      System.out.println("chosenparent is "+chosenParent);
	    //add the child under this parent's entry
	    Hashtable ph=(Hashtable) h.get(chosenParent);
	    ph.put(cuid,cmetrics);
	    h.put(chosenParent,ph);
	}
	return h;
    }

    public Hashtable findClustersUsingWeightedAverage(){
	Hashtable h=new Hashtable();
	//initialize h
	for(Enumeration pkeys=clusterParents.keys();pkeys.hasMoreElements();)
	    h.put(pkeys.nextElement(),new Hashtable());

	//    //first fill in the entries table with the uids and the metrics
	//    //we want to use for comparison.
	for(Enumeration ckeys=remaining.keys();ckeys.hasMoreElements();){
	    
	    //      //pick a parent and its position
	    Long cuid=(Long) ckeys.nextElement();
	    Hashtable cmetrics=(Hashtable)remaining.get(cuid);
   
      
      
	    //pick the parent closest to this receiver
	    Hashtable rh=rankReceivers(cuid,clusterParents);
	    //Hashtable h=new Hashtable();
	    //	  for(int i=0;i<clusterParents.size();i++){
	    Vector v=new Vector();
	    Hashtable rh2=new Hashtable();
	    int size=rh.size();
	    for(int i=0;i<size;i++){
		Long maxID=null;
		for(Enumeration keys=rh.keys();keys.hasMoreElements();){
		    maxID=weightedCompare(maxID, (Long) keys.nextElement(),rh);
		}
		v.addElement(maxID);
		rh2.put(maxID, receivers.get(maxID));
		rh.remove(maxID);
	    }

	    Long bestID=null;
	    for(int i=0;i<v.size();i++){
		Long id=(Long) v.elementAt(i);
		int clusterSize=((Hashtable) h.get(id)).size();
		if(clusterSize<=((remaining.size()/clusterParents.size())+1))
		    bestID=id;
	    }
	    //}
	    if(bestID==null) bestID=(Long)v.elementAt(0);
	    Hashtable ph=(Hashtable) h.get(bestID);
	    ph.put(cuid,cmetrics);
	    h.put(bestID, ph);
	    //rh.remove(maxID);
	}
	return h;
    }

    // Long chosenParent=(Long) pickBestWeightedSet(entries,oMetrics,1).keys().nextElement();
    //      System.out.println("chosenparent is "+chosenParent);
    //add the child under this parent's entry
    // Hashtable ph=(Hashtable) h.get(chosenParent);
    //	ph.put(cuid,cmetrics);
    //h.put(chosenParent,ph);
    // }
    // return h;
    //}

    /** 
     * Tester Method. Used when TwineClusterer is run independently
     * for simulation of clustering.
     * @param modemnum number of modems (56 Kbps)
     * @param cablesnum number of cable modems (1 Mbps)
     * @param dslnum number of DSL modems (1 Mbps)
     * @param ethernum number of ethernet machines (10 Mbps)
     * @param fast_ethernum number of fast ethernet machines (100 Mbps) 
     */
    public void testClustering(int modemnum, int cablesnum, int dslnum,
			       int ethernum, 
			       int fast_ethernum){
	int modems=modemnum;
	int modems_cap=56*1024;
	int cables=cablesnum;
	int cables_cap=1*1024*1024;
	int ether=ethernum;
	int ether_cap=10*1024*1024;
	int fast_ether=fast_ethernum;
	int fast_ether_cap=100*1024*1024;
	int dsl=dslnum;
	int dsl_cap=1*1024*1024;

	int size=modems+cables+ether+fast_ether+dsl;
	System.out.println(size);
	int max_rtt=500; //500 ms

	Vector uids=new Vector(); //longs
	Vector capacities=new Vector(); //ints
	Vector loads=new Vector(); //ints
	Vector positions=new Vector(); //Vectors of ints
	Vector packet_size=new Vector(); //ints
	Vector branch_number=new Vector(); //ints
	Vector source_number=new Vector(); //ints
	Vector coords=new Vector();

	Random r=new Random();

	float alpha=ALPHA;

	Long myINRuid = new Long((long)9999);
	int inSequenceNo = r.nextInt(10000); // random sequence number
	int clusterLevel = 1;

	for(int i=0;i<size;i++)
	    uids.addElement(new Long((long)(124567890+i)));
    
	for(int i=0;i<modems;i++){
	    capacities.addElement(new Integer(modems_cap));
	    loads.addElement(new Integer((int)(alpha*(float)(r.nextInt(modems_cap)))));
	}
	for(int i=0;i<cables;i++){
	    capacities.addElement(new Integer(cables_cap));
	    loads.addElement(new Integer((int)(alpha*(float)(r.nextInt(cables_cap)))));
	}
	for(int i=0;i<ether;i++){
	    capacities.addElement(new Integer(ether_cap));
	    loads.addElement(new Integer((int)(alpha*(float)(r.nextInt(ether_cap)))));
	}
	for(int i=0;i<fast_ether;i++){
	    capacities.addElement(new Integer(fast_ether_cap));
	    loads.addElement(new Integer((int)(alpha*(float)(r.nextInt(fast_ether_cap)))));
	}
	for(int i=0;i<dsl;i++){
	    capacities.addElement(new Integer(dsl_cap));
	    loads.addElement(new Integer((int)(alpha*(float)(r.nextInt(dsl_cap)))));
	}

	for(int i=0;i<size;i++)
	    packet_size.addElement(new Integer(r.nextInt( 40000)));
    
	for(int i=0;i<size;i++)
	    source_number.addElement(new Integer((r.nextInt(10))+1));

	for(int i=0;i<size;i++) {
	    int b = getTestBranchNumber(((Integer)capacities.elementAt(i)).intValue(),
					((Integer)loads.elementAt(i)).intValue(),
					((Integer)packet_size.elementAt(i)).intValue(),
					((Integer)source_number.elementAt(i)).intValue());
	    branch_number.addElement(new Integer(b));
	}

	for(int i=0;i<size;i++){
	    Vector v=new Vector();
	    int choice=r.nextInt(5);
	    int x1=0;
	    int y1=0;
	    if(choice==0){
		x1= r.nextInt(50);
		y1= r.nextInt(50);
		v.addElement(new Integer(x1));
		v.addElement(new Integer(y1));
		coords.addElement(v);
	    }else
		if(choice==1){
		    x1= r.nextInt(50);
		    y1= 450+r.nextInt(50);
		    v.addElement(new Integer(x1));
		    v.addElement(new Integer(y1));
		    coords.addElement(v);
		}else
		    if(choice==2){
			x1= 225+r.nextInt(50);
			y1= 225+r.nextInt(50);
			v.addElement(new Integer(x1));
			v.addElement(new Integer(y1));
			coords.addElement(v);
		    }else
			if(choice==3){
			    x1= 450+r.nextInt(50);
			    y1= r.nextInt(50);
			    v.addElement(new Integer(x1));
			    v.addElement(new Integer(y1));
			    coords.addElement(v);
			}else
			    if(choice==4){
				x1= 450+r.nextInt(50);
				y1= 450+r.nextInt(50);
				v.addElement(new Integer(x1));
				v.addElement(new Integer(y1));
				coords.addElement(v);
			    }
	     
	    
	    Vector pos=new Vector();

	    pos.addElement(new Integer(calcEuclidianDist(x1,y1,150,100)));
	    pos.addElement(new Integer(calcEuclidianDist(x1,y1,250,350)));
	    pos.addElement(new Integer(calcEuclidianDist(x1,y1,350,200)));
	    positions.addElement(pos);
	}
    
	Hashtable h2=new Hashtable();
 
	for(int i=0;i<size;i++){
	    Hashtable h3=new Hashtable();
	    h3.put("position",positions.elementAt(i));
	    h3.put("load",loads.elementAt(i));
	    h3.put("ave_pkt_size",packet_size.elementAt(i));
	    h3.put("branch_number",branch_number.elementAt(i));
	    h3.put("multicast_source_number",source_number.elementAt(i));
	    h3.put("capacity",capacities.elementAt(i));
	    h3.put("coords",coords.elementAt(i));
	    h2.put(uids.elementAt(i),h3);
	}
	Hashtable h1=new Hashtable();
	h1.put(myINRuid,h2);

	Hashtable myMetrics=new Hashtable();
	myMetrics.put("position", myTestPosition);
	myMetrics.put("load", new Integer(myTestLoad)); 
	myMetrics.put("ave_pkt_size", new Integer(myTestPktSize)); 
	myMetrics.put("branch_number", new Integer(myTestBranchNumber)); 
	myMetrics.put("multicast_source_number", new Integer(myTestMSN));  
	myMetrics.put("capacity", new Integer(myTestCap)); 
	myMetrics.put("coords", myCoords);
	recurseTest(1, myINRuid, myMetrics, h1, h2);
    }

    /**
     * Internal helper method to perform recursive testing during clustering simulation.
     */
	
    public void recurseTest(int clusterLevel, Long curINRuid, Hashtable curMetrics, 
			    Hashtable receivers, Hashtable allReceivers) {

	TwineClusterer tc=new TwineClusterer(true,true);
	tc.initializeLocalMetrics(curMetrics);
	tc.setReceivers(receivers);
	tc.maxNumClusterParents=tc.findMaxNumClusterParents(true);
	System.out.println("max number of parents="+tc.maxNumClusterParents);
	Hashtable clusters=tc.formClusters(true);
	
	// for each parent, construct packet and send
	for(Enumeration e = clusters.keys(); e.hasMoreElements(); ) {

	    Long parentINRuid = (Long)e.nextElement();

	    Hashtable children = (Hashtable)clusters.get(parentINRuid);
	    Hashtable parentMetrics = (Hashtable)allReceivers.get(parentINRuid);

	    int inSequenceNo = -1;

	    // MCAST OUT seqNum myINRuid curClusterLevel nextParentuid nextParentLoad nextParentPosition
	    log.printStatus("MCAST OUT " + inSequenceNo + 
			    " " + curINRuid +
			    " " + clusterLevel +
			    " " + parentINRuid +
			    " " + calcAvailCap(parentMetrics) +
			    printVector((Vector)parentMetrics.get("position")) +
			    printVector((Vector)parentMetrics.get("coords")),
			    TwineLogger.IMPORTANT_MSG);

	    Hashtable newCluster = new Hashtable();
	    newCluster.put(parentINRuid, children);

	    recurseTest(clusterLevel+1, parentINRuid, parentMetrics, newCluster, allReceivers);
	}
    }

    public static void initializeTestLocalMetrics() {
	int max_rtt=500; //500 ms

	Random r=new Random();

	myTestCap = 10*1024*1024;
	myTestLoad = r.nextInt((int)(TwineMetricEstimator.ALPHA*(float)(myTestCap)));
	myTestPktSize = r.nextInt(40000);
	myTestMSN = r.nextInt(10)+1;
	myTestBranchNumber = getTestBranchNumber(myTestCap, myTestLoad, myTestPktSize, myTestMSN);

	myTestPosition = new Vector();
	myCoords = new Vector();

	int x1= r.nextInt(50);
	int y1= 450+r.nextInt(50);
	myCoords.addElement(new Integer(x1));
	myCoords.addElement(new Integer(y1));

	myTestPosition.addElement(new Integer(calcEuclidianDist(x1,y1,150,100)));
	myTestPosition.addElement(new Integer(calcEuclidianDist(x1,y1,250,350)));
	myTestPosition.addElement(new Integer(calcEuclidianDist(x1,y1,350,200)));
	
	int myAvailCap = (int)(TwineMetricEstimator.ALPHA*(float)myTestCap)-myTestLoad-myTestMSN*myTestPktSize;

	log.printStatus("MCAST SRC " + myTestCap + " " +
			+ myTestLoad + " "
			+ myTestPktSize + " "
			+ myTestMSN + " "
			+ myTestBranchNumber 
			+ myAvailCap
			+ printVector(myTestPosition)
			+ printVector(myCoords),
			TwineLogger.IMPORTANT_MSG);
    }

    public void initializeLocalMetrics(Hashtable metrics) {
	myTestCap = ((Integer)metrics.get("capacity")).intValue();
	myTestLoad = ((Integer)metrics.get("load")).intValue();
	myTestPktSize = ((Integer)metrics.get("ave_pkt_size")).intValue();
	myTestMSN = ((Integer)metrics.get("multicast_source_number")).intValue();
	myTestBranchNumber = ((Integer)metrics.get("branch_number")).intValue();
	myTestPosition = (Vector)metrics.get("position");
	myCoords = (Vector)metrics.get("coords");
    }


    public static void main(String args[]){
	int modemnum=5;
	int cablenum=15;
	int dslnum=15;
	int ethernum=45;
	int fastethernum=20;
	
	for (int i = 0; i < args.length; i++) {
	    String arg = args[i];
	    if ( arg.equals("-m")) {
		modemnum = Integer.parseInt(args[i+1]);
		System.out.println("Arg: Modem = " + modemnum);
	    }
	    else if ( arg.equals("-c")) {
		cablenum = Integer.parseInt(args[i+1]);
		System.out.println("Arg: Cables = " + cablenum);
	    }
	    else if ( arg.equals("-d")) {
		dslnum = Integer.parseInt(args[i+1]);
		System.out.println("Arg: DSL = " + dslnum);
	    }
	    else if ( arg.equals("-e")) {
		ethernum = Integer.parseInt(args[i+1]);
		System.out.println("Arg: Ethernet = " + ethernum);
	    }
	    else if ( arg.equals("-f")) {
		fastethernum = Integer.parseInt(args[i+1]);
		System.out.println("Arg: FastEthernet = " + fastethernum);
	    }
	    else if ( arg.equals("-W1")) {
		W1 = Double.parseDouble(args[i+1]);
		System.out.println("Arg: W1 = " + W1);
	    }
	    else if ( arg.equals("-W2")) {
		W2 = Double.parseDouble(args[i+1]);
		System.out.println("Arg: W2 = " + W2);
	    }

	}

	initializeTestLocalMetrics();
      
	TwineClusterer tc=new TwineClusterer(true,true);
	tc.testClustering(modemnum,cablenum,dslnum,ethernum,fastethernum);
    }


    public static int getTestBranchNumber(int testCap, int testLoad, int testPktSize, int testMSN) {
	int max_branches = 10;
	float alpha=TwineMetricEstimator.ALPHA;
	int branches=(int) ((float)(alpha * ((float)testCap)-((float)testLoad)-
				    ((float) testPktSize)*((float)testMSN))/
			    ((float) testPktSize));
	
	if(branches<1) branches=1;
	else if (branches>max_branches) branches=max_branches;
	return branches;
    }

    public int calcAvailCap(Hashtable metrics) {
	int c = ((Integer)metrics.get("capacity")).intValue();
	int l = ((Integer)metrics.get("load")).intValue();
	int m = ((Integer)metrics.get("multicast_source_number")).intValue();
	int p = ((Integer)metrics.get("ave_pkt_size")).intValue();
	int acap = (int)(ALPHA*(float)c)-l-m*p;
	return ((acap < 0) ? 0 : acap);
    }

    public static int calcEuclidianDist(int x1, int y1, int x2, int y2) {
	return (int)Math.sqrt((x1-x2)*(x1-x2) + (y1-y2)*(y1-y2));
    }

    public static String printVector(Vector v) {
	String s = "";
	for(Enumeration e = v.elements(); e.hasMoreElements(); ) {
	    s = s + " " + e.nextElement();
	}
	return s;
    }
    
}//end TwineClusterer
