package ins.inr;

import ins.namespace.*;
import java.util.*;

/**
 * TwineStrandSplitter.java <br>
 *
 * This class takes name-specifiers and
 * splits them into strands which are
 * paths of attributes and values from 
 * the root of the name specifier to each 
 * leaf.
 * <br>
 * Created: Thu Jun 21 13:43:37 2001 <br>
 *
 * @author Magdalena Balazinska
 * 
 */

public class TwineFullStrands extends TwineStrandSplitter  {

    protected StrandManipulator sm;

    //-----------------------------------------------------
    /**
     */
    public TwineFullStrands() {
	
	switch ( TYPE_STRANDS ) {
	    
	case TwineStrandSplitter.ATTRIB_VALUE_STRANDS :
	    sm = (StrandManipulator)new AttribValueStrand();
	    break;
	case TwineStrandSplitter.ATTRIB_STRANDS :
	    sm = (StrandManipulator)new AttribStrand();	
	    break;
	default: 
	    sm = (StrandManipulator)new AttribStrand();	
	    break;
	}
    }
    
    
    //-----------------------------------------------------
    /**
     * This method splits a name-specifier into a vector of String[] 
     * corresponding to sequences of elements in its strands.
     * @param ns The name specifier that should be split
     * @return This method returns a Vector of Strands[]
     * corresponding to sequences of attributes.
     */
    public Vector splitToStrands(NameSpecifier ns) { 

	int index = 0;	
	Vector allStrands = new Vector();

	// Recursively split all root attribs into strands.
	for (Enumeration e = ns.getAVelements(); e.hasMoreElements(); ) {
	    AVelement av = (AVelement)e.nextElement(); 
	    extractStrands(av,allStrands,index);
	}
	log.printStatus(printStrandSet(allStrands),TwineLogger.TRACE_MSG);
	return allStrands;
    }


    //-----------------------------------------------------
    /**
     *  This method performs the actual splitting. The AVTree is 
     *  searched depth first. Everytime a leaf is encountered, a
     *  String[] is created for the corresponding attribute strand.
     *  Attributes are added to the strand on the return path.
     */
    protected Vector extractStrands(AVelement av, Vector allStrands, int index) {
	
	String[] strand = null;
	Vector strandSet = new Vector();
	Vector childStrandSet = null;

	if ( av != null ) {

	    // Create strands at leafs OR once a predefined maximal depth has been reached
	    if ( av.isLeaf() || (index >= MAX_DEPTH - 1) ) {
		strand = new String[sm.lastIndex(av,index)];
		sm.addLast(av,strand,index);
		if ( sm.longEnough(strand) ) {
		    allStrands.addElement(strand);
		    strandSet.addElement(strand);
		}
	    }

	    else {

		// For all children attributes
		for (Enumeration e = av.getAVelements(); e.hasMoreElements(); ) {     
		    AVelement childAV = (AVelement)e.nextElement();
		    childStrandSet = extractStrands(childAV,allStrands,sm.nextIndex(index));
		    
		    // For all strands created by the children
		    for (Enumeration enum = childStrandSet.elements(); enum.hasMoreElements(); ) {
			strand = (String[])enum.nextElement();
			sm.addElement(av,strand,index);
			strandSet.addElement(strand);
		    }
		}
		
	    }
	}
	return strandSet;
	    
    }




    //-----------------------------------------------------
    /** 
     * Strategy to exctract attributes and or values 
     * from AVTrees into strands
     */
    public interface StrandManipulator {
	public void addElement(AVelement av, String[] strand, int index);
	public void addLast(AVelement av, String[] strand, int index);
	public int nextIndex(int index);
	public int lastIndex(AVelement av, int index);
	public boolean longEnough(String[] strand);
    }


    //-----------------------------------------------------
    /** 
     * Class allowing to create ATTRIBUTE strands
     * 
     */
    public class AttribStrand implements StrandManipulator {


	public AttribStrand() {
	}

	public void addElement(AVelement av, String[] strand, int index) {

	    //String thisAttrib = "empty";
	    //if ( av.getAttribute() != null )
	    String thisAttrib = av.getAttribute().toString();
	    strand[index] = thisAttrib; 

	}  

	public void addLast(AVelement av, String[] strand, int index) {
	    addElement(av,strand,index);
	}
	
	public int nextIndex(int index) {
	    return index+1;
	}

	public int lastIndex(AVelement av, int index) {
	    return index+1;
	}
	
	/**
	 * If we're using only attributes, one is enough
	 */
	public boolean longEnough(String [] strand) {
	    if ( strand.length >= 1 )
		return true;
	    else return false;
	}
    }

    //-----------------------------------------------------
    /** 
     * Class allowing to create ATTRIBUTE-VALUE strands
     * 
     */
    public class AttribValueStrand implements StrandManipulator {

	public AttribValueStrand() {
	}

	/**
	 * There shouldn't be any "empty" attribute/values
	 */
	public void addElement(AVelement av, String[] strand, int index) {

	    //String thisAttrib = "empty";
	    //String thisValue = "empty";
	    //if ( av.getAttribute() != null )
	    String thisAttrib = av.getAttribute().toString();
	    //if ( av.getValue() != null )
	    String thisValue = av.getValue().toString();

	    strand[index] = thisAttrib; 
	    strand[index+1] = thisValue; 

	} 
 
	/**
	 * When adding the last element of an attribute/value tree
	 * the last value is discarded if it's a wild-card
	 */
	public void addLast(AVelement av, String[] strand, int index) {

	    //String thisAttrib = "empty";
	    //if ( av.getAttribute() != null )
	    String thisAttrib = av.getAttribute().toString();
	    strand[index] = thisAttrib; 

	    Value value = av.getValue();
	    if ( (value != null) && (!value.isWildcard()) ) {
		String thisValue = av.getValue().toString();
		strand[index+1] = thisValue;
	    }


	}  

	public int nextIndex(int index) {
	    return index+2;
	}

	public int lastIndex(AVelement av, int index) {

	    Value value = av.getValue();
	    if ( (value != null) && (!value.isWildcard()) )
		return index+2;
	    else
		return index+1;
	}

	/**
	 * If we're using values, then at least one is required
	 */
	public boolean longEnough(String [] strand) {
	    if ( strand.length >= 2 )
		return true;
	    else return false;
	}
    }


}
