package ins.inr;

import ins.inr.twineStrandMapper.*;
import ins.namespace.*;
import java.util.*;
import java.net.*;

/**
 * TwineInterResolver.java
 *
 * <br> 
 * Handles all communication between resolvers.
 * <br> 
 * This class inherits from RouteManager to
 * reuse some of the code.
 * Both classes are nevertheless very different.
 *
 * <br><br>
 * Created: Thu Jun 21 18:11:50 2001<br>
 * Modified: $Id: TwineInterResolver.java,v 1.8 2002/03/20 04:57:40 mbalazin Exp $
 *
 * @author Magdalena Balazinska
 * @author Kalpak Kothari
 * @version 1.0
 */

public class TwineInterResolver extends RouteManager {

    protected TwineStrandMapper strandMap;
    protected NameStoreInterface nameTree;
    protected TwineLogger log;
    protected AppManager appMn;
    protected TwineQueryManager queryManager;
    protected TwineAdvManager adManager;
    protected TwineNameTreeManipulator nameTreeManip;
    protected TwineKeyManager keyMn;
    protected TwineMetricManager tmm;	// added by kalpak
    protected TwineForwarder forwarder;	// added by kalpak
    protected Long myINRuid;		// added by kalpak
    protected RPCCommunicator rpcComm;

    protected NameSpecifier discoverRequestNS, earlybindRequestNS, latebindRequestNS;
    protected NameSpecifier discoverReplyNS, earlybindReplyNS, latebindReplyNS;
    protected NameSpecifier discoverIncompleteReplyNS, earlybindIncompleteReplyNS, 
	latebindIncompleteReplyNS;
    protected NameSpecifier nameUpdateRejectNS;
    protected NameSpecifier metricRequestNS, metricReplyNS;

    static Value discoverRequestVal = new Value("discovery-request");
    static Value earlybindRequestVal = new Value("early-binding-request");
    static Value latebindRequestVal = new Value("late-binding-request");

    static Value discoverReplyVal = new Value("discovery-reply");
    static Value earlybindReplyVal = new Value("early-binding-reply");
    static Value latebindReplyVal = new Value("late-binding-reply");

    // To signal incomplete responses
    static Value discoverIncompleteReplyVal = new Value("discovery-incomplete-reply");
    static Value earlybindIncompleteReplyVal = new Value("early-binding-incomplete-reply");
    static Value latebindIncompleteReplyVal = new Value("late-binding-incomplete-reply");

    // To reject name-updates
    static Value nameUpdateRejectVal = new Value("name-update-reject");

    // For metric control messages (for returning metrics)
    static Value metricRequestVal = new Value("metric-request");
    static Value metricReplyVal = new Value("metric-reply");

    public TwineInterResolver() {
	
	strandMap = (TwineStrandMapper) new TwineAtOnceStrandMapper();
	//strandMap = (TwineStrandMapper) new TwineConcatStrandMapper();
	log = new TwineLogger("InterResolver","TwineInterResolver.log");
	adManager = new TwineAdvManager();

    }

    /**
     * -----------------------------------------------------
     * Initialization section
     * -----------------------------------------------------
     */

    //-----------------------------------------------------
    /**
     *
     */
    public void init(Resolver r)
    {

	resolver = r;
	neighbors = null;
	nameTrees = null;
	comm = r.comm;
	rpcComm = ((TwineResolver)r).rpcComm;
	routeTables = null;
	defaultVSpace = null;
	queryManager = ((TwineResolver)r).queryManager;
	nameTreeManip = ((TwineResolver)r).nameTreeManip;
	keyMn = ((TwineResolver)r).keyMn;
	appMn = r.appMn;
	tmm = ((TwineResolver)r).tmm;	// added by kalpak
	forwarder = (TwineForwarder)r.forwarder;	// added by kalpak
	myINRuid = new Long(r.INRuid);	// added by kalpak
	adManager.init(r);

	// Twine specific resource
	nameTree = ((TwineResolver)r).nameTree;
	strandMap.init( ((TwineResolver)r).rpcComm,this);

	// Create the name specifier indicating name update
	nameNS = new NameSpecifier("[control-msg=name-update]");

	discoverRequestNS = new NameSpecifier();
	discoverRequestNS.addAVelement(new AVelement(controlmsgAttr, discoverRequestVal));

	earlybindRequestNS = new NameSpecifier();
	earlybindRequestNS.addAVelement(new AVelement(controlmsgAttr, earlybindRequestVal));

	// added by kalpak
	latebindRequestNS = new NameSpecifier();
	latebindRequestNS.addAVelement(new AVelement(controlmsgAttr, latebindRequestVal));
	// ----

	discoverReplyNS = new NameSpecifier();
	discoverReplyNS.addAVelement(new AVelement(controlmsgAttr, discoverReplyVal));

	earlybindReplyNS = new NameSpecifier();
	earlybindReplyNS.addAVelement(new AVelement(controlmsgAttr, earlybindReplyVal));

	// added by kalpak
	latebindReplyNS = new NameSpecifier();
	latebindReplyNS.addAVelement(new AVelement(controlmsgAttr, latebindReplyVal));
	// ----

	discoverIncompleteReplyNS = new NameSpecifier();
	discoverIncompleteReplyNS.addAVelement(new AVelement(controlmsgAttr, 
							     discoverIncompleteReplyVal));

	earlybindIncompleteReplyNS = new NameSpecifier();
	earlybindIncompleteReplyNS.addAVelement(new AVelement(controlmsgAttr, 
							      earlybindIncompleteReplyVal));

	// added by kalpak
	latebindIncompleteReplyNS = new NameSpecifier();
	latebindIncompleteReplyNS.addAVelement(new AVelement(controlmsgAttr, 
							     latebindIncompleteReplyVal));
	// ----

	nameUpdateRejectNS = new NameSpecifier();
	nameUpdateRejectNS.addAVelement(new AVelement(controlmsgAttr, 
						      nameUpdateRejectVal));

	// added by kalpak
	metricRequestNS = new NameSpecifier();
	metricRequestNS.addAVelement(new AVelement(controlmsgAttr, metricRequestVal));

	metricReplyNS = new NameSpecifier();
	metricReplyNS.addAVelement(new AVelement(controlmsgAttr, metricReplyVal));
	// ----

	sourceNS = new NameSpecifier("[host = " + comm.localhost.getHostAddress() +
				     "][UDPport = " + comm.UDPport + "]" +
				     "][TCPport = " + comm.TCPport + "]" +
				     "][RPCport = " + rpcComm.getRPCPort() + "]");

	// Adding everything to the name tree
	initNameTree();

	// In Twine, it's not the TwineInterResolver that checks for expired entries
	periodicThread = null;
	routeUpdateThread = null;
    }

    //-----------------------------------------------------
    /**
     * 
     */
    protected void initNameTree() {

	NameRecord nr = new NameRecord((IHandler)this, 0, -1, 0, 
				       false, resolver.INRuid);
	nameTree.addNameRecord(nameNS, nr);

	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTree.addNameRecord(discoverRequestNS, nr);

	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTree.addNameRecord(earlybindRequestNS, nr);

	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTree.addNameRecord(latebindRequestNS, nr);

	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTree.addNameRecord(discoverReplyNS, nr);

	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTree.addNameRecord(earlybindReplyNS, nr);

	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTree.addNameRecord(latebindReplyNS, nr);

	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTree.addNameRecord(discoverIncompleteReplyNS, nr);

	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTree.addNameRecord(earlybindIncompleteReplyNS, nr);

	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTree.addNameRecord(latebindIncompleteReplyNS, nr);

	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTree.addNameRecord(nameUpdateRejectNS, nr);

	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTree.addNameRecord(metricRequestNS, nr);

	nr = new NameRecord((IHandler)this, 0, -1, 0, false, resolver.INRuid);
	nameTree.addNameRecord(metricReplyNS, nr);

    }

    //-----------------------------------------------------
    /** 
     *  Hides a useless RouteManager method
     */
    protected void produceVspace(String vspace)
    {
    }

    /**
     * -----------------------------------------------------
     * Request generation section
     * -----------------------------------------------------
     */

    //-----------------------------------------------------
    /**
     * Messages to remove resources are not sent by splitting
     * into strands and computing sets of resolvers for each
     * strand. Instead, they are sent to the list of resolvers
     * that last received an update of the resource info. They
     * are the ones that should remove the info that they have.
     * @param seqNo New sequence number for this message
     * @param ns Description of resource to be removed
     * @param nr Name-record of resource to be removed
     * @param action Should be <code>NameUpdate.REMOVE_NAME</code>
     * @author Magdalena Balazinska
     */ 
    public void sendRemove(int seqNo, NameSpecifier ns, NameRecord nr, byte action) {

	NameUpdate nu = new NameUpdate(ns, nr, action);
	Packet packet = new Packet(sourceNS, nameNS, -1, Packet.toAll, 
				   false, zerobyte, nu.toBytes());
	
	TwineAdvertisement lastAd = adManager.getAdByNameRec(nr.getID());

	// For this advertisement, the packet is now a remove message
	if ( lastAd != null) {
	    lastAd.setPacket(packet);
	    strandMap.sendRemove(lastAd);
	    adManager.removeAdForNameRec(nr.getID());
	    
	    int nb = adManager.indexByNameRec.size();
	    log.printStatus("We currently have " + nb + " advertisements in memory",
			    TwineLogger.IMPORTANT_MSG);
	}
    }

    //-----------------------------------------------------
    /**
     * Forwards an advertisement to other resolvers.
     * <br>Note: When creating packet to be sent
     * <ul>
     * <li> sourceNS is the name spec of this resolver
     * <li> nameNS is [control-msg=name-update]
     * <li> -1 is the unused ttl
     * <li> Packet.toAll doesn't matter
     * <li> false means the packet doesn't come from an app
     * <li> zerobyte means there are no options
     * <li> nu.toBytes() is the actual data
     * </ul>
     * @param seqNo New sequence number for this message
     * @param ns Description of resource 
     * @param nr Name-record of resource 
     * @param action whether the message is an add or an update
     * @param nameSpecs set of strands extracted from description
     * of this resource
     * @author Magdalena Balazinska
     */
    public void sendNameUpdate(int seqNo, NameSpecifier ns, NameRecord nr, 
			       byte action, Vector nameSpecs) {

	NameUpdate nu = new NameUpdate(ns, nr, action);

	Packet packet = new Packet(sourceNS, nameNS, -1, Packet.toAll, 
				   false, zerobyte, nu.toBytes());

	TwineAdvertisement ad = adManager.newAdvertisement(seqNo,nr.getID(),
							   packet,nameSpecs.size());

	for (Enumeration e=nameSpecs.elements(); e.hasMoreElements();) {
	    
	    String[] attributeStrand = (String[])e.nextElement();
	    strandMap.send(attributeStrand,TwineResolver.ANNOUNCE_D,(TwineMessage)ad);

	}
    }


    //-----------------------------------------------------
    /** 
     * Sends a query to other resolvers
     * @param seqNo New sequence number for this message
     * @param ns The query name-specifier
     * @param nameSpecs set of strands extracted from the query
     * @author Magdalena Balazinska
     */
    public void discoverNames (int seqNo, NameSpecifier ns, Vector nameSpecs) {


	// construct data packet
	byte [] nsbytes = ns.toString().getBytes();
	byte [] bytes = new byte[nsbytes.length+11];

	Conversion.insertInt(bytes, 0, seqNo);
	System.arraycopy(rpcComm.localhost.getAddress(), 0, bytes, 4, 4);
	Conversion.insertIntLow16(bytes, 8, rpcComm.getRPCPort());
	bytes[10] = (byte)(4); // not used
	System.arraycopy(nsbytes, 0, bytes, 11, nsbytes.length);

	Packet packet = new Packet(sourceNS, discoverRequestNS, 8,
				   Packet.toAny, false, zerobyte,
				   bytes);

	sendQuery(nameSpecs,packet,seqNo);
    }


    //-----------------------------------------------------
    /** 
     * A query is transmitted to other resolvers using
     * only one strand from the query description
     * @author Magdalena Balazinska
     */
    protected void sendQuery(Vector nameSpecs, Packet packet, int seqNo) {

	if ( !nameSpecs.isEmpty() ) {
	    String[] strand = getLongestStrand(nameSpecs);
	    nameSpecs.remove(strand);
	    queryManager.setStrands(seqNo,nameSpecs);
	    queryManager.setPacket(seqNo,packet);
	    strandMap.send(strand,TwineResolver.QUERY_D,queryManager.getMessage(seqNo));
	}
	else {
	    queryManager.decPending(seqNo,discoverIncompleteReplyNS,
				    earlybindIncompleteReplyNS,
				    latebindIncompleteReplyNS);
	}
    }

    //-----------------------------------------------------
    /** 
     * Randomly selects one of the longest strands from
     * given set of strands.
     */
    public String[] getLongestStrand(Vector strands) {

	String[] choice = null;
        int max_length = 0;
        Vector choices = new Vector(strands.size());
        for (Enumeration e=strands.elements(); e.hasMoreElements();) {
            
            String[] strand = (String[])e.nextElement();
            if ( strand.length > max_length) {
                max_length = strand.length;
                choice = strand;
                choices.removeAllElements();
                choices.addElement(strand);
            }
            else if ( strand.length == max_length )
                choices.addElement(strand);
            
        }

        Random r = new Random(System.currentTimeMillis());
        int rand = r.nextInt(choices.size());
        
        String[] chosen = (String[])strands.elementAt(rand);
        log.printStatus("Chosen strand at index: " + rand + " " 
                        + TwineStrandSplitter.printOneStrand(chosen), 
			TwineLogger.TRACE_MSG);
        
        return chosen;

    }


    //-----------------------------------------------------
    /** 
     * Sends a query to other resolvers (early-binding query)
     * (Called from TwineAppMn)
     * @param seqNo New sequence number for this message
     * @param ns The query name-specifier
     * @param all Wheter all resources satisfying the query
     * are desired, or whether only one resource should be returned
     * @param nameSpecs set of strands extracted from the query
     * @author Magdalena Balazinska
     */
    public void earlyBinding (int seqNo, NameSpecifier ns, boolean all, Vector nameSpecs) {

	// construct data packet
	byte [] nsbytes = ns.toString().getBytes();
	byte [] bytes = new byte[nsbytes.length+11];

	Conversion.insertInt(bytes, 0, seqNo);
	System.arraycopy(rpcComm.localhost.getAddress(), 0, bytes, 4, 4);
	Conversion.insertIntLow16(bytes, 8, rpcComm.getRPCPort());
	bytes[10] = (byte)(all?1:0);
	System.arraycopy(nsbytes, 0, bytes, 11, nsbytes.length);

	Packet packet = new Packet(sourceNS, earlybindRequestNS, 8,
				   Packet.toAny, false, zerobyte,
				   bytes);

	sendQuery(nameSpecs,packet,seqNo);
    }

    //-----------------------------------------------------
    /** 
     * Called from AppMn to forward a late-binding query to other
     * resolvers
     *
     * @param seqNo	the sequence number for this query 
     * @param ns 	the name specifier being queried (this is packet.dNS
     * 			from the original late-binding packet that
     * 			arrives at this resolver)
     * @param all	boolean indicating if the query is anycast or multicast
     * @param nameSpecs the vector of strands of ns 
     * @author Kalpak Kothari
     */
    public void lateBinding (int seqNo, NameSpecifier ns, boolean all, Vector nameSpecs) {

	// construct data packet

	log.printStatus("TwineInterResolver::lateBinding() - ns="+ns+
			"\nnameSpecs="+nameSpecs+"\n", TwineLogger.TRACE_MSG);
	byte [] nsbytes = ns.toString().getBytes();
	byte [] bytes = new byte[nsbytes.length+11];

	Conversion.insertInt(bytes, 0, seqNo);
	System.arraycopy(rpcComm.localhost.getAddress(), 0, bytes, 4, 4);
	Conversion.insertIntLow16(bytes, 8, rpcComm.getRPCPort());
	bytes[10] = (byte)(all?1:0);
	System.arraycopy(nsbytes, 0, bytes, 11, nsbytes.length);

	Packet packet = new Packet(sourceNS, latebindRequestNS, 8,
				   all, false, zerobyte,
				   bytes);
	
	sendQuery(nameSpecs,packet,seqNo);
    }


    //-----------------------------------------------------
    /** 
     * Only once we're done computing the set of resolvers
     * that should resolve a query do we now the exact
     * number of resolvers in the set, and hence the
     * number of outstanding answers to expect.
     * @author Magdalena Balazinska
     */
    public void doneSending(int nbReqSent, int seqNo) {
	log.printStatus("Done sending " + nbReqSent + " seqNum " + seqNo,
			TwineLogger.TRACE_MSG);
	queryManager.addPending(seqNo,nbReqSent);
	if ( nbReqSent == 0 )
	    processIncomplete(seqNo);
    }

    /**
     * -----------------------------------------------------
     * Receive messages from other resolvers section
     * -----------------------------------------------------
     */


    //-----------------------------------------------------
    /** 
     *  A callback function: will be called every time a new
     *  INS packet for this handler arrives.
     *  (Interface implementation of IHandler)
     *
     *  @param msg Message from another resolver
     */
    public void INSreceive(Message msg) {

	log.printStatus("Received message through UDP!",TwineLogger.TRACE_MSG);

	boolean complete = true;
	AVelement av = msg.packet.dNS.getAVelement(controlmsgAttr);
	if (av != null) {
	    Value val = av.getValue();
	    if (val.equals(nameUpdateVal)) {
		processNameUpdate(msg.packet,msg.ia,msg.port);
	    } else if (val.equals(routeUpdateVal)) {
		log.printStatus("\nRoute updates not supported... dropping",
				TwineLogger.IMPORTANT_MSG);
	    } else if (val.equals(discoverRequestVal)) {
		processDiscoveryRequest(msg);
	    } else if (val.equals(earlybindRequestVal)) {
		processEarlyBindingRequest(msg);
	    } else if (val.equals(latebindRequestVal)) {
		processLateBindingRequest(msg);
	    } else if (val.equals(metricRequestVal)) {
		processMetricRequest(msg);
	    } else if (val.equals(discoverReplyVal)) {
		processDiscoveryReply(msg,complete);
	    } else if (val.equals(earlybindReplyVal)) {
		processEarlyBindingReply(msg,complete);
	    } else if (val.equals(latebindReplyVal)) {
		processLateBindingReply(msg,complete);
	    } else if (val.equals(metricReplyVal)) {
		processMetricReply(msg);
	    } else if (val.equals(discoverIncompleteReplyVal)) {
		processDiscoveryIncompleteReply(msg);
	    } else if (val.equals(earlybindIncompleteReplyVal)) {
		processEarlyBindingIncompleteReply(msg);
	    } else if (val.equals(latebindIncompleteReplyVal)) {
		processLateBindingIncompleteReply(msg);
	    } else if (val.equals(nameUpdateRejectVal)) {
		processNameUpdateReject(msg.packet,msg.ia,msg.port);
	    }
	}
	else {
            // actual late-binding payload packet
	    log.printStatus("\nReceived actual late-binding payload packet.", 
			    TwineLogger.IMPORTANT_MSG);
	    processLateBindingPayload(msg);
	}
    }

    //-----------------------------------------------------
    /** 
     *  A callback function: will be called every time a new
     *  INS packet for this handler arrives by TCP.
     *  (Interface implementation of IHandler)
     *  @param packet the INS packet
     *  @param from the neighbor sending the Packet packet     
     */
    public void INSreceiveByTCP(Packet packet, Node from)
    {
	log.printStatus("\nReceived a name-update through TCP:" 
			+ " Not supported... dropping",TwineLogger.IMPORTANT_MSG);

    }

    //-----------------------------------------------------
    /**
     * Returns true if it's time to refresh the advertisement
     * corresponding to this name-record.
     * @author Magdalena Balazinska
     */
    public boolean timeToRefresh(NameRecord nr) {

	TwineAdvertisement ad = adManager.getAdByNameRec(nr.getID());
	if ( ad == null )
	    return false;
	else return ad.getShouldBeRefreshed();
	
    }

    //-----------------------------------------------------
    /**
     * Processes an advertisement received from another
     * resolver.
     * @author Magdalena Balazinska
     */
    public void processNameUpdate(Packet packet, InetAddress fromAddr, int fromPort)
    {

	log.printStatus("\nReceived name update from InetAddr " 
			+ fromAddr + " port  " +fromPort,TwineLogger.INFO_MSG);

	NameUpdate nu = new NameUpdate(packet.data);
	NameRecord nr;

	if (nu.INRuid == resolver.INRuid) {
	    log.printStatus("WARNING: name update has my own INRuid, ignore it!",
			    TwineLogger.TRACE_MSG);
	    log.printStatus(nu.toString(),TwineLogger.TRACE_MSG);
	    return;
	}

	boolean storing = true;

	switch (nu.action) {

	case NameUpdate.ADD_NAME:
	case NameUpdate.UPDATE_NAME:

	    nr = nameTreeManip.getExactMatch(nu);
	    byte addOrUpdate = (nr == null) ? NameUpdate.ADD_NAME : 
		NameUpdate.UPDATE_NAME;

	    if ( nr == null ) {
		storing = keyMn.inc(packet.option);
		if ( storing ) {
		    nr = new NameRecord(nu, null,RouteManager.MAX_NAME_CORE_TTL);
		    nameTree.addNameRecord(nu.ns, nr);
		    keyMn.addNameRecord(nr,packet.option);
		}
		else 
		    reject(nu,fromAddr,fromPort);
	    }
	    else {
		if (nu.action == NameUpdate.ADD_NAME) {
		    log.printStatus("ASSERT: NameTree already has " + nu.ns.toString() 
				    + "...updating !", TwineLogger.INFO_MSG);
		}
		nr.update(nu, null, RouteManager.MAX_NAME_CORE_TTL);
	    }
	    break;

	    
	case NameUpdate.REMOVE_NAME: 
	   
	    storing = keyMn.dec(packet.option);
	    nr = nameTreeManip.getExactMatch(nu);
	    if ( nr == null ) {
		log.printStatus("Error! Trying to remove an unknown " 
				+ " resource: dropping",TwineLogger.IMPORTANT_MSG);
	    }
	    // Only the resolver that sent us the advertisement/update may ask
	    // to remove the resource
	    else if ( nr.getINRuid() == nu.INRuid ) {
		nameTree.removeNameRecord(nr);
	    }
	    else
		reject(nu,fromAddr,fromPort);
	    break;

	default: 
	    log.printStatus("WARNING: Unknown action in NameUpdate:",
			    TwineLogger.IMPORTANT_MSG);
	    log.printStatus(nu.toString(),TwineLogger.IMPORTANT_MSG);
	    
	} // switch
	
	log.printStatus(" Updated name tree: " + nameTree.toPrettyString(),
			TwineLogger.INFO_MSG);
    }

 
    //-----------------------------------------------------
    /**
     * Processing request received from another resolver
     * @author Magdalena Balazinska
     */
     public void processDiscoveryRequest(Message msg) {

	log.printStatus(" Processing discovery request ",TwineLogger.INFO_MSG);
	log.printStatus(" Resolving query: ",TwineLogger.IMPORTANT_MSG);

	Vector foundNSes = new Vector();
	if (msg.packet.data.length<11)
	    log.printStatus("\n Weird condition??? Dropping packet..",
			    TwineLogger.IMPORTANT_MSG);  
	else {	

	    // Get info from the message
	    int queryType = TwineQueryManager.DISCOVERY; // Type of query
	    TwineQuery query = queryManager.makeQueryFromPacket(msg.packet,queryType);
	    NameSpecifier ns = query.getNameSpec();
	   
	    // Do local lookup.
	    foundNSes = nameTreeManip.findNameSpecStrings(ns);

	    // Send a response
	    NameSpecifier destNS;
	    if ( keyMn.stillStoring(msg.packet.option))
		destNS = discoverReplyNS;
	    else 
		destNS = discoverIncompleteReplyNS;

	    sendDiscoveryResponse(foundNSes,query.getClientSeqNum(),
				  query.getAddr(),query.getPort(),
				  destNS);   
	}
	
    }

    //-----------------------------------------------------
    /**
     * Sending discovery response by RPC
     * @author Magdalena Balazinska
     */
    public void sendDiscoveryResponse(Vector nset, int sequenceNo,InetAddress addr, 
				      int port, NameSpecifier dest) {
	
	Packet packet = appMn.makeDiscoveryResponsePacket(nset,sequenceNo,
							  null,null,dest);
	if ( packet != null )
	    rpcComm.sendMessage(new Message(packet, addr, port));

    }



    //-----------------------------------------------------
    /**
     * Processing request received from another resolver
     * @author Magdalena Balazinska
     */
     public void processEarlyBindingRequest(Message msg) {

	log.printStatus(" Processing early binding request ",TwineLogger.INFO_MSG);
	log.printStatus(" Resolving a query: ",TwineLogger.IMPORTANT_MSG);

	Vector foundEBes = new Vector();
	if (msg.packet.data.length<11)
	    log.printStatus("\n Weird condition??? Dropping packet..",
			    TwineLogger.IMPORTANT_MSG);  
	else {	

	    // Get info from the message
	    int queryType = TwineQueryManager.EARLY_BINDING; // Type of query
	    TwineQuery query = queryManager.makeQueryFromPacket(msg.packet,queryType);
	    NameSpecifier ns = query.getNameSpec();
	    
	    // Do local lookup.
	    foundEBes = nameTreeManip.findHostRecords(ns);

	    // Send a response
	    NameSpecifier destNS;
	    if ( keyMn.stillStoring(msg.packet.option))
		destNS = earlybindReplyNS;
	    else 
		destNS = earlybindIncompleteReplyNS;

	    sendEarlyBindingResponse(foundEBes,query.getClientSeqNum(),
				     query.getAddr(),query.getPort(),
				     query.allResults(),destNS);   
	}

    }

    //-----------------------------------------------------
    /**
     * Send response by RPC
     * @author Magdalena Balazinska
     */
    public void sendEarlyBindingResponse(Vector rset, int sequenceNo, InetAddress addr, 
					 int port, boolean all, NameSpecifier dest) {
	
	Packet packet = appMn.makeEarlyBindingResponsePacket(rset,sequenceNo,
							     null,null,all,dest);
	if ( packet != null)
	    rpcComm.sendMessage(new Message(packet, addr, port));
	
    }

    //-----------------------------------------------------
    /**
     * Process late-binding request received from another resolver
     * @param msg Message from another resolver
     * @author Kalpak Kothari
     */
    public void processLateBindingRequest(Message msg) {

	log.printStatus("processLateBindingRequest enter",
			TwineLogger.INFO_MSG);

	// Get info from the message
	int queryType = TwineQueryManager.LATE_BINDING; // Type of query
	TwineQuery query = queryManager.makeQueryFromPacket(msg.packet,queryType);
	NameSpecifier ns = query.getNameSpec();

	// Do local lookup.
	Vector foundEBes = nameTreeManip.findNameRecords(ns);
	Vector foundLBes = new Vector();
	if(!msg.packet.all) {	// anycast case. keep only one (best) result.
	    int bestMetric=Integer.MIN_VALUE; NameRecord best = null;

	    for(Enumeration e = foundEBes.elements(); e.hasMoreElements(); ) {
		NameRecord nr = (NameRecord)e.nextElement();
		int thisMetric = nr.getAppMetric();
		if (thisMetric > bestMetric) {
		    best = nr; bestMetric = thisMetric;
		}
	    }
	    if(best != null) {
		TwineLBRecord tlbr = new TwineLBRecord(best.getAppMetric(), 
						       best.getINRuid(), 
						       tmm.getMetricNS(best.getINRuid()));
		foundLBes.addElement(tlbr);
	    }
	}
	else {	// multicast case. keep unique receivers.
	    Hashtable uniqueINRs = new Hashtable();
	    for(Enumeration e = foundEBes.elements(); e.hasMoreElements(); ) {
		NameRecord nr = (NameRecord)e.nextElement();
		long thisINRuid = nr.getINRuid();
		Long thisINRuidL = new Long(thisINRuid);
		if(!uniqueINRs.containsKey(thisINRuidL)) {
		    TwineLBRecord tlbr = new TwineLBRecord(nr.getAppMetric(), 
							   thisINRuid,
							   tmm.getMetricNS(thisINRuid));
		    uniqueINRs.put(thisINRuidL, tlbr);
		}
	    }

	    for(Enumeration e = uniqueINRs.elements(); e.hasMoreElements(); ) {
		foundLBes.addElement(e.nextElement());
	    }
	}

	// Send a response
	NameSpecifier destNS;
	if ( keyMn.stillStoring(msg.packet.option))
	    destNS = latebindReplyNS;
	else 
	    destNS = latebindIncompleteReplyNS;

	((TwineAppManager)appMn).
	    sendLateBindingResponse(foundLBes,query.getClientSeqNum(),
				    query.getAddr(),query.getPort(),
				    null,null,query.allResults(),destNS,null);
	log.printStatus("processLateBindingRequest exit",
			TwineLogger.INFO_MSG);

    }

    //-----------------------------------------------------
    /**
     * Processing answer received from another resolver
     * @param msg answer received from the other resolver
     * @param complete distinguishes between responses containing
     * a complete set of resources matching the query and those
     * giving only a subset of resources. This parameter is 
     * not used anymore.
     * @author Magdalena Balazinska
     */
    public void processDiscoveryReply(Message msg, boolean complete) {

	log.printStatus(" Processing discovery reply ",TwineLogger.INFO_MSG);
	 
	byte[] data = msg.packet.data;
	if (data.length<6)
	    return;

	int sequenceNo = Conversion.extract32toInt(data, 0);
	Vector result = new Vector();

	// Extract results received
	int pos = 4;	
	while (pos < data.length) {

	    int len = Conversion.extract16toIntLow16(data, pos);
	    if (pos+len+2>data.length || len==0) break;

	    String nsstr = new String(data, pos+2, len);
	    NameSpecifier ns = new NameSpecifier(nsstr);
	    result.addElement(ns.toString());
	    
	    pos+=(len+2);
	}

	// Add new results to pending query
	queryManager.addAnswer(result,sequenceNo);

    }

    //-----------------------------------------------------
    /**
     * Process incomplete answer received from another
     * resolver
     * @author Magdalena Balazinska
     */
    public void processDiscoveryIncompleteReply(Message msg) {

	// Bump the number of pending answers, so the queryManager
	// won't reply by itself
	int sequenceNo = Conversion.extract32toInt(msg.packet.data, 0);
	queryManager.addPending(sequenceNo,1);
	boolean complete = false;
	processDiscoveryReply(msg,complete);
	processIncomplete(sequenceNo);
    }

    //-----------------------------------------------------
    /**
     * When a resolver sends and incomplete result, a new
     * strand is chosen from the set of strands extracted
     * from the query. This new strand is then used
     * to select the new set of resolvers that should
     * resolve the query.
     * @author Magdalena Balazinska
     */
    protected void processIncomplete(int sequenceNo) {

	Vector strands = queryManager.getStrands(sequenceNo);
	
	log.printStatus("\nFurther processing incomplete...",
			TwineLogger.TRACE_MSG);
	
	// If there's no query pending, return
	// it means we're done or we've been called on an advertisement
	if ( strands == null ) {
	    log.printStatus("\nStrand vector is null... It's ok if this is an adv",
			    TwineLogger.TRACE_MSG);
	    return;
	}
	
	// Otherwise, check if there are any strands left
	if ( strands.isEmpty() ) {
	    log.printStatus("\nNo more strands... sending partial reply to client",
			    TwineLogger.TRACE_MSG);
	    queryManager.decPending(sequenceNo,discoverIncompleteReplyNS,
				    earlybindIncompleteReplyNS,
				    latebindIncompleteReplyNS);
	}
	// Try to resolve using antoher strand
	else {	
	    log.printStatus("\nSending for next strand",
			    TwineLogger.TRACE_MSG);
	
	    String[] strand = getLongestStrand(strands);
 	    strands.remove(strand);
 	    Packet packet = queryManager.getPacket(sequenceNo);
 	    strandMap.send(strand,TwineResolver.QUERY_D,
 			   (TwineMessage)queryManager.getMessage(sequenceNo));
	}
    }


    //-----------------------------------------------------
    /**
     * Processing answer received from another resolver
     * @author Magdalena Balazinska
     */
     public void processEarlyBindingReply(Message msg, boolean complete) {

	log.printStatus(" Processing early binding reply ",TwineLogger.IMPORTANT_MSG);
	byte[] data = msg.packet.data;
	if (data.length<10)
	    return;

	int sequenceNo = Conversion.extract32toInt(data, 0);
	Vector result = new Vector();
	
	int pos = 4;
	while (pos < data.length)
	{
	    int metric = Conversion.extract32toInt(data, pos);
	    int len = Conversion.extract16toIntLow16(data, pos+4);

	    if (pos+len+6>data.length || len==0) break;

	    byte [] hrbytes = new byte[len];
	    System.arraycopy(data, pos+6, hrbytes, 0, len);	    
	    result.addElement(new TwineEBRecord(metric, hrbytes));
	    pos+=(len+6);
	}

	// Add to query
	queryManager.addAnswer(result,sequenceNo);

    }

    //-----------------------------------------------------
    /**
     * Process late-binding reply received from another resolver
     * @param msg Message from another resolver
     * @param complete indicates whether all results were found
     * @author Kalpak Kothari
     */
    public void processLateBindingReply(Message msg, boolean complete) {

	log.printStatus("processLateBindingReply enter",
			TwineLogger.INFO_MSG);
	byte[] data = msg.packet.data;
	if (data.length<10)
	    return;

	boolean all = msg.packet.all;

	int sequenceNo = Conversion.extract32toInt(data, 0);
	Vector result = new Vector();

	int pos = 4;
	if(!all) { // anycast case
	    while(pos+12 <= data.length) {
		int metric = Conversion.extract32toInt(data, pos);
		long inruid = Conversion.extract64toLong(data, pos+4);
		result.addElement(new TwineLBRecord(metric, inruid, null));
		pos+=12;
	    }
	}
	else { // multicast case
	    while(pos+2 <= data.length) {
		int metriclen = Conversion.extract16toIntLow16(data, pos);
		if (pos+2+metriclen>data.length || metriclen==0) break;
		byte[] metricbytes = new byte[metriclen];
		System.arraycopy(data, pos+2, metricbytes, 0, metriclen);
		NameSpecifier metricNS = new NameSpecifier(new String(metricbytes));
		AVelement av = metricNS.getAVelement(tmm.INRuidAttr);
		long inruid = new Long(av.getValue().toString()).longValue();
		result.addElement(new TwineLBRecord(0, inruid, metricNS));
		pos += 2+metriclen;
	    }
	}

	// Add to query
	queryManager.addAnswer(result,sequenceNo);
	/**
	   if (complete) {
	   log.printStatus("complete! calling queryManager.sendResponse()",
	   TwineLogger.INFO_MSG);
	   
	   queryManager.sendResponse(sequenceNo);
	   }
	**/

	log.printStatus("processLateBindingReply exit",
			TwineLogger.INFO_MSG);

    }


    //-----------------------------------------------------
    /**
     * Process incomplete answer received from another
     * resolver.
     * @author Magdalena Balazinska
     */
    public void processEarlyBindingIncompleteReply(Message msg) {

	int sequenceNo = Conversion.extract32toInt(msg.packet.data, 0);
	queryManager.addPending(sequenceNo,1);
	boolean complete = false;
	processEarlyBindingReply(msg,complete);
	processIncomplete(sequenceNo);

    }

    //-----------------------------------------------------
    /**
     * Process incomplete late-binding reply received from another resolver
     * @param msg Message from another resolver
     * @author Kalpak Kothari
     */
    public void processLateBindingIncompleteReply(Message msg) {

	log.printStatus("processLateBindingIncompleteReply enter",
			TwineLogger.INFO_MSG);

	int sequenceNo = Conversion.extract32toInt(msg.packet.data, 0);
	queryManager.addPending(sequenceNo,1);
	boolean complete = false;
	processLateBindingReply(msg,complete);
	processIncomplete(sequenceNo);
	log.printStatus("processLateBindingIncompleteReply exit",
			TwineLogger.INFO_MSG);
	
    }


    //-----------------------------------------------------
    /**
     * If this resolver received too many advertisements for
     * a particular key, new advertisements are rejected.
     * (the resource information is not stored).
     * When this happens, an explicit message is sent back
     * to the resolver that sent the advertisement since 
     * it is possible that we should store the description
     * under another key as well. 
     * @author Magdalena Balazinska
     */
    public void reject(NameUpdate nu, InetAddress fromAddr, int fromPort) {

	log.printStatus("\nRejecting advertisement",TwineLogger.TRACE_MSG);
	Packet packet = new Packet(sourceNS, nameUpdateRejectNS, -1,
				   Packet.toAll, false, zerobyte,
				   nu.toBytes());
	Message msg = new Message(packet,fromAddr,fromPort);
	rpcComm.sendMessage(msg);

    }  

   //-----------------------------------------------------
    /**
     * Processing an advertisement rejection. When a
     * resolver rejects an advertisement, this resolver
     * checks whether the advertisement was supposed to
     * be sent under another key as well. The advertisement
     * might not be rejected for the other key.
     * @author Magdalena Balazinska
     */
    public void processNameUpdateReject(Packet packet, InetAddress fromAddr, 
					int fromPort) {

	log.printStatus("\nProcessing ad rejection",TwineLogger.TRACE_MSG);
	try {
	    NameUpdate nu = new NameUpdate(packet.data);
	    NameRecord nr = nameTreeManip.getExactMatch(nu);
	    TwineAdvertisement ad = adManager.getAdByNameRec(nr.getID());
	    KeyEntry ke = ad.getNextKeyForHost(fromAddr,fromPort);
	    if ( ke != null ) {
		log.printStatus("Trying to resend",TwineLogger.TRACE_MSG);
		strandMap.reSend(fromAddr,fromPort,ad.getPacket(),ke.getKey());
	    }
	} catch (Exception e) {
	    log.printStatus("Error processing adv rejection... dropping",
			    TwineLogger.KEY_MSG);
	}

    }

    //-----------------------------------------------------
    /**
     * Process metric request received from another resolver
     * @param msg Message from another resolver
     * @author Kalpak Kothari
     */
    public void processMetricRequest(Message msg) {
	log.printStatus("\nProcessing metric request",TwineLogger.TRACE_MSG);

	Packet p = msg.packet;

	if(p.data.length < 8) {	// packet must contain an inruid
	    log.printStatus("Dropping ill-formed message:\n" + msg,
			    TwineLogger.TRACE_MSG);
	}
	else {
	    long inruid = Conversion.extract64toLong(p.data, 0);
	    Hashtable metrics = tmm.getMetrics(inruid);
	    if(metrics == null) {
		log.printStatus("no metrics for inruid = " + inruid,
				TwineLogger.TRACE_MSG);
		return;
		
	    }

	    log.printStatus("Request for inruid = " + inruid,TwineLogger.TRACE_MSG);

	    // construct metric reply
	    NameSpecifier myReplyNS = tmm.getMetricNS(inruid);
	    // send metric reply
	    byte[] bytes = myReplyNS.toString().getBytes();
	    Packet packet = new Packet(sourceNS, metricReplyNS, 8,
				       Packet.toAny, false, zerobyte,
				       bytes);
	    rpcComm.sendMessage(new Message(packet, msg.ia, msg.port));
	}
    }

    //-----------------------------------------------------
    /**
     * Process metric reply received from another resolver
     * @param msg Message from another resolver
     * @author Kalpak Kothari
     */
    public void processMetricReply(Message msg) {
	 log.printStatus("Processing metric reply:",TwineLogger.TRACE_MSG);

	 Packet p = msg.packet;
	 String replyStr = new String(p.data);
	 NameSpecifier replyNS = new NameSpecifier(replyStr);
	 tmm.setMetricNS(replyNS);	// set new metric info in TMM
	 log.printStatus("Set Metric: " + replyNS.toString(),TwineLogger.TRACE_MSG);
    }

    //-----------------------------------------------------
    /**
     * Process late-binding payload received from another resolver
     * @param msg Message from another resolver
     * @author Kalpak Kothari
     */
    public void processLateBindingPayload(Message msg) {
	log.printStatus("\nProcessing incoming late-binding payload",
			TwineLogger.TRACE_MSG);

	Packet packet = msg.packet;

	// Do local lookup of destination NS
	Vector foundNRs = nameTreeManip.findNameRecords(packet.dNS);

	if(!packet.all) {
	    // anycast case, so size the set down to one max
	    int bestMetric=Integer.MIN_VALUE; NameRecord best = null;

	    for(Enumeration e = foundNRs.elements(); e.hasMoreElements(); ) {
		NameRecord thisItem = (NameRecord)e.nextElement();
		int thisMetric = thisItem.getAppMetric();
		if (thisMetric > bestMetric) {
		    best = thisItem; bestMetric = thisMetric;
		}
	    }
	    foundNRs.removeAllElements();
	    foundNRs.addElement(best);

	}

	// local payload delivery
	Packet localpacket = new Packet(packet);
	
	// remove all options. fix this! (actually, desired behavior is to only
	// remove multicast-forwarding related option
	localpacket.option = new byte[0]; 
	for(Enumeration e = foundNRs.elements(); e.hasMoreElements(); ) {
	    NameRecord nr = (NameRecord)e.nextElement();
	    long id = nr.getINRuid();
	    if(id == resolver.INRuid) {
		// send payload to this locally-connected destination
		HostRecord hr = nr.getHostRecord();
		InetAddress destaddr = hr.getInetAddress();
		int destport = hr.UDPport;

		//fix this! put proper log output.
		//log.printStatus("Delivering late-binding payload to local destination - " +
		log.printStatus("Local Dest Send -> " +
				destaddr + ":" + destport, TwineLogger.IMPORTANT_MSG);


		comm.sendMessage(new Message(localpacket, destaddr, destport));
	    }
	}

	// handle multicast case when we are a parent of a cluster

	if(packet.all && (packet.option.length > 0)) { 
	    if(packet.option[0] != 2) { // fix this!! add "kind" in Packet.java
		return;
	    }
	    byte[] option = packet.option;
	    int pos=1; // current index in option byte array
	    int len = Conversion.extract16toIntLow16(option, pos);
	    pos+=2;

	    // for testing only: extract sequenceNo, clusterLevel, increment it.
	    // fix this!
	    int inSequenceNo = Conversion.extract32toInt(option,pos);
	    int clusterLevel = Conversion.extract16toIntLow16(option, pos+4);
	    pos+=6;

	    // construct receivers Hashtable so we can form new clusters
	    Hashtable h1 = new Hashtable();
	    Hashtable receivers = new Hashtable();

	    while(pos+2 <= len) {
		int childlen = Conversion.extract16toIntLow16(option, pos);
		if (pos+2+childlen>len || childlen==0) break;
		
		byte[] childNSbytes = new byte[childlen];
		System.arraycopy(option, pos+2, childNSbytes, 0, childlen);
		pos += 2+childlen;
		NameSpecifier childMetricNS = new NameSpecifier(new String(childNSbytes));
		Vector childMetricV = tmm.metricNStoHashtable(childMetricNS);
		if(((Long)childMetricV.elementAt(0)).equals(myINRuid)) {
		    log.printStatus("weird: we are our own child.", TwineLogger.IMPORTANT_MSG);
		    continue;
		}
		receivers.put(childMetricV.elementAt(0), 
			      childMetricV.elementAt(1));

	    }
	    //////////////
	    h1.put(myINRuid, receivers);
	    //log.printStatus("inPayload: H input=" + h1, TwineLogger.IMPORTANT_MSG);
	    
	    log.printStatus("MCAST I " + inSequenceNo + 
			    " " + (++clusterLevel) +
			    " " + h1, TwineLogger.IMPORTANT_MSG);

	    // perform clustering of receivers
	    TwineClusterer tc = new TwineClusterer((TwineResolver)resolver, 
						   h1, true, 1); // change to true
	    Hashtable clusters = tc.formClusters(true); // set true later (useMetric)
	    //log.printStatus("inPayload: H output=" + clusters, TwineLogger.IMPORTANT_MSG);

	    log.printStatus("MCAST O " + inSequenceNo + 
			    " " + clusterLevel +
			    " " + clusters, TwineLogger.IMPORTANT_MSG);
	    
	    // for each parent, construct packet and send
	    for(Enumeration e = clusters.keys(); e.hasMoreElements(); ) {

		// get the payload
		Packet newpacket = new Packet(packet); // make new packet
		byte[] packetbytes = newpacket.toBytes();
		int optionmaxlen = (Packet.MAX_PACKET_SIZE - 20 -
				    newpacket.sNS.toString().length() - 
				    newpacket.dNS.toString().length() -
				    newpacket.data.length);
		byte[] tempoptions = new byte[optionmaxlen];
		byte multicast_forward_kind = 2;
		tempoptions[0] = multicast_forward_kind;
		/// remember to fill optionlen in tempoptions[1,2]
		Conversion.insertInt(tempoptions, 3, inSequenceNo); // fix this! testing only
		Conversion.insertIntLow16(tempoptions, 7, clusterLevel); // fix this! testing only
		// current index in option byte array
		int ct=9; 	// fix this! was 3 (remove sequenceNo,
		    		// and level after testing )

		Long parentINRuid = (Long)e.nextElement();

		// for testing only. fix this!!
		// MCAST OUT seqNum myINRuid curClusterLevel nextParentuid
		log.printStatus("MCAST OUT " + inSequenceNo + 
				" " + myINRuid +
				" " + clusterLevel +
				" " + parentINRuid, TwineLogger.IMPORTANT_MSG);

		Hashtable children = (Hashtable)clusters.get(parentINRuid);
		for(Enumeration f = children.keys(); f.hasMoreElements(); ) {
		    Long childINRuid = (Long)f.nextElement();
		    Hashtable childMetrics = (Hashtable)children.get(childINRuid);
		    NameSpecifier childMetricNS = tmm.metricHTtoNS(childINRuid, childMetrics);
		    byte[] childMetricbytes = childMetricNS.toString().getBytes();
		    int metriclen = childMetricbytes.length;
			
		    if(ct+metriclen > optionmaxlen) {
			log.printStatus("WARNING: option field for multicast packet "+
					"has exceeded its max length",
					TwineLogger.IMPORTANT_MSG);
			log.printStatus(".. only " + ct + " option bytes will be sent",
					TwineLogger.IMPORTANT_MSG);
			break;
		    }
		    else {
			Conversion.insertIntLow16(tempoptions, ct, metriclen);
			System.arraycopy(childMetricbytes, 0, tempoptions, ct+2, metriclen);
			ct += 2+metriclen; // space for len + actual length
		    }
		}
		// insert total option length
		Conversion.insertIntLow16(tempoptions, 1, ct);

		// construct final options byte array
		byte[] newoption = new byte[ct];
		System.arraycopy(tempoptions, 0, newoption, 0, ct);
		// set new options in packet
		newpacket.option = newoption;

		// send payload off to corresponding resolver
		long parentuid = parentINRuid.longValue();
		InetAddress destaddr = forwarder.extractAddrFromINRuid(parentuid);
		int destport = forwarder.extractPortFromINRuid(parentuid);
		    
		/// fix problem with options. should maintain all
		// other options from original packet.
		// currently, it is creating a completely new option header.

		// indicate packet being sent from resolver and not application
		newpacket.fromApp = false;
		    
		log.printStatus("MULTICAST: destination resolver ->" +
				destaddr + " : " + destport,
				TwineLogger.IMPORTANT_MSG);
		    
		rpcComm.sendMessage(new Message(newpacket, destaddr, destport));

		////
	    }
	}
	
    }
    
}


    
