package ins.inr;

import ins.namespace.*;
import java.util.*;
import cryptix.util.core.Hex; // For debugging

/**
 * TwineKeyManager.java <br>
 *
 * Allows a resolver to keep track of the number of 
 * resources received under different keys.
 * <br>
 * When more resources are known than a predetermined
 * threshold, further advertisements received under the
 * same key are rejected.
 * <br>
 * Created: Wed Oct 3 2001 <br>
 * Modified: $Id: TwineKeyManager.java,v 1.7 2002/03/21 00:01:55 mbalazin Exp $
 * @author Magdalena Balazinska
 */

public class TwineKeyManager {

    public static int INITIAL_SIZE = 2000;
    public static int MAX_INSERTS_PERKEY = 1000;
    Hashtable keys = new Hashtable(INITIAL_SIZE);
    Hashtable namerecords = new Hashtable(INITIAL_SIZE);
    protected TwineLogger log; 
    protected TwineStats stats;

    public TwineKeyManager() {
	log = new TwineLogger("KeyManager");	
    }

    public void init(Resolver r) {
	stats = ((TwineResolver)r).stats;
    }
    
    public synchronized boolean inc(byte[] key) {

	Integer n = getHashCode(key);
	TwineKeyElement element = (TwineKeyElement)keys.get(n);
	if ( element == null ) {
	    element = new TwineKeyElement(key,log);
	    keys.put(n,element);
	}
	else if ( !element.storing())
	    log.printStatus("Not storing that key",TwineLogger.TRACE_MSG);
	else if ( element.inc() > MAX_INSERTS_PERKEY ) {
	    element.stopStoring();
	    stats.stopStoringOneKey();
	    element.dec(); // now we have exactly MAX_INSERTS_PERKEY
	}
	return element.storing();

    }


    public synchronized boolean dec(byte[] key) {

	Integer n = getHashCode(key);
	TwineKeyElement element = (TwineKeyElement)keys.get(n);
	if ( ( element != null ) && ( element.dec() < MAX_INSERTS_PERKEY ))
	    element.startStoring();

	return element.storing();
    }

    public synchronized boolean stillStoring(byte[] key) {
	
	Integer n = getHashCode(key);
	TwineKeyElement element = (TwineKeyElement)keys.get(n);
	if ( element == null || element.storing())
	    return true;
	else return false;

    }

    public synchronized TwineKeyManager getElement(byte[] key) {
	Integer n = getHashCode(key);
	return (TwineKeyManager)keys.get(n);
    }

    public synchronized void addNameRecord(NameRecord nr, byte[] key) {

	Integer id = new Integer (nr.getID());
	NameRecord old = (NameRecord)namerecords.get(id);
	if ( old == null ) {
	    namerecords.put(id,key);
	}

    }

    public synchronized void removeNameRecord(NameRecord nr) {

	byte[] value = (byte[])namerecords.get(new Integer(nr.getID()));
	if ( value != null ) {
	    namerecords.remove(nr);
	    dec(value);
	}
    }
    
    public Integer getHashCode(byte[] key) {

	if ( key == null)
	    return null;

	// Using the last 4 bytes as the integer hashcode
	return new Integer(Conversion.extract32toInt(key,key.length-4-1));

    }
}

//-----------------------------------------------------
/**
 *
 */
class TwineKeyElement {

    int counter;
    byte[] key;
    boolean store;
    String stringKey;
    TwineLogger log;

    public TwineKeyElement(byte[] theKey, TwineLogger logger) {

	counter  = 1;
	key = theKey;
	store = true;
	log = logger;

	// For debugging
	stringKey = cryptix.util.core.Hex.toString(key);
	log.printStatus("\n Start storing key " + stringKey,TwineLogger.TRACE_MSG);
    }

    public int inc() {
	counter++;
	return counter;
    }
    
    public int dec() {
	counter--;
	return counter;
    }

    public boolean storing() {
	return store;
    }

    public void startStoring() {
	log.printStatus("\n Continue or re-start storing key " + stringKey,
			TwineLogger.TRACE_MSG);
	store = true;
    }

    public void stopStoring() {
	log.printStatus("\n Stop storing key " + stringKey,
			TwineLogger.TRACE_MSG);
	store = false;
    }


}
