package ins.inr;

import java.util.*;

/** This class estimates the metrics for a specific resolver.
    The metrics include:<br>
    1. "position"<br>
    2. "load" (bits per second of messages going in and out)<br>
    3. "ave_pkt_size" (average packet size in bits)<br>
    4. "branch_number" (the number of resolvers it will branch a message to
       from a multicast source). That number is per multicast source.<br>
    5. "multicast_source_number" (the number of multicast sources
       sending to this resolver)<br>
    6. "capacity" (the link capacity in bits per second: 56,000 ,
       1,000,000 , 10,000,000 , 100,000,000 for example)<br>
 
   @author Nikos Michalakis
   @author Kalpak Kothari
   @version 1.0
*/

class TwineMetricEstimator extends TimerTask{
  
  TwineResolver r;
  Timer t;

  TwineLogger log; // added by kalpak

  NodeSet pingStations;
  public static final long TIME_INTERVAL=2000; //2 seconds
  public static final int PING_ROUNDS=30; //5;  //number of rounds to wait for 2nd task
  public static final float ALPHA=(float)0.2; //max percent of capacity for multicast
  //TODO:make this number a function of capacity
  public static final int max_branches=5; // was 10. made 5 for testing. fix this!
  //time to live for multicast source entries. Entries expire after TTL*ROUNDS*TIME_INTERVAL
  //milliseconds
  public static final int TTL_ROUNDS=15;
  public static final int TTL=1;
 
  boolean estimateMetrics;
  int currentPINGRound;
  int currentTTLRound;
  Hashtable msources;
  //metrics
  Vector position;
  int load;
  int ave_pkt_size;
  int branch_number;
  int multicast_source_number;
  int capacity;
  
  public TwineMetricEstimator(){
    this.estimateMetrics=true;
    position=new Vector();
    load=0;
    ave_pkt_size=0;
    branch_number=0;
    multicast_source_number=0;
    capacity=0;
    currentPINGRound=0;
    currentTTLRound=0;
    msources=new Hashtable();
    log = new TwineLogger("TME","TwineMetricEstimator.log"); // kalpak
  }
  
  public TwineMetricEstimator(boolean estimateMetrics){
    this.estimateMetrics=estimateMetrics;
    position=new Vector();
    load=0;
    ave_pkt_size=0;
    branch_number=0;
    multicast_source_number=0;
    capacity=0;
    currentPINGRound=0;
    currentTTLRound=0;
    msources=new Hashtable();
    log = new TwineLogger("TME","TwineMetricEstimator.log"); // kalpak
  }
 public TwineMetricEstimator(boolean estimateMetrics,NodeSet pstations){
    this.estimateMetrics=estimateMetrics;
    position=new Vector();
    load=0;
    ave_pkt_size=0;
    branch_number=0;
    multicast_source_number=0;
    capacity=0;
    currentPINGRound=0;
    currentTTLRound=0;
    msources=new Hashtable();
    this.pingStations=pstations;
    log = new TwineLogger("TME","TwineMetricEstimator.log"); // kalpak
  }

  public TwineMetricEstimator( Vector position,
			      int load, int ave_pkt_size, int branch_number,
			      int multicast_source_number, int capacity,
			       boolean estimateMetrics,NodeSet pstations){

    this.position=position;
    this.load=load;
    this.ave_pkt_size=ave_pkt_size;
    this.branch_number=branch_number;
    this.multicast_source_number=multicast_source_number;
    this.capacity=capacity;
    this.estimateMetrics=estimateMetrics;
    this.pingStations=pstations;
    currentPINGRound=0;
    currentTTLRound=0;
    msources=new Hashtable();
    log = new TwineLogger("TME","TwineMetricEstimator.log"); // kalpak
  }

  public void init(TwineResolver r){
    this.branch_number=max_branches; // fix this! is this a bug?
    this.r=r;
    this.t=r.timer;
    t.scheduleAtFixedRate(this,TIME_INTERVAL,TIME_INTERVAL);
    this.setCapacity();
}
  public void run(){
    if(estimateMetrics){

      float alpha=ALPHA;
      long interval=TIME_INTERVAL;
      //      int rounds=ROUNDS;

      //the short cycle actions: (every TIME_INTERVAL ms)

      //set the load in bits per second
      int l=(int) ((((float) 1000)/((float)interval))
		 *8*((float) r.comm.getBytesSent()));
      setLoad(l);
      //set the average packet size in bits
      int p=(int)(8*((float) r.comm.getBytesSent())/
		  ((float) r.comm.getPacketsSent()));
      setPacketSize(p);
      //set the multicast source number
      setMulticastSourceNumber(getLiveMulticastSources());

      //set the branch number
      //int b=(int) (alpha*((float)getCapacity())/((float)getPacketSize())/
      //       ((float) getMulticastSourceNumber()));
      //System.out.println("Capacity= "+getCapacity());
      //	System.out.println("Packet size= "+getPacketSize());
      //System.out.println("Source number= "+getMulticastSourceNumber());
      //System.out.println("setting branch number="+b);
      int b=(int) ((float)(alpha* ((float)getCapacity())-((float)getLoad())-
		   ((float) getPacketSize())*((float)getMulticastSourceNumber()))/
	((float) getPacketSize()));
      //	if(b>max_branches) b=max_branches;
      //else if(b<1) b=1;
	  
	setBranchNumber(b);
	  
	//the long cycle actions: (every TIME_INTERVAL*ROUNDS ms)
	if(currentPINGRound >=PING_ROUNDS){
	  currentPINGRound=0;
	  setPosition();
	}

	if(currentTTLRound >=TTL_ROUNDS){
	  currentTTLRound=0;
	  decreaseMulticastTTL();
	}
	currentPINGRound++;
	currentTTLRound++;
    }

    // log all metrics to file
    printAllMetrics();

    //necessary to reset the value in the Communicator so it doesn't grow
    //huge.
    r.comm.resetBytesSent();
  }

    //modifiers

  /** Sets the position metric and does the same for TMM
      @param position The position to set to
  **/
  public void setPosition(Vector position){
      //      log.printStatus("setPosition:" + position, TwineLogger.TRACE_MSG);
    this.position=position;
    r.tmm.setPosition(r.INRuid,this.position);
    
  }

  /** Sets the position metric using PositionThread which calls
      setPosition(Vector) with the vector it calculates
  **/
  public void setPosition(){
    (new PositionThread()).start();
  }

  /** Sets the load metric and does the same for TMM
      @param load The load to set to 
  **/
  public void setLoad(int load){
    this.load=load;
    r.tmm.setLoad(r.INRuid,this.load);
  }

  /** Sets the average packet size  and does the same for TMM
      @param packet_size The average packet size to set to
  **/
  public void setPacketSize(int packet_size){
    if(packet_size<1) packet_size=1;
    else this.ave_pkt_size=packet_size;
    r.tmm.setPacketSize(r.INRuid,this.ave_pkt_size);
  }

  /** Sets the number of branches from this resolver
       and does the same for TMM
      @param The number of branches
  **/
  public void setBranchNumber(int branches){
    if(branches<1) this.branch_number=1;
    else if (branches>max_branches) this.branch_number=max_branches;
    else this.branch_number=branches;
    r.tmm.setBranchNumber(r.INRuid,this.branch_number);
  }
  
  /** Sets the number of multicast sources that send to this resolver
       and does the same for TMM
      @param sourceNumber
  **/
  public void setMulticastSourceNumber(int sourceNumber){
    this.multicast_source_number=sourceNumber;
    r.tmm.setMulticastNumber(r.INRuid,multicast_source_number);
  }

  /** Sets the capacity of the link the resolver is using
      and does the same for TMM
      @param capacity
  **/
  public void setCapacity(int capacity){
    this.capacity=capacity;
    r.tmm.setCapacity(r.INRuid,this.capacity);
  }

  /** Sets the capacity of the link the resolver is using
      by calling getCapacity() from the TwineMetricManager
  **/
  public void setCapacity(){
    this.capacity=r.tmm.getCapacity(r.INRuid);
  }

  /** Sets the ping station nodes so that we
      know where to calculate coordinates from.
      @param pingStations A NodeSet of resolvers we will ping
  **/
  public void setPingStations(NodeSet pingStations){
    this.pingStations=pingStations;
  }

  //observers

  /** Gets the position metric
      @return position
  **/
  public Vector getPosition(){
    return this.position;
  }

  /** Gets the load in bps
      @return load 
  **/
  public int getLoad(){
    return this.load;
  }

  /** Gets the average packet size in bits 
      @return ave_pkt_size
  **/
  public int getPacketSize(){
    return this.ave_pkt_size;
  }

  /** Gets the number of branches from this resolver
      @return branch_number
  **/
  public int getBranchNumber(){
    return this.branch_number;
  }
  
  /** Gets the number of multicast sources that send to this resolver
      @return multicast_source_number
  **/
  public int getMulticastSourceNumber(){
    return this.multicast_source_number;
  }

  /** Gets the capacity of the link the resolver is using
      @return capacity
  **/
  public int getCapacity(){
    return this.capacity;
  }

  /** Gets the number of ping station nodes
      @return pingStations The size of a NodeSet of resolvers
  **/
  public int getPingStationsNumber(){
    if(pingStations!=null)
    return this.pingStations.countNodes();
    else return 3; // 0;
  }

  //special purpose functions:

  /** Decreases the multicast source entry TLL by 1
      until it reaches 0.
  **/
  public void decreaseMulticastTTL(){
    for(Enumeration skeys=msources.keys();skeys.hasMoreElements();){
      Long skey=(Long) skeys.nextElement();
      int ttl=((Integer)msources.get(skey)).intValue();
      if(ttl>0) ttl--;
      else msources.remove(skey);
      msources.put(skey,new Integer(ttl));
    }
  }

  /** Inserts a new source to the list of multicast sources with
      ttl=TTL. If the source is already in the list then it only
      updates ttl=TTL
      @param uid The uid of the multicast source to be updated
  **/
  public void updateMulticastSources(long uid){
    this.msources.put(new Long(uid),new Integer(TTL));
  }
  
  /** Returns the number of live multicast sources, i.e.
      those that have ttl>0.
  **/
  public int getLiveMulticastSources(){
    int i=0;
    for(Enumeration mkeys=msources.keys();mkeys.hasMoreElements();){
      if(((Integer)this.msources.get(mkeys.nextElement())).intValue()>0)
	i++;
    }
    return i;
  }

  /** Print all metrics to log file (for testing purposes)
  **/
  public void printAllMetrics(){
      long time = System.currentTimeMillis();
      
      Vector mypos = getPosition();
      StringBuffer myposStr = new StringBuffer();
      if((mypos != null) && (mypos.size()>0)) {
	  for(Enumeration e = mypos.elements(); e.hasMoreElements(); )
	      myposStr.append(e.nextElement() + ":");
	  myposStr = myposStr.deleteCharAt(myposStr.length() - 1); // get rid of end ":"
      }
      else myposStr.append("null");       // default (error)
      
      log.printStatus(time + " " + 
		      getLoad() + " " + 
		      getCapacity() + " " + 
		      myposStr + " " + 
		      getBranchNumber() + " " + 
		      getPacketSize() + " " + 
		      getMulticastSourceNumber(),
		      TwineLogger.IMPORTANT_MSG);
  }


  /** This class is used for pinging the stations and setting the position
      for the TwineMetricEstimator.
  **/
  class PositionThread extends Thread{
    public void run(){
     setPosition(this.findPosition());
    }
	
    /** Find the position of this resolver by pinging the stations
	@return The vector that contains the RTTs from all stations
    **/
    public Vector findPosition(){
      Vector v=new Vector();
      //      log.printStatus("enter findPosition", TwineLogger.TRACE_MSG);

      if(pingStations != null) {
	  //	  log.printStatus("pingStations not null", TwineLogger.TRACE_MSG);
	  for(Enumeration stations=pingStations.getNodes();stations.hasMoreElements();){
	      Node station=(Node)stations.nextElement();
	      //	      log.printStatus("pinging:" + station.toString(), TwineLogger.TRACE_MSG);
	      //	      String vspace=r.FAKE_VSPACE; // (String)((Enumeration)station.getVspaces()).nextElement();
	      //ping station
	      r.prober.determineRTT(station,r.FAKE_VSPACE);
	      v.addElement(new Integer(station.rtt));
	      //	      log.printStatus("findPosition rtt:" + station.rtt, TwineLogger.TRACE_MSG);
	  }
      }
      return v;
    }
    
  }
}//end TwineMetricEstimator
