package ins.inr;

import ins.namespace.*;
import java.util.*;
import java.net.*;
import java.util.Timer;
import java.util.TimerTask;

/**
 * TwineMetricManager.java
 *
 * This class periodically collects metric information of resolvers
 * whose applications' information this resolver is storing.
 * <br><br>
 * Created: Wed Nov 21 02:25:02 EST 2001
 * $Id: TwineMetricManager.java,v 1.3 2002/03/18 19:43:16 kalpak Exp $
 * @author Kalpak Kothari
 * @version 1.0
 */

public class TwineMetricManager extends TimerTask {
  
    public static int DEFAULT_SIZE = 50;
    public static final long REFRESH_INTERVAL = 30000;	// 30 seconds
  
    /* constants that define keys for the metrics Hashtable */
    public static final String loadKey = "load";
    public static final String capacityKey = "capacity";
    public static final String positionKey = "position";
    //added by nikos
    public static final String branchKey = "branch_number";
    public static final String packetSizeKey = "ave_pkt_size";
    public static final String multicastKey = "multicast_source_number";

    /* constants that define attributes for metric name specifiers */
    static final Attribute INRuidAttr = new Attribute("INRuid");
    static final Attribute loadAttr = new Attribute(loadKey);
    static final Attribute capacityAttr = new Attribute(capacityKey);
    static final Attribute positionAttr = new Attribute(positionKey);
    static final Attribute branchAttr = new Attribute(branchKey);
    static final Attribute packetSizeAttr = new Attribute(packetSizeKey);
    static final Attribute multicastAttr = new Attribute(multicastKey);

    protected Hashtable INRinfo;
  
    protected TwineLogger log;
    protected Resolver resolver;
    protected Forwarder forwarder;
    protected Communicator comm;
    protected RPCCommunicator rpcComm;
    protected NameStoreInterface nameTree;
    protected long myINRuid;
    final static byte[] zerobyte = new byte[0];
    protected NameSpecifier sourceNS, metricRequestNS;
  
    protected Timer timer;
  
    public TwineMetricManager() {
	log = new TwineLogger("MetricManager","TwineMetricManager.log");
    }
  
    public void init(Resolver r) {
	resolver = r;
	forwarder = r.forwarder;
	comm = r.comm;
	rpcComm = ((TwineResolver)r).rpcComm;
	nameTree = ((TwineResolver)r).nameTree;
	myINRuid = r.INRuid;
	sourceNS = ((TwineInterResolver)r.routeMn).sourceNS;
	metricRequestNS = ((TwineInterResolver)r.routeMn).metricRequestNS;

	INRinfo = new Hashtable(DEFAULT_SIZE);

	//initial metrics
	Hashtable temp = new Hashtable();
	temp.put(loadKey, new Integer(0));
	temp.put(capacityKey, new Integer(10*1024*1024));
	temp.put(branchKey, new Integer(5)); // was 10. fix this!
	temp.put(packetSizeKey, new Integer(0));
	temp.put(multicastKey, new Integer(0));
	Vector v = new Vector();
	temp.put(positionKey, v);
	INRinfo.put(new Long(myINRuid), temp);
	// Schedule periodic metric collection 
	timer = ((TwineResolver)r).timer;
	timer.scheduleAtFixedRate(this,REFRESH_INTERVAL,REFRESH_INTERVAL);
    }
  
    public void run() {
	// do stuff
	Vector INRs = new Vector();
    
	// do local NameTree lookup and pick all INRuids
	for(Enumeration e = nameTree.getNameRecords(); e.hasMoreElements(); ) {
	    
	    NameRecord nr = (NameRecord)e.nextElement();
	    long id = nr.getINRuid();
	    if(id == 0)
		continue;
	    Long lid = new Long(id);
	    if((id != myINRuid) && !INRs.contains(lid)) {
		INRs.addElement(lid);
	    }
	}
    
	for(Enumeration e = INRs.elements(); e.hasMoreElements(); ) {
	    // send metric request message to relevant inr's
	    long inruid = ((Long)e.nextElement()).longValue();
	    sendMetricRequest(inruid);
	    log.printStatus("request: INRuid=" + inruid + "\n",TwineLogger.TRACE_MSG);
	}

    }


    /** Send a metric request control message to a resolver
	@param inruid The remote resolver UID
     */
    public void sendMetricRequest(long inruid) {

	// construct data packet
	byte [] bytes = new byte[8];
	Conversion.insertLong(bytes, 0, inruid); // insert dest. inruid

	Packet packet = new Packet(sourceNS, metricRequestNS, 8,
				   Packet.toAny, false, zerobyte,
				   bytes);
	
	// extract addr and port from inruid
	InetAddress destaddr = forwarder.extractAddrFromINRuid(inruid);
	int destport = forwarder.extractPortFromINRuid(inruid);
	
	// send request
	rpcComm.sendMessage(new Message(packet, destaddr, destport));
    }

  
    /****
	 Methods to set various metrics
    ****/
  
    /** Set all metrics at once
	@param inruid The Resolver UID
	@param m The hastable of metrics
    **/
    public void setMetrics(long inruid, Hashtable m) {
	INRinfo.put(new Long(inruid), m);
    }

    /** Set all metrics at once
	@param inruid The Resolver UID
	@param l load
	@param c capacity
	@param b branching number
	@param s packet size
	@param m multicast source number
	@param p position vector
    **/
    public void setMetrics(Long inruid, Integer l, Integer c, Integer b, 
			   Integer s, Integer m, Vector p) {
	Hashtable h = new Hashtable();
	h.put(loadKey, l);
	h.put(capacityKey, c);
	h.put(branchKey, b);
	h.put(packetSizeKey, s);
	h.put(multicastKey, m);
	h.put(positionKey, p);
	INRinfo.put(inruid, h);
    }
  
    /** Set Load
	@param inruid The Resolver UID
	@param load The new load
    **/
    public void setLoad(long inruid, int load) {
	Hashtable metrics = (Hashtable)INRinfo.get(new Long(inruid));
	metrics.put(loadKey, new Integer(load));
    }
  
    /** Set Capacity
	@param inruid The Resolver UID
	@param capacity The new capacity
    **/
    public void setCapacity(long inruid, int capacity) {
	Hashtable metrics = (Hashtable)INRinfo.get(new Long(inruid));
	metrics.put(capacityKey, new Integer(capacity));
    }
  
    /** Set Position Coordinates
	@param inruid The Resolver UID
	@param pos The vector containing position co-ordinates
    **/
    public void setPosition(long inruid, Vector pos) {
	Hashtable metrics = (Hashtable)INRinfo.get(new Long(inruid));
	metrics.put(positionKey, pos);
    }

    //nikos
    /** Set Number of branches to immediate children that will be parents for clusters
	@param inruid The Resolver UID
	@param branches The number of branches
    **/
    public void setBranchNumber(long inruid, int branches) {
	Hashtable metrics = (Hashtable)INRinfo.get(new Long(inruid));
	metrics.put(branchKey, new Integer(branches));
    }

    //nikos
    /** Set the packet size the resolver is seeing at the moment (sliding average)
	@param inruid The Resolver UID
	@param pkt_size The packet size
    **/
    public void setPacketSize(long inruid, int pkt_size) {
	Hashtable metrics = (Hashtable)INRinfo.get(new Long(inruid));
	metrics.put(packetSizeKey, new Integer(pkt_size));
    }

    //nikos
    /** Set The number of multicast sources using this resolver at the moment
	@param inruid The Resolver UID
	@param m_sources The number o multicast sources sending packets to this resolver
    **/
    public void setMulticastNumber(long inruid, int m_sources) {
	Hashtable metrics = (Hashtable)INRinfo.get(new Long(inruid));
	metrics.put(multicastKey, new Integer(m_sources));
    }
  
    /**
       Call this function to force immediate information collection
    */
    public void runNow() {
	run();
    }
  
    /**** 
	  Methods to get various metrics
    ****/
  
    /** Get all metrics at once
	@param inruid The Resolver UID
	@return metrics hashtable
    **/
    public Hashtable getMetrics(long inruid) {
	return (Hashtable)INRinfo.get(new Long(inruid));
    }
  
    /** Get load
     * @return Load if given inruid exists
     * <br>
     * -1 if error
     **/
    public int getLoad(long inruid) {
	int l = -1;
	try {
	    l = ((Integer)((Hashtable)INRinfo.get(new Long(inruid))).get(loadKey)).intValue(); 
	} catch(NullPointerException e) {
	    log.printStatus("getLoad(), Unknown INRuid:" + inruid, 
			    TwineLogger.TRACE_MSG);
	}
    
	return l;
    }
  
    /** Get capacity
     * @return Capacity if given inruid exists
     * <br>
     * -1 if error
     **/
    public int getCapacity(long inruid) {
	int c = -1;
	try {
	    c = ((Integer)((Hashtable)INRinfo.get(new Long(inruid))).get(capacityKey)).intValue();
	} catch(NullPointerException e) {
	    log.printStatus("getCapacity(), Unknown INRuid:" + inruid, 
			    TwineLogger.TRACE_MSG);
	}
	return c;
    }
  
    /** Get position vector
     * @return Position Vector if given inruid exists
     * <br>
     * null if error
     **/
    public Vector getPosition(long inruid) {
	Vector v = null;
	try {
	    v = (Vector)((Hashtable)INRinfo.get(new Long(inruid))).get(positionKey);
	} catch(NullPointerException e) {
	    log.printStatus("getPosition(), Unknown INRuid:" + inruid, 
			    TwineLogger.TRACE_MSG);
	}
	return v;
    }

    //nikos
    /** Get branching number
     * @return number of branches if given inruid exists
     * <br>
     * -1 if error
     **/
    public int getBranchNumber(long inruid) {
	int c = -1;
	try {
	    c = ((Integer)((Hashtable)INRinfo.get(new Long(inruid))).get(branchKey)).intValue();
	} catch(NullPointerException e) {
	    log.printStatus("getBranchNumber(), Unknown INRuid:" + inruid, 
			    TwineLogger.TRACE_MSG);
	}
	return c;
    }

    //nikos
    /** Get packet size
     * @return packet size the resolver sees if given inruid exists
     * <br>
     * -1 if error
     **/
    public int getPacketSize(long inruid) {
	int c = -1;
	try {
	    c = ((Integer)((Hashtable)INRinfo.get(new Long(inruid))).get(packetSizeKey)).intValue();
	} catch(NullPointerException e) {
	    log.printStatus("getPacketSize(), Unknown INRuid:" + inruid, 
			    TwineLogger.TRACE_MSG);
	}
	return c;
    }

    //nikos
    /** Get multicast source number
     * @return number of multicast sources that use this resolver if given inruid exists
     * <br>
     * -1 if error
     **/
    public int getMulticastNumber(long inruid) {
	int c = -1;
	try {
	    c = ((Integer)((Hashtable)INRinfo.get(new Long(inruid))).get(multicastKey)).intValue();
	} catch(NullPointerException e) {
	    log.printStatus("getMulticastNumber(), Unknown INRuid:" + inruid, 
			    TwineLogger.TRACE_MSG);
	}
	return c;
    }

    /**
     * The following two methods are used by TwineInterResolver
     **/

    /**
     * Constructs a simple name specifier of all metrics.
     * @param inruid The Resolver UID
     * @return a name specifier containing all the metrics
     **/
    public NameSpecifier getMetricNS(long inruid) {
	// construct metric reply
	NameSpecifier myReplyNS = new NameSpecifier();
	myReplyNS.addAVelement(new AVelement(INRuidAttr, 
					     new Value(""+inruid)));
	myReplyNS.addAVelement(new AVelement(loadAttr, 
					     new Value(""+getLoad(inruid))));
	myReplyNS.addAVelement(new AVelement(capacityAttr, 
					     new Value(""+getCapacity(inruid))));

	Vector mypos = getPosition(inruid);
	StringBuffer myposStr = new StringBuffer();
	if((mypos != null) && (mypos.size()>0)) {
	    for(Enumeration e = mypos.elements(); e.hasMoreElements(); )
		myposStr.append(e.nextElement() + ":");
	    myposStr = myposStr.deleteCharAt(myposStr.length() - 1); // get rid of end ":"
	}
	else myposStr.append("null");	// default (error)
	    
	myReplyNS.addAVelement(new AVelement(positionAttr, 
					     new Value(myposStr.toString())));
	myReplyNS.addAVelement(new AVelement(branchAttr, 
					     new Value(""+getBranchNumber(inruid))));
	myReplyNS.addAVelement(new AVelement(packetSizeAttr, 
					     new Value(""+getPacketSize(inruid))));
	myReplyNS.addAVelement(new AVelement(multicastAttr, 
					     new Value(""+getMulticastNumber(inruid))));
	return myReplyNS;
    }

    /**
     * Set metrics of given inruid
     * @param replyNS The metric name specifier for a resolver
     **/
    public void setMetricNS(NameSpecifier replyNS) {
	Vector v = metricNStoHashtable(replyNS);
	try {
	    Long inruid = (Long)v.elementAt(0);
	    Hashtable metrics = (Hashtable)v.elementAt(1);
	    // add to metrics hashtable
	    INRinfo.put(inruid, metrics);
	} catch(ArrayIndexOutOfBoundsException e) {
	    log.printStatus("setMetricNS(): invalid input.",
			    TwineLogger.TRACE_MSG);
	}
    }

    /**
     * Converts metric NS to a metric Hashtable
     * @param replyNS The metric name specifier for a resolver
     * @return Vector with element1=inruid and element2=metrics_hashtable
     **/
    public Vector metricNStoHashtable(NameSpecifier replyNS) {
	AVelement av;
	Vector replyV = new Vector();
	try {	// parse name specifier for inruid and metrics 
	    av = replyNS.getAVelement(INRuidAttr);
	    Long r_inruid = new Long(av.getValue().toString());

	    av = replyNS.getAVelement(loadAttr);
	    Integer r_load = new Integer(av.getValue().toString());

	    av = replyNS.getAVelement(capacityAttr);
	    Integer r_capacity = new Integer(av.getValue().toString());

	    av = replyNS.getAVelement(branchAttr);
	    Integer r_branch = new Integer(av.getValue().toString());
	     
	    av = replyNS.getAVelement(packetSizeAttr);
	    Integer r_packetSize = new Integer(av.getValue().toString());
	     
	    av = replyNS.getAVelement(multicastAttr);
	    Integer r_multicast = new Integer(av.getValue().toString());

	    av = replyNS.getAVelement(positionAttr);
	    String r_position = av.getValue().toString();

	    Vector v = new Vector();
	    if(!r_position.equals("null")) {
		StringTokenizer st = new StringTokenizer(r_position, ":");
		while (st.hasMoreTokens()) {
		    String pos = st.nextToken();
		    v.addElement(new Integer(pos));
		}
	    }

	    // add to metrics hashtable
	    Hashtable h = new Hashtable();
	    h.put(loadKey, r_load);
	    h.put(capacityKey, r_capacity);
	    h.put(branchKey, r_branch);
	    h.put(packetSizeKey, r_packetSize);
	    h.put(multicastKey, r_multicast);
	    h.put(positionKey, v);

	    replyV.addElement(r_inruid);
	    replyV.addElement(h);
	    
	} catch(NullPointerException e) {
	    log.printStatus("metricNStoHashtable(): required attribute not found in: "+
			    replyNS.toString(), TwineLogger.TRACE_MSG);
	}
	return replyV;
    }

    /**
     * Constructs a simple name specifier from given metrics Hashtable
     * @param inruid The Resolver UID
     * @param metricsHT The metrics Hashtable
     * @return a name specifier containing all the metrics
     **/
    public NameSpecifier metricHTtoNS(Long inruid, Hashtable metricsHT) {
	// construct metric reply
	NameSpecifier myReplyNS = new NameSpecifier();
 	myReplyNS.addAVelement(new AVelement(INRuidAttr, 
					     new Value(""+inruid)));

	int l = -1;
	int c = -1;
	int b = -1;
	int s = -1;
	int m = -1;

	try {
	    l = ((Integer)(metricsHT.get(loadKey))).intValue(); 
	    c = ((Integer)(metricsHT.get(capacityKey))).intValue(); 
	    b = ((Integer)(metricsHT.get(branchKey))).intValue(); 
	    s = ((Integer)(metricsHT.get(packetSizeKey))).intValue(); 
	    m = ((Integer)(metricsHT.get(multicastKey))).intValue();

	    Vector mypos = (Vector)(metricsHT.get(positionKey));
	    StringBuffer myposStr = new StringBuffer();
	    if((mypos != null) && (mypos.size()>0)) {
		for(Enumeration e = mypos.elements(); e.hasMoreElements(); )
		    myposStr.append(e.nextElement() + ":");
		myposStr = myposStr.deleteCharAt(myposStr.length() - 1); // get rid of end ":"
	    }
	    else myposStr.append("null");	// default (error)
	    
	    myReplyNS.addAVelement(new AVelement(loadAttr, 
						 new Value(""+l)));
	    myReplyNS.addAVelement(new AVelement(capacityAttr, 
						 new Value(""+c)));
	    myReplyNS.addAVelement(new AVelement(positionAttr, 
						 new Value(myposStr.toString())));
	    myReplyNS.addAVelement(new AVelement(branchAttr, 
						 new Value(""+b)));
	    myReplyNS.addAVelement(new AVelement(packetSizeAttr, 
						 new Value(""+s)));
	    myReplyNS.addAVelement(new AVelement(multicastAttr, 
						 new Value(""+m)));
	} catch(NullPointerException e) {
	    log.printStatus("Did not find desired metric in: " + metricsHT, 
			    TwineLogger.TRACE_MSG);
	}
	return myReplyNS;
    }
}




