package ins.inr;

import ins.namespace.*;
import java.util.*;
import java.net.*;

/**
 * TwineNameTreeManipulator.java <br>
 *
 * Provides utility functions to manipulate
 * the name-tree. Many of these functions come
 * from different classes
 * <p>
 * Created: Tue Aug 14 10:52 2001 <br>
 * Modified:  $Id: TwineNameTreeManipulator.java,v 1.6 2002/03/21 00:01:55 mbalazin Exp $
 * @author Magdalena Balazinska
 * @version 1.0
 */

public class TwineNameTreeManipulator extends TimerTask {

    Resolver resolver;
    NameStoreInterface nameTree;
    TwineKeyManager keyMn;
    boolean checkingExpireNames;
    Timer timer;
  
    public TwineNameTreeManipulator() {
	checkingExpireNames = true;

    }

    public void init(Resolver r) {
	
	resolver = r;
	nameTree = ((TwineResolver)r).nameTree;
	keyMn = ((TwineResolver)r).keyMn;
	timer = ((TwineResolver)r).timer;

	// To reduce load during experiments
	timer.scheduleAtFixedRate(this,RouteManager.EXPIRE_CHECK_INTERVAL,
				  RouteManager.EXPIRE_CHECK_INTERVAL);
	
    }

    
    //-----------------------------------------------------
    /** 
     * Returns name record corresponding to given description
     * and resource identifier.
     * Similar to: lookupExactMatch in class <code>VSNameTree</code>
     * @param nu update message containing resource description
     * and identifier.
     * @retun NameRecord corresponding to the resource or 
     * <code>null</code> if no matches are found
     * @author Magdalena Balazinska
     */
    public NameRecord getExactMatch(NameUpdate nu) {
	
	NameRecord result = null;
	NameRecord[] rset = nameTree.lookup(nu.ns);
	
	if (rset == null)
	    return null;
	
	for (int i = 0; i < rset.length; i++) {
	    
	    NameRecord curRecord = rset[i];
	    NameSpecifier curNS = nameTree.extract(curRecord);
	  
	    if ( ( curNS.equals(nu.ns) ) && (curRecord.getAnnouncer() == nu.announcer) ) {
		return curRecord;
	    }    
	}
	return null;
    }

    //-----------------------------------------------------
    /**
     * Creates new name-record for resource specified by
     * the name-update.
     * @author Magdalena Balazinska
     */
    protected NameRecord createNameRecord(NameUpdate nu) {

	return new NameRecord(resolver.INRuid,null, nu.hostRecord,
			      nu.appMetric, RouteManager.MAX_NAME_EDGE_TTL, 
			      0, true, nu.announcer);
    }



    //-----------------------------------------------------
    /** 
     * Verifies whether something has changed in the resource
     * description. Updates the name-record accordingly.
     * This code is recycled from within a very long method
     * in class <code>AppManager</code>
     */
    protected boolean updatedResource(NameUpdate nu, NameRecord nr) {

	boolean diffNameRecord = false;

	// Update name record to reflect possibly new IP address
	// and port of app if mobile app...
	HostRecord hr = nr.getHostRecord();
	HostRecord newhr = nu.hostRecord;

	if (!Conversion.byteCompare(newhr.address, hr.address, 4)) {
	    hr.address = newhr.address;
	    diffNameRecord = true;
	}

	if (hr.UDPport != newhr.UDPport) {
	    hr.UDPport = newhr.UDPport;
	    diffNameRecord = true;
	}

	if (hr.TCPport != newhr.TCPport) {
	    hr.TCPport = newhr.TCPport;
	    diffNameRecord = true;
	}

	// this assumes that an add/delete do not happen simultaneously
	if (hr.transport.size() !=  newhr.transport.size()) {
	    hr.transport = newhr.transport;
	    hr.port = newhr.port;
	    diffNameRecord=true;
	}

	// the first-hop INR is responsible for soft-state
	// needs to refresh entry's TTL
	nr.setExpireTime(RouteManager.MAX_NAME_EDGE_TTL);

	// mobility case: if nr.getINRuid() is different from my uid
	// (i.e., resolver.uid) that means app has attached to me 
	// now from different INR.
	if (nr.getINRuid() != resolver.INRuid) {
	    nr.setINRuid(resolver.INRuid);
	    diffNameRecord = true;
	}

	return diffNameRecord;
    }


    //-----------------------------------------------------
    /**
     * Extracts complete descriptions matching the partial
     * description given in parameter.
     * @author Magdalena Balazinska
     */
    public Vector findNameSpecStrings(NameSpecifier ns) {

	Vector foundNSes = new Vector();
	NameRecord[] rset = nameTree.lookup(ns);
	for (int i=0; i<rset.length; i++) {
	    NameSpecifier curNS = nameTree.extract(rset[i]);
	    foundNSes.addElement(curNS.toString());
	}
	return foundNSes;
    }

    //-----------------------------------------------------
    /**
     * Returns a vector of host-records corresponding
     * to resources whose descriptions match the 
     * name-specifier given in parameter.
     * @author Magdalena Balazinska
     */
    public Vector findHostRecords(NameSpecifier ns) {

	Vector foundEBes = new Vector();
	NameRecord[] rset = nameTree.lookup(ns);
	for (int i=0; i<rset.length; i++) {	
	    TwineEBRecord ebr = new TwineEBRecord(rset[i].getAppMetric(),
						  rset[i].getHostRecord().toBytes());
	    foundEBes.addElement(ebr);
	}
	
	return foundEBes;
    }

    //-----------------------------------------------------
    /**
     * Find and return all matching NameRecords. Added by Kalpak Kothari.
     * @param ns name specifier for local lookup
     * @return all matching results as a vector of NameRecords
     * @author Kalpak Kothari
     */
    public Vector findNameRecords(NameSpecifier ns) {
	
	Vector foundLBes = new Vector();
	NameRecord[] rset = nameTree.lookup(ns);
	for (int i=0; i<rset.length; i++) {
	    foundLBes.addElement(rset[i]);
	}
	
	return foundLBes;
    }

    //-----------------------------------------------------
    /**
     * This is a periodic task that cleans-up expired
     * name-records. 
     * @author Magdalena Balazinska
     */
    public void run() {

	for (Enumeration eNR = nameTree.getNameRecords(); eNR.hasMoreElements(); ) {

	    NameRecord nr = (NameRecord)eNR.nextElement();
	    NameSpecifier ns = nameTree.extract(nr);

	    if (nr.testExpiration()) {
		
		nameTree.removeNameRecord(nr);
		keyMn.removeNameRecord(nr);
		// If we are the edge resolver for this resource, remove it explicitely
		// from the rest of the network
		if ( nr.getINRuid() == resolver.INRuid ) {
		    int mySeqNum = resolver.dsrMn.getNextSequenceNo();
		    ((TwineInterResolver)resolver.routeMn).sendRemove(mySeqNum,ns,nr,
								      NameUpdate.REMOVE_NAME);
		}
	    }
	    
	}    
    }

}
