package ins.inr;

import ins.namespace.*;
import java.util.*;
import java.net.*;

/**
 * TwineQueryManager.java
 * <br>
 * Holds pending queries as soft state.
 * <ul><li>When new queries arrive: adds them to database.
 * <li>When a query is forwarded to resolvers: remembers number requests sent.
 * <li>When some answer arrives: merges with existing answers and checks number
 * of pending answers.
 * <li>Every time interval: expires queries and invokes an empty send response 
 * </ul>
 * Created: Mon Aug 13 15:09:34 2001<br>
 * Modified: $Id: TwineQueryManager.java,v 1.9 2002/03/21 00:01:55 mbalazin Exp $
 * @author Magdalena Balazinska
 * @author Kalpak Kothari
 * @version 1.0
 */
public class TwineQueryManager extends TimerTask {

    public static int DISCOVERY = 0;
    public static int EARLY_BINDING = 1;
    public static int LATE_BINDING = 2;
    public static int DEFAULT_SIZE = 50;
    protected Resolver resolver;
    protected AppManager appMn;
    protected TwineInterResolver tir;  // When query expires, use next strand
    protected Hashtable queries;
    protected Hashtable latebindPackets;
    protected TwineLogger log;
    protected Timer timer;

    //-----------------------------------------------------
    /**
     * Class constructor
     */
    public TwineQueryManager() {
	log = new TwineLogger("QueryManager","TwineQueryManager.log");
    }

    //-----------------------------------------------------
    /**
     *
     */
    public void init(Resolver r) {
	resolver = r;
	appMn = r.appMn;
	tir = (TwineInterResolver)r.routeMn;
	queries = new Hashtable(DEFAULT_SIZE);
	latebindPackets = new Hashtable(DEFAULT_SIZE);
	timer = ((TwineResolver)r).timer;
	timer.scheduleAtFixedRate(this,TwineResolver.QUERY_TO/2,
				  TwineResolver.QUERY_TO/2);
    }

    //-----------------------------------------------------
    /**
     * Initializes some of the attributes of the query using
     * the packet in which the query was received.
     * @param packet Packet containing the query
     * @param queryType The type of query: may be DISCOVERY or 
     * EARLY_BINDING
     * @author Magdalena Balazinska
     */
    public TwineQuery makeQueryFromPacket(Packet packet, int queryType) {

	String nsstr = new String(packet.data, 11, packet.data.length-11);
	NameSpecifier ns = new NameSpecifier(nsstr);
	int seqNum = Conversion.extract32toInt(packet.data, 0);
	InetAddress addr = Conversion.extractInetAddress(packet.data, 4);
	int udpport = Conversion.extract16toIntLow16(packet.data, 8);
	boolean all = true;
	
	if ( queryType == EARLY_BINDING )
	    all = (packet.data[11]!=0);

	return new TwineQuery(ns,seqNum,addr,udpport,queryType,all);

    }

    //-----------------------------------------------------
    /**
     * Create a search query from a late-binding packet using its
     * destination name specifier.
     * @author Kalpak Kothari 
     * @param packet should be a late-binding packet
     * @param queryType should be LATE_BINDING
     * @return a search query which can be sent to other resolvers
     */
    public TwineQuery makeSearchQueryFromPacket(Packet packet, int queryType) {

	if ( queryType != LATE_BINDING ) {
	    log.printStatus("Packet is not late-binding type",
			    TwineLogger.TRACE_MSG);
	}
	
	NameSpecifier ns = packet.dNS;	// destination ns
	//	int seqNum = Conversion.extract32toInt(packet.data, 0);	// fix this! fake seqnum
	int seqNum = appMn.dsrMn.getNextSequenceNo(); // get unique sequence number
	InetAddress addr = resolver.comm.localhost;
	int udpport = resolver.comm.UDPport;
	
	return new TwineQuery(ns,seqNum,addr,udpport,queryType,packet.all);

    }

    //-----------------------------------------------------
    /**
     * Adds a new query to the query manager
     * @param mySeqNumber unique ID for the query is the sequence
     * number used when sending it to other resolvers
     * @param partialRes list of resources matching the query
     * @param query the query itself
     * @author Magdalena Balazinska
     */
    public synchronized void addQuery(int mySeqNumber, Vector partialRes, 
				      TwineQuery query) {

	// Add the query
	if ( partialRes == null)
	    partialRes = new Vector();
	
	query.prepareToStore(mySeqNumber,partialRes);
	
	// If only one answer is sought, we might already have it.
	//if ( !query.allResults() && (partialRes.size()> 0))
	//sendResponse(query, null);
	//else
	queries.put(query.getID(),query);
	
    }

    //-----------------------------------------------------
    /**
     * Add a search query along with the payload message for late-binding.
     * @param mySeqNumber unique sequence number for this query
     * @param partialRes partial results vector
     * @param query the search query to be added
     * @param msg the payload that was sent as a late-binding message by an application
     * @author Kalpak Kothari
     */
    public synchronized void addQuery(int mySeqNumber, Vector partialRes, 
				      TwineQuery query, Message msg) {
    
	// Add the query
	if ( partialRes == null)
	    partialRes = new Vector();
    
	query.prepareToStore(mySeqNumber,partialRes);

	//if ( !query.allResults() && (partialRes.size()> 0))
	//    sendResponse(query, msg);
	//else {
	queries.put(query.getID(),query);
	latebindPackets.put(query.getID(),msg);
	//}
    }


    //-----------------------------------------------------
    /**
     * Returns the query matching the given sequence number 
     * (unique ID) as the more generic type <code>TwineMessage</code>
     * @author Magdalena Balazinska
     */
    public synchronized TwineMessage getMessage(int mySeqNumber) {
	Integer key = new Integer(mySeqNumber);
  	return (TwineMessage)queries.get(key);
    }

    //-----------------------------------------------------
    /**
     * Adds a new pending query
     * @param number number of pending answers for this query
     * @param mySeqNumber unique ID of whole pending query
     * @author Magdalena Balazinska
     */
    public synchronized void addPending(int mySeqNumber,int number) {
	
	Integer key = new Integer(mySeqNumber);
	TwineQuery query = (TwineQuery)queries.get(key);
	if ( query != null) {
	    query.addPendingAnswers(number);
	    log.printStatus("Incremented pending answers in QueryManager: " 
			    + toString(),TwineLogger.TRACE_MSG);
	}

    }


    //-----------------------------------------------------
    /**
     * When an answer comes back for a query, the number
     * of outstanding responses is decremented.
     * @param mySeqNumber unique ID of the pending query
     * @param incompleteDisc one of possible destination name-specs.
     * The destination name-spec depends on the type of query that
     * was issued.
     * @param incompleteEarly one of possible destination name-specs
     * @param incompleteLate one of possible destination name-specs
     * @author Magdalena Balazinska
     */
    public synchronized void decPending(int mySeqNumber, NameSpecifier incompleteDisc,
					NameSpecifier incompleteEarly, 
					NameSpecifier incompleteLate) {
	
	Integer key = new Integer(mySeqNumber);
	TwineQuery query = (TwineQuery)queries.get(key);
	if ( query != null) {
	    query.decPending();
	    if ( query.getPending() <= 0 ) {	
		if ( query.getType() != LATE_BINDING )
		    sendPartialResponse(query,incompleteDisc,incompleteEarly, incompleteLate, null);
		else
		    sendPartialResponse(query,incompleteDisc,incompleteEarly, 
					incompleteLate, (Message)latebindPackets.get(key));
	    }
	}
    }


    //-----------------------------------------------------
    /**
     * Adds a set of answers to the set accumulated until now.
     * Compares all new answers with existing ones to avoid
     * keeping duplicates.
     * @author Magdalena Balazinska
     */
    public synchronized void addAnswer(Vector partialRes, int mySeqNum) {

	// Merge partial results with set of answers
	Integer key = new Integer(mySeqNum);
	TwineQuery query = (TwineQuery)queries.get(key);

	// If the query is not in the table, we have already
	// sent a response so we can ignore the new one
	if ( query != null) {

	    Vector old = query.getResults(); 
	    for ( Enumeration novel = partialRes.elements(); novel.hasMoreElements();) {
		Object current = novel.nextElement();
		if ( ! old.contains(current))
		    old.addElement(current);
	    }
	    
	    // Decrease number of pending answers.
	    query.decPending();

	    // If no more pending requests or if only one answer was
	    // desired and obtained, then send response and clean-up.
	    if ( ( query.getPending() <= 0 ) 
		 || ( !query.allResults() && (old.size() > 0))) {
		if ( query.getType() != LATE_BINDING )
		    sendResponse(query, null);
		else 
		    sendResponse(query, (Message)latebindPackets.get(key));
	    }
	}

	log.printStatus("Added answer to QueryManager",TwineLogger.TRACE_MSG);
    }


    //-----------------------------------------------------
    /**
     * Each query is associated with the set of strands
     * extracted from the name-specifier describing the
     * desired resource. Strands from this set determine
     * what resolvers will collaborate in solving the query.
     * @author Magdalena Balazinska
     */
    public synchronized void setStrands(int mySeqNumber, Vector strands) {
	
	Integer key = new Integer(mySeqNumber);
	TwineQuery query = (TwineQuery)queries.get(key);
	if ( query != null)
	    query.setStrands(strands);
    }

    //-----------------------------------------------------
    /**
     *
     */
    public synchronized Vector getStrands(int mySeqNumber) {

	Vector strands = null;
	Integer key = new Integer(mySeqNumber);
	TwineQuery query = (TwineQuery)queries.get(key);
	if ( query != null)
	    strands = query.getStrands();
	return strands;
    }

    //-----------------------------------------------------
    /**
     * The packet associated with a query is the exact message
     * transmitted to other resolvers.
     * @author Magdalena Balazinska
     */
    public synchronized void setPacket(int mySeqNumber, Packet p) {
	
	Integer key = new Integer(mySeqNumber);
	TwineQuery query = (TwineQuery)queries.get(key);
	if ( query != null)
	    query.setPacket(p);
    }

    //-----------------------------------------------------
    /**
     *
     */
    public synchronized Packet getPacket(int mySeqNumber) {

	Packet p = null;
	Integer key = new Integer(mySeqNumber);
	TwineQuery query = (TwineQuery)queries.get(key);
	if ( query != null)
	    p = query.getPacket();
	return p;
    }


    //-----------------------------------------------------
    /**
     * Once a query is solved, or when a query expires
     * a response is sent to the client. The response
     * will contain all results accumulated until now
     * @param mySeqNumber Unique ID for the query
     * @author Magdalena Balazinska
     */
    public synchronized void sendResponse(int mySeqNumber) {
	
	Integer key = new Integer(mySeqNumber);
	TwineQuery query = (TwineQuery)queries.get(key);
	if ( query != null) {
	    if ( query.getType() != LATE_BINDING )
		sendResponse(query, null);
	    else // this is a late-binding response
		sendResponse(query, (Message)latebindPackets.get(key));
	}
	else
	    log.printStatus("Sending response query is null.",
			    TwineLogger.TRACE_MSG);

    }

    //-----------------------------------------------------
    /**
     * Sends response to query back to the client.
     * Modified by Kalpak to handle late-binding responses
     * @author Magdalena Balazinska
     * @author Kalpak Kothari
     */
    protected synchronized void sendResponse(TwineQuery query, Message msg)
    {
	
	log.printStatus("\n Sending response from QueryManager " + 
			toString(),TwineLogger.TRACE_MSG);

	if ( query.getType() == DISCOVERY )
	    appMn.sendDiscoveryResponse(query.getResults(),query.getClientSeqNum(),
					query.getAddr(),query.getPort(),null,null);
	
	else if ( query.getType() == EARLY_BINDING )
	    appMn.sendEarlyBindingResponse(query.getResults(),query.getClientSeqNum(),
					   query.getAddr(),query.getPort(),null,null,
					   query.allResults());
	else { // otherwise query type is late binding 
	    log.printStatus("payload= " + msg, TwineLogger.TRACE_MSG);
	    ((TwineAppManager)appMn).
		sendLateBindingResponse(query.getResults(),query.getID().intValue(), // getClientSeqNum(),
					query.getAddr(),query.getPort(),null,null,
					query.allResults(), msg);
	}
	    
	// Once a response a sent, the query HAS to be removed
	queries.remove(query.getID());
    }

    //-----------------------------------------------------
    /**
     * Sends a response to a query flagging it as incomplete.
     * This type of responses may be returned for queries
     * entirely composed of overly popular strands.
     * Modified by Kalpak to handle late-binding responses
     * @author Magdalena Balazinska
     * @author Kalpak Kothari
     */
    protected synchronized void sendPartialResponse(TwineQuery query, 
						    NameSpecifier incompleteDisc,
						    NameSpecifier incompleteEarly,
						    NameSpecifier incompleteLate,
						    Message msg)
    {
	
	log.printStatus("\n Sending partial response from QueryManager " + toString(),
			TwineLogger.TRACE_MSG);
	if ( query.getType() == DISCOVERY )
	    appMn.sendDiscoveryResponse(query.getResults(),query.getClientSeqNum(),
					query.getAddr(),query.getPort(),null,null,
					incompleteDisc);
	
	else if ( query.getType() == EARLY_BINDING ) 
	    appMn.sendEarlyBindingResponse(query.getResults(),query.getClientSeqNum(),
					   query.getAddr(),query.getPort(),null,null,
					   query.allResults(),incompleteEarly);
	else  // otherwise query type is late binding 
	    ((TwineAppManager)appMn).
		sendLateBindingResponse(query.getResults(),query.getID().intValue(), //getClientSeqNum(),
					query.getAddr(),query.getPort(),null,null,
					query.allResults(),incompleteLate, msg);

	// Once a response a sent, the query HAS to be removed
	queries.remove(query.getID());
    }
 
    //-----------------------------------------------------
    /**
     */
    public String toString() {

	String output = "\n TwineQueryManager: ";
	for ( Enumeration e = queries.elements(); e.hasMoreElements();) {
	    TwineQuery q = (TwineQuery)e.nextElement();
	    output = output + q.toString();
	}
	return output;
    }
    

    //-----------------------------------------------------
    /**     
     * Performs clean up to remove expired queries.
     * @author Magdalena Balazinska
     */
    protected synchronized void cleanUp() {

 	for ( Enumeration enum = queries.elements(); enum.hasMoreElements();) {
	    
 	    long time = System.currentTimeMillis();
 	    TwineQuery query = (TwineQuery)enum.nextElement();
 	    if (query.expired(time)) {
 		sendResponse(query, null);	// changed by kalpak
 	    }
 	}
    }

    //-----------------------------------------------------
    /**
     * Performs clean up to remove expired queries. This method is
     * similar to cleanUp() but it also handles late-binding queries.
     * @author Kalpak Kothari 
     */
    protected synchronized void cleanUpNew() {

	for ( Enumeration enum = queries.keys(); enum.hasMoreElements();) {
	    Integer key = (Integer)enum.nextElement();
	    TwineQuery query = (TwineQuery)queries.get(key);
	    
	    long time = System.currentTimeMillis();
	    if (query.expired(time)) {
		sendResponse(query, (Message)latebindPackets.get(key));
	    }
	}
    }
    
    
    //-----------------------------------------------------
    /**
     * Clean-up expired queries and send responses accumulated
     * until now.
     * @author Magdalena Balazinska
     */
    public void run() {
	cleanUpNew();
    }
}
 
