package ins.inr;

import java.util.*;
import java.net.*;
import java.io.*;
import ins.namespace.*;
import ins.inr.twineKeyRouter.*;
import ins.inr.interResolverSvc.*;

/**
 * TwineResolver.java
 *
 * The main executable class to run a Twine resolver.
 * <p>
 * Created: Tue Jun 19 10:48:35 2001 <br>
 * Modified:  $Id: TwineResolver.java,v 1.13 2002/03/25 20:36:52 nikos Exp $
 *
 * @author Magdalena Balazinska
 */

public class TwineResolver extends Resolver {

    /** 
     * Twine resolvers advertise a fake vspace so DSRs do not 
     * have to be modified to support Twine nodes. The fake vspace is set
     * to "default" to match the applications defaults 
     */
    public static String FAKE_VSPACE = "default";
    public static boolean DISPLAY = false;

    public static long QUERY_TO = 1500; // By default queries timeout in 1.5 seconds
    public static int  QUERY_D = 1;     // By default 1 resolver solves a query
    public static int  ANNOUNCE_D = 1;  // By default announcing to 1 resolver per strand
    public static int myCapacity = 10485760; // default network capacity // kalpak
    public static String PSPATH=null; //path for the pingstation file //nikos

    // Switch for turning on/off debugging messages level (0-5) // kalpak
    public static final int DEBUG = 2; // for testing, only print IMPORTANT_MSGs

    // A Twine resolver has a single name tree
    protected NameStoreInterface nameTree;

    // Other resources
    protected TwineGUI gui;
    protected TwineStats stats;
    protected TwineQueryManager queryManager;
    protected TwineNameTreeManipulator nameTreeManip;
    protected TwineKeyManager keyMn;
    protected RPCCommunicator rpcComm;

    // added by kalpak
    protected TwineMetricManager tmm;	
    protected TwineMetricEstimator tme;
    protected NodeSet nodeset;
    
    protected Timer timer;
    protected static int rpcPort = -1;

    // -----------------------------------------------
    /**
     * 
     */
    public TwineResolver() throws Exception {

	super();
	
    }

    // ----------------------------------------------- 
    /**
     * 
     */
    public TwineResolver(/*String[]*/ Vector vspaces, Vector aggregateVspaces,
		    int mode, int udp, int tcp,
		    String[] dsrs, String [] neighs)
	throws Exception
    {
	super(vspaces, aggregateVspaces, mode, udp, tcp, dsrs, neighs);
    }


    //-----------------------------------------------
    /** 
     * Overridding the init method of the parent.
     * Most resources of a Twine resolver are subtypes
     * of corresponding resources in the old version of 
     * the resolver.
     */
    void init(/*String[]*/ Vector vspaces, Vector aggregateVspaces, 
	      int mode, int udp, int tcp,
	      String[] dsrs, String [] neighs)
	throws Exception
    {

    	// Set system output stream to LoggedPrintStream
	loggedStream = new LoggedPrintStream(System.out);	
    	try {System.setOut(loggedStream); System.setErr(loggedStream);}
    	catch (SecurityException e) {e.printStackTrace();}
    	
	// Create sockets for communications
	comm = new Communicator(udp, tcp);
	if (rpcPort > 0)
	    rpcComm = new RPCCommunicator(rpcPort);
	else
	    usage();

	// Create uid
	byte[] addr = comm.localhost.getAddress();
	Date now = new Date();
	INRuid = (long)(Conversion.extract32toInt(addr, 0));
	// obsolete legacy udp port	INRuid = (INRuid << 16) | ((long)comm.UDPport & 0xFFFFl);
	// use new RPC port (kalpak)
	INRuid = (INRuid << 16) | ((long)rpcComm.getRPCPort() & 0xFFFFl);
	printStatus("INR uid is " + INRuid);

	// Create other resources

	// Those nameTrees are created only for compatibility
	// with the DSRManager
	nameTrees = (VSNameTree)new TwineVSNameTree(vspaces);	
	routeTables = null;
	neighbors = null;
	resolvers = null;
	forwarder = (Forwarder)new TwineForwarder();
	printStatus("Resources created...");

	nameTree = (NameStoreInterface)new NameTree();
	gui = new TwineGUI();
	stats = new TwineStats();
	queryManager = new TwineQueryManager();
	nameTreeManip = new TwineNameTreeManipulator();
	keyMn = new TwineKeyManager();
	timer = new Timer();
	
	// Create handlers
	overlayMn = null;
	routeMn = (RouteManager)new TwineInterResolver();
	appMn = (AppManager)new TwineAppManager();
	prober = (NetProber)new TwineNetProber();
	dsrMn = (DSRManager)new TwineDSRManager(dsrs);
	printStatus("Handlers created...");

	// Create Metric Manager (added by kalpak)
	tmm = new TwineMetricManager();
       
	//extracting pingstations from the ping.stations file and passing it to 
	//the Metric estimator. (nikos)
	nodeset=getPingStations(PSPATH);
	// Create Local Metric Estimator (added by kalpak)
	tme=new TwineMetricEstimator(true,nodeset);

	// DSRManager now does search for unknown dsr in domain
	defaultVSpace = FAKE_VSPACE;

	// Init resources	
	comm.init(this);
	forwarder.init(this);
	rpcComm.init(this);
	printStatus("Resources initialized...");

	// Init handlers
	appMn.init(this);
	prober.init(this);
	dsrMn.init(this);
	routeMn.init(this);
	queryManager.init(this);
	nameTreeManip.init(this);

 	// Init metric manager (added by kalpak)
  	tmm.init(this);
  	tmm.setCapacity(INRuid, myCapacity);
  	// Init metric estimator (added by kalpak)
  	tme.init(this);

	gui.init(this);
	stats.init(this); // Init last
	keyMn.init(this);

	printStatus("Handlers initialized...");

	printStatus("*********************************************");

	// Commented out interactive mode to run resolvers with "nohup"
	//new Thread(this).start();
    }

  /**
   * Method for reading the ping.stations file and putting
   * the pingstations into a nodeset.
   */
    private NodeSet getPingStations(String pspath){

      if(pspath!=null){

      NodeSet nodeset=new NodeSet();
      RandomAccessFile raf=null;
      try{
	raf=new RandomAccessFile(pspath,"r");
      }catch(FileNotFoundException e){
	System.err.println("ping.stations file:" +pspath+" not found");
      }
      String line=null;
      do{
	try{
	  line=raf.readLine();
	  //System.out.println(line);
	}catch(IOException e){ System.err.println("IO error in reading file line");}
	if(line!=null){
	  StringTokenizer st=new StringTokenizer(line);
	  if(st.countTokens()==2){
	  
	    InetAddress inet=null;
	    try{
	    inet=InetAddress.getByName(st.nextToken());
	    System.out.println(inet);
	    }catch(UnknownHostException e){}
	    String token=st.nextToken();
	    //System.out.println("token*"+token+"*");
	    int port=Integer.parseInt(token);
	    //System.out.println("port"+port);
	    Node node = new Node(inet, port, 0);
	    //System.out.println("node="+ node);
	    nodeset.put(node);
	  }
	}
      }while(line!=null);
      try{
	raf.close();
      }catch(IOException e){}
      return nodeset;
      }else return null;
    }

    /** 
     * Main method to run a Twine resolver.
     *
     */
    public static void main(String[] args)
    {
	
	IResolverFactory res;
	try {
	    res = (IResolverFactory)new TwineResolverFactory();

	    // Twine options have to be processed before a Resolver is created
	    twineOptions(args);

	    // Faking the use of a vspace unless specified in arguments
	    String[] moreArgs = args;
	    if ( (args.length == 0) || (args[0].startsWith("-")) ) {
		System.out.println("\nFaking the virtual space");
		moreArgs = new String[args.length+1];
		System.arraycopy(args,0,moreArgs,1,args.length);
		moreArgs[0] = TwineResolver.FAKE_VSPACE;
	    }
	    
	    // Calling the original main method of Resolver
	    mainMethod(moreArgs,res);

	} catch (Exception e) {
	    System.out.println("The INR could not be started " +
			       "for the following reason:");
	    System.out.println(e.getMessage());
	    e.printStackTrace();
	}

    }


    //-----------------------------------------------
    /**  
     * Additional parameters for Twine resolvers.
     * See method <code>usage</code> for details.
     */
    public static void twineOptions(String[] args) {

	rpcPort = -1;
	int keyrouterport = -1;
	try {
	    for (int i = 0 ; i < args.length; i++) {
		String arg = args[i];
		if (arg.equals("--display")) {
		    TwineResolver.DISPLAY = true; 
		} else if (arg.equals("-qt")) {
		    TwineResolver.QUERY_TO = Integer.parseInt(args[i+1]); 
		} else if (arg.equals("-qd")) {
		    TwineResolver.QUERY_D = Integer.parseInt(args[i+1]); 
		} else if (arg.equals("-ad")) {
		    TwineResolver.ANNOUNCE_D = Integer.parseInt(args[i+1]); 
		} else if (arg.equals("-h")) {
		    usage();
		} else if (arg.equals("-rpc")) {
		    rpcPort = Integer.decode(args[i+1]).intValue();
		} else if (arg.equals("-keyrouter")) {
		    keyrouterport = Integer.decode(args[i+1]).intValue();
		} else if (arg.equals("--no-values")) {
		    TwineStrandSplitter.TYPE_STRANDS = TwineStrandSplitter.ATTRIB_STRANDS;
		} else if (arg.equals("--cap")) {
		    myCapacity = Integer.parseInt(args[i+1]);
		}
		//setting pingstations from file (nikos)
		else if(arg.equals("-pingstations")){
		  TwineResolver.PSPATH= args[i+1];
		}
	    }
	} catch (Exception e) {
	    usage();
	}
	
	if ( (rpcPort == -1) || (keyrouterport == -1))
	    usage();
	else {
	    TwineKeyRouter.setPortsOffset(rpcPort,keyrouterport);
	}
    }


    //-----------------------------------------------
    /**  
     * 
     */
    static void usage()
    {
	System.err.println("Usage: java " +
			   "ins.inr.TwineResolver [vspace1,vspace2,...] "
			   + " -rpc myport -keyrouter port [-udp myport]"
			   + " [-d dsr] [-l] [-qt timeout] [-ad nbresolvers] [-qd nbresolvers]  "
			   + " [-pingstations pspath] " // (nikos)
			   + " [--display] [--no-values] [--cap capacity] ");
	System.err.println("  -rpc \t\t Use this port for communication between resolvers");
	System.err.println("  -keyrouter \t Connect to this ckr process (running on the same host)");
	System.err.println("  -udp \t\t Use this port to listen for client messages");
	System.err.println("  \t\t {resolver,keyrouter} ports MUST have the same offset for each pair");
	System.err.println("  -d  \t\t Use dsr as the DSR");
	System.err.println("  -l  \t\t Turn on packet-logging mode");
	System.err.println("  -qt \t\t Use this timeout for queries (ms)");
	System.err.println("  -ad \t\t Replicate each key that many times");
	System.err.println("  -qd \t\t Submit each query to this many resolvers at once. ");
	System.err.println("  \t\t Value should be less than the one for the -ad option.");
	System.err.println("  -pingstations \t\t use pspath to specify a ping.stations file.");
	System.err.println("  --display \t Start a graphical window displaying the name tree");
	System.err.println("  --no-values \t Make resolver use only attributes of res. descr.");
	System.err.println("  --cap network capacity in bits/second (default: " + myCapacity + ")");
	System.exit(1);
  	
    }
}
 

//-----------------------------------------------
/** 
 *
 * Resolver factory producing Twine resolvers.
 *
 */ 
class TwineResolverFactory implements IResolverFactory {

    public TwineResolverFactory() {
    }

    public Resolver makeResolver(/*String[]*/ Vector vspaces, 
				 Vector aggregateVspaces, 
				 int mode, int udp, int tcp,
				 String[] dsrs, String [] neighs)
	throws Exception {
	
	return new TwineResolver(vspaces, aggregateVspaces,mode,udp,tcp,
				 dsrs, neighs);
	
    }

}
