package ins.inr;

import java.util.*;
import java.net.InetAddress;
import ins.namespace.*;
import ins.namespace.index.*;

/** 
 * VSNameTree is a set of virtual spaces and their (separate) name trees.
 * It also serves an authoritative list of the vspaces which we host 
 *    locally on this INR. (e.g. getVspaces)
 *
 * Although not explicitly a subclass of VSpaceSet, it more or less 
 * follows the same model as VSpaceSet of having a list of vspaces and
 * associated elements.
 */

public class VSNameTree  {

    private Hashtable rep;

    public VSNameTree () {
	rep = new Hashtable(47);
    }

    /**
     * adds a Vspace to the system. Note: this should only be done
     *  when the INR agrees to host the vspace.
     *
     * @param vspace Vspace to add to the INR
     * @return Whether it could be added
     */
    public synchronized boolean addVspace(String vspace, NameStoreInterface nametree) 
    {
	if (rep.containsKey(vspace))
	    return false;

	rep.put(vspace, nametree);
	return true;
    }

    /**
     * Removes the vspace from the list and returns whether or not 
     * it was successful
     *
     * @param vspace The vspace to remove.
     * @return Whether the vspace could be removed.
     */
    public synchronized boolean removeVspace(String vspace) 
    {
	return (rep.remove(vspace) != null);
    }


    /**
     * Looks up the a NameSpecifier in the proper nametree,
     * as known by the vspace. If the vspace is not specified,
     * defaultTo will be used as the vspace
     *
     * @param ns        NameSpecifier to look up
     * @param defaultTo Vspace to default to
     */
    public NameRecord[] lookup(NameSpecifier ns, String defaultTo)
    {
	Object item = null;
	
	String vspace = ns.getVspace();
	if (vspace != null)
	    item = rep.get(vspace);

	if (item==null && defaultTo!=null) 
	    item = rep.get(defaultTo);

	if (item==null) 
	    return null;

	return ((NameStoreInterface)item).lookup(ns);
    }


    /**
     * Extracts the NameSpecifier given its corresponding NameRecord.
     * The vspace must be specified. 
     *
     * @param rec NameRecord to look up
     * @param vspace The virtual space to look in
     * @return NameSpecifier it corresponds to, or null
     */
    public NameSpecifier extract(NameRecord rec, String vspace)
    {
	if (vspace == null) 
	    return null;

	NameStoreInterface rt = (NameStoreInterface)rep.get(vspace);

	return rt.extract(rec);
    }

    /** 
     * Adds a NameRecord to the proper name tree
     *
     * @param ns NameSpecifier to add
     * @param defaultVspace vspace to use if the ns has none
     * @param rec The corresponding name record.
     */
    public void addNameRecord(NameSpecifier ns, String defaultVspace,
			      NameRecord rec)
    {    
	String vspace = ns.getVspace();
	NameStoreInterface nt = null;

	if (vspace != null)
	    nt = getNameTree(vspace);
	
	if (nt == null && defaultVspace != null)
	    nt = getNameTree(defaultVspace);

	if (nt != null)
	    nt.addNameRecord(ns, rec);
    }


    /**
     * Removes a NameRecord from the proper tree.
     *
     * @param vspace The vspace to look under
     * @param rec The NameRecord to remove
     */
    public void removeNameRecord(String vspace, NameRecord rec)
    {
	NameStoreInterface nt = null;

	if (vspace != null)
	    nt = getNameTree(vspace);

	if (nt != null)
	    nt.removeNameRecord(rec);
	else
	{
	    System.out.println("VSNameTree: WARNING: removeNameRecord() results in null NameTree");
	}
    }

    /**
     * Returns the NameTree for a given virtual space.
     *
     * @param vspace The vspace to use.
     * @return NameTree that corresponds to it, or null.
     */
    public NameStoreInterface getNameTree(String vspace) 
    {
	if (vspace == null) return null;
	
	return (NameStoreInterface)rep.get(vspace);
    }


    /** 
     * Sets up a vspace if it doesn't exist. 
     * 
     * @param vspace The vspace to add.
     * @return Did it add the NameTree?
     */
    boolean produceVspace(String vspace) 
    {
	NameStoreInterface nt;
	if ((System.getProperty("useIndex", "false")).equals("true")) {
	    System.out.println("<-- System property 'useIndex' equals 'true' -->");
	    System.out.println("**********************************************************");
	    System.out.println("* Instantiating NameIndex based replacement for NameTree *");
	    System.out.println("**********************************************************");
	    nt = new NameIndex(100); 
	}
	else {
	    
	    nt = new NameTree();
	}
	nt.setVspace(vspace);

	return addVspace(vspace, nt);
    }


    /**
     * Returns an enumeration of all the vspace names.
     *
     * @return Enumeration of the String vspace names.
     */
    public Enumeration getVspaces() 
    {
	return rep.keys();
    }

    /**
     * Returns whether we host that vspace. Uses hash lookup.
     */
    public boolean containsVspace(String vspace)
    {
	return rep.containsKey(vspace);
    }


    /**
     * Does a lookup on the NameTree such that the item exactly
     *  matches ns, and that the announcer is equal.
     * This is more expensive than a regular lookup() since it
     *  extracts each of the results and does a compare.
     *
     * @param ns NameSpecifier to match against
     * @param announcer INR Announcer value to match against.
     */
    public Vector lookupExactMatch(NameSpecifier ns, long announcer)
    {
	String vspace = ns.getVspace();
	if (vspace == null)
	{
	    System.out.println("No VSpace info in name, stop looking up:");
	    System.out.println(ns.toString());
	    return null;
	}

	Vector result = new Vector(2);

	NameStoreInterface nt = getNameTree(vspace);
	if (nt == null) 
	{
	    System.out.println("WARNING: NameTree for vspace "+vspace+ 
		" does not exist!");
	    return result;
	}

	NameRecord[] rset = nt.lookup(ns);
	if (rset == null) 
	    return result;

	for (int i = 0; i < rset.length; i++) {
	    NameRecord curRecord = rset[i];
	    
	    // Skip this if it's not an exact match
	    NameSpecifier curNS = nt.extract(curRecord);
	    if (! curNS.equals(ns)) {
		continue;
	    }
	    
	    // Skip it if it's not from the same announcer
	    if (curRecord.getAnnouncer() != announcer) {
	        continue;
	    }

	    if (Resolver.DEBUG >= 5) {
	       System.out.println("These two are considered equal:");
	       System.out.println(curNS.toString() + " with announcer " + 
				  curRecord.getAnnouncer());
	       System.out.println(ns.toString() + " with announcer " + 
				  announcer);
	    }

	    result.addElement(curRecord);
	}
	return result;
    }

    /**
     * Returns a String representation of the VSNameTree.
     *
     * @return String representing the structure.
     */
    public String toString()
    {
	StringBuffer strbuf = new StringBuffer();

	for (Enumeration enum = rep.keys(); enum.hasMoreElements(); )
	{
	    Object vspace = enum.nextElement();
	    Object nametree = rep.get(vspace);

	    strbuf.append(nametree.toString() + "\n");
	}
	return strbuf.toString();
    }

}

