package ins.inr;

import java.net.InetAddress;
import java.util.*;


/**
 * VSNeighbors is a list of all this node's neighbors for each vspace.
 *
 * The list of neighbors is a subset of the list of resolvers
 * (e.g. A, B, C, D are all neighbors in this vspace, but we might only
 *  be neighbors with B and D).
 *
 * This only consists of nodes in vspaces with which we are member of. 
 * VSResolvers consists of all nodes in a vspace, and also contains 
 * "cached" vspaces that we are not a member of.
 *
 * IMPORTANT: The Nodes inside this structure should only be pointers to 
 *    the same element in VSResolvers. That way, when the vspaces or
 *    TTL on one is updated, the same consistent value is maintained.
 *   THUS, addNeighbor() should only be called on objects returned
 *      from VSResolvers
 */

class VSNeighbors
    extends VSpaceSet
{
    // VARIABLES
    NodeSet allNeighbors;   // NodeSet for all VSpaces this
			    // INR belongs to
			    // AllNeighbors is a Vector of Node(s)

    VSNameTree nameTrees;

    // CONSTRUCTORS
    VSNeighbors()
    {
	super();
	allNeighbors = new NodeSet();
    }

    void init(Resolver r)
    {
	nameTrees = r.nameTrees;
    }

    /**
     * Adds a VSpace to the list.
     * if it already exists, returns false (otherwise true)
     *
     * @param vspace The vspace to add.
     * @param nodeset The NodeSet to associate with the vspace.
     * @return Whether it could be successfully added.
     */
    public synchronized boolean addVspace(String vspace, NodeSet nodeset)
    {
	return super.addVspace(vspace, (Object)nodeset);
    }

    /**
     * Removes the vspace from the list and returns whether or not 
     * it was successful
     *
     * @param vspace The vspace to remove
     * @return Whether the vspace could be removed
     */
    public synchronized boolean removeVspace(String vspace) 
    {
	return super.removeVspace(vspace);
    }

    /**
     * Returns the NodeSet object associated with vspace
     *
     * @param vspace The vspace to look up
     * @return NodeSet of those in vspace, with expired elements pruned
     */
    public synchronized NodeSet getNeighbors(String vspace)
    {
	return pruneExpired((NodeSet)super.getVSpace(vspace));
    }

    /** 
     * Prunes all the expired nodes from the list and also
     *  the allResolvers list and applicable individual vspace
     *  lists. Should be called on anything returned to the
     *  user so that expired entries are never returned. 
     */
    protected synchronized NodeSet pruneExpired(NodeSet nodeset) 
    {
	if (nodeset == null) return null;

	long time = System.currentTimeMillis();

	for (int i = 0; i<nodeset.countNodes(); i++)
	{
	    Node n = (Node)nodeset.nodeNumber(i);
	    if (pruneExpiredNode(n, time)) i--;
	}

	return nodeset;
    }


    public boolean pruneExpiredNode(Node n, long time)
    {
	// is the node expired?
	if (time>n.expireTime) {
	    	
	    System.out.println("REALLY PRUNING NODE!!");

	    for (int i=0; i<n.vspaces.size(); i++)
	    {
		// check each vspace
		String vspaceToCheck = (String)n.vspaces.elementAt(i);
		
		if (!nameTrees.containsVspace(vspaceToCheck))
		{
		    System.out.println("removing "+vspaceToCheck);
		    NodeSet neigh = (NodeSet)rep.get(vspaceToCheck);
		    neigh.remove(n);
		    
		    if (neigh.countNodes() == 0) 
			rep.remove(vspaceToCheck);
		    
		    n.removeVspace(vspaceToCheck);
		    i--;
		}
	    }

	    if (n.countVspaces()==0) 
	    {
		totallyRemoveNeighbor(n);
	    } else {
		n.touchTTL(); // make sure nobody bothers this for a while
	    }

	    return true;
	} else {
	    return false;
	}
    }


    synchronized void totallyRemoveNeighbor(Node node)
    {
	allNeighbors.remove(node);
		
	// remove it from the vspace lists that it's in
	Enumeration vses = node.getVspaces();

	while (vses.hasMoreElements()) {
	    String vs = (String)vses.nextElement();
		    
	    NodeSet nst = (NodeSet)rep.get(vs);
			
	    if (nst != null) { // this _should_ be the case
		nst.remove(node); // take it away!
			
		// if it's empty now, delete it!
		if (nst.countNodes() == 0) {
		    rep.remove(vs);
		}
	    }
	} // while (vses.hasMoreElements())
    }


    /**
     * Adds a neighbor node. If it already exists, it will ensure that
     *  the vspace is there, and that the ttl is updated.
     * For consistency, this should be a neighbor retrieved from the
     *   VSResolvers set.
     *
     * @param vspace Appropriate vspace
     * @param newNode Node to be added.
     * @return Whether it was new.
     */
    synchronized boolean addNeighbor(String vspace, Node newNode)
    {
	// add to node
	newNode.addVspace(vspace);
	
	NodeSet neigh = (NodeSet)rep.get(vspace);

	if (neigh != null) // does the vspace already exist??
	{	    
	    Node n = neigh.get(newNode);

	    if (n == null)	// nope, then add it
	    {
		// Check if AllNeighbors already contains it
		Node existingNode = allNeighbors.get(newNode);
		if (existingNode != null)	// newNode exists AllNeighbors
		{
		    existingNode.addVspace(vspace);
		    if (!nameTrees.containsVspace(vspace))
			existingNode.updateTTL(newNode.getTTL());
		    neigh.put(existingNode);
		}
		else		// AllNeighbors doesn't have newNode
		{
		    allNeighbors.put(newNode);
		    neigh.put(newNode);
		}
	    } // (n == null)
	    else
	    {
		if (!nameTrees.containsVspace(vspace))
		    n.updateTTL(newNode.getTTL());
		return false;	// since ia:port already in vspace
	    }
	}
	else 	// vspace doesn't exist, create a new VSpace
	{
	    // Check if AllNeighbors already contains it
	    Node existingNode = allNeighbors.get(newNode);
	    if (existingNode != null)	// newNode exists AllNeighbors
	    {
		existingNode.addVspace(vspace);
		if (!nameTrees.containsVspace(vspace))
		    existingNode.updateTTL(newNode.getTTL());
		neigh = new NodeSet(existingNode);
	    }
	    else		// AllNeighbors doesn't have newNode
	    {
		allNeighbors.put(newNode);
		neigh = new NodeSet(newNode);
	    }

	    rep.put(vspace, neigh);
	}
	return true;
    }


    /** 
     * Accepts a string in the form of host:UDPport:TCPport:vspace:ttlms
     *  Used more for NodeConsole/high-level interaction
     *  Returns the node that was added or null if it couldn't be parsed.
     *
     * Be careful how to use this... the result Node should be explicitly
     *   added to the resolvers list to avoid inconsistency.
     */
    Node addNeighbor(String s)
    {
	java.util.StringTokenizer st = 
	    new java.util.StringTokenizer(s,":;\n\r\t ");
	
	if (st.countTokens()<5) return null;
	try { 
	    InetAddress ia = InetAddress.getByName(st.nextToken()); 
	    int udp = Integer.parseInt(st.nextToken());
	    int tcp = Integer.parseInt(st.nextToken());
	    String vspace = st.nextToken();
	    long ttl = Long.parseLong(st.nextToken());
	    Node node = new Node(ia, udp, tcp, vspace, ttl);
	    addNeighbor(vspace, node);
	    node = getNeighbor(node);
	    return node;
	}
	catch (Exception e) { return null; }
    }


    /**
     * Gets a neighbor given matching characteristics.
     * Slightly more efficient than the one without the vspace parameter.
     *
     * @param vspace The vspace that must be matched
     * @param ia     InetAddress of the matching resolver.
     * @param udp    UDP port to match (0 = any)
     * @param tcp    TCP port to match (0 = any)
     */
    synchronized Node getNeighbor
	(String vspace, InetAddress ia,	int udp, int tcp)
    {
	NodeSet nodeset = (NodeSet)rep.get(vspace);
	
	if (nodeset == null)
	    return null;

	Node n = nodeset.get(new Node(ia, udp, tcp, vspace));

	if (n != null)
	{
	    if (pruneExpiredNode(n, System.currentTimeMillis()))
		{
		    if (n.countVspaces() == 0) return null;
		}
	}

	return n;
    }


    /**
     * Gets a neighbor given matching characteristics.
     *
     * @param ia     InetAddress of the matching resolver.
     * @param udp    UDP port to match (0 = any)
     * @param tcp    TCP port to match (0 = any)
     */
    synchronized Node getNeighbor
	(InetAddress ia, int udp, int tcp)
    {
	Node n = allNeighbors.get(new Node(ia, udp, tcp));

	if (n != null)
	{
	    if (pruneExpiredNode(n, System.currentTimeMillis()))
		{
		    if (n.countVspaces() == 0) return null;
		}
	}

	return n;
    }


    /**
     * Gets a neighbor matching the node, slightly more efficient
     *  than the other format if a relevant vspace is known
     *
     * @param node Node to match against.
     */
    synchronized Node getNeighbor(String vspace, Node node)
    {
	NodeSet neigh = (NodeSet)rep.get(vspace);
	if (neigh == null) return null;

	Node n = neigh.get(node);

	if (n != null)
	{
	    if (pruneExpiredNode(n, System.currentTimeMillis()))
		{
		    if (n.countVspaces() == 0) return null;
		}
	}

	return n;
    }


    /**
     * Gets a neighbor matching the node
     *
     * @param node Node to match against.
     */
    synchronized Node getNeighbor(Node node)
    {
	Node n = allNeighbors.get(node);

	if (n != null)
	{
	    if (pruneExpiredNode(n, System.currentTimeMillis()))
		{
		    if (n.countVspaces() == 0) return null;
		}
	}

	return n;
    }


    /**
     * Removes a neighbor from a given vspace 
     *  (but doesn't entirely destroy it)
     *
     * @param vspace The vspace that must be matched
     * @param ia     InetAddress of the matching resolver.
     * @param udp    UDP port to match (0 = any)
     * @param tcp    TCP port to match (0 = any)
     */
    synchronized boolean removeNeighbor
	(String vspace, InetAddress ia,	int udp, int tcp)
    {
	NodeSet neigh = (NodeSet)rep.get(vspace);
	if (neigh == null)
	    return false;

	Node rmNode = new Node(ia, udp, tcp, vspace);

	if (!neigh.remove(rmNode))
	    return false;

	// if that was the last node in the vspace, it no longer exists.
	if (neigh.countNodes() == 0)
	    rep.remove(vspace);
	
	// Check allNeighbors
	Node existingNode = allNeighbors.get(rmNode);
	if (existingNode == null)	// should not be null BAD BAD BAD
	    return false;

	existingNode.removeVspace(vspace);
	if (existingNode.countVspaces() == 0)
	{
	    allNeighbors.remove(existingNode);
	}
       
	return true;
    }


    /**
     * Removes a neighbor from a given vspace 
     *  given that this is the same Node found in the neighbor list
     *  (be careful!! do _not_ do removeNeighbor(vspace, new Node(...)) )
     *   if in doubt use the other removeNeighbor
     *
     * @param vspace The vspace that must be matched
     * @param node   Exact node present in this data structure
     */
    synchronized boolean removeNeighbor
	(Node node, String vspace)
    {
	NodeSet neigh = (NodeSet)rep.get(vspace);
	if (neigh == null)
	    return false;

	if (!neigh.remove(node))
	    return false;

	// if that was the last node in the vspace, it no longer exists.
	if (neigh.countNodes() == 0)
	    rep.remove(vspace);
	
	node.removeVspace(vspace);
	if (node.countVspaces() == 0)
	{
	    allNeighbors.remove(node);
	}
       
	return true;
    }




    /** 
     * Sets up a vspace if it doesn't exist. 
     * 
     * @param vspace The vspace to add.
     * @return Did it add the NameSet?
     */
    boolean produceVspace(String vspace) 
    {
	return addVspace(vspace, new NodeSet());
    }


    /**
     * Returns a String representation of the VSNeighbors.
     *
     * @return String representing the structure.
     */
    public String toString()
    {
	StringBuffer strbuf = new StringBuffer();

	strbuf.append("VSNeighbors has " + rep.size() + " vspaces.\n");
	for (Enumeration enum = rep.keys(); enum.hasMoreElements(); )
	{
	    Object vspace = enum.nextElement();
	    Object nodeset = rep.get(vspace);

	    strbuf.append("VSPACE " + vspace.toString() + "\n");
	    strbuf.append(nodeset.toString() + "\n");
	}
	return strbuf.toString();
    }

}	   
