package ins.inr;

import java.net.InetAddress;
import java.util.Hashtable;
import java.util.Enumeration;

/**
 * VSResolvers is a set of all the resolvers in each of a set of vspaces.
 *
 * Whereas Neighbors consists only of neighboring/peering nodes, this
 * consists of any resolver, including "cached" resolvers from non-local
 * vspaces.
 *
 * The authoritative list of which NameTrees we host is on VSNameTrees,
 * and the list of our peers is in VSNeighbors.
 *
 * IMPORTANT: The Nodes between VSResolvers and VSNeighbors should be
 *   the same, in order that updates to the hosted vspaces, ttl, etc.
 *   will be reflected on both and be consistent.
 */


class VSResolvers
    extends VSNeighbors
{
    // VARIABLES
    NodeSet allResolvers;   // NodeSet for all VSpaces this
			   // INR belongs to
			   // AllNeighbors is a Vector of Node(s)

    // CONSTRUCTORS
    VSResolvers()
    {
	super();
	allResolvers = new NodeSet();
    }

    void init(Resolver r)
    {
	super.init(r);
    }

    /**
     * Adds a VSpace to the list.
     * if it already exists, returns false (otherwise true)
     *
     * @param vspace The vspace to add.
     * @param nodeset The NodeSet to associate with the vspace.
     * @return Whether it could be successfully added.
     */
    public synchronized boolean addVspace(String vspace, NodeSet n)
    {
	return super.addVspace(vspace, n);
    }


    /**
     * Removes the vspace from the list and returns whether or not 
     * it was successful
     *
     * @param vspace The vspace to remove
     * @return Whether the vspace could be removed
     */
    public synchronized boolean removeVspace(String vspace) 
    {
	return super.removeVspace(vspace);
    }


    /**
     * Returns the NodeSet object associated with vspace
     *
     * @param vspace The vspace to look up
     * @return NodeSet of those in vspace, with expired elements pruned
     */
    public synchronized NodeSet getResolvers(String vspace)
    {
	return super.getNeighbors(vspace);
    }

    /**
     * Adds a resolver node. If it already exists, it will ensure that
     *  the vspace is there, and that the ttl is updated.
     *
     * @param vspace Appropriate vspace
     * @param ia     InetAddress of the INR
     * @param udp    UDP port for the INR
     * @param tcp    TCP port for the INR
     * @param ttl    ttl for the INR in ms, Long.MAX_VALUE = never expire,
     *                   ~ 1 hour recommended.
     * @return Whether it was new.
     */
    boolean addResolver(String vspace, 
	InetAddress ia,	int udp, int tcp, long ttl)
    {
	Node newNode = new Node(ia, udp, tcp, vspace, ttl);
	return super.addNeighbor(vspace, newNode);
    }

    /**
     * Adds a resolver node. If it already exists, it will ensure that
     *  the vspace is there, and that the ttl is updated.
     *
     * @param vspace Appropriate vspace
     * @param newNode Node to be added.
     * @return Whether it was new.
     */
    synchronized boolean addResolver(String vspace, Node n)
    {
	return super.addNeighbor(vspace, n);
    }

    /** 
     * Accepts a string in the form of host:UDPport:TCPport:vspace:ttlms
     *  Used more for NodeConsole/high-level interaction
     *  Returns Node that was added.
     */
    Node addResolver(String s)
    {
	return super.addNeighbor(s);
    }

    /**
     * Gets a resolver given matching characteristics.
     * Slightly more efficient than the one without the vspace parameter.
     *
     * @param vspace The vspace that must be matched
     * @param ia     InetAddress of the matching resolver.
     * @param udp    UDP port to match (0 = any)
     * @param tcp    TCP port to match (0 = any)
     */
    synchronized Node getResolver(String vspace, 
	InetAddress ia,	int udp, int tcp)
    {
	return super.getNeighbor(vspace, ia, udp, tcp);
    }

    /**
     * Gets a resolver given matching characteristics.
     *
     * @param ia     InetAddress of the matching resolver.
     * @param udp    UDP port to match (0 = any)
     * @param tcp    TCP port to match (0 = any)
     */
    synchronized Node getResolver(InetAddress ia, int udp, int tcp)
    {
	return super.getNeighbor(ia, udp, tcp);
    }

    /**
     * Gets a resolver matching the node
     *
     * @param node Node to match against.
     */
    synchronized Node getResolver(Node node)
    {
	return super.getNeighbor(node);
    }

    /**
     * Gets a resolver matching the node
     * Slightly more efficient than the other form if vspace is known.
     *
     * @param node Node to match against.
     */
    synchronized Node getResolver(String vspace, Node node)
    {
	return super.getNeighbor(vspace, node);
    }


    /**
     * Removes a resolver from a given vspace 
     *  (but doesn't entirely destroy it)
     *
     * @param vspace The vspace that must be matched
     * @param ia     InetAddress of the matching resolver.
     * @param udp    UDP port to match (0 = any)
     * @param tcp    TCP port to match (0 = any)
     */
    synchronized boolean removeResolver(String vspace, 
	InetAddress ia,	int udp, int tcp)
    {
	return super.removeNeighbor(vspace, ia, udp, tcp);
    }

    /**
     * Removes a resolver from a given vspace 
     *  given that this is the same Node found in the neighbor list
     *  (be careful!! do _not_ do removeNeighbor(vspace, new Node(...)) )
     *   if in doubt use the other removeNeighbor
     *
     * @param vspace The vspace that must be matched
     * @param node   Exact node present in this data structure
     */
    synchronized boolean removeResolver
	(Node node, String vspace)
    {
	return super.removeNeighbor(node, vspace);
    }

    /**
     * Removes a resolver from all the relevant vspaces.
     *
     * @param node The resolver node to remove.
     */
    synchronized void totallyRemoveResolver(Node node)
    {
	super.totallyRemoveNeighbor(node);

    }

    /**
     * Returns a String representation of the VSResolvers.
     *
     * @return String representing the structure.
     */
    public String toString()
    {

	StringBuffer strbuf = new StringBuffer();

	strbuf.append("VSResolvers has " + rep.size() + " vspaces.\n");
	for (Enumeration enum = rep.keys(); enum.hasMoreElements(); )
	{
	    Object vspace = enum.nextElement();
	    Object nodeset = rep.get(vspace);

	    strbuf.append("VSPACE " + vspace.toString() + "\n");
	    strbuf.append(nodeset.toString() + "\n");
	}
	return strbuf.toString();
    }

}	   
