package ins.inr.twineKeyRouter;
import ins.inr.*;
import ins.inr.twineStrandMapper.*;
import ins.keyrouters.javakeyrouter.*;
import java.net.*;
import java.util.*;
import cryptix.util.core.Hex;
import cryptix.provider.md.*;

/**
 * TwineKeyRouter.java
 * <br>
 * Given a key and a message (value), this
 * method uses an underlying distributed 
 * hash-table process to determine what 
 * resolvers on the network should receive 
 * the message.<br>
 *
 * Created: Fri Jul 27 15:17:17 2001 <br>
 * Modified: $Id: TwineKeyRouter.java,v 1.8 2002/03/21 00:01:55 mbalazin Exp $
 * @author Magdalena Balazinska
 */
public class TwineKeyRouter  {

    public static byte PROT_VERSION = 2;
    // The port offset is used to match resolver with keyrouters
    // This makes it possible to run several pairs on a single host
    protected static int port_offset = -1;
    protected static int keyrouterport = -1;

    protected TwineStrandMapper strandMapper;
    protected TwineLogger log;
    protected RPCCommunicator rpcComm;

    protected KeyRouter kr;
    public InetAddress localhost;
    protected SimpleThreadPool pool;
    protected static int NB_THREADS = 2;

    /**
     * Class constructor
     */
    public TwineKeyRouter() {	
	log = new TwineLogger("TwineKR","TwineKeyRouter.log");
    }

    /**
     * Given a pair of ports, one for a resolver
     * process and one for a keyrouter process, this method
     * computes the offset that will later be applied to 
     * all resolver/keyrouter pairs
     */
    public static void setPortsOffset(int resolver, int keyrouter) {
	port_offset = resolver - keyrouter;
    }

    /**
     * Computes port used by the resolver mapping
     * to the keyrouter process running on port given
     * in parameter.
     */
    public static int resolverPortFromKeyrouterPort(int port) {
	return port + port_offset;
    }
 
    /**
     * Computes keyrouter port given corresponding
     * resolver port
     */
    public static int keyrouterPortFromResolverPort(int port) {
	return port - port_offset;
    }


    public void init(RPCCommunicator c, TwineStrandMapper sm) {
	rpcComm = c;
	strandMapper = sm;
	localhost = c.getLocalhost();
	keyrouterport = keyrouterPortFromResolverPort(rpcComm.getRPCPort());
	kr = new KeyRouter("localhost",keyrouterport);	
	pool = new SimpleThreadPool(log,NB_THREADS);
    }

    /**
     * Messages are transmitted to other resolvers asynchronously. 
     * This message will be processed by the thread pool as soon 
     * as a thread becomes available.
     */
    public void send(byte[] key, int howMany, TwineMessage tm ) {	
	KeyRoutingTask t = new KeyRoutingTask(key,howMany,tm,this);
	pool.addOneTask(t);
    }

    /**
     * Explicit remove messages are sent directly to the set
     * of resolvers that last received advertisements. This
     * set is known and does not need to be computed again
     */
    public void sendRemove(TwineMessage tm ) {	

	if ( tm instanceof TwineAdvertisement) {
	    TwineAdvertisement ta = (TwineAdvertisement)tm;
	    Packet pa = ta.getPacket();
	    Enumeration enum = ta.getSetAllResolvers();
	    if ( (enum != null) && (enum.hasMoreElements()) ) {
		while ( enum.hasMoreElements()) {
		    String hostKey = (String)enum.nextElement();
		    sendOneRemove(ta,hostKey,pa);
		}
	    }
	}
    }

    /**
     *
     */
    public void sendOneRemove(TwineAdvertisement ta, String hostKey, Packet pa) {

	String address = TwineAdvertisement.hostAddrFromHashKey(hostKey);
	int port = TwineAdvertisement.portFromHashKey(hostKey);
	try {
	    InetAddress addr = InetAddress.getByName(address);
	    byte[] byteKey = ta.getByteKeyForHost(addr,port);
	    if ( byteKey == null)
		byteKey = new byte[]{ 0 };
	    log.printStatus("Sending remove to " + address + ":" + port,
			    TwineLogger.IMPORTANT_MSG);
	    sendToResolver(addr,port,pa,byteKey);
	} catch ( UnknownHostException e ) {
	    log.printStatus("Error looking up: " + address + ":" + port,
			    TwineLogger.KEY_MSG);
	}
    }

    /**
     * The number of resolvers that actually received a message
     * is known only once the set of resolvers has been computed
     * for all keys making up a message
     */
    public void doneSending(int howMany, int seqNum) {	
	strandMapper.doneSending(howMany,seqNum);		      
    }

    /**
     *
     */
    public void reSend(InetAddress addr,int port, Packet p,byte[] key) {
	sendToResolver(addr,port,p,key);
    }


    /**
     * Transmits a message to another resolver. This method
     * does not block since <code>rpcComm</code> puts the
     * message into a queue and returns.
     */
    public boolean sendToResolver(InetAddress addr, int port,
				  Packet packetBase, byte[] key) {
	
	// The key is sent as an option
	Packet packet = (Packet)packetBase.clone();
	packet.option = key;
	packet.version = PROT_VERSION;
	boolean sent = false;
	try {
	    // Do not send to ourselves!
	    int myport = rpcComm.getRPCPort();
	    if ( addr.equals(localhost) && ( port == myport) ) {
		log.printStatus("localhost is destination: not sending",
				TwineLogger.IMPORTANT_MSG);
	    } else {
		Message msg = new Message(packet,addr,port);
		log.printStatus("sending one packet...", TwineLogger.IMPORTANT_MSG);
		rpcComm.sendMessage(msg);
		sent = true;
	    }	    
	} catch ( Exception e ) { e.printStackTrace(System.out); }
	return sent;
    }  

}


//-----------------------------------------------------
/** 
 * This class implements one task corresponding to sending 
 * an advertisement or a query to a set of resolvers.
 * The taks includes the computation of the resolvers set
 * given a single key.
 * @author: Magdalena Balazinska
 */
class KeyRoutingTask extends Task {

    String stringKey; // The key to use to compute the set of resolvers
    byte[] byteKey;   // Same key in different format
    int howMany;      // Desired replication level
    TwineMessage tm;  // The message to transmit
    TwineKeyRouter tkr;

    //-----------------------------------------------------
    /** 
     */
    public KeyRoutingTask(byte[] key, int nb, TwineMessage tmsg, 
			  TwineKeyRouter twineKeyRouter) {

	stringKey = cryptix.util.core.Hex.toString(key); 
	byteKey = key;
	howMany = nb;
	tm = tmsg;
	tkr = twineKeyRouter;
    }

    //-----------------------------------------------------
    /** 
     * For every resolver from the set of resolvers that
     * should receive a message, this method first determines
     * whether the message was already sent or whether it
     * should be sent. It then sends the message or not.
     * @param byteKey The key that was used to determine that
     * this resolver should receive the message
     * @param hostname Host where the resolver process is running
     * @param resolverport Port on which the destinatin resolver 
     * is listening
     * @return <code>true</code> if the message will be sent
     * to the resolver and <code>false</code> if the message
     * was already sent using another key.
     */
    protected boolean tryToSend(byte[] byteKey, String hostname, 
				int resolverport) throws Exception {
	
	tkr.log.printStatus(" -> Trying to send to: " + hostname 
			    + ":"+resolverport + "\n",TwineLogger.IMPORTANT_MSG);
	InetAddress addr = InetAddress.getByName(hostname);
	if ( tm.shouldSend(byteKey,addr,resolverport) ) {
	    if (tkr.sendToResolver(addr,resolverport,tm.getPacket(),byteKey)) {
		return true;
	    }
	}
	return false;
    }

    //-----------------------------------------------------
    /** 
     * Main method corresponding to executing a transmission 
     * task. Given a key, this method computes the set of 
     * resolvers that should receive the message. It then 
     * transmits the message to these resolvers.
     */
    public void run() {

	tkr.log.printStatus("Searching for " + howMany + " nodes for key " 
			    + stringKey +"\n", TwineLogger.IMPORTANT_MSG);

	int nbReqSent = 0;
	keyrouter_nodeinfo_res result = tkr.kr.getNode(stringKey);
	if ( result.status == keyrouter_stat.KEYROUTER_ERR) {
	    tkr.log.printStatus("*** Error finding node *** ",TwineLogger.IMPORTANT_MSG);
	    return;
	}

	try {
	
	    // Extracting detailed results
	    String hostname = result.resok.addr.hostname;
	    int keyrouterport = Integer.decode(result.resok.addr.port).intValue();
	    int resolverport = TwineKeyRouter.resolverPortFromKeyrouterPort(keyrouterport);
	    String key = result.resok.key;
	    tkr.log.printStatus("\nSuccessor of: " + stringKey + " is " + hostname + ":" 
	    + keyrouterport,TwineLogger.IMPORTANT_MSG);

	    // Sending to resolver in charge of that key
	    if (tryToSend(byteKey,hostname,resolverport))
		nbReqSent++;

	    // We may need to send to more than one resolver
	    howMany--;
	    if ( howMany > 0 ) {
		keyrouter_knodeinfo_res r =  tkr.kr.getKNext(hostname,keyrouterport,
							     key,howMany);
		if ( r.status == keyrouter_stat.KEYROUTER_ERR) {
		    tkr.log.printStatus("*** Error finding destination node *** ",
					TwineLogger.IMPORTANT_MSG);
		    return;
		} else {
		    for ( int j = 0; j < r.resok.howMany; j++ ) {
			hostname = r.resok.list[j].addr.hostname;
			keyrouterport = Integer.decode(r.resok.list[j].addr.port).intValue();
			resolverport = TwineKeyRouter.resolverPortFromKeyrouterPort(keyrouterport);
			if (tryToSend(byteKey,hostname,resolverport))
			    nbReqSent++;
		    }
		}
	    }
	    
	} catch (Exception e) { e.printStackTrace(System.out);}
	tkr.log.printStatus("Sent " + nbReqSent + " messages for seqnum " +tm.getSeqNo(),
			    TwineLogger.IMPORTANT_MSG);
	tkr.doneSending(nbReqSent,tm.getSeqNo());

	if ( tm instanceof TwineAdvertisement) 
	    processEndOfAdvertisement();
	

    }

    //-----------------------------------------------------
    /** 
     * Once a resource has been re-advertised, this method 
     * checks if some resolvers were dropped from the list
     * of resolvers responsible for that resource. It then
     * sends explicit remove messages.
     */
    public void processEndOfAdvertisement() {
	
	TwineAdvertisement ta = (TwineAdvertisement)tm;
	boolean allDone = ta.incNbKeysAndTest();
	
	if ( allDone ) {
	    
	    Enumeration enum = ta.getSetGoneResolvers();
	    if ( (enum != null) && (enum.hasMoreElements()) ) {
		
		// Clone the advertisement and transform into a delete
		Packet pa = new Packet(ta.getPacket());
		transformIntoRemoveMessage(pa);
		
		// Send remove messages to all resolvers in the list
		for ( ; enum.hasMoreElements();) {
		    String hostKey = (String)enum.nextElement();
		    tkr.sendOneRemove(ta,hostKey,pa);
		}
	    }
	}
    }

    //-----------------------------------------------------
    /**
     * Transforms a resource advertisement into a remove
     * message.
     */
    public void transformIntoRemoveMessage(Packet packet) {

	NameUpdate nu = new NameUpdate(packet.data);
	nu.action = NameUpdate.REMOVE_NAME;
	packet.data = nu.toBytes();
	
    }

}



