#include "chord_interface.h"

extern "C" void usage();

//-----------------------------------------------------------
/** 
 * Methods communicating with the Chord node
 * @author Magdalena Balazinska
 */
//-----------------------------------------------------------

//-----------------------------------------------------------
/* 
 * Constructor
 */
chord_interface::chord_interface(int argc, char** argv) {

  nb_vnodes = 0;
  sfsconst_init ();
  random_init ();
  configureNode(argc,argv);

}

//-----------------------------------------------------------
/* 
 * Send information about a chord node back to RPC client
 * 
 */
void chord_interface::answerOK(svccb* sbp, chordID succ) {
    
  warnx << "\nSuccessor is: " << succ << "\n";
  keyrouter_nodeinfo_res res(KEYROUTER_OK);
  prepareNodeInfoRes(res.resok,succ);
  sbp->reply(&res);

}

//-----------------------------------------------------------
/**
 * Send information about a set of chord nodes back to RPC client
 */
void chord_interface::answerOK(svccb* sbp, chordID* successors, int howMany) {

  keyrouter_knodeinfo_res kres(KEYROUTER_OK);
  kres.resok->howMany = howMany;
  
  kres.resok->list.setsize(howMany);

  for ( int j = 0; j < howMany; j++ ) {
    chordID succ = successors[j];
    prepareNodeInfoRes(&(kres.resok->list[j]),succ);  
  }
  sbp->reply(&kres);
	     
}

//-----------------------------------------------------------
/**
 */
void
chord_interface::prepareNodeInfoRes(keyrouter_nodeinfo_res_ok* resok, chordID succ) {

  // Get address from ChordID
  net_address address = chordnode->locations->getaddress(succ);

  // Fill out the result structure
  resok->addr.hostname = address.hostname;
  char* temp = New char[PORT_LENGTH];
  sprintf(temp,"%u",address.port+PORT_OFFSET);
  resok->addr.port = temp;
  resok->key =  (char*)succ.cstr();

  warnx << "Preparing info for: " << address.hostname << ":" << temp << "\n";
}


//-----------------------------------------------------------
/**
 * Returns an error message to RPC client
 */
void chord_interface::answerERR(svccb* sbp) {

 keyrouter_nodeinfo_res res(KEYROUTER_ERR);
 sbp->reply(&res);

}

//-----------------------------------------------------------
/* 
 * Receives the node associated with a key
 * Second argument is actual argument, the rest is mandatory
 */
void chord_interface::getNode_cb(svccb* sbp, chordID succ, route path, chordstat err) {

  if (err) {
    warn << "error finding sucessor\n";
    answerERR(sbp);
  }
  else {
    answerOK(sbp,succ);
  }
}


//-----------------------------------------------------------
/* 
 * Gets the node associated with a key
 */
void chord_interface::getNode(const char* key, svccb *sbp) {

  chordID n;
  char* properLengthKey = convertLength(key);
  initID(properLengthKey,&n);
  delete [] properLengthKey;
  warnx << "\nSearching for successor of string key: " 
	<< key << "\n...or equivalent chordID: " << n << "\n";
  chordnode->find_successor (n, wrap(this, &chord_interface::getNode_cb, sbp));

}

//-----------------------------------------------------------
/* 
 * Picking the right virtual node for the given ID
 */
vnode* 
chord_interface::pickVnode(const char* key) {

  chordID n;
  initID(key,&n);

  for ( int i = 0; i < nb_vnodes; i++ ) {
    vnode* p_vnode = vnodes[i];
    if ( p_vnode->myID == n ) {
      warnx << "\nThat's the node... now checking it's successor(s) \n"; 
      return p_vnode; 
    }
  }
  return NULL;

}

//-----------------------------------------------------------
/* 
 * Gets direct successor of this node
 */
void chord_interface::getNext(const char* key, svccb *sbp) {

  vnode* p_vnode = pickVnode(key);
  if ( p_vnode) {
    chordID succ = p_vnode->my_succ();
    answerOK(sbp,succ);
  } else {
    answerERR(sbp);
  }
}

//-----------------------------------------------------------
/* 
 * Gets K direct successors of this node
 */
void chord_interface::getKNext(const char* key, int howMany, svccb *sbp) {

  if ( howMany < 1 ) {
    answerERR(sbp);
  } else {

    vnode* p_vnode = pickVnode(key);
    if ( p_vnode) {
      chordID successors[howMany];
      for ( int j = 0; j < howMany; j++ )
	successors[j] = p_vnode->nth_successorID(j+1);
      answerOK(sbp,successors,howMany);
    }
    else {
      answerERR(sbp);
    }

  }

}

//-----------------------------------------------------------
/* 
 * If necessary pad key with zeros to get correct length
 */
char* chord_interface::convertLength(const char* key) {

  char* fullKey = new char[KEY_LENGTH+1];
  int initialKeyLength = strlen(key);

  for ( int i = 0; i < KEY_LENGTH; i++) {
    if ( i < initialKeyLength ) {
      fullKey[i] = key[i];
    }
    else {
      fullKey[i] = '0';
    }
  }
  fullKey[KEY_LENGTH] = '\0';
  return fullKey;

}


//-----------------------------------------------------------
/* 
 * Prints the info of this chord node
 */
void chord_interface::printNodeInfo() {

  chordnode->print();

}

//-----------------------------------------------------------
/* 
 * The following comes from sfsnet/lsd/lsd.C
 * Currently, I'm not doing anything with
 * the vnodes and the stat???
 */
void chord_interface::newvnode_cb (int n, vnode *my, chordstat stat) {

  if (stat != CHORD_OK) {
    warnx << "newvnode_cb: status " << stat << "\n";
    fatal ("unable to join\n");
  }

  vnodes[nb_vnodes] = my;
  nb_vnodes++;

  warnx << "Added vnode of ID " << my->myID << " at index " << nb_vnodes << "\n";

  if (n > 0 && n < MAX_VNODES ) 
    chord_interface::chordnode->newvnode (wrap (this, &chord_interface::newvnode_cb, n-1));

}

//-----------------------------------------------------------
/* The following comes from sfsnet/lsd/lsd.C
 * Input should be of the form 0x12345
 */
void chord_interface::initID (char* key, chordID *ID) {

  str keystr(key);
  initID(keystr,ID);

}


//-----------------------------------------------------------
/* The following comes from sfsnet/lsd/lsd.C
 * Input should be of the form 0x12345
 */
void chord_interface::initID (str s, chordID *ID)
{
  chordID n (s);
  chordID b (1);
  b = b << NBIT;
  b = b - 1;
  *ID = n & b;
  warnx << "\n String is:" << s << " and chordID is " << *ID;
}


//-----------------------------------------------------------
/* The following comes from sfsnet/lsd/lsd.C
 * It is not an exact copy of parseconfigfile, though
 * Removed dhash options and added the number of virtual nodes
 * as a config file option.
 */
void chord_interface::configureNode(int argc, char** argv) {

  int vnode = 1;
  int ch;
  int ss_mode = 0;

  int myport = 0;
  // max_loccache is very important, without it the Chord circle
  // doesn't get formed
  int max_loccache = 1000;
  str wellknownhost;
  int wellknownport = 0;

  while ((ch = getopt (argc, argv, "v:j:p:n:t:")) != -1)
    switch (ch) {
    case 't': // Already taken care during binding
      break;
    case 'p':
      myport = atoi (optarg) - PORT_OFFSET;
      break;
    case 'j': 
      {
	char *bs_port = strchr(optarg, ':');
	*bs_port = 0;
	bs_port++;
	if (inet_addr (optarg) == INADDR_NONE) {
	  struct hostent *h = gethostbyname (optarg);
	  if (!h) {
	    warn << "Invalid address or hostname: " << optarg << "\n";
	    usage ();
	  }
	  struct in_addr *ptr = (struct in_addr *)h->h_addr;
	  wellknownhost = inet_ntoa (*ptr);
	} else
	  wellknownhost = optarg;

	if (bs_port)
	  wellknownport = atoi (bs_port) - PORT_OFFSET;
	
	break;
      }
    case 'v':
      vnode = atoi (optarg);
      break;
    default:
      usage ();
      break;
    }

  if (wellknownport == 0) usage ();

  warn << "Connecting to " << wellknownhost << ":" << wellknownport << "\n";
  chordnode = New refcounted<chord> (wellknownhost, wellknownport, 
				     myport,max_loccache,ss_mode);


  chordnode->newvnode (wrap (this, &chord_interface::newvnode_cb, vnode-1));

}


