/*
 *  This source file is copyright (C) 1999 by joeshmoe.com
 *  It may be freely used for non-commercial / development purposes.
 *  Commericial use requires the payment of a nominal license fee.
 *  Contact license@joeshmoe.com to purchase a license for commericial use.
 *
 *  It is requested that all users who modify this source code send their
 *  changes to joeshmoe.com
 * 
 *  Bugs reports / modifications can sent to bugs@joeshmoe.com.
 */

package joeshmoe.mpeg;

/**
 * A class which implements this interface contains information about an
 * MPEG audio file.
 * @see MPEGProber
 **/

public interface MPEGInfo {
    /**
     * Returns the version of the MPEG audio file.
     * <BLOCKQUOTE><B>1</B> - MPEG1<br><B>2</B> - MPEG2<br></BLOCKQUOTE>
     * Currently, only MPEG1 and MPEG2 are supported.  MPEG2.5 files will
     * simple be classified as MPEG2 and certain pieces of information about
     * the files may be incorrect.
     * @see ID3Tagger
     **/
    public int getVersion ();

    /**
     * Returns the MPEG audio layer number.  Valid values are 1, 2, and 3.
     **/
    public int getLayer ();

    /**
     * Returns true if the MPEG audio file has a variable bit rate (VBR), false
     * if the bit rate is constant (CBR).
     **/
    public boolean isVBR ();

    /**
     * Returns (in KB/s) the bitrate of the MPEG audio file.  For CBR
     * files, this is the actual bitrate of the file.  For VBR, this is the
     * average bitrate across the entire file.
     **/
    public int getBitrate ();

    /**
     * Returns the sampling rate of the MPEG audio file.  The value is in
     * samples per second.
     **/
    public int getSamplingRate ();

    /**
     * Returns the channel mode of the file.  Valid values are:
     * <BLOCKQUOTE>
     * <B>1</B> - True stereo<br>
     * <B>2</B> - Joint stereo<br>
     * <B>3</B> - Dual channel (2 mono channels)<br>
     * <B>4</B> - Single channel mono<br></blockquote>
     **/
    public int getChannelMode ();

    /**
     * Returns true if the file uses intensity stereo.  This only applies
     * if the channel mode is joint stereo.  All other channel modes will
     * return false.  Layer 1 and layer 2 MPEG audio files will always return
     * true.  For these files, you can call getIntensityBands to determine
     * the frequency bands where intensity stereo is applied.  For layer 3
     * files, this method may return false.  
     **/
    public boolean isIntensityStereo ();
		
    /**
     * Returns true if the file uses m/s stereo.  This may return true
     * if the file is layer 3 and channel mode is joint stereo.  
     * All other channel modes and layers will always return false.  
     **/
    public boolean isMSStereo ();

    /**
     * Returns a numeric code indicating the frequency bands for which 
     * intensity stereo was applied.  This will return -1 for all files
     * that are not encoded in joint stereo or for all layer 3 joint stereo
     * files.  The valid return values are:<blockquote>
     * <B>-1</B> - not applicable<br>
     * <B>1</B> &nbsp;- bands 4 to 31<br>
     * <B>2</B> &nbsp;- bands 8 to 31<br>
     * <B>3</B> &nbsp;- bands 12 to 31<br>
     * <B>4</B> &nbsp;- bands 16 to 31<br>
     * </blockquote>
     **/
    public int getIntensityBands ();

    /**
     * Returns the length of the audio data in milliseconds.
     **/
    public long getLength ();

    /**
     * Returns the number of bytes of audio data in this file.  This number is
     * not always equal to the actual size of the file because of extra
     * header information and/or an ID3 tag.
     **/
    public long getFileSize ();

    /**
     * Returns true if the file contains CRC protection data, false otherwise.
     **/
    public boolean isCRCProtected ();
		
    /**
     * Returns true if the copyright bit is set, false otherwise.
     **/
    public boolean isCopyright ();

    /**
     * Returns true if the original bit is set, false otherwise.
     **/
    public boolean isOriginal ();

    /**
     * Returns the ID3 tag contained inside the file.  If there is no ID3
     * tag, null is returned.
     **/
    public ID3Tag getID3 ();
}
